# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import EntityMib
import Fru
import Tac
import Tracing
from FruPlugin.Health import registerHealthSource

__defaultTraceHandle__ = Tracing.Handle( "Fru.PowerSupplyExternal" )
t0 = Tracing.trace0

class PowerSupplyExternalDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::PowerSupply::PowerSupplyExternal"
   managedApiRe = "$"

   def __init__( self, psInv, parentEntmib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, psInv, parentEntmib, parentDriver, driverCtx )

      # Create the HW config
      config = driverCtx.sysdbRoot.entity[ "hardware/powerSupply/external/config" ]
      config.psName = psInv.name
      for i in [ 'A', 'B', 'C', 'D' ]:
         config.idGpos[ i ] = psInv.idGpos[ i ].systemName
         config.idSetupGpos[ i ] = psInv.idSetupGpos[ i ].systemName
      invScd = driverCtx.sysdbRoot.entity[
            "hardware/inventory/fixedSystem" ].component[ "scd" ]
      config.powerGoodInterrupt = invScd.psuPowerGoodInterrupt.fruIntCtrl
      config.powerGoodGpo = invScd.psuPowerGoodGpo.systemName
      config.powerGoodChangedClearGpo = invScd.psuPowerGoodChangedClearGpo.systemName

      registerHealthSource( psInv, psInv.name )

      # Default power supply to slot 1 for these systems with an external adapter
      psEntmib = parentEntmib.powerSupplySlot[ 1 ]
      if not psEntmib.powerSupply:
         physicalIndex = EntityMib.IndexAllocator.physicalIndex(
               psEntmib, "PowerSupply", psEntmib.relPos )
         Fru.Dep( psEntmib, psInv ).powerSupply = (
               physicalIndex, psEntmib.relPos, "PowerSupply" )
         psEntmib.powerSupply.description = config.psName
         psEntmib.powerSupply.swappability = "notSwappable"
         psEntmib.powerSupply.mfgName = "Arista Networks"

      EntityMib.populateMib( psEntmib.powerSupply, psInv )

      # Create the launcherConfig that launches PowerSupplyExternal agent
      lcDir = driverCtx.sysdbRoot.entity[
            "hardware/powerSupply/external/launcherConfig" ]
      lcDir.newEntity( "Tac::Dir", config.psName )
      # pylint: disable-next=consider-using-f-string
      t0( "Instantiated launcherConfig for %s" % config.psName )
      archerStatusDir = driverCtx.sysdbRoot.entity[
            "environment/archer/power/status/powerSupply" ]
      # Instantiate the reactors to handle updating the EntityMib
      self.dirReactor = PowerSupplyStatusDirReactor( archerStatusDir, psEntmib )

class PowerSupplyStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = "Tac::Dir"
   controllerStatusType = Tac.Type( "Environment::Power::SupplyStatus" )

   def __init__( self, archerStatusDir, psEntmib ):
      Tac.Notifiee.__init__( self, archerStatusDir )
      self.archerStatusDir = archerStatusDir
      self.psEntmib = psEntmib
      self.supplyStatusReactor = None

   @Tac.handler( 'entityPtr' )
   def handlePowerSupply( self, key ):
      status = self.archerStatusDir.entityPtr.get( key )
      statusEntry = self.archerStatusDir.entryState.get( key )

      if status and statusEntry and isinstance( status, self.controllerStatusType ):
         self.supplyStatusReactor = PowerSupplyStatusReactor(
               status, self.psEntmib )

class PowerSupplyStatusReactor( Tac.Notifiee ):
   """Reactor that updates the power supply modelname in the EntityMib after
   PowerSupplyExternal agent determines the capacity.
   """
   notifierTypeName = "Environment::Power::SupplyStatus"

   def __init__( self, status, psEntmib ):
      Tac.Notifiee.__init__( self, status )
      self.psEntmib = psEntmib
      self.handleCapacity()

   @Tac.handler( 'capacity' )
   def handleCapacity( self ):
      if self.notifier_.capacity == float( "-Inf" ):
         return
      # Using dominant field to identify an external PSU
      # Currently, all external PSU are dominant
      if not self.notifier_.dominant:
         return
      capacity = int( self.notifier_.capacity )
      # These rudimentary external power supplies are only differentiated by their
      # capacity. The modelname will always follow the format below.
      self.psEntmib.powerSupply.modelName = f"PWR-{capacity}-ADP"

def Plugin( ctx ):
   ctx.registerDriver( PowerSupplyExternalDriver )
   mg = ctx.entityManager.mountGroup()
   mg.mount( "hardware/powerSupply/external/config",
             "Hardware::PowerSupplyExternal::Config", "w" )
   mg.mount( "hardware/powerSupply/external/launcherConfig", "Tac::Dir", "wi" )
   mg.close( None )
