# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Fru, EntityMib, Tac
import FruPlugin.Smbus as Smbus # pylint: disable=consider-using-from-import
from EntityMib import IndexAllocator
from FruPlugin.Health import registerHealthSource

import Tracing
t0 = Tracing.trace0

class FixedDirDriver( Fru.FruDriver ):
   """ Manages the PowerSupplyFixedDir object, create corresponding fixed power 
   supply objects, to enable Fixed Power Agent startup """

   managedTypeName = "Inventory::PowerSupply::FixedDir"
   managedApiRe = "$"

   def __init__( self, fixedDir, parentEntmib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, fixedDir, parentEntmib, parentDriver, driverCtx )

      # -------------------------------------------------------------------
      # Create the Hardware::PowerSupplyFixed::PowerSupplyFixedConfig config
      config = driverCtx.sysdbRoot.entity[ 'hardware/powerSupply/fixed/config' ]

      for ( slotId, fixedInv ) in fixedDir.powerSupplyFixed.items():
         # pylint: disable-next=consider-using-f-string
         psName = "PowerSupply%d" % fixedInv.slotId
         # pylint: disable-next=consider-using-f-string
         t0( "Handling config for %s" % psName )
         registerHealthSource( fixedInv, psName )
         powerOkGpo = fixedInv.powerOk.systemName if fixedInv.powerOk else ''

         # create and update a config fixed inventory
         fixedConfig = config.newFixedConfig( slotId, psName, powerOkGpo,
                                              fixedInv.capacity )
         fixedConfig.dominant = fixedInv.dominant

         # --------------------------------------------------------
         # Create the power supply entity mib object
         powerSupplySlotEntmib = parentEntmib.powerSupplySlot[ slotId ]
         if not powerSupplySlotEntmib.powerSupply:
            physicalIndex = IndexAllocator.physicalIndex \
                            ( powerSupplySlotEntmib,
                              'PowerSupply',
                              powerSupplySlotEntmib.relPos )
            Fru.Dep( powerSupplySlotEntmib, fixedInv ).powerSupply = (
               physicalIndex,
               powerSupplySlotEntmib.relPos,
               "PowerSupply" )
            powerSupplySlotEntmib.powerSupply.description = psName
            powerSupplySlotEntmib.powerSupply.swappability = "notSwappable"
            powerSupplySlotEntmib.powerSupply.mfgName = fixedInv.mfgName
            powerSupplySlotEntmib.powerSupply.modelName = fixedInv.modelName

         powerSupplyEntmib = powerSupplySlotEntmib.powerSupply
         powerSupplyEntmib.label = powerSupplySlotEntmib.label
         EntityMib.populateMib( powerSupplyEntmib, fixedInv )
         powerSupplyEntmib.initStatus = "ok"

         # --------------------------------------------------------
         # Create PowerMonitor config
         if fixedInv.powerMonitor:
            fixedInv.powerMonitor.managingCellId = Cell.cellId()

            topology = driverCtx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
            # pylint: disable-next=consider-using-f-string
            name = "%sAhamDesc" % fixedInv.powerMonitor.name
            ahamArgs = ( name, ) + \
               Smbus.ahamDesc( topology, fixedInv.powerMonitor.deviceBase )

            fixedConfig.powerMonitorConfig = ( fixedInv.powerMonitor.name, )
            fixedConfig.powerMonitorConfig.model = fixedInv.powerMonitor.model
            fixedConfig.powerMonitorConfig.psuModel = fixedInv.modelName
            fixedConfig.powerMonitorConfig.label = powerSupplyEntmib.label
            fixedConfig.powerMonitorConfig.powerMonitorAhamDesc = ahamArgs

def Plugin( ctx ):
   ctx.registerDriver( FixedDirDriver )
   
   mg = ctx.entityManager.mountGroup()

   mg.mount( 'hardware/powerSupply/fixed/config',
             'Hardware::PowerSupplyFixed::Config', 'w' )

   mg.close( None )

