# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import Fru
import Tac
import os

traceHandle = Tracing.Handle( "Fru.PtpTimeSync" )
t0 = traceHandle.trace0

def broadSyncSupported( hamImpl, base ):
   revReg = Tac.Value( 'PtpTimeSync::BroadSyncCoreRevision' )
   addr = base + revReg.address
   version = 0

   # In a workspace dut test fake the broadsync supported.
   # This should not be used in ptp package testing since the simulation
   # of this functionality is supported there
   if any( name in os.environ
           for name in [ "PMBUS_SIMULATION",
                         "WS_CHASSIS_SIMULATION",
                         "SIMULATION_VMID" ] ):
      t0( "In simulation, faking broadsync supported" )
      return True

   try:
      version = hamImpl.data[ addr ]
   except IndexError:
      # If we can't read the version, assume we don't have BroadSync support
      version = 0
   return version >= revReg.minSupportedRev

class PtpTimeSyncMasterHamReactor( Tac.Notifiee ):
   notifierTypeName = "Fru::HamImpl"

   def __init__( self, scdHamImpl, master, ctx ):
      Tac.Notifiee.__init__( self, scdHamImpl )
      self.master = master
      self.ctx = ctx

      fruBase = Fru.fruBase( master )
      self.cellId = fruBase.managingCellId
      self.sliceId = fruBase.sliceId
      t0( "creating reactor for scd ham" )

      self.initializePtpTimeSyncMaster()


   @Tac.handler( 'hardwarePresent' )
   def initializePtpTimeSyncMaster( self ):
      if not self.notifier_.hardwarePresent:
         t0( self.sliceId, "hardware not present, not adding master." )
         return

      if not self.master.block.bypassRevCheck and \
         not broadSyncSupported( self.notifier_, self.master.block.offset ):
         t0( self.sliceId, "BroadSync not supported, not adding master." )
         return

      tSyncConfig = self.ctx.sysdbRoot[ 'ptpTimeSync' ][ 'config' ]
      # If a PTP TimeSync capable Scd shows up:
      # 1. Mark Ptp TimeSync as supported.
      # 2. Create the ScdConfig object.
      tSyncConfig.supported = True
      tSyncConfig.pidControllerSupported = self.master.pidControllerSupported
      tSyncConfig.internalSyncMode = self.master.internalSyncMode
      timeSyncVersion = "unknown"
      if self.master.timeSyncVersion == 1:
         timeSyncVersion = "version1"
      interruptController = None
      if self.master.interruptController:
         interruptController = self.master.interruptController.fruIntCtrl
      genId = Fru.fruBase( self.master ).generationId
      t0( self.sliceId, "hardware present, adding master." )
      config = Fru.Dep( tSyncConfig.scd, self.master ).newMember(
         self.sliceId,
         timeSyncVersion,
         self.master.block.offset,
         self.master.delayOffset,
         self.master.outputFrequency,
         self.master.alwaysOnClock,
         self.master.requiresInternalPllReset,
         self.cellId,
         self.master.block.ham,
         interruptController, genId,
         self.master.internalSyncMode,
         self.master.gpioSampleStatusOffset,
         self.master.standbyInputClkDisabled,
         self.master.counterInputCounterFreqInMhz )

      if self.master.startupOp is not None:
         for op in self.master.startupOp.values():
            config.startupOp.enq( op )

class PtpTimeSyncMasterDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::PtpTimeSync::Master"
   managedApiRe = ".*"

   def __init__( self, master, parentMibEntity, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, master, parentMibEntity, parentDriver, ctx )
      self.masterReactor = PtpTimeSyncMasterHamReactor( master.block.ham.hamImpl,
                                                        master, ctx )

class PtpTimeSyncSupportedReactor( Tac.Notifiee ):
   notifierTypeName = "PtpTimeSync::Config"

   def __init__( self, config, slave ):
      Tac.Notifiee.__init__( self, config )
      self.fruBase = Fru.fruBase( slave )
      self.sliceId = self.fruBase.sliceId
      self.slave = slave
      self.config = config

      self.initializePtpTimeSyncSlave()

   @Tac.handler( 'supported' )
   def initializePtpTimeSyncSlave( self ):
      if not self.config.supported:
         t0( self.sliceId, "PTP not supported, switch:", self.slave.name )
         return

      if not self.notifier_.supported:
         t0( self.sliceId, "Broadsync not supported, switch:", self.slave.name )
         return

      t0( self.sliceId, "PTP supported, switch:", self.slave.name )
      slotDelayIfAny = 0
      try:
         slotDesc = getattr( self.fruBase, 'slotDesc' )
         slotDelayIfAny = slotDesc.ptpTimeSyncDelayOffset
      except AttributeError:
         # We're on a fixed system
         pass

      tSyncDelay = slotDelayIfAny + self.slave.timeSyncDelay

      # assert invertFpgaSupeSel is True only if topology is fpgaPlusPll
      if self.slave.invertFpgaSupeSel:
         assert self.slave.topology == 'fpgaPlusPll'

      Fru.Dep( self.notifier_.switchAsic, self.slave ).newMember(
         self.slave.name, Fru.powerGenerationId( self.slave ),
         self.slave.topology, tSyncDelay,
         self.slave.inputFreqMultiplier, self.slave.pllIsStandalone,
         self.slave.invertFpgaSupeSel, self.slave.internalSyncMode,
         self.slave.phaseOffsetErrorThreshold, self.slave.reqsAlwaysOnTimeSync,
         self.slave.sourceClockTimestamp,
         self.slave.supeSelectOffset,
         self.slave.gpioStatusRegisterAddress,
         self.slave.counterInputCounterFreqInMhz, self.slave.supeSelectInfo,
         self.slave.slotId )

   @Tac.handler( 'switchAsic' )
   def handleSwitchAsic( self, name ):
      insert = name in self.config.switchAsic
      t0( "inserting" if insert else "removing", "switch:", name )
      alwaysOn = False
      for switchAsic in self.config.switchAsic.values():
         alwaysOn = alwaysOn or switchAsic.reqsAlwaysOnTimeSync
      self.config.alwaysOn = alwaysOn

class PtpTimeSyncSlaveDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::PtpTimeSync::Slave"
   managedApiRe = ".*"

   def __init__( self, slave, parentMibEntity, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, slave, parentMibEntity, parentDriver, ctx )

      tSyncConfig = ctx.sysdbRoot[ 'ptpTimeSync' ][ 'config' ]
      self.slaveReactor = PtpTimeSyncSupportedReactor( tSyncConfig, slave )

def Plugin( context ):
   context.registerDriver( PtpTimeSyncMasterDriver )
   context.registerDriver( PtpTimeSyncSlaveDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'ptpTimeSync/config', 'PtpTimeSync::Config', 'wi' )

   mg.close( None )
