#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Fru, Tracing, Cell
from TypeFuture import TacLazyType

from HwL1TopologyComponentLib import Components

__defaultTraceHandle__ = Tracing.Handle( "PhyTopoFru" )

t0 = Tracing.trace0
t1 = Tracing.trace1

SerdesDescriptor = TacLazyType( "PhyEee::SerdesDescriptor" )
class StaticSerdesMapDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::L1TopologyStaticSerdesComponentDir"
   managedApiRe = "$"

   provides = [ "StaticSerdesMaps" ]
   requires = []

   def __init__( self, invStaticSerdesDir, fruEntMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, invStaticSerdesDir, fruEntMib,
                              parentDriver, driverCtx )

      sliceId = Fru.fruBase( invStaticSerdesDir ).sliceId
      if not sliceId:
         assert Cell.cellType() == "fixed"
         sliceId = "FixedSystem"

      t0( "Handling Inventory::L1TopologyStaticSerdesComponentDir for ", sliceId )

      fruModelDir = driverCtx.entity( "hardware/l1/fru/topology" )
      fruModelWriter = Tac.newInstance(
            'Hardware::L1Topology::FruModelWriter', fruModelDir )

      def getSerdesDesc( componentDef, componentId, sliceId,
                         phyScope, serdesId, isTx ):

         # TODO BUG489342: To calculate a global phyCoreId within a component type,
         # we assume that all componentIds are of the same component type given
         # by componentDef.
         # This calculation will be replaced once chip is added to the descriptor
         # hierarchy.
         ( coreId, physicalLaneId ) = componentDef.getSerdes( serdesId )
         phyCoreId = componentId * componentDef.getComponentCoreIdOffset() + coreId
         return SerdesDescriptor( sliceId,
                                  componentDef.name,
                                  componentId,
                                  componentDef.getComponentType( coreId ),
                                  phyCoreId,
                                  phyScope,
                                  physicalLaneId,
                                  isTx )

      for componentId, serdesMap in \
            invStaticSerdesDir.componentStaticSerdesMaps.items():
         componentClass = Components.getComponent( invStaticSerdesDir.chipType )
         assert componentClass, \
                f"Invalid component {invStaticSerdesDir.chipType}"

         # Change the coreModes to specify None rather than '' since going through
         # the inventory models changes everything to strings
         coreModes = { k: ( v or None ) for k, v in
                       invStaticSerdesDir.coreMode.items() }
         componentDef = componentClass( mode=invStaticSerdesDir.mode or None,
                                        coreModes=coreModes or None )

         for phyScope, laneMap in serdesMap.staticSerdesLaneMap.items():
            for serdesId, laneId in laneMap.txLogicalLane.items():
               serdesDesc = getSerdesDesc( componentDef, componentId, sliceId,
                                           phyScope, serdesId, True )
               fruModelWriter.addStaticLaneMap( serdesDesc, laneId )

            for serdesId, laneId in laneMap.rxLogicalLane.items():
               serdesDesc = getSerdesDesc( componentDef, componentId, sliceId,
                                           phyScope, serdesId, False )
               fruModelWriter.addStaticLaneMap( serdesDesc, laneId )

         for phyScope, polarityMap in serdesMap.staticSerdesPolarityMap.items():
            for serdesId, polarity in polarityMap.txPolarity.items():
               serdesDesc = getSerdesDesc( componentDef, componentId, sliceId,
                                           phyScope, serdesId, True )
               fruModelWriter.addStaticPolarityMap( serdesDesc, polarity )

            for serdesId, polarity in polarityMap.rxPolarity.items():
               serdesDesc = getSerdesDesc( componentDef, componentId, sliceId,
                                           phyScope, serdesId, False )
               fruModelWriter.addStaticPolarityMap( serdesDesc, polarity )

def Plugin( context ):
   """Register the plugins."""
   context.registerDriver( StaticSerdesMapDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( "hardware/l1/fru/topology", "Tac::Dir", "wi" )
   mg.close( None )

