# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Cell
import EntityMib
import Fru
import StorageInvLib
import Tracing
import Tac
from Toggles.StorageDevicesToggleLib import                                        \
      toggleStorageDevicesDetectionSyslogsEnabled
from Toggles.SwagCoreToggleLib import toggleMultiSwagBringupEnabled

import os

__defaultTraceHandle__ = Tracing.Handle( "Fru.Storage" )
t0 = Tracing.trace0

def getStorageEntMib( deviceName, fixedSystemMib ):
   for entityMibDevice in fixedSystemMib.storageDevices.values():
      if entityMibDevice.tag == deviceName:
         return entityMibDevice

   # If there are existing device ids, add one to the highest value
   devId = max( fixedSystemMib.storageDevices, default=0 ) + 1

   physicalIndex = EntityMib.IndexAllocator.collectionItemPhysicalIndex(
      fixedSystemMib.storageDevices, devId )
   storageDevicesMibEnt = fixedSystemMib.newStorageDevices( physicalIndex,
      devId, deviceName )

   return storageDevicesMibEnt

class StorageDeviceDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::StorageDeviceDir"
   managedApiRe = "$"
   def __init__( self, invStorageDeviceDir, fixedSystemMib, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invStorageDeviceDir, fixedSystemMib,
                              parentDriver, ctx )
      assert invStorageDeviceDir, 'storage device dir does not exist'

      cellId = Fru.fruBase( invStorageDeviceDir ).managingCellId
      # TODO: BUG 998472: remove after swagifying StorageDevices
      # We hardcode cellId = 3 in bi-swag where we assume the swag worker has a
      # member ID of 3. It has no cellId because its fruBase is a non-supe card
      if toggleMultiSwagBringupEnabled() and not cellId:
         cellId = Fru.fruBase( invStorageDeviceDir ).smid

      sysdbRoot = ctx.sysdbRoot
      hwCell = sysdbRoot.entity[ 'hardware' ][ 'cell' ][ str( cellId ) ]
      sdDir = hwCell.mkdir( 'storageDevices' )

      expectedMounts = [ 'flash' ]
      if invStorageDeviceDir.storageMount:
         expectedMounts += list( invStorageDeviceDir.storageMount )

      # In some btests (Yosemite/test/M4FruTest.py) there is a race between
      # Sysdb creating fruConfig via preinit profile and the access here.
      # Instead of using a preinit profile, just create the entity here.
      fruConfig = sdDir.createEntity( "StorageDevices::FruConfig", "fruConfig" )
      fruConfig.expectedMounts.clear()
      for mount in expectedMounts:
         fruConfig.expectedMounts.add( mount )

      # We should always start the StorageDevices agent. We have a conditional
      # runnability so that it's not started in namespace duts and to ensure
      # fruConfig is pre-populated.
      sdLauncherConfigDir = sdDir.mkdir( 'launcherConfig' )
      sdLauncherConfigDir.mkdir( 'storageDevices' )
      envTempConfigDir = sysdbRoot.entity[ 'environment' ][ 'temperature' ]\
            [ 'config' ]

      if cellId == Cell.cellId():
         self.populator = Tac.newInstance(
            "StorageDevicesFruPluginHelper::FruPluginHelperSm",
            sysdbRoot.entity[ Cell.path( "hardware/storageDevices/status" ) ],
            fixedSystemMib, envTempConfigDir, invStorageDeviceDir, fruConfig )
      else:
         # We are managing storage inventory for the peer under SSO.
         self.populator = Tac.newInstance(
            "StorageDevicesFruPluginHelper::PeerFruPluginHelper", fixedSystemMib,
            ctx.entityManager.cEntityManager(), envTempConfigDir,
            invStorageDeviceDir, fruConfig )

      # Keep running inventory directly in Fru for now. This code will still syslog
      # in certain failure cases, and the StorageDevices agent doesn't do this on
      # it's own
      if not toggleStorageDevicesDetectionSyslogsEnabled():
         self.inventory( expectedMounts, fixedSystemMib )

   def inventory( self, expectedMounts, fixedSystemMib ):
      if "SIMULATION_VMID" in os.environ:
         # In btests, we can't expect any mounts to exists so skip populating
         # this inventory.
         return
      mounts = StorageInvLib.Mounts()
      devices = StorageInvLib.DeviceFiles( mounts )
      possibleMounts = [ 'flash', 'drive', 'drive2' ]
      for mount in possibleMounts:
         mountExpected = mount in expectedMounts
         devices.deviceFactory( mount, mountExpected=mountExpected )

def Plugin( context ):
   context.registerDriver( StorageDeviceDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( Cell.path( 'hardware/storageDevices/status' ),
             'StorageDevices::Status', 'r' )
   mg.close( None )
