#!/usr/bin/env python3
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
import Cell
import EntityMib
from EntityMib import IndexAllocator
import Fru
import FruPlugin.Gpio
import FruPlugin.PowerController
import FruPlugin.Smbus
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "Fru.Ucd9012" )
t0 = Tracing.trace0

# BUG_dov: this is predecated on Fru re-execing the fdl if the agent restarts
cellDirReactor = None
statusDirReactorMap = {}
controllerNameToChipMib = {}

class UcdDriver( FruPlugin.PowerController.Driver ):
   ''' 
   This is the parent driver to the drivers for Ucd90120, Ucd90320, and Psoc.
   '''
   # Pylint can't find attrs defined by FruDriver.
   # pylint: disable-msg=E1101

   requires = [ "StandbyCpld", "SouthBridge", "CpuComplex",
                FruPlugin.Gpio.pcaGpioInit ]
   provides = [ FruPlugin.Gpio.gpioInit ]

   def __init__( self, invPowerController, parentMibEntity, parentDriver,
                 driverCtx ):
      FruPlugin.PowerController.Driver.__init__(
         self, invPowerController, parentMibEntity, parentDriver, driverCtx )
      # pylint: disable-msg=E1101
      sliceId = Fru.fruBase(invPowerController).sliceId

      cell = Fru.fruBase( invPowerController ).managingCellId
      config = self._config()

      environmentDir = driverCtx.sysdbRoot[ 'environment' ]

      powerEnvConfig = environmentDir[ 'power' ][ 'config' ]

      name = self._name()
      self._chipName = name
      customerName = sliceId

      t0( "Creating power controller hw model for", name )

      ahamDesc = self._ahamDesc( config, name )

      if invPowerController.resetGpioPin:
         resetGpioPinName = invPowerController.resetGpioPin.systemName
         assert resetGpioPinName
      else:
         resetGpioPinName = ''

      #
      # Create the associated EntityMib object
      #
      label = name
      chipMib = None
      self._chipId = None
      for chip in self.parentMibEntity_.chip.values():
         if chip.label == label:
            chipMib = chip
            self._chipId = chip.relPos
            break
      else:
         self._chipId = FruPlugin.PowerController.powerControllerChipId(
                                  'Ucd9012', self.parentMibEntity_.chip )
         physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                         ( self.parentMibEntity_.chip, self._chipId )
         chipMib = self.parentMibEntity_.newChip( physicalIndex,
                                                  self._chipId,
                                                  self.modelName + "-" +  
                                                  str( self._chipId % 10 ) ) 
         chipMib.label = label
         chipMib.description = "Ucd Chip %s" % self._chipId
      EntityMib.populateMib( chipMib, invPowerController )

      t0( "Populating controllerNameToChipMib:", name )
      controllerNameToChipMib[ name ] = chipMib

      configColl = self._getInstCollection( config )
      hwUcdConfig = Fru.Dep( configColl, invPowerController ).newMember( name,
            customerName, ahamDesc, resetGpioPinName )
      Fru.Dep( config.controller, invPowerController ).addMember( hwUcdConfig )

      hwUcdConfig.chipId = self._chipId % 100

      hwUcdConfig.modelName = self.modelName

      invPowerController.controllerName = name

      hwUcdConfig.generationId = Fru.fruBase( invPowerController ).generationId

      hwUcdConfig.autoStartPowerProgram = invPowerController.autoStartPowerProgram
      if hwUcdConfig.autoStartPowerProgram and cell:
         hwUcdConfig.controllerState = "powerOn"
      # On Yellowstone cards, there is nothing to set the state to 'powerOn'
      # for power controllers hanging off the secondary smbus.   
      elif invPowerController.inMainPowerDomain:
         hwUcdConfig.inMainPowerDomain = invPowerController.inMainPowerDomain
         hwUcdConfig.controllerState = "powerOn"
         t0( "Set powerOn for secondary DPM", name )
      
      hwUcdConfig.agentControlsPower = invPowerController.agentControlsPower

      shortName = invPowerController.name.capitalize()
      shortName = re.sub( "power", "Pwr", shortName, flags=re.IGNORECASE )
      shortName = re.sub( "controller", "Con", shortName, flags=re.IGNORECASE )

      for rail in invPowerController.rail.values():
         # Note: don't create voltage rails or sensors for the status.
         # This is a transitional period where Fru creates a dummy status
         # for the sake of NorCalCard, but the real status is created by the agent.
         # The cli checks the Fru-created voltage sensors first, then falls back to
         # the archer path.
         voltageRailId = rail.id
         sensorId = rail.sensorId
         sensorName = EntityMib.componentNameFromParent( self.parentMibEntity_,
                                                         "VoltageSensor",
                                                         sensorId )
         envSensorConfig = Fru.Dep(
            powerEnvConfig.voltageSensor,
            invPowerController ).newMember( sensorName,
                                            rail.maxMarginHighPercent,
                                            rail.maxMarginLowPercent )
         envSensorConfig.expectedVoltage = rail.expectedVoltage
         envSensorConfig.description = rail.description
         envSensorConfig.marginLevelPercent = rail.defaultMarginPercent
         envSensorConfig.marginSource = rail.marginSource
         envSensorConfig.dmon = rail.dmon
         envSensorConfig.uplink = rail.uplink
         envSensorConfig.hasCurrentSensor = rail.hasCurrentSensor
         envSensorConfig.sliceName = Fru.fruBaseName( rail )
         hwUcdRailConfig = hwUcdConfig.newVoltageRail( voltageRailId,
                                                       envSensorConfig )
         hwUcdRailConfig.overrideVoutUvLimit = rail.overrideVoutUvLimit
         hwUcdRailConfig.overrideVoutOvLimit = rail.overrideVoutOvLimit
         hwUcdRailConfig.overridePowerGoodOff = rail.overridePowerGoodOff
         hwUcdRailConfig.tolerance = rail.tolerance
         if invPowerController.pollingEnabled:
            sensorMib = self.parentMibEntity_.sensor.get( sensorId )
            if sensorMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  self.parentMibEntity_.sensor, sensorId )
               sensorMib = Fru.Dep(
                  self.parentMibEntity_.sensor, invPowerController ).newMember(
                     physicalIndex, sensorId, "VoltageSensor" )

            sensorMib.description = "%s Rail%s %s" % ( shortName,
                                                       voltageRailId,
                                                       rail.description )
            sensorMib.label = str( sensorId )
            sensorMib.initStatus = "ok"
         if rail.hasCurrentSensor:
            currentSensorName = EntityMib.componentNameFromParent(
               self.parentMibEntity_, "CurrentSensor", sensorId )
            envCurrentConfig = Fru.Dep(
               powerEnvConfig.currentSensor,
               invPowerController ).newMember( currentSensorName )
            envCurrentConfig.description = rail.description
            hwUcdConfig.newCurrentSensor( voltageRailId, envCurrentConfig )

      hwUcdConfig.reloadCauseSortKey = invPowerController.reloadCauseSortKey

      # Load the Reload cause in the HwUcdConfig
      for ( i, cause ) in invPowerController.cause.items():
         hwUcdConfig.reloadCause[ i ] = Tac.Value( 
            "Hardware::Ucd9012::UcdCause",
            cause.reloadCauseKey, cause.type, cause.id, cause.isRemote )

      for ( i, cause ) in invPowerController.psocCause.items():
         assert cause.bitNum.bit_length() <= 16
         hwUcdConfig.psocReloadCause[ i ] = Tac.Value( 
            "Hardware::Ucd9012::PsocCause",
            cause.reloadCauseKey, cause.bitNum, cause.origin )
      
      for ( i, cause ) in invPowerController.pageFaultCause.items():
         hwUcdConfig.pageFaultCause[ i ] = Tac.Value(
            "Hardware::Ucd9012::PageFaultCause",
            cause.reloadCauseKey, cause.id, cause.faultMask )
   
      for ( i, rail ) in invPowerController.externalVoltageRail.items():
         hwUcdConfig.externalVoltageRail[ i ] = Tac.Value(
            "Hardware::Ucd9012::ExternalVoltageRail", rail.railNum, rail.faultType )

      for reg in invPowerController.overrideReg.values():
         newReg = Tac.Value( "Hardware::Ucd9012::OverrideReg", reg.addr, reg.val )
         hwUcdConfig.overrideReg.enq( newReg )

      if invPowerController.kernelPanicEeprom is not None:
         if isinstance( invPowerController.kernelPanicEeprom, 
                        Tac.Type( "Inventory::Seeprom" ) ):
            t0( "kernelPanicEeprom is type Inventory::Seeprom" )
            topology = (
               self.driverCtx_.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ] )

            ahamDescArgs = \
               FruPlugin.Smbus.ahamDesc( topology, 
                                         invPowerController.kernelPanicEeprom )
            hwUcdConfig.kernelPanicEepromSmbusAham = ( name, ) + ahamDescArgs
            hwUcdConfig.kernelPanicEepromAham = \
                  hwUcdConfig.kernelPanicEepromSmbusAham 
         else:
            t0( "kernelPanicEeprom is not of type Inventory::Seeprom" )         
            hwUcdConfig.kernelPanicEepromI2cAham = ( name, 
                  invPowerController.kernelPanicEeprom.master.id,
                  invPowerController.kernelPanicEeprom.deviceId )
            hwUcdConfig.kernelPanicEepromI2cAham.waitBetweenTransactions = \
               invPowerController.kernelPanicEeprom.waitBetweenTransactions
            hwUcdConfig.kernelPanicEepromAham = \
                  hwUcdConfig.kernelPanicEepromI2cAham 

         wpGpo = invPowerController.kernelPanicEepromWriteProtect
         wpGpio = invPowerController.kernelPanicEepromWriteProtectGpio
         # check that if a write protect pin is specified it is either a gpo or gpio,
         # but not both
         assert wpGpo is None or wpGpio is None
         if wpGpo is not None:
            t0( "kernelPanicEeprom write protect uses synchronous gpo" )
            hwUcdConfig.kernelPanicEepromWpGpo = wpGpo.systemName
         elif wpGpio is not None:
            t0( "kernelPanicEeprom write protect uses asynchronous gpio" )
            hwUcdConfig.kernelPanicEepromWpGpio = wpGpio.systemName

      if invPowerController.description:
         hwUcdConfig.description = invPowerController.description
      hwUcdConfig.isMainController = invPowerController.isMainController
      hwUcdConfig.altIndicatorSource = invPowerController.altIndicatorSource
      hwUcdConfig.location = invPowerController.location or name
      hwUcdConfig.shutdownEnabled = invPowerController.shutdownEnabled
      hwUcdConfig.pollingEnabled = invPowerController.pollingEnabled

      if cell:
         dirName = str( cell )
         archerCellStatusDir = driverCtx.sysdbRoot.entity[
               "hardware/archer/powercontroller/status/cell" ]
         global cellDirReactor
         if not cellDirReactor:
            cellDirReactor = Ucd9012ParentStatusDirReactor( archerCellStatusDir )
      else:
         dirName = 'system'
         archerSystemStatusDir = driverCtx.sysdbRoot.entity[
               "hardware/archer/powercontroller/status/system" ]
         if dirName not in statusDirReactorMap:
            statusDirReactorMap[ dirName ] = Ucd9012StatusDirReactor(
                  archerSystemStatusDir )

      if dirName in statusDirReactorMap:
         statusDirReactorMap[ dirName ].handlePowerController( hwUcdConfig.name )

      gpioInitDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]
      for ( pinId, invPin ) in invPowerController.gpioPin.items():
         if invPin:
            systemName = "%s-%s-%d" % ( name, invPin.pinName, pinId )
            pinInitType = 'Hardware::Gpio::ArcherPinInit'
            pinInit = Fru.Dep( gpioInitDir,
                               invPowerController ).newEntity( pinInitType,
                                                               systemName )
            pinInit.pinName = invPin.pinName
            pinInit.pinId = pinId
            pinInit.count = invPin.count
            pinInit.activeType = invPin.activeType
            pinInit.setToInputIfInactive = invPin.setToInputIfInactive
            pinInit.setToInputIfActive = invPin.setToInputIfActive
            pinInit.direction = invPin.direction
            pinInit.defaultVal = invPin.defaultVal
            invPin.systemName = systemName
            hwUcdConfig.pinName[ systemName ] = True

      hwUcdConfig.needTonDelayUpdate = invPowerController.needTonDelayUpdate

      hwUcdConfig.configReady = True

   def _config( self ):
      cell = Fru.fruBase( self.invEntity_ ).managingCellId
      if cell:
         # Create hardware/cell/<cellid>/powercontroller/config/ucd9012
         hwDir = self.driverCtx_.sysdbRoot[ 'hardware' ][ 'cell' ][ str( cell ) ]
         configDir = hwDir.mkdir( 'powercontroller/config' )
         config = configDir.newEntity( 'Hardware::Ucd9012::Config', 'ucd90120' )
      else:
         hwDir = self.driverCtx_.sysdbRoot[ 'hardware' ]
         config = hwDir[ 'powercontroller' ][ 'config' ][ 'ucd90120' ]

      return config

   def _name( self ):
      sliceId = Fru.fruBase( self.invEntity_ ).sliceId
      return sliceId + self.invEntity_.name

   def _ahamDesc( self, config, name ):
      topology = self.driverCtx_.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = [ "%sAhamDesc" % name ]
      ahamArgs.extend( FruPlugin.Smbus.ahamDesc( topology, self.invEntity_ ))
      ahamDesc = Fru.Dep( config.ahamDesc, self.invEntity_ ).newMember( *ahamArgs )
      return ahamDesc
   
class Ucd90120Driver( UcdDriver ):
   """
   This driver handles the Ucd90120 power controller chip found on
   Raven, Nevada, and most products since.
   """
   managedTypeName = "Inventory::Ucd9012Controller"
   modelName = "Ucd9012"

   def _getInstCollection( self, config ):
      return config.realUcd9012Controller
   
class Ucd90320Driver( UcdDriver ):
   '''
   This driver handles the Ucd90320 power controller chip found 
   on Oakland and possibly other products in the future.
   '''
   managedTypeName = "Inventory::Ucd9032Controller"
   modelName = "Ucd9032"

   def _getInstCollection( self, config ):
      return config.realUcd9032Controller

class PsocPowerDriver( UcdDriver ):
   managedTypeName = "Inventory::PsocPowerController"
   modelName = "Psoc"

   def _getInstCollection( self, config ):
      return config.realPsocController

class PsocPmbusDriver( PsocPowerDriver ):
   managedTypeName = "Inventory::PsocPmbusController"

   def _config( self ):
      cell = Fru.fruBase( self.invEntity_ ).managingCellId
      if cell:
         path = 'hardware/cell/%s/powercontroller/config/ucd90120' % str( cell )
      else:
         path = 'hardware/powercontroller/config/ucd90120'
      return self.driverCtx_.sysdbRoot.entity[ path ]

   def _name( self ):
      return self.invEntity_.name

   def _ahamDesc( self, config, name ):
      # Create ahamDesc using the parent Psoc inv-tac-model. This is so one
      # psoc model can represent both the ucd9012 and psocFanController in the smbus
      # topology. For details, see review 18986.
      invPsoc = self.invEntity_.parent
      topology = self.driverCtx_.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = [ "%sAhamDesc" % invPsoc.name ]
      ahamArgs.extend( FruPlugin.Smbus.ahamDesc( topology, invPsoc ) )
      return Fru.Dep( self._config().ahamDesc, invPsoc ).newMember( *ahamArgs )

class Ucd9012ParentStatusDirReactor( Tac.Notifiee ):
   """ 
   Instantiates a Ucd9012StatusDirReactor for each cell under
   "hardware/archer/powercontroller/status/cell/..."
   When this code runs there is no guarantee all the cells will be
   there, so a reactor needs to be set up for when they come in.
   """
   notifierTypeName = "Tac::Dir"

   def __init__( self, cellDir ):
      Tac.Notifiee.__init__( self, cellDir )
      t0( "Creating Ucd9012ParentStatusDirReactor" )
      for cell in self.notifier_:
         self.handleCell( cell )

   @Tac.handler( 'entityPtr' )
   def handleCell( self, cell ):
      t0( "handleCell() called for cell %s" % cell )
      cellPtr = self.notifier_.get( cell )
      cellEntry = self.notifier_.entryState.get( cell )
      if not cellPtr and not cellEntry:
         t0( "Deleting cell reactor for cell %s" % cell )
         del statusDirReactorMap[ cell ]
      elif cellPtr and cellEntry:
         t0( "Creating cell reactor for cell %s" % cell )
         statusDirReactorMap[ cell ] = Ucd9012StatusDirReactor( cellPtr )

class Ucd9012StatusDirReactor( Tac.Notifiee ):
   notifierTypeName = "Tac::Dir"
   controllerStatusType = Tac.Type( "Hardware::Ucd9012::ControllerStatus" )

   def __init__( self, archerStatusDir ):
      Tac.Notifiee.__init__( self, archerStatusDir )
      # Note: do NOT iterate over the current entries in archerStatusDir.
      # The Fru restart case is handled by calling handlePowerController
      # in the inventory driver for a specific controller, right after populating
      # the controllerNameToChipMib collection for that controller.
      self.archerStatusDir_ = archerStatusDir
      self.controllerNameToVersionReactor = {}

   @Tac.handler( 'entityPtr' )
   def handlePowerController( self, key ):
      status = self.archerStatusDir_.entityPtr.get( key )
      statusEntry = self.archerStatusDir_.entryState.get( key )

      if ( not status and
           not statusEntry and
           key in self.controllerNameToVersionReactor ):
         t0( "Deleting reactor for: '%s' " % key )
         del self.controllerNameToVersionReactor[ key ]
      elif ( status and
             statusEntry and
             isinstance( status, self.controllerStatusType ) and
             key not in self.controllerNameToVersionReactor ):
         t0( "Creating reactor for: '%s' " % key )
         self.controllerNameToVersionReactor[ key ] = \
                           Ucd9012VersionReactor( status,
                                                  controllerNameToChipMib[ key ] )

class Ucd9012VersionReactor( Tac.Notifiee ):
   """propagates UCD9012 version from hardware to entmib"""
   notifierTypeName = "Hardware::Ucd9012::ControllerStatus" 

   def __init__( self, hwUcdStatus, chipMib ):
      Tac.Notifiee.__init__( self, hwUcdStatus )
      self.chipMib = chipMib
      self.updateFirmwareRev()
   
   def updateFirmwareRev( self ): 
      firmwareRev = self.notifier_.controllerVersion.strip() 
      if self.notifier_.firmwareVersion:
         firmwareRev += " (%s)" % self.notifier_.firmwareVersion.strip()
      elif self.notifier_.firmwareMfrDate:
         firmwareRev += " (%s)" % self.notifier_.firmwareMfrDate.strip()

      self.chipMib.firmwareRev = \
         firmwareRev.encode( "utf-8", "ignore" ).decode( "utf-8" )
      t0( "Publishing firmware info", firmwareRev, "for", self.notifier_.name ) 

   # applies to both Ucd and PsoC chips 
   @Tac.handler( 'controllerVersion' )
   def handleControllerVersion( self ):
      t0( "Controller version changed" )
      self.updateFirmwareRev()

   # only applies to Ucd
   @Tac.handler( 'firmwareVersion' ) 
   def handleFirmwareVersion( self ):
      t0( "Firmware version changed" )
      self.updateFirmwareRev()

   # only applies to PsoC
   @Tac.handler( 'firmwareMfrDate' )
   def handleFirmwareMfrDate( self) : 
      t0( "Firmware mfr date changed" )
      self.updateFirmwareRev()

def Plugin( ctx ):

   ctx.registerDriver( Ucd90120Driver )
   ctx.registerDriver( Ucd90320Driver )
   ctx.registerDriver( PsocPowerDriver )
   ctx.registerDriver( PsocPmbusDriver )
   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/archer/powercontroller/status/cell', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/archer/powercontroller/status/system', 'Tac::Dir', 'ri' )
   mg.mount( 'environment/archer/power/status/voltageSensor/cell',
             'Tac::Dir', 'ri' )
   mg.mount( 'environment/archer/power/status/voltageSensor/system',
             'Tac::Dir', 'ri' )
   mg.mount( 'environment/power/config', 'Environment::Power::Config', 'w' )
   mg.mount( "hardware/powercontroller/config/ucd90120",
             'Hardware::Ucd9012::Config', "w" )
   mg.mount( "hardware/cell/%d/powercontroller/config/ucd90120" % Cell.cellId(),
             'Hardware::Ucd9012::Config', "w" )
   mg.mount( 'hardware/archer/gpio/init', 'Tac::Dir', 'wi' )
   mg.close( None )
