# Copyright (c) 2010-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Fru
import Tac
import XcvrLib

def createXcvrCtrlPaths( fruBase, ctx ):
   configDir = ctx.sysdbRoot[ 'hardware' ][ 'xcvrController' ][ 'config' ]
   cellId = fruBase.managingCellId
   sliceId = fruBase.sliceId
   if cellId:
      cellConfigDir = configDir.mkdir( 'cell' ).mkdir( str( cellId ) )
      cellConfigDir.newEntity( "Hardware::XcvrController::CellScdInputConfig",
                                            "config" )
   elif sliceId:
      sliceConfigDir = configDir.mkdir( 'slice' )
      linecardConfigDir = \
            sliceConfigDir.newEntity( "Tac::Dir", sliceId )
      linecardConfigDir.newEntity( "Hardware::XcvrController::CellScdInputConfig",
                                   "config" )
   else:
      assert 0

def addXcvr( xcvrInv, fruBase, scdInv, driverCtx, xcvrType, modular ):
   """
   Adds a transceiver slot to the system using Inventory data

   Parameters
   ----------
   xcvrInv : Inventory::XcvrController

   fruBase : Inventory::FruBase
      Contains attributes populated by the Fru that are used by the XcvrAgent
      to determine if we are adding xcvrs to a fixed or modular system

   scdInv : Inventory::PciScd

   driverCtx : Fru.Driver.DriverContext

   xcvrType : str
      String representation of the eype of the transceiver, for example
      "osfp", "cfp2", etc.

   modular : bool

   Returns
   -------
   xcvrConfig : Hardware::XcvrController::XxxxControllerConfig type object
      Returns the ControllerConfig object correctly matching the passed in
      transceiver type.  For example, if transceiver type is 'cfp2', then a
      Cfp2ControllerConfig object is returned.
   """

   def enumerateXcvrIntfs( xcvrControllerInv ):
      """
      Enumerate all interfaces and corresponding link indexes ( XCVR lanes ) given an
      XCVR controller inventor.

      Parameters
      ----------
      xcvrInv : Inventory::XcvrController

      Returns
      -------
      The ( linkIndex, intfId ) tuple.
      """
      for conn in xcvrControllerInv.xcvrConnection.values():
         # On older systems, the interface ID is derived from the connection object's
         # name.  This is quite dangerous as the interface ID present in the
         # inventory is created completely independently of the rest of the ports
         # infrastructure and so there is an implicit reliance on the stars
         # aligning.
         if not conn.port:
            yield ( conn.linkIndex, conn.name )
            continue

         # If the port corresponding to an interface has been marked as unbound then
         # for all intents and purposes it does not exist.
         if not conn.port.bound:
            continue

         yield ( conn.linkIndex, conn.port.intfId )

   def unbound( xcvrControllerInv ):
      """
      Determines if all interfaces on the corresponding XCVR slot have been unbound.

      Parameters
      ----------
      xcvrInv : Inventory::XcvrController

      Returns
      -------
      True if all interfaces on the XCVR slot are unbound else False
      """
      # If the XCVR has no connections at all, we assume the module is in
      # an auxiliary slot, which is not connected to an asic/phy. This slot
      # is always considered bound.
      if not xcvrControllerInv.xcvrConnection.values():
         return False

      for conn in xcvrControllerInv.xcvrConnection.values():
         # On older systems the port inventories are not linked to the XCVR slot
         # inventories. For such systems L1 profiles is unsupported so the XCVR slot
         # is by definition bound.
         if not conn.port:
            return False

         # If any interface on the XCVR slot is still bound then the entire XCVR slot
         # is bound.
         if conn.port.bound:
            return False

      return True

   if unbound( xcvrInv ):
      return None

   # add the to the system directory, the policy agent uses this to control
   # the led's properties
   configDir = driverCtx.sysdbRoot[ 'hardware' ][ 'xcvrController' ][ 'config' ]
   # Cell Id is 0 ( the default value ) if we're adding xcvrs to a a modular system
   cellId = fruBase.managingCellId
   sliceId = fruBase.sliceId
   if cellId:
      cellConfigDir = configDir.mkdir( 'cell' ).mkdir( str( cellId ) )
      scdInputConfig = \
            cellConfigDir.newEntity( "Hardware::XcvrController::CellScdInputConfig",
                                     "config" )
      allConfigDir = \
            configDir.newEntity( "Hardware::XcvrController::AllConfigDir", "all" )

   elif sliceId:
      sliceConfigDir = configDir.mkdir( 'slice' )
      linecardConfigDir = \
            Fru.Dep( sliceConfigDir, xcvrInv ).newEntity( "Tac::Dir", sliceId )
      scdInputConfig = Fru.Dep( linecardConfigDir, xcvrInv ).newEntity(
         "Hardware::XcvrController::CellScdInputConfig", "config" )
      allConfigDir = Fru.Dep( linecardConfigDir, xcvrInv ).newEntity(
         "Hardware::XcvrController::AllConfigDir", "xcvrConfig" )
   else:
      # If we don't have a cell id or a slice id, we cannot determine
      # If we're using a fixed or modular system, which means FRU forgot to
      # populate some essential information
      assert 0

   # Global name of the transceiver slot
   name = xcvrInv.name

   # Only the CMIS modules use the new Hw agnostic SCD config object
   xcvrScdConfig = None
   if xcvrType == 'sfpPlus':
      xcvrConfig = Fru.Dep( scdInputConfig.sfpPlusControllerConfig,
                            scdInv ).newMember( name )
      # add the xcvrController to the all collection
      Fru.Dep( allConfigDir.sfpPlusControllerConfig,
               scdInv ).addMember( xcvrConfig )
   elif xcvrType == 'qsfpPlus':
      xcvrConfig = Fru.Dep( scdInputConfig.qsfpPlusControllerConfig,
                            scdInv ).newMember( name )
      # add the xcvrController to the all collection
      Fru.Dep( allConfigDir.qsfpPlusControllerConfig,
               scdInv ).addMember( xcvrConfig )
      # copy the link index to port name mapping
      for linkIndex, intfName in enumerateXcvrIntfs( xcvrInv ):
         xcvrConfig.link[ linkIndex ] = intfName
      if xcvrInv.capabilities and xcvrInv.capabilities.slotPowerControlSupported:
         xcvrConfig.powerEnableSupported = True
   elif xcvrType == 'cfp2':
      xcvrConfig = Fru.Dep( scdInputConfig.cfp2ControllerConfig,
                            scdInv ).newMember( name )
      # add the xcvrController to the all collection
      Fru.Dep( allConfigDir.cfp2ControllerConfig, scdInv ).addMember( xcvrConfig )
      # copy the link index to port name mapping
      for linkIndex, intfName in enumerateXcvrIntfs( xcvrInv ):
         xcvrConfig.link[ linkIndex ] = intfName
      if xcvrInv.capabilities and xcvrInv.capabilities.slotPowerControlSupported:
         xcvrConfig.powerEnableSupported = True
   elif XcvrLib.isCmisTypeStr( xcvrType ):
      # SFP-DD form factor has different hwXcvrController objects from normal
      # CMIS types
      if xcvrType in ( 'sfpDd', ):
         xcvrScdConfig = Fru.Dep( scdInputConfig.sfpDdScdInputConfig,
                                  scdInv ).newMember( name )
         xcvrConfig = Fru.Dep( allConfigDir.sfpDdControllerInputConfig,
                               scdInv ).newMember( name )
      else:
         xcvrScdConfig = Fru.Dep( scdInputConfig.cmisScdInputConfig,
                                  scdInv ).newMember( name )
         xcvrConfig = Fru.Dep( allConfigDir.cmisControllerInputConfig,
                               scdInv ).newMember( name )
      # copy the link index to port name mapping
      for linkIndex, intfName in enumerateXcvrIntfs( xcvrInv ):
         xcvrConfig.link[ linkIndex ] = intfName
   else:
      # we don't support any other transceiver type
      assert 0

   fruPowerGenerationId = Fru.powerGenerationId( scdInv )
   if scdInputConfig:
      scdInputConfig.generationId = fruPowerGenerationId

   # If we're using the new SCD input config object, we will set the
   # generation id there and copy it over to the xcvrConfig object
   if xcvrScdConfig:
      xcvrScdConfig.generationId = fruPowerGenerationId

   xcvrConfig.generationId = fruPowerGenerationId
   allConfigDir.generationId = fruPowerGenerationId
   # populate sliceId
   if cellId:
      allConfigDir.sliceId = "FixedSystem"
   else:
      allConfigDir.sliceId = sliceId

   return xcvrConfig

def getAllConfigDir( fruBase, ctx ):
   """
   This function is used to find and return the current instance
   of AllConfigDir stored in Sysdb. AllConfigDir stores the ahamDesc
   objects as a collection indexed by port name, which are then passed
   to each individual Xcvr Input Config object as a pointer.

   Parameters:
   ----------
   fruBase : Inventory::FixedSystem
   ctx : Fru.Driver.DriverContext

   Returns:
   allConfigDir : Hardware::XcvrController::AllConfigDir
   """
   cellId = fruBase.managingCellId
   xcvrController = ctx.sysdbRoot[ 'hardware' ][ 'xcvrController' ]
   xcvrControllerConfig = xcvrController[ 'config' ]
   # Need to find allConfigDir to store ahamDesc for all config entities
   if cellId:
      allConfigDir = xcvrControllerConfig[ 'all' ]
   else:
      sliceId = fruBase.sliceId
      allConfigDir = xcvrControllerConfig[ 'slice' ][ sliceId ][ "xcvrConfig" ]
   return allConfigDir
