#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Fru
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "XcvrLaneMgrFru" )

t0 = __defaultTraceHandle__.trace0
t1 = __defaultTraceHandle__.trace1

class PortLaneConfigDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::XcvrLaneMgr::PortLaneConfigDir"
   provides = [ ]
   requires = [ Fru.FruDriver.interfaceInit, Fru.FruDriver.systemInit, "xcvr" ]

   def __init__( self, portLaneConfigDir, parentMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, portLaneConfigDir, parentMib,
                              parentDriver, driverCtx )
      t0( "Handling %s for %s" % # pylint: disable=consider-using-f-string
          ( self.managedTypeName, Fru.fruBaseName( portLaneConfigDir ) ) )
      hwDir = driverCtx.sysdbRoot[ "hardware" ]
      laneMgrConfig = hwDir[ "xcvrLaneMgr" ][ "config" ]
      assert laneMgrConfig

      # If we are bypassing the the creation of L1 Topology intventory models from
      # this FRU plugin then dont pass an L1 Topology root directory to the helper.
      #
      # TODO BUG664124: As we migrate platforms towards using the new L1 Interface
      #                 model generation scheme described in AID 9905, we should see
      #                 less platforms rely on XCVR lane manager's FRU plugin for L1
      #                 Topology interface model generation. Evenrually, we should
      #                 remove this functionality from this file.
      topoDir = None
      if portLaneConfigDir.populateL1Topology:
         topoDir = driverCtx.entity( "hardware/l1/topology" )

      fruHelper = Tac.newInstance( "XcvrLaneMgr::FruPluginHelper",
                                   portLaneConfigDir, laneMgrConfig, Fru.fruUtil(),
                                   topoDir )

      # SliceId seems to have become valid even on fixed systems and now contains
      # 'FixedSystem' on those platforms.
      cellId = fruHelper.fruUtil.fruBase( portLaneConfigDir ).managingCellId
      sliceId = fruHelper.fruUtil.fruBase( portLaneConfigDir ).sliceId
      # pylint: disable-next=consider-using-f-string
      t0( "cell %s slice %s" % ( cellId, sliceId ) )

      fruHelper.doPopulateHardware()

def Plugin( context ):
   context.registerDriver( PortLaneConfigDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/xcvrLaneMgr/config', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/l1/topology', 'Tac::Dir', 'wi' )
   mg.close( None )
