#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This plugin provides a FruFactory that is used when EOS in running
# inside a Container. We synthesize a FDL based on what we can read out of
# the Linux kernel. We assume that we're on a fixed system.
# Interfaces that begin with 'docker' are assumed to be Management Ethernet
# All other interfaces are numbered.

import os
from shlex import quote as shquote
import socket

import Ark
import Arnet.Device
import Cell
import EventLib
import Tac
import Tracing
import Fru
from Fru.FruBaseVeosDriver import parseVeosConfig, handleEtbaOverride
from FruPlugin.EosInitFruPluginLib import (
      VEosLabCEosLabBaseDriver,
      Config,
      JsonIntfConfig,
      IntfType,
)
import SimpleConfigFile
import Url
from UrlPlugin.FlashUrl import flashFileUrl
from EbraTestBridgeConstants import bridgingEtbaConfigPath, bridgingEtbaConfigType

__defaultTraceHandle__ = Tracing.Handle( "Fru.vEOScEOSLab" )
t0 = Tracing.trace0
t1 = Tracing.trace1

ztpConfigForTest = None

class CEosConfig( Config ):
   def __init__( self, ceosConfigPath=None, systemMacAddrPath=None ):
      t0( 'CEosConfig: Building a list of interfaces for cEOS Lab' )
      if ceosConfigPath is None:
         ceosConfigPath = '/mnt/flash/ceos-config'
      kwargs = { 'veosConfigPath': ceosConfigPath }
      if systemMacAddrPath is not None:
         kwargs[ 'systemMacAddrPath' ] = systemMacAddrPath

      self.ceosConfig = parseVeosConfig( **kwargs )
      t1( 'CEosConfig = ', self.ceosConfig )

   def getSerialNumber( self ):
      return self.ceosConfig[ "SERIALNUMBER" ]

   def getSystemMacAddr( self ):
      return self.ceosConfig[ 'SYSTEMMACADDR' ]

   def getMode( self ):
      return self.ceosConfig[ 'MODE' ]

   def handleEtbaOverride( self ):
      handleEtbaOverride( self.ceosConfig )

class cEOSLabDriver( VEosLabCEosLabBaseDriver ):
   managedTypeName = "Eos::CEOSLabFru"
   managedApiRe = ".*"
   driverPriority = 1

   def __init__( self, cEOSLab, parentMibEntity, parentDriver, driverCtx ):
      t0( "cEosLabDriver: Initializing" )
      config = CEosConfig()
      # This is called INTFTYPE but is actually a kernel device prefix, like et.
      self.myIntfType = os.environ.get( "INTFTYPE" )
      self.mapEth0Intf = os.environ.get( "MAPETH0" )
      self.mgmtIntf = os.environ.get( "MGMT_INTF" )
      self.maIntfAllowed = os.environ.get( "MA_INTF_ALLOWED" )
      self.etba = os.environ.get( "ETBA" )
      self.validDevicePfxs = { 'Ethernet', 'eth', 'et', 'ma' }
      self.mapping = {}
      self.sfpXcvrConfig = None
      self.initXcvrConfig( driverCtx )
      self.handleZeroTouchConfig( driverCtx )
      VEosLabCEosLabBaseDriver.__init__( self,
                                         cEOSLab,
                                         parentMibEntity,
                                         parentDriver,
                                         driverCtx,
                                         modelName="cEOSLab",
                                         description="cEOSLab",
                                         config=config )
      #
      # Communicate the shutdown and watchdog-kill commands to
      # Fru/SysReloadScheduler.tin ReloadReactor::handleClock
      os.environ[ "SHUTDOWN_CMD" ] = ( 'bash -c "echo Simulating reboot by '
                                          'restarting ProcMgr; '
                                       'nohup systemctl restart ProcMgr"' )
      os.environ[ "WATCHDOGKILL_CMD" ] = 'echo No watchdog to kill on cEOS-lab'

   def initXcvrConfig( self, driverCtx ):
      # This is cEOS-lab. We don't have any optics plugged in. But there's
      # a desire to expose some information via telemetry for which we will
      # need to set up the state such that the desired information is
      # exported. To do this, we pretend our interface to contain an SFP.
      # The actual work happens when we handle a given device but before that
      # we need to set up some state in Sysdb: we need to instantiate the
      # sfp cell config. The actual use of this happens when we deal with a given
      # device (see handleDevice)
      xcvrConfig = driverCtx.entity( 'hardware/xcvr/config' )
      sfp = xcvrConfig.get( 'sfp' )
      if sfp is None:
         # The xcvr plugin isn't loaded, we are probably in a test
         t0( 'initXcvrConfig: no sfp directory, not creating xcvrs' )
         return
      sfp.newEntity( 'Tac::Dir', 'cell' )
      cell = sfp[ 'cell' ]
      sfpCellConfig = cell.newEntity( 'Xcvr::SfpCellConfig',
                                      str( Cell.cellId() ) )
      self.sfpXcvrConfig = sfpCellConfig.xcvrConfig

   def handleZeroTouchConfig( self, driverCtx ):
      # We handle two separate items from /mnt/flash/zerotouch-config:
      # 1. Any ZEROTOUCH_ or SZTP_ variables: we insert these into the
      #    environment of the ZeroTouch process
      # 2. SYNTHETIC_*: if any of these are set, we create an on-boot
      #    task to run zerotouch-dhclient-script with these values.
      if ztpConfigForTest:
         zeroTouchConfig = ztpConfigForTest
      else:
         zeroTouchConfigPath = flashFileUrl( "zerotouch-config" )
         urlContext = Url.Context( None, disableAaa=True )
         zeroTouchConfigUrl = Url.parseUrl( zeroTouchConfigPath, urlContext )
         try:
            zeroTouchConfig = SimpleConfigFile.SimpleConfigFileDict(
               zeroTouchConfigUrl.localFilename() )
         except ( IOError, SimpleConfigFile.ParseError ) as e:
            # No config file, or error in config file
            t0( 'No zerotouch:', e )
            return
      t1( 'zerotouch config =', zeroTouchConfig )
      agentEnv = {}
      for k in zeroTouchConfig:
         if k.startswith( ( 'SZTP_', 'ZEROTOUCH_' ) ):
            agentEnv[ k ] = zeroTouchConfig[ k ]
      if agentEnv:
         t0( "Configure agent environment", agentEnv )
         envConfig = " ".join( "%s=%s" % ( k, v )
                               for ( k, v ) in agentEnv.items() )
         driverCtx.entity( 'agent/config' ).newAgentGlobalConfig(
                              'ZeroTouch' ).environment = envConfig
      syntheticEnv = {}
      for k in zeroTouchConfig:
         if k.startswith( 'SYNTHETIC_' ):
            syntheticEnv[ k.replace( 'SYNTHETIC_', '' ) ] = zeroTouchConfig[ k ]
      if syntheticEnv:
         t0( "Configure on-logging event with env", syntheticEnv )
         env = " ".join( shquote( "%s=%s" % ( k, v ) )
                         for k, v in syntheticEnv.items() )
         cmd = "env " + env + " /usr/bin/zerotouch-dhclient-script"
         t0( "Configure on-logging event with command", cmd )
         eventConfig = driverCtx.entity( 'sys/event/config' )
         # We create an event handler that claims to receive the
         # synthetic response every time ZTP logs that it's sending
         # a DHCPv4_QUERY.  (This works for IPv6 too since this log
         # message just indicates the beginning of the DHCP sequence)
         params = {
               'triggerType': 'on-logging',
               'operstatus': False,
               'ip': False,
               'ip6': False,
               'intfName': '',
               'actionKind': 'bash',
               'command': cmd,
               'delay': 1,      # Delay after it says it sent the request
               'repeatInterval': 2, # If sending multiple requests, don't run
                                    # multiple times
               'maxRepeatActionCount': eventConfig.defaultMaxRepeatActionCount,
               'asynchronous': False,
               'timeout': eventConfig.defaultTimeout,
               'thresholdCount': eventConfig.defaultThresholdCount,
               'threshold': eventConfig.defaultThreshold,
               'pollInterval': eventConfig.defaultPollInterval,
               'countersCondition': eventConfig.defaultCountersCondition,
               'logRegex': 'ZTP-6-DHCPv4_QUERY',
               'maintenanceUnitName': eventConfig.defaultMaintenanceUnitName,
               'maintenanceOper': eventConfig.defaultMaintenanceOper,
               'maintenanceStage': eventConfig.defaultMaintenanceStage,
               'maintenanceBgpPeer': eventConfig.defaultMaintenanceBgpPeer,
               'vrfName': eventConfig.defaultVrfName,
               'metricName': eventConfig.defaultMetricName,
               'hasSubhandlers': True,
               'runUponConfig': eventConfig.defaultRunUponConfig,
            }
         EventLib.builtinEventHandlerIs( eventConfig,
                                         "ZeroTouchSyntheticResult",
                                         params )

   def getValidDevices( self ):
      if self.myIntfType:
         self.validDevicePfxs.add( self.myIntfType )
      # Eventually we filter out all ethernet devices except for those if type
      # INTFTYPE, but we have to include them all for now in case they are mapped
      # to a management interface.
      return self.getDevices( self.validDevicePfxs )

   def parseKernelDevices( self, devNames, config ):
      devConfig = []
      intfConfig = JsonIntfConfig()
      nameMapping = intfConfig.getNameMapping()
      if nameMapping is None:
         for devName in devNames:
            labels = self.getLabels( devName )
            if labels is None:
               # Can't get labels, unsupported format (ex. et2.4096 kernel subintf)
               continue
            devType, label, subLabel, subSubLabel = labels
            intfType = self.intfTypeFromDevType( devType, config,
                                                 self.validDevicePfxs - { 'ma' } )
            if self.mgmtIntf and devName == self.mgmtIntf:
               # Ethernet device mapped management interface
               intfType = IntfType.management
            elif intfType is IntfType.management:
               if not self.maIntfAllowed:
                  continue
            elif not self.myIntfType or not devType == self.myIntfType:
               # Filter out ethernet devices not mapped to management interfaces or
               # of the form specified by INTFTYPE
               continue
            devConfig.append( ( devName, intfType, label, subLabel, subSubLabel ) )
      else:
         for devName in devNames:
            desiredIntfName = nameMapping.get( devName, None )
            if not desiredIntfName:
               # Device isn't mapped, just ignore.
               t0( devName, ' does not have a mapping, ignoring' )
               continue
            labels = self.getLabels( desiredIntfName )
            if labels is None:
               # Can't get labels, malformed desired interface name
               continue
            intfPfx, label, subLabel, subSubLabel = labels
            intfType = self.intfTypeFromDesiredIntfPrefix( intfPfx, config )
            devConfig.append( ( devName, intfType, label, subLabel, subSubLabel ) )

      # If there are configured MAC addresses, apply them now.
      for ( devName, _, _, _, _ ) in devConfig:
         mac = config.ceosConfig.get( '%s_MACADDR' % devName.upper() )
         if mac is not None:
            try:
               t0( "Setting MAC address of", devName, "to", mac )
               Arnet.Device.Device( devName, hw=mac )
            except Exception as e: # pylint: disable-msg=broad-except
               t0( "Got exception", e )
               try:
                  t0( "Making sure", devName, "is up after failure" )
                  # It's just a side effect of instantiating the Arnet.Device
                  Arnet.Device.Device( devName )
               except Exception as e2: # pylint: disable-msg=broad-except
                  t0( "Got exception", e2 )
      return devConfig

   def handleDevice( self, devName, intfType, label, subLabel, subSubLabel,
                     mac, portId, etbaConfig, ethPortDir, pciPortDir,
                     phyEthtoolPhyDir ):
      t0( 'handleDevice: ', devName, intfType, label, subLabel, subSubLabel )
      if intfType is IntfType.management:
         if label in self.mapping and self.mgmtIntf:
            # prefer MGMT_INTF over interface 'maX', i.e, when MGMT_INTF env
            # variable is set to 'eth0' and 'ma0' interface is also present
            # then use 'eth0'.
            if self.mapping[ label ] != self.mgmtIntf:
               # Delete conflicting interface that we already created
               existingPhyName = "Phy%s" % ( self.mapping[ label ] )
               existingPhyEnt = phyEthtoolPhyDir.phy.get( existingPhyName )
               existingPort = existingPhyEnt.port
               del phyEthtoolPhyDir.phy[ existingPhyName ]
               del pciPortDir.port[ existingPort.id ]
               del existingPort
               del existingPhyEnt
            else:
               # We already created the one we desired, skip the conflicting one
               return

         assert not subSubLabel
         port = pciPortDir.newPort( portId )
         port.description = "Management Ethernet Port %d" % label
         port.role = "Management"
         # Don't set a pci address, the EthPciPort fru driver will assign
         # things correctly based on the device name

         # Create an Inventory::PhyEthtoolPhy
         phy = phyEthtoolPhyDir.newPhy( "Phy%s" % ( devName ) )
         phy.port = port
         self.mapping[ label ] = devName

      elif intfType is IntfType.ethernet:
         # If Ethernet510 is being used, report conflict and terminate the test
         if label == 510 and self.mapEth0Intf:
            assert False, "eth0 is conflict with %s" % devName

         # eth0 interface on the host is not avaliable in EOS for configurations
         # Rename eth0 to unused Ethernet510 in order to manipulate the interface
         if not label and self.mapEth0Intf:
            label = 510

         # The upper limit of ethernet interface portID is 511
         # If portID exceed 511, it is a unvalid port
         # The lower limit is 1, label can not be zero
         if label > 511 or label == 0:
            return

         # Create an Inventory::VirtualEthPort
         port = ethPortDir.newPort( portId )
         port.description = "Front-Panel Port %d" % label
         port.role = "Switched"
         tapDevice = Tac.Value( "Bridging::Etba::Config::TapDevice" )
         tapDevice.name = devName
         tapDevice.fileno = 0
         hardwarePort = "Ethernet%d" % label
         intfName = "Ethernet%d" % label
         if subSubLabel:
            intfName += "/%d/%d" % ( subLabel, subSubLabel )
            hardwarePort += "/%d" % subLabel
         elif subLabel:
            intfName += "/%d" % subLabel

         etbaConfig.extTapDevice[ intfName ] = tapDevice

         # When we have 2 level interface (names), we assume that the associated
         # "hardware" is up to the 1st level else just the base interface.
         # Let's take an example: Linecard 5, port 35. If this has 10G and maybe
         # can be broken into 1Gx10, then you will have interfaces like Et5/35/1
         # Et5/35/2 and so on. The "hardware port" in this case is Et5/35.
         # However, if it is not broken out, then it is "Et5/35" as well.
         # But if it's just a single interface (fixed config box) then it is Et5

         # However, we don't understand breakouts in ceos-lab. We are simply
         # modeling the hardware port based on some hard coded logic. In our logic
         # if the interface had a sub-sub-level, then the hardware-port is up to
         # the sublevel or else simply one level up (which in the above example
         # would be "Et5")

         # So this is keyed by the hardware port. The actual config
         # has a "intfName" which has the actual intf name elements.
         # for example on a fixed config box, with "Ethernet1" as the
         # hardware port, we will see:
         # /ar/Sysdb/hardware/xcvr/config/qsfp/cell/1/xcvrConfig/Ethernet1
         # and then underneath it intfName with n entries:
         # intfName/0 is 'Ethernet1/1', intfName/1 is 'Ethernet1/2' and so on.
         # All of this is based on my understanding of what we see on physical
         # devices.
         sfpXcvrConfig = self.sfpXcvrConfig
         if sfpXcvrConfig is not None:
            ethSfpConfig = sfpXcvrConfig.newMember( hardwarePort )
            ethSfpConfig.intfName[ len( ethSfpConfig.intfName ) ] = intfName

      else:
         assert False, "Unknown port %s" % devName

      port.macAddr = mac
      port.label = label
      if subLabel:
         port.subLabel = subLabel
      if subSubLabel:
         port.subSubLabel = subSubLabel

   def setDeviceName( self, phyEthtoolPhyDir ):
      t0( 'Setting deviceName in PhyEthtool' )
      # For all front-panel interfaces managed by PhyEthtool,
      # we have to set the deviceName on the EthIntfStatus
      # ourselves (just like we do in Fru for the management
      # interfaces).
      if phyEthtoolPhyDir:
         for invPhy in phyEthtoolPhyDir.phy.values():
            intfStatus = invPhy.port.intfStatus
            if not intfStatus.deviceName:
               intfStatus.deviceName = self.mapping[ invPhy.port.label ]

   def setBridging( self, etbaConfig, linuxConfig, config ):
      # In cEOS-lab, we largely use environment variables instead of
      # config like we use in vEOS-lab.  In this case, the ETBA
      # environment variable is used instead of config.getMode == 'test'.
      etbaConfig.complete = self.etba not in ( None, '0' )
      linuxConfig.enabled = not etbaConfig.complete

   def setHostname( self, netConfig ):
      # Docker and k8s both set the container hostname
      hostname = socket.gethostname()
      if not hostname.startswith( 'localhost' ) and not netConfig.hostname:
         netConfig.hostname = hostname

def containerFactory( parent, fdl, idInParent ):
   assert idInParent is None
   cEOSLab = parent.newEntity( "Eos::CEOSLabFru", "cEOSLab" )
   cEOSLab.component = ( "component", )
   cEOSLab.api = Ark.getPlatform() or ""
   cEOSLab.managingCellId = Cell.cellId()
   cEOSLab.valid = True
   cEOSLab.generationId = 1
   return cEOSLab

def Plugin( context ):
   context.registerDriver( cEOSLabDriver )
   context.registerFruFactory( containerFactory, Fru.containerFactoryId )
   mg = context.entityManager.mountGroup()
   mg.mount( bridgingEtbaConfigPath, bridgingEtbaConfigType, 'w' )
   mg.mount( 'bridging/linux/config', 'Bridging::Linux::Config', 'w' )
   mg.mount( 'hardware/xcvr/config', 'Tac::Dir', 'wi' )
   # to start ZeroTouch with the SZTP_MANUF_CERT environment
   mg.mount( 'agent/config', 'Agent::GlobalConfigDir', 'w' )
   # to create an on-boot event handler
   mg.mount( 'sys/event/config', 'Event::Config', 'w' )
   mg.close( None )
