#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import os
import re
import requests
import shutil
import ArPyUtils.Decorators
import Tac
import EosCloudInitLib as ecil

configs = {}

METADATA_KEY_VAL_RE = re.compile( r'([^:]+):(.*)' )
instanceIdUrl = "http://%s/computeMetadata/v1/instance/id" % ecil.metadataIp
instanceIdFile = Tac.Type( "CloudUtils::CloudAttr" ).instanceIdFile

def parseMetadata( metadata ): # pylint: disable=inconsistent-return-statements
   m = METADATA_KEY_VAL_RE.match( metadata )
   try:
      groups = m.groups()
      return ( groups[ 0 ], groups[ 1 ] )
   # pylint: disable-msg=W0702
   except:
      ecil.cleanupExit( 'Could not parse key value from metadata: %s' % metadata,
                        rc=1 )

def getMetadata( path, source='instance', attempts=10, forceFail=True ):
   @ArPyUtils.Decorators.retry( retryCheckFunc=lambda response: not response.ok,
                                attempts=attempts, retryInterval=5 )
   def f():
      return requests.get(
         f'http://{ecil.metadataIp}/computeMetadata/v1/{source}/{path}',
         headers={ 'Metadata-Flavor': 'Google' } )

   response = f()
   if response.ok:
      text = response.text
      if not isinstance( text, str ):
         text = str( text, 'utf8', errors='replace' )
      return text
   elif forceFail:
      ecil.cleanupExit( 'Could not get metadata response for path: %s' % path, rc=1 )
   return None

def handleUserData():
   userData = ( getMetadata( 'attributes/user-data', attempts=1, forceFail=False )
                or ecil.emptyUserData )
   ecil.writeFile( ecil.userData, userData )

def handleSshKeys():
   configLines = []
   users = []

   instanceSshKeys = getMetadata( 'attributes/ssh-keys', source='instance',
                                  attempts=2, forceFail=False ) or ''

   # grab project ssh-keys if they're not blocked
   blockProjectSshKeysMetadata = getMetadata( 'attributes/block-project-ssh-keys',
                                              attempts=1, forceFail=False )
   blockProjectSshKeys = blockProjectSshKeysMetadata == 'true'

   projectSshKeys = ''
   if not blockProjectSshKeys:
      projectSshKeys = getMetadata( 'attributes/ssh-keys', source='project',
                                    attempts=2, forceFail=False ) or ''

   if not instanceSshKeys and not projectSshKeys:
      if blockProjectSshKeys:
         ecil.cleanupExit( 'No instance ssh-keys were present, '
                           'and project ssh-keys were blocked', rc=1 )
      else:
         ecil.cleanupExit( 'Neither instance nor project ssh-keys were present',
                           rc=1 )

   for sshKeys in ( instanceSshKeys, projectSshKeys ):
      for sshKey in sshKeys.splitlines():
         userName, key = parseMetadata( sshKey )
         if userName in users:
            continue
         formattedKey = ' '.join( key.split( ' ' )[ :2 ] )
         if 'ssh-rsa' not in formattedKey:
            continue

         keyName = '%s_key.pub' % userName
         keyPath = os.path.join( '/mnt/flash', keyName )
         ecil.writeFile( keyPath, '%s\n' % formattedKey )

         configLines.extend(
            ecil.generateUserConfig( userName, None, keyName=keyName ) )

         users.append( userName )

   return configLines

def handleHostname():
   configLines = []

   hostname = getMetadata( 'hostname' )
   configLines.append( 'hostname %s\n' % hostname )

   return configLines

def cloudInit():
   ecil.setupConnection( ping=False )
   instanceId, rc = ecil.runCmd( [ "curl", "-H", "Metadata-Flavor: Google",
                                   instanceIdUrl ] )
   if rc != 0:
      ecil.cleanupExit( "Could not get the instanceId. Skipping downloading " +
                        "user data. RC %d" % rc, rc )
   try:
      with open( instanceIdFile, "w" ) as fd:
         fd.write( instanceId )
   except OSError:
      ecil.cleanupExit( "Failed to write instance id file" )

   handleUserData()

   if ecil.firstTimeBoot() or ecil.isForcedUserData():
      configLines = []

      configLines.extend( handleSshKeys() )
      configLines.extend( handleHostname() )

      configLines = [ line for line in configLines if line ]
      configLines.append( 'ip name-server %s\n' % ecil.getDnsIp() )

      # in GCP image the default startup config is stored in kickstart
      # file, so need to copy over
      shutil.copyfile( ecil.kickstartConfig, ecil.defaultStartupConfig )

      ecil.processUserData( configs, extraConfig=configLines )
