# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import pwd

import CliSave
import GitLib
import Tac
import TacUtils
import Tracing

t0 = Tracing.trace0

def saveRunningConfigToGit( entityManager, subject, author=GitLib.UNKNOWN_GIT_USER,
      trailers=None, allowEmptyCommit=False, maxHistorySize=None, tag=None ):

   sysname = entityManager.sysname()
   runningConfigFilename = os.path.join( GitLib.getGitRepoPath( sysname ),
         GitLib.CONFIG_FILE_NAME )
   try:
      with open( runningConfigFilename, 'w' ) as f:
         CliSave.saveRunningConfig( entityManager, f )
   except IOError:
      t0( 'unable to write', runningConfigFilename )
      return None

   return GitLib.gitCommit( sysname, subject, author, trailers=trailers,
         allowEmptyCommit=allowEmptyCommit, tag=tag, maxHistorySize=maxHistorySize )

def getCurrentUser( mode ):
   aaaUser = mode.session.aaaUser()
   return aaaUser.user \
         if aaaUser and aaaUser.user else pwd.getpwuid( os.getuid() )[ 0 ]

def getCommit( sysname, changeId ):
   for commit in GitLib.getGitCommits( sysname, trailerKeys=( 'Change-Id', ) ):
      if commit[ 'trailers' ][ 'Change-Id' ] == changeId:
         return commit
   return None

def savePreCommitRunningConfig( mode, sessionName, maxHistorySize=None ):
   return saveRunningConfigToGit( mode.entityManager,
         subject=f"running-config changes before session {sessionName}",
         trailers={ "Session-name": sessionName,
                    "Commit-type": "pre-commit" },
         maxHistorySize=maxHistorySize )

def savePostCommitRunningConfig( mode, sessionName, description=None,
      commitTimeExpiry=None, maxHistorySize=None, tagCommit=False ):
   description = description if description else ''
   trailers = {
      "Session-name": sessionName,
      "Commit-type": "post-commit",
      'Description': description,
      'Commit-time-expiry': commitTimeExpiry if commitTimeExpiry else 0
   }

   return saveRunningConfigToGit( mode.entityManager,
         subject=f"running-config changes after session {sessionName}",
         author=getCurrentUser( mode ),
         trailers=trailers,
         allowEmptyCommit=True,
         maxHistorySize=maxHistorySize,
         tag=f'session:{sessionName}' if tagCommit else None )
