# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import os
import re
import time
import uuid

import Tac
import TacUtils
import Tracing
import Url

t0 = Tracing.trace0

CONFIG_REPO_NAME = ".config-repo"
CONFIG_FILE_NAME = "running-config"
UNKNOWN_GIT_USER = "eos"
XDG_CONFIG_HOME_ROOT = "/var/run/git"

lastChangeId = None
commitHistorySize = None

def getGitRepoPath( sysname ):
   if Url.fsRoot() is None:
      return None
   return os.path.join( Url.fsRoot(), "flash", f'{CONFIG_REPO_NAME}-{sysname}' )

def getXdgConfigHome( sysname ):
   return os.path.join( XDG_CONFIG_HOME_ROOT, sysname )

def runGitCommand( sysname, cmd, **kwargs ):
   repo = getGitRepoPath( sysname )
   if repo is None:
      return None
   if not os.path.exists( os.path.join( repo, ".git" ) ):
      return None

   kwargs.setdefault( 'env', {
      'PATH': os.getenv( 'PATH', '' ),
      'XDG_CONFIG_HOME': getXdgConfigHome( sysname )
   } )
   kwargs.setdefault( 'stdout', Tac.CAPTURE )
   kwargs.setdefault( 'stderr', Tac.CAPTURE )
   kwargs.setdefault( 'cwd', repo )
   t0( 'running git cmd', cmd, 'with args', kwargs )
   try:
      output = Tac.run( cmd, **kwargs )
      t0( 'output of running cmd', cmd, 'is', output )
      return output
   except TacUtils.SystemCommandError as e:
      t0( 'Error seen while running cmd', cmd, 'with args', kwargs )
      t0( 'Error seen', e.error )
      t0( 'Error output', e.output )
      raise

def _isVfatFs():
   if Url.fsRoot() is None:
      return False
   try:
      Tac.run( [ 'df', '-t', 'vfat', os.path.join( Url.fsRoot(), "flash" ) ],
            stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      return True
   except TacUtils.SystemCommandError:
      return False

def maybeCreateGitRepo( sysname ):
   global commitHistorySize

   repo = getGitRepoPath( sysname )
   if repo is None:
      return

   if not os.path.exists( os.path.join( repo, ".git" ) ):
      try:
         Tac.run( [ 'mkdir', '-m', '777', '-p', repo ], asRoot=True )
         gitInitCmd = [ 'git', 'init', '-b', 'main' ]
         if not _isVfatFs():
            # on a vfat file system we can't change permissions. On a physical
            # EOS switch we use a vfat file system and should by default be g+w.
            # However on non-vfat file systems, such as veos, we need to set the
            # permissions to group
            gitInitCmd.append( '--shared=0666' )
         Tac.run( gitInitCmd, cwd=repo, stdout=Tac.CAPTURE,
               stderr=Tac.CAPTURE, asRoot=True )
         Tac.run( [ 'git', 'config', 'user.email', '' ], cwd=repo, asRoot=True )
         Tac.run( [ 'git', 'config', 'user.name', UNKNOWN_GIT_USER ],
               cwd=repo, asRoot=True )
         # Garbage collection is done manually whenever history is compacted
         Tac.run( [ 'git', 'config', 'gc.auto', '0' ], cwd=repo, asRoot=True )
         Tac.run( [ 'git', 'config', '--add', 'safe.directory', repo ],
               cwd=repo, asRoot=True )
      except TacUtils.SystemCommandError as e:
         t0( 'unable to instantiate git dir', e )
         return

   xdgConfigDir = os.path.join( getXdgConfigHome( sysname ), 'git' )
   gitConfigPath = os.path.join( xdgConfigDir, 'config' )
   if not os.path.exists( gitConfigPath ):
      try:
         Tac.run( [ 'mkdir', '-m', '777', '-p', xdgConfigDir ], asRoot=True )
      except TacUtils.SystemCommandError as e:
         t0( 'unable to create git xdgConfigDir', e )
         return
      try:
         Tac.run( [ 'touch', gitConfigPath ], umask=0 )
      except TacUtils.SystemCommandError as e:
         t0( 'unable to create gitConfigPath', e )
         return
      try:
         with open( gitConfigPath, 'w' ) as f:
            f.write( f"\n[safe]\n        directory = {repo}" )
      except IOError:
         t0( 'unable to write', gitConfigPath )

   runningConfigFilename = os.path.join( repo, CONFIG_FILE_NAME )
   if not os.path.exists( runningConfigFilename ):
      t0( 'running-config doesnt exist. creating it...' )
      # if running-config doesn't exist we add it to the repo and commit it
      try:
         Tac.run( [ 'touch', runningConfigFilename ], umask=0 )
      except TacUtils.SystemCommandError as e:
         t0( 'Unable to create running-config error:', e )

      try:
         runGitCommand( sysname, [ 'git', 'add', runningConfigFilename ] )
      except TacUtils.SystemCommandError as e:
         t0( 'Unable to add running-config error:', e )
         return

      commitHistorySize = 0 # new repo, start count at 0

      t0( 'checking in initial running-config' )
      gitCommit( sysname, 'initial checkin', UNKNOWN_GIT_USER,
            trailers={ 'Commit-type': 'creation' } )
   else:
      t0( 'running-config exists in the system' )

def _writeShallowFile( sysname, commitHash ):
   # write a shallow file with this commit hash to limit commit history
   repo = getGitRepoPath( sysname )
   if repo is None:
      return

   gitDir = os.path.join( repo, ".git" )
   if not os.path.exists( gitDir ):
      return

   shallowFilename = os.path.join( gitDir, 'shallow' )
   if not os.path.exists( shallowFilename ):
      # if shallow file doesn't exist create it, and change permissions
      try:
         Tac.run( [ 'touch', shallowFilename ], umask=0 )
      except TacUtils.SystemCommandError as e:
         t0( 'unable to create shallowFilename', e )
         return

   try:
      with open( shallowFilename, 'w' ) as f:
         f.write( commitHash )
   except IOError:
      t0( 'unable to write', shallowFilename )

def _deleteUnreferenceTags( sysname, commitHashes ):
   # tags may refer to commits that are no longer visible in the show logs because
   # of creating the shallow file. Remove tags to these otherwise unreachable commits
   # so that the commits can be cleaned up
   try:
      hashTags = runGitCommand( sysname, [ 'git', 'show-ref', '--tags' ] )
   except TacUtils.SystemCommandError as e:
      t0( 'unable to get git refs error', e )
      return

   tagsToDelete = []
   for line in hashTags.split( '\n' ):
      if not line:
         continue
      commitHash, refTag = line.split( ' ' )
      if commitHash in commitHashes:
         continue
      tagsToDelete.append( refTag.split( '/' )[ -1 ] )

   if not tagsToDelete:
      # avoid running the git tag delete command if there are no tags to delete
      return

   try:
      runGitCommand( sysname, [ 'git', 'tag', '-d' ] + tagsToDelete )
   except TacUtils.SystemCommandError as e:
      t0( 'unable to delete tags', tagsToDelete, 'with error', e )
      return

def _getGitRepoSize( sysname ):
   try:
      commitHashes = runGitCommand( sysname, [ 'git', 'log', '--format=%H' ] )
      return len( commitHashes.split( '\n' ) ) - 1
   except TacUtils.SystemCommandError as e:
      t0( 'unable to get git log error', e )
      return 0

def maybeSquashGitHistory( sysname, maxHistorySize ):
   global commitHistorySize
   if maxHistorySize is None:
      return

   if commitHistorySize is None:
      commitHistorySize = _getGitRepoSize( sysname )

   t0( 'maybeSquashGitHistory commitHistorySize', commitHistorySize,
      'requested maxHistorySize', maxHistorySize )
   if commitHistorySize <= maxHistorySize:
      t0( 'maybeSquashGitHistory not squashing history' )
      return

   try:
      commitHashes = runGitCommand( sysname, [ 'git', 'log', '--format=%H' ] )
   except TacUtils.SystemCommandError as e:
      t0( 'unable to get git log error', e )
      return

   commitHashes = commitHashes.split( '\n' )

   # compress the GIT repo to 80% of the size to prevent significant trashing
   if maxHistorySize > 2:
      newHistorySize = int( maxHistorySize * .8 )
   else:
      # if maxHistorySize <= 2, don't want it to get smaller
      newHistorySize = max( maxHistorySize, 1 )
   assert newHistorySize <= commitHistorySize
   t0( 'maybeSquashGitHistory squashing git history to', newHistorySize )
   _writeShallowFile( sysname, commitHashes[ newHistorySize - 1 ] )
   commitHistorySize = newHistorySize
   _deleteUnreferenceTags( sysname, commitHashes[ : newHistorySize ] )
   try:
      runGitCommand( sysname, [ 'git', 'gc', '--prune=1.day.ago' ] )
   except TacUtils.SystemCommandError as e:
      t0( 'unable to git gc', e )

def gitDiff( sysname, commitHash1, commitHash2=None ):
   try:
      t0( 'getting diff for hashs', commitHash1, commitHash2 )
      if commitHash2 is None:
         lines = runGitCommand( sysname, [ 'git', 'diff', f'{commitHash1}^!' ] )
         t0( 'diff for hash', commitHash1, lines )
      else:
         lines = runGitCommand( sysname,
               [ 'git', 'diff', commitHash2, commitHash1 ] )
      t0( 'diff for hash', commitHash1, commitHash2, lines )

   except TacUtils.SystemCommandError as e:
      t0( 'Error getting diff for hash', commitHash1, commitHash2, e )
      return None

   return '\n'.join( ( lines.split( '\n' )[ 4 : ] ) )

def gitShow( sysname, commitHash, filename ):
   commitHash = commitHash.replace( ':', '%3A' )
   try:
      t0( 'git show', commitHash )
      lines = runGitCommand( sysname, [ 'git', 'show', f'{commitHash}:{filename}' ] )
      t0( 'git show', commitHash, lines )

   except TacUtils.SystemCommandError as e:
      t0( 'Error git show for hash', commitHash, e )
      return None

   return '\n'.join( ( lines.split( '\n' ) ) )

def gitCatFileSize( sysname, commitHash, filename ):
   commitHash = commitHash.replace( ':', '%3A' )
   try:
      t0( 'git cat-file', commitHash )
      output = runGitCommand( sysname,
            [ 'git', 'cat-file', '-s', f'{commitHash}:{filename}' ] )
      t0( 'git cat-file', commitHash, output )
   except TacUtils.SystemCommandError as e:
      t0( 'Error git cat-file for hash', commitHash, e )
      return 0

   return int( output )

def _getLastChangeId( sysname ):
   previousCommit = getGitCommits( sysname,
         trailerKeys=( 'Change-Id', ),
         maxNumEntries=1 )
   if previousCommit:
      # return changeId of the current head
      return previousCommit[ 0 ][ 'trailers' ][ 'Change-Id' ]
   t0( 'Unable to determine last change in the system' )
   return None

def gitCommit( sysname, subject, author, trailers=None, allowEmptyCommit=False,
      tag=None, maxHistorySize=None ):
   global lastChangeId
   global commitHistorySize

   # git users must have an email in <>
   commitCmd = [ 'git', 'commit', '-m', subject, '-a', f"--author=\"{author} <>\"" ]

   # cli might want to commit even if it's empty
   if allowEmptyCommit:
      t0( 'empty commit allowed' )
      commitCmd.append( '--allow-empty' )

   if trailers is None:
      trailers = {}

   # add timestamp with nanoseconds precision
   trailers[ 'Timestamp-ns' ] = time.time_ns()

   assert 'Change-Id' not in trailers, 'Change-Id must not be present'
   trailers[ 'Change-Id' ] = str( uuid.uuid4() )
   for trailerName, trailerValue in trailers.items():
      t0( 'adding trailer', trailerName, ':', trailerValue )
      commitCmd.append( '--trailer' )
      # encode : since this is a reserved char in git trailers
      commitCmd.append(
            f'{trailerName}:{str( trailerValue ).replace( ":", "%3A" )}' )

   t0( 'git commit command', commitCmd )
   try:
      runGitCommand( sysname, commitCmd, umask=0 )
   except TacUtils.SystemCommandError as e:
      t0( 'Unable to commit running-config error:', e )
      if 'nothing to commit' in e.output:
         # nothing changed, this means use the last change ID
         if lastChangeId is None:
            lastChangeId = _getLastChangeId( sysname )
         return lastChangeId
      return None

   if commitHistorySize is None:
      commitHistorySize = _getGitRepoSize( sysname )
   else:
      commitHistorySize += 1
   lastChangeId = trailers[ 'Change-Id' ]
   if tag:
      gitTag( sysname, tag )

   maybeSquashGitHistory( sysname, maxHistorySize )
   return lastChangeId

def gitTag( sysname, tag, commitHash=None, force=True ):
   cmd = [ 'git', 'tag' ]
   tag = tag.replace( ':', '%3A' ) # encode : as this isn't a valid tag char
   if force:
      # overwrite if already present
      cmd.append( '-f' )
   cmd.append( tag )
   if commitHash:
      # tag the commit with the requested tag, otherwise last commit
      cmd.append( commitHash )
   try:
      runGitCommand( sysname, cmd, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
   except TacUtils.SystemCommandError as e:
      t0( 'Unable to add tag error:', e )

def getGitCommits( sysname, trailerKeys=None, maxNumEntries=None, commitHash=None ):
   commitHash = commitHash.replace( ':', '%3A' ) if commitHash else None
   trailerOptions = []
   trailerOptions.append( "separator=%x2C" )
   if trailerKeys:
      for trailerKey in trailerKeys:
         trailerOptions.append( f'key={trailerKey}' )

   trailerOptionStr = ','.join( trailerOptions )

   if commitHash is None:
      cmd = [ 'git', 'log' ]
      if maxNumEntries:
         cmd.append( '-n' )
         cmd.append( f'{maxNumEntries}' )
   else:
      cmd = [ 'git', 'log', f'{commitHash}', '-n', '1' ]
   cmd.append( f'--format='
      f'{{'
      f'   "commitHash":"%h",'
      f'   "author":"%an",'
      f'   "commiter":"%cn",'
      f'   "commitTime":%at,'
      f'   "subject":"%s",'
      f'   "trailers": "%(trailers:{trailerOptionStr})",'
      f'   "tags": "%d"'
      f'}}'
   )
   t0( 'git log command', cmd )
   try:
      commitLines = runGitCommand( sysname, cmd )
   except TacUtils.SystemCommandError as e:
      t0( 'Unable to parse git log', e )
      return []

   if commitLines is None:
      t0( 'Unable to parse git log' )
      return []

   commits = []
   for commitLine in commitLines.split( '\n' ):
      if not commitLine.strip():
         continue
      t0( repr( commitLine ) )
      try:
         commitInfo = json.loads( commitLine.strip() )
      except:
         t0( 'exception!!!!', repr( commitLine ) )
         raise
      t0( commitInfo )
      # squash commits don't have trailers, skip those
      if not commitInfo[ 'trailers' ]:
         continue
      parsedTrailers = { trailer.split( ':' )[ 0 ]: trailer.split( ':' )[ 1 ].strip()
            for trailer in commitInfo[ 'trailers' ].split( ',' ) }
      commitInfo[ 'trailers' ] = parsedTrailers
      for parsedTrailer in parsedTrailers:
         parsedTrailers[ parsedTrailer ] = parsedTrailers[ parsedTrailer ].replace(
               '%3A', ':' )

      # fix up tags to be a list of tags
      commitInfo[ 'tags' ] = commitInfo[ 'tags' ].strip()
      if commitInfo[ 'tags' ]:
         commitInfo[ 'tags' ] = re.findall( r'tag: ([^,)]+)', commitInfo[ 'tags' ] )
         commitInfo[ 'tags' ] = [ tag.replace( '%3A', ':' )
               for tag in commitInfo[ 'tags' ] ]
      else:
         commitInfo[ 'tags' ] = []
      t0( commitInfo )
      commits.append( commitInfo )

   return commits
