# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AirStreamLib
import CliSession
import Tac
import Tracing
from abc import ABCMeta, abstractmethod

t0 = Tracing.trace0
t5 = Tracing.trace5
t9 = Tracing.trace9

# This implements a registration mechanism that allows the plugins to register
# a callback function before the current session is committed.
#
# The typical usage is to do one-time-sync from external entity to internal entity
# before the session commit.
# For now, the preCommitHandler is only run when `config session type gnmi` is set.
#
# The handler must derive from PreCommitHandler class and defines:
# - metaDataAppKey : a static variable that captures the application key for metadata
# - externalPathList : a static variable listing all external paths.
#                      The handler is run if any of external paths are modified.
# - nativePathList : a static variable listing all native paths modified by the
#                    handler.
# - run( cls, sessionName ) : a class method that run the handler logic.
#           It takes two arguments:
#               * cls - the class name used for raising exception
#               * sessionName - the session name used to access session entities
# - setMetaData( cls, sessionName, groupKey, data ) : a class method to set metaData
#           It takes four arguments:
#               * cls - the class name used for raising exception
#               * sessionName - the session name used to access session entities
#               * groupKey - the key for the metadata
#               * data - the metaData string
#
# Multiple handlers may tie to the same external path and vice versa.
# In all cases, any handler runs only once when session is committed.

class PreCommitHandler( metaclass=ABCMeta ):

   # metadata application key to be set if required
   metaDataAppKey = None

   # Variables to be set
   externalPathList = []
   nativePathList = []

   @classmethod
   @abstractmethod # TODO: unnecessary, since we never instantiate subclasses.
   def run( cls, sessionName ):
      raise NotImplementedError(
         "Child class of PreCommitHandler must implement run()" )

   @classmethod
   def setMetaData( cls, sessionName, groupKey, data ):
      assert cls.metaDataAppKey, "AppKey not set for this handler,"\
         " cannot set meataData without AppKey"
      handler = _metaDataHandler( sessionName )
      assert handler, "MetaDataHandler not initialized for this session"
      handler.setMetaData( groupKey, cls.metaDataAppKey, data )

# A dict mapping the external path to a set of handler classes tied to such path
preCommitHandlers = {}

# A dict of all the initHandlers running per session
metaDataHandlers = {}

def _metaDataHandler( sessionName ):
   return metaDataHandlers.get( sessionName, None )

def registerPreCommitHandler( handlerClass ):
   """ Adds the handlers to be run before commit."""
   assert all( not path.startswith( '/' ) for path in handlerClass.externalPathList
            ), "Remove prefix slash from externalPathList of " + \
            str( handlerClass ) + " : " + str( handlerClass.externalPathList )
   assert all( not path.startswith( '/' ) for path in handlerClass.nativePathList ),\
            "Remove prefix slash from nativePathlist of " + str( handlerClass ) + \
            str( handlerClass.nativePathList )
   for path in handlerClass.externalPathList:
      assert ( not path.endswith( '/' ) ) and ( "." not in path ), \
          "Error with external path: '" + path + "' "\
          "Path must not have . or end with slash. Traveling from root, it must "\
          "be the first non-Tac::Dir entity that is handled by the preCommitHandler"
      preCommitHandlers.setdefault( path, set() ).add( handlerClass )

def relativePath( path, sessionPath ):
   if not path.startswith( sessionPath ):
      raise Exception( "sessionPath: " + sessionPath + " does not exist in " + path )
   return path[ len( sessionPath ) : ]

def _runPreCommitHandlers( handlerClasses, em, sessionName, path,
                           processedHandlers ):
   t9( 'Running handlers for path ' + path )
   for handlerClass in handlerClasses:
      nativePathList = handlerClass.nativePathList
      if handlerClass in processedHandlers:
         t9( 'already run handler: ' + str( handlerClass ) )
         continue
      t9( f'running handler: {handlerClass} path: {path} '
          f'externalPathList {handlerClass.externalPathList} '
          f'nativePathList {handlerClass.nativePathList}' )
      # Add all external path to session root.
      # This is to cover the case where one syncher have multiple external paths
      # but OC only modify subset of it.  In such case, OC only call
      # 'session gnmi add path {PATH...}' only on the paths it directly modified.
      # Hence, only such paths were added to the session root. The syncher
      # requires all external paths to exist inside the session.
      # We add them all here.  This is idempotent if the path was already added.
      # Optimization can be done at 'gnmi add path' time by identifying the
      # matched syncher and add all external paths there. Keep it simple for now.
      for externalPath in handlerClass.externalPathList:
         CliSession.addSessionRoot( em, externalPath, sessionName )
      for nativePath in nativePathList:
         CliSession.addSessionRoot( em, nativePath, sessionName )
      handlerClass.run( sessionName )
      processedHandlers.add( handlerClass )

def _getHandlerClasses( state, newState=None ):
   if newState:
      state = newState
      dirtyPaths = state.path
   else:
      dirtyPaths = state.dirtyPath

   for path in dirtyPaths:
      if not newState:
         dirtyPath = relativePath( path, state.sessionPath )
      else:
         # newState already contains relative path
         dirtyPath = path
      if dirtyPath in preCommitHandlers:
         yield ( dirtyPath, preCommitHandlers[ dirtyPath ] )

def _createMetaDataHandler( em, sessionName ):
   # create the gnmiSetMetaDataConfig
   gnmiSetMetaDataConfig = Tac.newInstance( "AirStreamLib::GnmiSetMetaDataConfig" )
   # create the cliConfig
   cliConfigPath = 'cli/config'
   CliSession.addSessionRoot( em, cliConfigPath, sessionName )
   cliConfig = AirStreamLib.getSessionEntity( em, sessionName, cliConfigPath )
   # create the metaDataHandler here
   handler = Tac.newInstance(
      "AirStreamLib::GnmiSetMetaDataHandler", gnmiSetMetaDataConfig, cliConfig )
   # add the metaDataHandler to session
   metaDataHandlers[ sessionName ] = handler

def _finalizeMetaDataHandler( sessionName, processedHandlers ):
   handler = _metaDataHandler( sessionName )
   # set the dirty appKeys
   for preCommitHandler in processedHandlers:
      if preCommitHandler.metaDataAppKey:
         handler.gnmiSetMetaDataConfig.gnmiSetDirtyAppKeySet.add(
            preCommitHandler.metaDataAppKey )
   # call finalize of metaDataHandler
   handler.finalize()
   # cleanup the metaDataHandler for this session
   metaDataHandlers.pop( sessionName, None )

def runToNativeSyncers( em, sessionName, newState=None ):
   if not newState:
      t5( 'Run airstream gnmi set pre-commit handlers' )
      state = Tac.singleton( "AirStream::GnmiSetSessionState" )
      if not state.sessionName:
         t5( "Not a gnmi config session. Skip running ToNativeSyncers." )
         return
      elif sessionName != state.sessionName:
         # Aborted gNMI sessions don't reset the GnmiSetSessionState singleton. Check
         # for that by comparing the singleton's sessionName the current session.
         t5( "Not a gnmi config session, Skip running ToNativeSyncers. "
             "Aborted gNMI session detected, reset GnmiSetSessionState. " )
         resetGnmiSetState()
         return
   else:
      state = newState
   # Keep handlers that already run for the current commit.
   processedHandlers = set()

   _createMetaDataHandler( em, sessionName )

   # AirStream adds the first non-Tac::Dir from root to dirtyPath collection.
   # The preCommitHandler path is also the non-Tac::Dir so we can do exact
   # match here.
   for path, handlerClasses in _getHandlerClasses( state, newState ):
      _runPreCommitHandlers( handlerClasses, em, sessionName, path,
                             processedHandlers )
      # Some path may be modified but doesn't have corresponding preCommitHandler
      # This is a valid case when OpenConfig modifies native paths directly

   _finalizeMetaDataHandler( sessionName, processedHandlers )

# When gNMI updates are made for a very small set of attributes of a nested entity
# of a mounted entity. The pre-commit handler is an overhead in such case since
# every attribute of the external entity is synced with the native entity over
# GenericIf. In such cases, a pre-commit SM can be used which is run when a gNMI
# update is received for such external entity received. It is stopped when config
# session is complete. The SM should not sync native and external entities during
# init, which will defeat the purpose.
#
# GnmiSetCliSessionSm is a base class for all pre-commit SMs of OpenConfig features.
# Following is the expected method of implementation:
# 1. The child class should set 'entityPaths' to the mount paths of all entities that
#    the SM needs.
# 2. The run() method should be overridden. This is the method where the SM should be
#    instantiated. The SM can be one or more instances of
#    a. GenericReactor or other Tac.Notifiee class
#    b. Tac::Constrainer or other notifying class
#    NOTE: These SM classes SHOULD pass callBackNow=False, if it's python class
#          and for Tacc constrainer class, 'initially' or 'handleInitialized' method
#          SHOULD NOT be implemented
# 3. When the run() method is run, the entities (config roots) corresponding to the
#    mount paths inside the config session will be available in 'sessionEntities'
#    dict keyed by the mount paths, which can be used to instantiate the classes as
#    described in the previous point.
#
class GnmiSetCliSessionSm( metaclass=ABCMeta ):
   # Paths of all entities that the SM requires. An SM may only need read access to
   # an entity but that entity can be modified by other CLI in the same session.
   # So, even the read-only entites will be added to the session before SM is
   # instantiated.
   entityPaths = []
   # A dictionary of entityPaths mapped to the session entity objects. It'll be
   # populated by the PreCommitSmRunner after the SM is instantiated.
   sessionEntities = {}

   @abstractmethod
   def run( self ):
      assert False, "Deriving class should override this method"

   def stop( self ):
      # A derived class will have a choice of relying on python's garbage collection
      # of C++ SM object, if any.
      pass

   def __del__( self ):
      self.stop()

# Used to manage the pre-commit SMs viz., registering the SM class, instantiating
# the object, running it and stopping it.
class PreCommitSmRunner:
   featureSms = {}

   def registerSm( self, smClass ):
      assert smClass not in self.featureSms
      self.featureSms[ smClass ] = None

   def runSm( self, em, sessionName, path ):
      t9( "Check if an SM should be run for path:", path, "in session", sessionName )
      for smCls, smObj in self.featureSms.items():
         if path not in smCls.entityPaths:
            continue
         if smObj:
            t9( "SM is already running" )
            continue
         smObj = smCls()
         for p in smObj.entityPaths:
            CliSession.addSessionRoot( em, p, sessionName )
            smObj.sessionEntities[ p ] = \
                  AirStreamLib.getSessionEntity( em, sessionName, p )
         t9( "Starting instance of SM", smCls.__name__ )
         smObj.run()
         self.featureSms[ smCls ] = smObj

   def reset( self ):
      t5( "Reset PreCommitSm instances" )
      self.featureSms = dict.fromkeys( self.featureSms, None )

# A global singleton object that lives during the lifetime of ConfigAgent
SmRunner = PreCommitSmRunner()

def registerPreCommitSm( smClass ):
   t9( "registerPreCommitSm", smClass.__name__ )
   SmRunner.registerSm( smClass )

# pkgdeps: rpm AirStreamGnmiSet
def resetGnmiSetState( mode=None, sessionName=None ):
   currSessionName = sessionName or CliSession.cachedSessionName()
   t0( f'resetGnmiSetState {currSessionName}' )
   newState = Tac.singleton( "AirStream::GnmiSetState" )
   if newState.sessionName:
      t5( f"Reset GnmiSetState for session {newState.sessionName}. " )
      newState.reset()

   state = Tac.singleton( "AirStream::GnmiSetSessionState" )
   if state.sessionName:
      state.lastSessionName = state.sessionName
      t5( f"Reset GnmiSetSessionState for session {currSessionName}. "
          f"Most recent gnmi set session is {state.lastSessionName}." )
      state.reset()

   # Stop all preCommit state-machines
   SmRunner.reset()
