# Copyright (c) 2008-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#
# Cli extension module: handling hostname-or-IP-address tokens
#

import enum
import re
import socket

import Arnet
import BasicCliModes
import CliParser
import CliMatcher
import Tac

hostnameErrorMsg = "Invalid hostname. Host names must contain only alphanumeric \
characters, '.' and '-', and begin/end with an alphanumeric character."
_hostnameOrIpv4Regex = r'[^:|> ]+'
# pylint: disable-next=consider-using-f-string
_hostnameRegex = r'(%s|%s)' % ( _hostnameOrIpv4Regex, Arnet.Ip6AddrRe )

def validateHostname( hostname ):
   """Validates hostname as specified by RFC1123/RFC952 and returns True or False
    accordingly.From RFC
    <hostname> ::= <name>*["."<name>]
    <name>  ::= <let-or-digit>[*[<let-or-digit-or-hyphen>]<let-or-digit>]"""

   # hostnameRegex = '^([a-zA-Z0-9]([a-zA-Z0-9\-]*[a-zA-Z0-9])*\.)*'+\
   #                     '([a-zA-Z0-9]([a-zA-Z0-9\-]*[a-zA-Z0-9])*)$'

   # The above commented regex matches same pattern as below but suffers
   # from catastrophic backtracking problem.
   hostnameRegex = r'^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9-]*[a-zA-Z0-9])\.)*' + \
                   '([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9-]*[a-zA-Z0-9])$'
   x = re.match( hostnameRegex, hostname )
   # Technically even 200.200.200.200 is valid hostname. In fact fc14 allows this
   # as hostname. But obviously this doesn't make sense and we will prevent it.
   # In fact, we will prevent all the strings which inet_aton() think, are 
   # IPv4 addresses. This is consistent with IpAddrorHostnameRule as well.
   if not ( x and x.group() == hostname ):
      return False
   else:
      try:
         Arnet.IpAddress( hostname )
         return False
      except ValueError:
         return True
   return True

class Resolution( enum.IntEnum ):
   UNRESOLVED = 0
   RESOLVED = 1
   SKIPPED = 2

def resolveHostname( mode, ipAddrOrHostname, doWarn=True ):
   """Attempts to resolve the hostname and possibly warn about any errors.

   This is typically called when a config command handler tries to validate a
   hostname provided by the user.

   In non-interactive config sessions, it skips resolution because it can be
   slow or have false alarms (e.g., when loading startup-config) and returns
   Resolution.SKIPPED.  In these cases, the config command should accept the
   hostname.

   Otherwise, it returns Resolution.RESOLVED or Resolution.UNRESOLVED.
   """

   if ( isinstance( mode, BasicCliModes.ConfigModeBase ) and
        mode.session.skipConfigCheck() ):
      return Resolution.SKIPPED

   try:
      socket.getaddrinfo( ipAddrOrHostname, None )
   except socket.gaierror:
      if doWarn:
         mode.addWarning(
            # pylint: disable-next=consider-using-f-string
            "'%s' does not appear to be a valid IP address or hostname. "
            "Trying to use it anyway." % ipAddrOrHostname )
      return Resolution.UNRESOLVED
   return Resolution.RESOLVED

def ipAddrOrHostnameValueFunc( mode, match ):
   return handleIpAddrOrHostname( mode, match.group() )

def handleIpAddrOrHostname( mode, ipAddrOrHostname ):
   """A value function for use in IpAddrOrHostnameRule: canonicalizes an IP address 
   if it receives one."""
   if ':' in ipAddrOrHostname:
      # It's an IPv6 address, make sure it's valid
      try:
         socket.getaddrinfo( ipAddrOrHostname, None )
      except socket.gaierror:
         mode.addError( "Invalid IPv6 address" )
         raise CliParser.AlreadyHandledError() # pylint: disable=raise-missing-from
   else:
      # It turns out that inet_aton() has some bizarre corner cases. For example,
      # it happily accepts the string "1.1" and converts it into the IP address
      # "1.0.0.1". Therefore, we try to do this conversion here before saving the
      # raw string, so we convert these weirdo (but apparently valid??) strings
      # into "canonical" IP addresses before we save them in the config.
      try:
         addr = Arnet.IpAddress( ipAddrOrHostname )
         ipAddrOrHostname = str( addr )
      except ValueError:
         # It wasn't a valid IPv4 address, even by inet_aton()' standards,
         # so assume it's a hostname.
         if not validateHostname( ipAddrOrHostname ): 
            mode.addError( hostnameErrorMsg )
            # pylint: disable-next=raise-missing-from
            raise CliParser.AlreadyHandledError()
   return ipAddrOrHostname

class IpAddrOrHostnameMatcher( CliMatcher.PatternMatcher ):
   def __init__( self, ipv6=False, **kargs ):
      kargs.setdefault( 'value', ipAddrOrHostnameValueFunc )
      kargs.setdefault( 'helpname', 'WORD' )
      if 'helpdesc' not in kargs:
         helpdesc = "Hostname or A.B.C.D"
         if ipv6:
            helpdesc += " or A:B:C:D:E:F:G:H"
         kargs[ 'helpdesc' ] = helpdesc
      pattern = _hostnameRegex if ipv6 else _hostnameOrIpv4Regex
      CliMatcher.PatternMatcher.__init__( self, pattern, rawResult=True, **kargs )

def hostnameValueFunc( mode, match ):
   """A value function for use in HostnameRule: Validates the hostname."""
   if not validateHostname( match.group() ):
      mode.addError( hostnameErrorMsg )
      raise CliParser.AlreadyHandledError()
   return match.group()

class HostnameMatcher( CliMatcher.PatternMatcher ):
   """Rule that captures a hostname. The default value function validates
   the passed string for correct syntax"""
   def __init__( self, value=None, **kargs ):
      value = value or hostnameValueFunc
      CliMatcher.PatternMatcher.__init__( self, r'\S+', value=value,
                                          rawResult=True, **kargs )
