#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import OrderedDict
from CliCommon import ResponseFormats # pylint: disable=unused-import
from PyArCrypto import OPENSSL_TO_RFC_NAMES # pylint: disable=no-name-in-module
import DefaultSslLib
import Tac

SslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

class ServerConstants:
   """ General constants for the HTTP server. """
   
   # CORS constants
   ALLOW_ALL_CROSS_ORIGIN_DOMAINS = "all"
   ALLOW_ALL_CROSS_ORIGIN_DOMAINS_STAR = "*"
   CORS_PREFLIGHT_CACHE_TIME = 600
   CORS_EXPOSE_HEADERS = ( "Cache-control", "Pragma", "Location", "WWW-Authenticate",
                          "Content-type", "Content-length", "Connection" )
   CORS_ALLOW_HEADERS = ( "Authorization", "Content-Type", "Accept", "Origin",
                         "User-Agent", "DNT", "Cache-Control", "X-Mx-ReqToken",
                         "Keep-Alive", "X-Requested-With", "If-Modified-Since" )

   # NGINX constants
   NGINX_CONF_FILE = "/etc/nginx/nginx.conf"
   NGINX_LOG_FILE = "/var/log/nginx-access.log"
   NGINX_ERROR_LOG_FILE = "/var/log/nginx-error.log"
   NGINX_STATUS_UDS = "/var/run/nginx_stub_status_%s.sock"
   NGINX_STATUS_ENDPOINT = "http://localhost/nginx_status"
   
   def nginxSecurityHeaders(self, space=7): 
      return ("# Other security headers\n" +
             space*' ' + 'add_header X-XSS-Protection "1; mode=block" always;\n' +
             space*' ' + 'add_header Strict-Transport-Security "max-age=31536000;' +
             'includeSubDomains" always;')

   # SSL/TLS constants
   BASE_SSL_FILE_DIR = "/persist/secure/"
   SSL_CERT_FILE = "capi.pem"
   SSL_KEY_FILE = "capikey.pem"
   SSL_CERT_DAYS = 30000

   # Server Error Messages
   CAPI_VRF_ERROR = 'VRF %r is either not defined or not active.'
   INVALID_SSL_KEY = 'SSL key error: Invalid Private Key'
   INVALID_SSL_CERT = 'SSL certificate error: Invalid Certificate'
   INVALID_SSL_CERT_KEY = ( 'SSL certificate/key error: Certificate and Private '
                            'Key do not match' )

   # The default TLS version is different on different platform 
   # [ 1.0, 1.1, 1.2 ] on EL7
   # [ 1.0, 1.1, 1.2, 1.3 ] on EL9
   DEFAULT_TLS_PROTOCOLS = DefaultSslLib.tlsVersionMaskToStrList(
                                          SslConstants.allTlsVersion ) 

   serviceName = "httpServer"

class CapiCipherSuites:
   CIPHERS = OrderedDict ( [
      ( "aes256-cbc",
         { "helpdesc" : "Advanced Encryption Standard (256 bit, CBC mode)",
           "filter" : "AES256" } ),
      ( "aes128-cbc",
         { "helpdesc" : "Advanced Encryption Standard (128 bit, CBC mode)",
           "filter" : "AES128" } )
   ] )

   KEYEXCHANGE = OrderedDict( [
      ( "rsa", { "helpdesc" : "RSA certificate", "filter" : "" } ),
      ( "diffie-hellman-ephemeral-rsa",
         { "helpdesc" : "Diffie Hellman paramers signed by RSA certificate",
           "filter" : "DHE-RSA" } )
   ] )

   MAC = OrderedDict( [
      ( "hmac-sha1",
         { "helpdesc" : "Hash Message Authentication Code SHA-1",
           "filter" : "SHA" } )
   ] )

   cipherSuiteFilterToNameMap = OPENSSL_TO_RFC_NAMES
