#!/usr/bin/env python3
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-from-import

'''
Generate valid ID EEPROM contents against which we can run some Xcvr
breadth tests, and which we can use on a real switch to simulate the
presence of such transceivers.

Templates for the ID EEPROM contents are available in the Sol package,
under manufacturing/XcvrIdprom/...
'''

import datetime
import random
import Tac
import XcvrLib
import IdpromTemplates.Acacia as Acacia
import IdpromTemplates.Blanco as Blanco
import IdpromTemplates.Cfp2 as Cfp2
import IdpromTemplates.Dsfp as Dsfp
import IdpromTemplates.Osfp as Osfp
import IdpromTemplates.Qsfp as Qsfp
import IdpromTemplates.QsfpCmis as QsfpCmis
import IdpromTemplates.QsfpDd as QsfpDd
import IdpromTemplates.Sfp as Sfp
import IdpromTemplates.SfpDd as SfpDd

idPromLen = 128
emptyEprom = [ 0x0 ] * idPromLen

# The revision number is defined as an Enum in /Xcvr/HalSff8436.tac
Sff8636RevEnum = Tac.Type( "Xcvr::Sff8636Revisions" )
sff8636rev15 = Tac.enumValue( Sff8636RevEnum, Sff8636RevEnum.sff8636rev15 )
sff8636rev20 = Tac.enumValue( Sff8636RevEnum, Sff8636RevEnum.sff8636rev20 )
sff8636rev28 = Tac.enumValue( Sff8636RevEnum, Sff8636RevEnum.sff8636rev28 )

# This is for QSFP+.  This page reflects only the page 00 upper page
# (address 128-255) as this is the only part we look at for basic
# id eeprom information. The lower page is control and status and
# the upper page, provided here, is the static information.
qsfpVendorNameOffset = 148 - 128
qsfpVendorNameLength = 16
qsfpVendorPartNumOffset = 168 - 128
qsfpVendorPartNumLength = 16
qsfpVendorRevNumOffset = 184 - 128
qsfpVendorRevNumLength = 2
qsfpSerialNumOffset = 196 - 128
qsfpSerialNumLength = 16
qsfpDateCodeOffset = 212 - 128
qsfpDateCodeLength = 8
qsfpBaseChecksumOffset = 191 - 128
qsfpExtChecksumOffset = 223 - 128

sff8436VendorInfoDefs = {
   'vendorName': ( qsfpVendorNameOffset, qsfpVendorNameLength ),
   'vendorPN': ( qsfpVendorPartNumOffset, qsfpVendorPartNumLength ),
   'vendorRev': ( qsfpVendorRevNumOffset, qsfpVendorRevNumLength ),
   'vendorSN': ( qsfpSerialNumOffset, qsfpSerialNumLength ),
   'vendorDateCode': ( qsfpDateCodeOffset, qsfpDateCodeLength ) }

# trailing spaces in these names are required to meet the 16 byte
# field size.
eepromTemplates = {
   "empty": ( "sfp", emptyEprom, None ),
   "BoboSRv1001": ( "sfp", Sfp.SRv1001SFP, "BoboCorp        " ),
   "BoboSRv1001nondom": ( "sfp", Sfp.SRv1001nondomSFP, "BoboCorp        " ),
   "ArastraSRv1001": ( "sfp", Sfp.SRv1001SFP, "Arastra, Inc.   " ),
   "AristaSRv1001": ( "sfp", Sfp.SRv1001SFP, "Arista Networks " ),
   "AristaSRv1002": ( "sfp", Sfp.SRv1002SFP, "Arista Networks " ),
   "AristaSRv1003": ( "sfp", Sfp.SRv1003SFP, "Arista Networks " ),
   "AristaSRNSRBv1003": ( "sfp", Sfp.SRv1003NOSOLREGBASESFP, "Arista Networks " ),
   "AristaSRv1001nondom": ( "sfp", Sfp.SRv1001nondomSFP, "Arista Networks " ),
   "AristaSRv1001External": ( "sfp", Sfp.SRv1001SFPExternal, "Arista Networks " ),
   "AristaSRv1001BadCc": ( "sfp", Sfp.SRv1001SFPBadCc, "Arista Networks " ),
   "OemBXD": ( "sfp", Sfp.BXDSFP, "OEM             " ),
   "OemBXU": ( "sfp", Sfp.BXUSFP, "OEM             " ),
   "HPTwinAx": ( "sfp", Sfp.HPTwinAx, "Hewlett Packard " ),
   "AristaLRv1001": ( "sfp", Sfp.LRv1001SFP, "Arista Networks " ),
   "AristaLRMv1001": ( "sfp", Sfp.LRMv1001SFP, "Arista Networks " ),
   "AristaCR": ( "sfp", Sfp.CRSFP, "Arista Networks " ),
   "AristaCRA": ( "sfp", Sfp.CRSFPA, "Arista Networks " ),
   "AristaCRP": ( "sfp", Sfp.CRSFPP, "Arista Networks " ),
   "AristaCR25G": ( "sfp", Sfp.CRSFP25G, "Arista Networks " ),
   "AristaCR25GS": ( "sfp", Sfp.CRSSFP25G, "Arista Networks " ),
   "AristaCR25GN": ( "sfp", Sfp.CRNSFP25G, "Arista Networks " ),
   "AristaCRP25G": ( "sfp", Sfp.CRSFPP25G, "Arista Networks " ),
   "AristaCR50G": ( "sfp", Sfp.CRSFP50G, "Arista Networks " ),
   "AristaCRP50G": ( "sfp", Sfp.CRSFPP50G, "Arista Networks " ),
   "AristaLXv1001": ( "sfp", Sfp.LXv1001SFP, "Arista Networks " ),
   "AristaLXv1001nondom": ( "sfp", Sfp.LXv1001nondomSFP, "Arista Networks " ),
   "Arista10GRA1GLX": ( "sfp", Sfp.LX10GRA1GSFP, "Arista Networks " ),
   "AristaLX10": ( "sfp", Sfp.LX10SFP, "Arista Networks " ),
   "BoboSRDualSpeed": ( "sfp", Sfp.SRDualSpeed, "BoboCorp        " ),
   "BoboLRDualSpeed": ( "sfp", Sfp.LRDualSpeed, "BoboCorp        " ),
   "AristaTv2": ( "sfp", Sfp.Tv2SFP, "Arista Networks " ),
   "Molex_Tan": ( "sfp", Sfp.Molex_74765_0902, "Molex Inc.      " ),
   "AristaDWDMER": ( "sfp", Sfp.SFPDWDM, "Arista Networks " ),
   "AristaDWDMZR": ( "sfp", Sfp.SFPDWDMZR, "Arista Networks " ),
   "AristaDWDMZT": ( "sfp", Sfp.SFPDWDMZT, "Arista Networks " ),
   "AristaSRAOC": ( "sfp", Sfp.AOCSFP, "Arista Networks " ),
   "AristaSRAOCNoEnhOpt": ( "sfp", Sfp.AOCSFPNoEnhOpt, "Arista Networks " ),
   "AristaSRL": ( "sfp", Sfp.SRLSFP, "Arista Networks " ),
   "AristaLRL": ( "sfp", Sfp.LRLSFP, "Arista Networks " ),
   "AristaER": ( "sfp", Sfp.ERSFP, "Arista Networks " ),
   "AristaERBDU": ( "sfp", Sfp.ERBDUSFP, "Arista Networks " ),
   "AristaERBDD": ( "sfp", Sfp.ERBDDSFP, "Arista Networks " ),
   "AristaERLBDU": ( "sfp", Sfp.ERLBDUSFP, "Arista Networks " ),
   "AristaERLBDD": ( "sfp", Sfp.ERLBDDSFP, "Arista Networks " ),
   "AristaER25G": ( "sfp", Sfp.ERSFP25G, "Arista Networks " ),
   "AristaZR": ( "sfp", Sfp.ZRSFP, "Arista Networks " ),
   "AristaSR25G": ( "sfp", Sfp.SRSFP25G, "Arista Networks " ),
   "AristaSRE25G": ( "sfp", Sfp.SRESFP25G, "Arista Networks " ),
   "AristaXSR25G": ( "sfp", Sfp.XSRSFP25G, "Arista Networks " ),
   "AristaAR25G": ( "sfp", Sfp.ARSFP25G, "Arista Networks " ),
   "3rdPartyAR25G": ( "sfp", Sfp.ARSFP25G3RDPTY, "Arista Networks " ),
   "AristaAR25GN": ( "sfp", Sfp.ARNSFP25G, "Arista Networks " ),
   "AristaLR25G": ( "sfp", Sfp.LRSFP25G, "Arista Networks " ),
   "AristaLRE25G": ( "sfp", Sfp.LRESFP25G, "Arista Networks " ),
   "AristaLRL25G": ( "sfp", Sfp.LRLSFP25G, "Arista Networks " ),
   "Unknown25GSfp": ( "sfp", Sfp.UNKNOWNSFP25G, "Unknown Corp " ),
   "AristaT10G": ( "sfp", Sfp.TSFP10G, "Arista Networks " ),
   "AristaT10GMRA": ( "sfp", Sfp.TSFP10GMRA, "Arista Networks " ),
   "AristaT10GRP": ( "sfp", Sfp.TSFP10GRP, "Arista Networks " ),
   "3rdPartyT10GSR": ( "sfp", Sfp.AQS107_TSFP10GSR, "AQUANTIA" ),
   "Arista100FX": ( "sfp", Sfp.FXSFP, "Arista Networks " ),
   "AristaMRSR25G": ( "sfp", Sfp.MRSRSFP25G, "Arista Networks" ),
   "AristaMRLR25G": ( "sfp", Sfp.MRLRSFP25G, "Arista Networks" ),
   "AristaMRSR50G": ( "sfp", Sfp.MRSRSFP50G, "Arista Networks" ),
   "AristaMRLR50G": ( "sfp", Sfp.MRLRSFP50G, "Arista Networks" ),
   "WncLoopback": ( "sfp", Sfp.WncLoopbackSfp, None ),
   # Useful shorthands for the command-line 'xcvr' tool
   "CR": ( "sfp", Sfp.CRSFP, "Arista Networks " ),
   "3rdPartyCR": ( "sfp", Sfp.CRSFP, "BoboCorp" ),
   "SR": ( "sfp", Sfp.SRv1001SFP, "Arista Networks " ),
   "SRnondom": ( "sfp", Sfp.SRv1001nondomSFP, "Arista Networks " ),
   "LR": ( "sfp", Sfp.LRv1001SFP, "Arista Networks " ),
   "LRM": ( "sfp", Sfp.LRMv1001SFP, "Arista Networks " ),
   "SX": ( "sfp", Sfp.SXSFP, "Arista Networks " ),
   "LX": ( "sfp", Sfp.LXv1001SFP, "Arista Networks " ),
   "LXnondom": ( "sfp", Sfp.LXv1001nondomSFP, "Arista Networks " ),
   "T": ( "sfp", Sfp.Tv2SFP, "Arista Networks " ),
   "3rdPartyT": ( "sfp", Sfp.TSFP3RDPTY, "Arista Networks " ),
   "AristaSRAOCNoDom": ( "sfp", Sfp.AOCSFPNoDOM, "Arista Networks " ),

   # Qsfp types.
   "AristaSR4Qsfp": ( "qsfp", Qsfp.SRQSFP, "Arista Networks " ),
   "AristaSR4Qsfp1002": ( "qsfp", Qsfp.SRQSFP1002, "Arista Networks " ),
   "AristaSR4Qsfp1003": ( "qsfp", Qsfp.SRQSFP1003, "Arista Networks " ),
   "AristaSR4Qsfp1004": ( "qsfp", Qsfp.SRQSFP1004, "Arista Networks " ),
   "AristaSR4QsfpNSRB1003": ( "qsfp", Qsfp.SRQSFPNOSOLREGBASE1003,
                              "Arista Networks " ),
   "AristaXSR4Qsfp": ( "qsfp", Qsfp.XSR4QSFP, "Arista Networks " ),
   "AristaSR4DQsfp": ( "qsfp", Qsfp.SR4DQSFP, "Arista Networks " ),
   "AristaSRBDQsfp": ( "qsfp", Qsfp.SRBDQSFP, "Arista Networks " ),
   "3rdPartySRBDQsfp": ( "qsfp", Qsfp.SRBDQSFP3RDPTY, "BoboCorp        " ),
   "BoboSR4Qsfp": ( "qsfp", Qsfp.SRQSFP, "BoboCorp        " ),
   "AristaAR4Qsfp": ( "qsfp", Qsfp.AR4QSFP, "Arista Networks " ),
   "AristaCR4Qsfp": ( "qsfp", Qsfp.CR4QSFP, "Arista Networks " ),
   "3rdPartyCR4Qsfp": ( "qsfp", Qsfp.CR4QSFP3RDPTY, "BoboCorp        " ),
   "AristaLR4Qsfp": ( "qsfp", Qsfp.LR4QSFP, "Arista Networks " ),
   "AristaLR4Qsfp1002": ( "qsfp", Qsfp.LR4QSFP1002, "Arista Networks " ),
   "AristaLRL4Qsfp": ( "qsfp", Qsfp.LRL4QSFP, "Arista Networks " ),
   "AristaPLR4Qsfp": ( "qsfp", Qsfp.PLR4QSFP, "Arista Networks " ),
   "AristaPLRL4Qsfp": ( "qsfp", Qsfp.PLRL4QSFP, "Arista Networks " ),
   "3rdPartyPLR4Qsfp": ( "qsfp", Qsfp.PLR4QSFP3PTY, "BoboCorp        " ),
   "Psm4Qsfp": ( "qsfp", Qsfp.PSM4QSFP, "Psm4 Corp      " ),
   "AristaUNIVQsfp": ( "qsfp", Qsfp.UNIVQSFP, "Arista Networks " ),
   "AristaER4Qsfp": ( "qsfp", Qsfp.ER4QSFP, "Arista Networks " ),
   "Arista100GSR4Qsfp": ( "qsfp", Qsfp.SR4QSFP100G, "Arista Networks " ),
   "Arista100GSR4EQsfp": ( "qsfp", Qsfp.SR4EQSFP100G, "Arista Networks " ),
   "Arista100GXSR4Qsfp": ( "qsfp", Qsfp.XSR4QSFP100G, "Arista Networks " ),
   "3rdParty100G40GSR4Qsfp": ( "qsfp", Qsfp.SR4QSFP100G40G3RDPTY,
                               "BoboCorp        " ),
   "Arista100GLR4Qsfp": ( "qsfp", Qsfp.LR4QSFP100G, "Arista Networks " ),
   "Arista100GLRL4Qsfp": ( "qsfp", Qsfp.LRL4QSFP100G, "Arista Networks " ),
   "3rdParty100G40GLR4Qsfp": ( "qsfp", Qsfp.LR4QSFP100G40G3RDPTY,
                               "BoboCorp        " ),
   "Arista100GER4Qsfp": ( "qsfp", Qsfp.ER4QSFP100G, "Arista Networks " ),
   "Arista100GAR4Qsfp": ( "qsfp", Qsfp.AR4QSFP100G, "Arista Networks " ),
   "3rdParty100G40GAR4Qsfp": ( "qsfp", Qsfp.AR4QSFP100G40G3RDPTY,
                               "BoboCorp        " ),
   "Arista100GAR4NQsfp": ( "qsfp", Qsfp.AR4NQSFP100G, "Arista Networks " ),
   "3rdParty100G40GAR4NQsfp": ( "qsfp", Qsfp.AR4NQSFP100G40G3RDPTY,
                                "BoboCorp        " ),
   "Arista100GCR4Qsfp": ( "qsfp", Qsfp.CR4QSFP100G, "Arista Networks " ),
   "Arista100GCR4SQsfp": ( "qsfp", Qsfp.CR4SQSFP100G, "Arista Networks " ),
   "Arista100GCR4NQsfp": ( "qsfp", Qsfp.CR4NQSFP100G, "Arista Networks " ),
   "Arista100GCRA4Qsfp": ( "qsfp", Qsfp.CRA4QSFP100G, "Arista Networks " ),
   "Arista50GCR2LQsfp": ( "qsfp", Qsfp.CR2LQSFP50G, "Arista Networks " ),
   "Arista50GCR2SQsfp": ( "qsfp", Qsfp.CR2SQSFP50G, "Arista Networks " ),
   "Arista50GCR2NQsfp": ( "qsfp", Qsfp.CR2NQSFP50G, "Arista Networks " ),
   "3rdParty100GCR4Qsfp": ( "qsfp", Qsfp.CR4QSFP100G3RDPTY, "BoboCorp        " ),
   "3rdParty100G40GCR4Qsfp": ( "qsfp", Qsfp.CR4QSFP100G40G3RDPTY,
                               "BoboCorp        " ),
   "Arista100GPSM4Qsfp": ( "qsfp", Qsfp.PSM4QSFP100G, "Arista Networks " ),
   "Arista100GPLRL4Qsfp": ( "qsfp", Qsfp.PLRL4QSFP100G, "Arista Networks " ),
   "Arista100GPLR4Qsfp": ( "qsfp", Qsfp.PLR4QSFP100G, "Arista Networks " ),
   "Arista100GCLR4Qsfp": ( "qsfp", Qsfp.CLR4QSFP100G, "Arista Networks " ),
   "Arista100GCWDM4Qsfp": ( "qsfp", Qsfp.CWDM4QSFP100G, "Arista Networks " ),
   "Arista100GEDWDM2Qsfp": ( "qsfp", Qsfp.DWDM2QSFP100G, "Arista Networks " ),
   "Arista100GSWDM4Qsfp": ( "qsfp", Qsfp.SWDM4QSFP100G, "Arista Networks " ),
   "Arista100GSRBDQsfp": ( "qsfp", Qsfp.SRBDQSFP100G, "Arista Networks " ),
   "Arista100GERL4Qsfp": ( "qsfp", Qsfp.ERL4QSFP100G, "Arista Networks " ),
   "Arista100GDRQsfp": ( "qsfp", Qsfp.DRQSFP100G, "Arista Networks " ),
   "Arista100GFRQsfp": ( "qsfp", Qsfp.FRQSFP100G, "Arista Networks " ),
   "Arista100GFRQsfpEnhanceDom": ( "qsfp",
      Qsfp.FRQSFP100GENDOM, "Arista Networks" ),
   "Arista100GLRQsfp": ( "qsfp", Qsfp.LRQSFP100G, "Arista Networks " ),
   "Arista100GFRFecQsfp": ( "qsfp", Qsfp.FRFECQSFP100G, "Arista Networks " ),
   "Arista100GLRFecQsfp": ( "qsfp", Qsfp.LRFECQSFP100G, "Arista Networks " ),
   "Arista100GERQsfp": ( "qsfp", Qsfp.ERQSFP100G, "Arista Networks " ),
   "Arista100GERFecQsfp": ( "qsfp", Qsfp.ERFECQSFP100G, "Arista Networks " ),
   "Arista100GXCWDM4Qsfp": ( "qsfp", Qsfp.XCWDM4QSFP100G, "Arista Networks " ),
   "Arista100GZR4Qsfp": ( "qsfp", Qsfp.ZR4QSFP100G, "Arista Networks " ),
   "AristaTxPowerQsfp": ( "qsfp", Qsfp.SRQSFP, "Arista Networks " ),
   "Arista400g2QqsfpSide": ( "qsfp", Qsfp.Qsfp400g2QqsfpSide, "Arista Networks " ),
   "Arista400g2QqsfpSide40G": ( "qsfp", Qsfp.Qsfp400g2QqsfpSide40G,
                                 "Arista Networks " ),
   "Arista400g2QqsfpSideDepop": ( "qsfp", Qsfp.Qsfp400g2QqsfpSideDepop,
                                   "Arista Networks " ),
   "Qsfp400gCr4": ( 'qsfp', Qsfp.Qsfp400GCr4Upper00, "BoboCorp        " ),
   "Qsfp400gCr4NoCompCode": ( 'qsfp',
                     Qsfp.Qsfp400gCr4NoCompCodeUpper00, "BoboCorp        " ),
   "UnknownQsfp": ( "qsfp", Qsfp.UNKNOWNQSFP, "Unknown Corp " ),
   "Unknown100GQsfp": ( "qsfp", Qsfp.UNKNOWNQSFP100G, "Unknown Corp " ),
   "Unknown100GQsfpOptics": ( "qsfp", Qsfp.UNKNOWNQSFP100GOPTICS,
                              "Unknown Corp " ),
   "FaultyQsfp": ( "qsfp", Qsfp.FAULTYQSFP, "Faulty Corp     " ),
   "AristaQsfp56Sff8636Cu200G": ( "qsfp", Qsfp.Qsfp56Sff8636Cu200GUpperPage0,
                                  "Arista Networks" ),
   "3pxQsfp56Sff8636Cu200G": ( "qsfp", Qsfp.Qsfp56Sff8636Cu200GUpperPage0,
                               "Bobo Corp" ),
   "AristaQsfp56Sff8636Sr4": ( "qsfp", Qsfp.Qsfp56Sff8636Sr4UpperPage0,
                               "Arista Networks" ),
   "AristaQsfp56Sff8636Fr4": ( "qsfp", Qsfp.Qsfp56Sff8636Fr4UpperPage0,
                               "Arista Networks" ),
   "AristaQsfp56Sff8636Ar4": ( "qsfp", Qsfp.Qsfp56Sff8636Ar4UpperPage0,
                               "Arista Networks" ),
   "AristaQsfp56Sff8636Sr1_2": ( "qsfp", None, "Arista Networks " ),
   "AristaQsfp56Sff8636Sr1_2Lower00h": ( "qsfp", Qsfp.Qsfp100gSr1_2Lower00h,
                                    "Arista Networks " ),
   "AristaQsfp56Sff8636Sr1_2Upper00h": ( "qsfp", Qsfp.Qsfp100gSr1_2Upper00h,
                                    "Arista Networks " ),

   # CFP2 LR4 ( base entry + EEPROM tables )
   "AristaLR4Cfp2": ( 'cfp2', None, "Arista Networks " ),
   "AristaLR4Cfp2_table8000": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4Table8000,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_table8080": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4Table8080,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_table8420": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4Table8420,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA000": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4TableA000,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA200": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4TableA200,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA280": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4TableA280,
                                 "Arista Networks " ),
   "AristaLR4Cfp2_tableA400": ( 'cfp2', Cfp2.Cfp2MsaMis100GLR4TableA400,
                                 "Arista Networks " ),
   # 3rd party CFP2 LR4 ( base entry + EEprom tables )
   "BoboLR4Cfp2": ( 'cfp2', None, "BoboCorp        " ),
   "BoboLR4Cfp2_table8000": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4Table8000,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_table8080": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4Table8080,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_table8420": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4Table8420,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA000": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4TableA000,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA200": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4TableA200,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA280": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4TableA280,
                               "BoboCorp        " ),
   "BoboLR4Cfp2_tableA400": ( 'cfp2', Cfp2.Cfp2MsaMisBobo100GLR4TableA400,
                               "BoboCorp        " ),
   # CFP2 SR10 ( base entry + EEPROM tables )
   "AristaSR10Cfp2": ( 'cfp2', None, "Arista Networks " ),
   "AristaSR10Cfp2_table8000": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10Table8000,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_table8080": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10Table8080,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_table8420": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10Table8420,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA000": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10TableA000,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA200": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10TableA200,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA280": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10TableA280,
                                  "Arista Networks " ),
   "AristaSR10Cfp2_tableA400": ( 'cfp2', Cfp2.Cfp2MsaMis100GSR10TableA400,
                                  "Arista Networks " ),
   # CFP2 ER4 ( base entry + EEPROM tables )
   "AristaER4Cfp2": ( 'cfp2', None, "Arista Networks " ),
   "AristaER4Cfp2_table8000": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4Table8000,
                                  "Arista Networks " ),
   "AristaER4Cfp2_table8080": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4Table8080,
                                  "Arista Networks " ),
   "AristaER4Cfp2_table8420": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4Table8420,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA000": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4TableA000,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA200": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4TableA200,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA280": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4TableA280,
                                  "Arista Networks " ),
   "AristaER4Cfp2_tableA400": ( 'cfp2', Cfp2.Cfp2MsaMis100GER4TableA400,
                                  "Arista Networks " ),
   # 3rd party CFPX ( base entry + EEprom tables )
   "BoboDwdmCoherentCfp2": ( 'cfp2', None, "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8000": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTable8000,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8080": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTable8080,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8180": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTable8180,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_table8420": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTable8420,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB000": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableB000,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB180": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableB180,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB300": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableB300,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB400": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableB400,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableB480": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableB480,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableBA80": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableBA80,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableBB00": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableBB00,
                     "BoboCorp        " ),
   "BoboDwdmCoherentCfp2_tableBB80": ( 'cfp2',
                     Cfp2.Cfp2MsaMis100GBoboDwdmCoherentTableBB80,
                     "BoboCorp        " ),
   # CFP2 Dwdm Coherent limiting module ( base entry + EEPROM tables )
   "AristaDwdmCoherentCfp2": ( 'cfp2', None, "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8000": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTable8000,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8080": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTable8080,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8180": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTable8180,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_table8420": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTable8420,
                                          "AristaNetworks " ),
   "AristaDwdmCoherentCfp2_tableB000": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableB000,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB180": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableB180,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB300": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableB300,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB400": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableB400,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableB480": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableB480,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableBA80": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableBA80,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableBB00": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableBB00,
                                          "Arista Networks " ),
   "AristaDwdmCoherentCfp2_tableBB80": ( 'cfp2',
                                          Cfp2.Cfp2MsaMis100GDwdmCoherentTableBB80,
                                          "Arista Networks " ),
   # CFP2 Dwdm Coherent linear module ( base entry + EEPROM tables ) Oclaro
   "AristaDwdmCoherentECfp2": ( 'cfp2', None, "Arista Networks " ),
   "AristaDwdmCoherentECfp2_table8180": ( 'cfp2',
                                           Cfp2.Cfp2MsaMis100GDwdmCoherentETable8180,
                                           "Arista Networks " ),
   "AristaDwdmCoherentECfp2_tableB180": ( 'cfp2',
                                           Cfp2.Cfp2MsaMis100GDwdmCoherentETableB180,
                                           "Arista Networks " ),
   "AristaDwdmCoherentECfp2_tableBB00": ( 'cfp2',
                                           Cfp2.Cfp2MsaMis100GDwdmCoherentETableBB00,
                                           "Arista Networks " ),
   # CFP2 Dwdm Coherent linear module V2 ( base entry + EEPROM tables ) Acacia
   "AristaDwdmCoherentECfp2V2": ( 'cfp2', None, "Arista Networks " ),
   "AristaDwdmCoherentECfp2V2_table8000": ( 'cfp2',
                        Cfp2.Cfp2MsaMis100GDwdmCoherentEV2Table8000,
                        "Arista Networks " ),
   "OclaroDwdmCoherentECfp2": ( 'cfp2', None, "OCLARO          " ),
   "OclaroDwdmCoherentECfp2_table8000": ( 'cfp2',
               Cfp2.Cfp2MsaMis100GOclaroDwdmCoherentETable8000,
               "OCLARO          " ),
   "AcaciaDwdmCoherentECfp2": ( 'cfp2', None, "Acacia Comm Inc." ),
   "AcaciaDwdmCoherentECfp2_table8000": ( 'cfp2',
               Cfp2.Cfp2MsaMis100GAcaciaDwdmCoherentETable8000,
               "Acacia Comm Inc." ),
   # CFP2 DCO module ( base entry + EEPROM tables )
   "AristaCfp2DwdmDco": ( 'cfp2', None, "Arista Networks " ),
   "AristaCfp2DwdmDco_table8000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTable8000, "Arista Networks " ),
   "AristaCfp2DwdmDco_table8080": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTable8080, "Arista Networks " ),
   "AristaCfp2DwdmDco_table8180": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTable8180, "Arista Networks " ),
   "AristaCfp2DwdmDco_table9000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTable9000, "Arista Networks " ),
   "AristaCfp2DwdmDco_table9100": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTable9100, "Arista Networks " ),
   "AristaCfp2DwdmDco_tableB000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTableB000, "Arista Networks " ),
   "AristaCfp2DwdmDco_tableB480": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDcoTableB480, "Arista Networks " ),
   # CFP2 DCO module 3rd party ( base entry + EEPROM tables ) Acacia
   "AcaciaCfp2DwdmDco": ( 'cfp2', None, "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table8000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTable8000,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table8080": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTable8080,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table8180": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTable8180,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_table9000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTable9000,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_tableB000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTableB000,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DwdmDco_tableB480": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTableB480,
                        "Acacia Comm Inc " ),
   # CFP2 Dual laser DCO module ( base entry + EEPROM tables )
   "AcaciaCfp2DlDwdmDco": ( 'cfp2', None, "Arista Networks " ),
   "AcaciaCfp2DlDwdmDco_table8000": (
      'cfp2', Cfp2.Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable8000, "Acacia Comm Inc " ),
   "AcaciaCfp2DlDwdmDco_table8080": (
      'cfp2', Cfp2.Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable8080, "Acacia Comm Inc " ),
   "AcaciaCfp2DlDwdmDco_table8180": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTable8180,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DlDwdmDco_table9B80": (
      'cfp2', Cfp2.Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9B80, "Acacia Comm Inc " ),
   "AcaciaCfp2DlDwdmDco_table9000": (
      'cfp2', Cfp2.Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9000, "Acacia Comm Inc " ),
   "AcaciaCfp2DlDwdmDco_tableB000": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTableB000,
                        "Acacia Comm Inc " ),
   "AcaciaCfp2DlDwdmDco_tableB480": ( 'cfp2',
                        Cfp2.Cfp2MsaMisCfp2DwdmDco3rdPartyTableB480,
                        "Acacia Comm Inc " ),

   # CFP2 XSR10 ( base entry + EEPROM tables )
   "AristaXSR10Cfp2": ( 'cfp2', None, "Arista Networks " ),
   "AristaXSR10Cfp2_table8000": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10Table8000,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_table8080": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10Table8080,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_table8420": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10Table8420,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA000": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10TableA000,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA200": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10TableA200,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA280": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10TableA280,
                                  "Arista Networks " ),
   "AristaXSR10Cfp2_tableA400": ( 'cfp2', Cfp2.Cfp2MsaMis100GXSR10TableA400,
                                  "Arista Networks " ),

   # OSFP 400G-CR8 ( base entry + Id EEPROM pages lowerPage00 and upperPage00 )
   "AristaOsfp400gCr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gCr8Lower00": ( 'osfp', Osfp.Osfp400gCr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gCr8Upper00": ( 'osfp', Osfp.Osfp400gCr8UpperPage00,
                                  "Arista Networks " ),
   "AristaCmis4Osfp400gCr8": ( 'osfp', None, "Arista Networks " ),
   "AristaCmis4Osfp400gCr8Lower00": ( 'osfp', Osfp.Cmis4Osfp400gCr8LowerPage00,
                                  "Arista Networks " ),
   "AristaCmis4Osfp400gCr8Upper00": ( 'osfp', Osfp.Cmis4Osfp400gCr8UpperPage00,
                                  "Arista Networks " ),
   "VeridianOsfp400gCr8": ( 'osfp', None, "VeridianDynamics" ),
   "AristaQsfpDd400gCr8": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gCr8Lower00": ( 'qsfpDd', QsfpDd.QsfpDd400gCr8LowerPage00,
                                    "Arista Networks " ),
   "AristaQsfpDd400gCr8Upper00": ( 'qsfpDd', QsfpDd.QsfpDd400gCr8UpperPage00,
                                    "Arista Networks " ),
   "VeridianQsfpDd400gCr8": ( 'qsfpDd', None, "VeridianDynamics" ),
   "AristaQsfpCmis200gCr4": ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis200gCr4Lower00": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gCr4LowerPage00,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gCr4Upper00": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gCr4UpperPage00,
                                      "Arista Networks " ),
   "VeridianQsfpCmis200gCr4": ( 'qsfpCmis', None, "VeridianDynamics" ),
   "AristaOsfp400gSr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gSr8Lower00": ( 'osfp', Osfp.Osfp400gSr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper00": ( 'osfp', Osfp.Osfp400gSr8UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper01": ( 'osfp', Osfp.Osfp400gSr8UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper02": ( 'osfp', Osfp.Osfp400gSr8UpperPage02,
                                  "Arista Networks " ),
   "AristaOsfp400gSr8Upper10Bank0": ( 'osfp', Osfp.Osfp400gSr8UpperPage10Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gSr8Upper11Bank0": ( 'osfp', Osfp.Osfp400gSr8UpperPage11Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gSr8Cmis3": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gSr8Cmis3Lower00": ( 'osfp', Osfp.Osfp400gSr8Cmis3LowerPage00,
                                       "Arista Networks " ),
   "AristaOsfp400gTo4Qsfp": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gTo4QsfpLower00": ( 'osfp', Osfp.Osfp400gTo4QsfpLowerPage00,
                                      "Arista Networks " ),
   # Dr4
   "AristaOsfp400gDr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gDr4Lower00": ( 'osfp', Osfp.Osfp400gDr4LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper00": ( 'osfp', Osfp.Osfp400gDr4UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper01": ( 'osfp', Osfp.Osfp400gDr4UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper02": ( 'osfp', Osfp.Osfp400gDr4UpperPage02,
                                  "Arista Networks " ),
   "AristaOsfp400gDr4Upper10Bank0": ( 'osfp', Osfp.Osfp400gDr4UpperPage10Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gDr4Upper11Bank0": ( 'osfp', Osfp.Osfp400gDr4UpperPage11Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gXdr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gXdr4Upper01": ( 'osfp', Osfp.Osfp400gXdr4UpperPage01,
                                   "Arista Networks " ),
   "AristaOsfp4x100gDr": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp4x100gDrLower00": ( 'osfp', Osfp.Osfp4x100gDrLowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp4x100gDrUpper01": ( 'osfp', Osfp.Osfp4x100gDrUpperPage01,
                                  "Arista Networks " ),
   # Fr4
   "AristaOsfp400gFr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gFr4Lower00": ( 'osfp', Osfp.Osfp400gFr4LowerPage00,
                                  "Arista Networks " ),
   # Plr4
   "AristaOsfp400gPlr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gPlr4Lower00": ( 'osfp', Osfp.Osfp400gPlr4LowerPage00,
                                  "Arista Networks " ),
   # 2Fr4
   "AristaOsfp400g2Fr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400g2Fr4Lower00": ( 'osfp', Osfp.Osfp400g2Fr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper00": ( 'osfp', Osfp.Osfp400g2Fr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper01": ( 'osfp', Osfp.Osfp400g2Fr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper02": ( 'osfp', Osfp.Osfp400g2Fr4UpperPage02,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper10Bank0": ( 'osfp', Osfp.Osfp400g2Fr4UpperPage10Bank0,
                                    "Arista Networks " ),
   "AristaOsfp400g2Fr4Upper11Bank0": ( 'osfp', Osfp.Osfp400g2Fr4UpperPage11Bank0,
                                    "Arista Networks " ),
   "VeridianOsfp400gSr8": ( 'osfp', None, "VeridianDynamics" ),
   # OsfpAr8 borrows Sr8's 10h and 11h pages
   "AristaOsfp400gAr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gAr8Lower00": ( 'osfp', Osfp.Osfp400gAr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper00": ( 'osfp', Osfp.Osfp400gAr8UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper01": ( 'osfp', Osfp.Osfp400gAr8UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper02": ( 'osfp', Osfp.Osfp400gAr8UpperPage02,
                                  "Arista Networks " ),
   "AristaOsfp400gAr8Upper10Bank0": ( 'osfp', Osfp.Osfp400gSr8UpperPage10Bank0,
                                       "Arista Networks " ),
   "AristaOsfp400gAr8Upper11Bank0": ( 'osfp', Osfp.Osfp400gSr8UpperPage11Bank0,
                                       "Arista Networks " ),
   "VeridianOsfp400gAr8": ( 'osfp', None, "VeridianDynamics" ),
   "AristaOsfp400gCra8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gCra8Lower00": ( 'osfp', Osfp.Osfp400gCra8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gCra8Upper00": ( 'osfp', Osfp.Osfp400gCra8UpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gCra8Upper01": ( 'osfp', Osfp.Osfp400gCra8UpperPage01,
                                  "Arista Networks " ),
   "AristaOsfpAmpZr": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfpAmpZrLower00": ( 'osfp', Osfp.OsfpAmpZrLowerPage00,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpper00": ( 'osfp', Osfp.OsfpAmpZrUpperPage00,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpper01": ( 'osfp', Osfp.OsfpAmpZrUpperPage01,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpper02": ( 'osfp', Osfp.OsfpAmpZrUpperPage02,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpper11Bank0": ( 'osfp', Osfp.OsfpAmpZrUpperPage11Bank0,
                                     "Arista Networks " ),
   "AristaOsfpAmpZrUpperC0": ( 'osfp', Osfp.OsfpAmpZrUpperPageC0,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpperC1": ( 'osfp', Osfp.OsfpAmpZrUpperPageC1,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpperD0": ( 'osfp', Osfp.OsfpAmpZrUpperPageD0,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpperD1": ( 'osfp', Osfp.OsfpAmpZrUpperPageD1,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpperD8": ( 'osfp', Osfp.OsfpAmpZrUpperPageD8,
                                "Arista Networks " ),
   "AristaOsfpAmpZrUpperD9": ( 'osfp', Osfp.OsfpAmpZrUpperPageD9,
                                "Arista Networks " ),
   "AristaOsfp400gSrbd": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gSrbdLower00": ( 'osfp', Osfp.Osfp400gSrbdLowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gSrbdUpper00": ( 'osfp', Osfp.Osfp400gSrbdUpperPage00,
                                  "Arista Networks " ),
   "AristaOsfp400gSrbdUpper01": ( 'osfp', Osfp.Osfp400gSrbdUpperPage01,
                                  "Arista Networks " ),
   'AristaQsfpDd400gZr': ( 'qsfpDd', None, "Arista Networks " ),
   'AcaciaQsfpDd400gZr': ( 'qsfpDd', None, "Acacia Networks " ),
   'AristaQsfpDd400gZrLower00': ( 'qsfpDd', Acacia.QsfpDd400gZrLowerPage00,
                                   "Arista Networks " ),
   'AristaQsfpDd400gZrUpper00': ( 'qsfpDd', Acacia.QsfpDd400gZrUpperPage00,
                                   "Arista Networks " ),
   'AristaQsfpDd400gZrUpper01': ( 'qsfpDd', QsfpDd.QsfpDd400gZrUpperPage01,
                                   "Arista Networks " ),
   'AristaQsfpDd400gZrUpper04': ( 'qsfpDd', QsfpDd.QsfpDd400gZrUpperPage04,
                                   "Arista Networks " ),
   "AristaQsfpDd400gZrUpper12Bank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage12Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper20Bank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage20Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper21Bank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage21Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper22Bank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage22Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper23Bank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage23Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper28Bank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage28Bank0,
                                        "Arista Networks " ),
   "AristaQsfpDd400gZrUpper2FBank0": ( 'qsfpDd',
                                        QsfpDd.QsfpDd400gZrUpperPage2FBank0,
                                        "Arista Networks " ),

   'AristaOsfp400gZr': ( 'osfp', None, "Arista Networks " ),
   'AristaOsfp400gZrLower00': ( 'osfp', Acacia.Osfp400gZrLowerPage00,
                                 "Arista Networks " ),
   'AristaOsfp400gZrUpper00': ( 'osfp', Acacia.Osfp400gZrUpperPage00,
                                 "Arista Networks " ),
   'AristaOsfp400gZrUpper01': ( 'osfp', QsfpDd.QsfpDd400gZrUpperPage01,
                                 "Arista Networks " ),
   'AristaOsfp400gZrUpper13': ( 'osfp', Osfp.CmisUpperPage13Bank0,
                                 "Arista Networks " ),
   'AristaQsfpDd400gZrP': ( 'qsfpDd', None, 'Arista Networks ' ),
   'AristaQsfpDd400gZrPLower00': ( 'qsfpDd', Acacia.QsfpDd400gZrPLowerPage00,
                                    "Arista Networks " ),
   'AristaQsfpDd400gZrPUpper00': ( 'qsfpDd', Acacia.QsfpDd400gZrPUpperPage00,
                                    "Arista Networks " ),
   'AristaQsfpDd400gZrPUpper01': ( 'qsfpDd', Acacia.QsfpDd400gZrPUpperPage01,
                                    "Arista Networks " ),
   'AristaQsfpDd400gZrPUpper04': ( 'qsfpDd', Acacia.QsfpDd400gZrPUpperPage04,
                                    "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper12Bank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage12Bank0,
                                         "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper20Bank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage20Bank0,
                                         "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper21Bank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage21Bank0,
                                         "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper22Bank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage22Bank0,
                                         "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper23Bank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage23Bank0,
                                         "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper28Bank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage28Bank0,
                                         "Arista Networks " ),
   "AristaQsfpDd400gZrPUpper2FBank0": ( 'qsfpDd',
                                         Acacia.QsfpDd400gZrPUpperPage2FBank0,
                                         "Arista Networks " ),
   'AristaOsfp400gZrP': ( 'osfp', None, "Arista Networks " ),
   'AristaOsfp400gZrPLower00': ( 'osfp', Acacia.Osfp400gZrPLowerPage00,
                                  "Arista Networks " ),
   'AristaOsfp400gZrPUpper00': ( 'osfp', Acacia.Osfp400gZrPUpperPage00,
                                  "Arista Networks " ),

   "GoogleOM8FNT": ( 'osfp', None, "Google" ),
   "GoogleOM8FNTUpper00": ( 'osfp', Osfp.GoogleOM8FNTUpperPage00, "Google" ),

   "GoogleOsfp8x10gLr": ( "osfp", None, "Arista Networks" ),
   "GoogleOsfp8x10gLrLower00": ( "osfp", Osfp.GoogleOsfp8x10gLrLowerPage00,
                                 "Arista Networks" ),
   "GoogleOsfp8x10gLrUpper00": ( "osfp", Osfp.GoogleOsfp8x10gLrUpperPage00,
                                 "Arista Networks" ),

   "AristaQsfpDd400gSr8": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gSr8Lower00": ( 'qsfpDd', QsfpDd.QsfpDd400gSr8LowerPage00,
                                  "Arista Networks " ),
   "AristaQsfpDd400gSr8Upper00": ( 'qsfpDd', QsfpDd.QsfpDd400gSr8UpperPage00,
                                  "Arista Networks " ),
   "AristaQsfpDd8x25gSr": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd8x25gSrLower00": ( 'qsfpDd', QsfpDd.QsfpDd8x25gSrLowerPage00,
                                   "Arista Networks " ),
   "AristaQsfpDd8x25gSrUpper00": ( 'qsfpDd', QsfpDd.QsfpDd8x25gSrUpperPage00,
                                   "Arista Networks " ),
   "AristaQsfpDd8x25gSrUpper01": ( 'qsfpDd', QsfpDd.QsfpDd8x25gSrUpperPage01,
                                   "Arista Networks " ),
   "AristaQsfpDd400gSrbd": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gSrbdLower00": ( 'qsfpDd', QsfpDd.QsfpDd400gSrbdLowerPage00,
                                  "Arista Networks " ),
   "AristaQsfpDd400gSrbdUpper00": ( 'qsfpDd', QsfpDd.QsfpDd400gSrbdUpperPage00,
                                  "Arista Networks " ),
   "AristaQsfpDd400gSrbdUpper01": ( 'qsfpDd', QsfpDd.QsfpDd400gSrbdUpperPage01,
                                  "Arista Networks " ),
   "AristaQsfpCmis200gFr4": ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis200gFr4Lower00": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gFr4LowerPage00,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper00": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gFr4UpperPage00,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper01": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gFr4UpperPage01,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper02": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gFr4UpperPage02,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper10": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gFr4UpperPage10,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gFr4Upper11": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gFr4UpperPage11,
                                      "Arista Networks " ),

   "AristaQsfpCmis400gAr4": ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis400gAr4Lower00": ( 'qsfpCmis',
                                     QsfpCmis.QsfpCmis400gAr4LowerPage00,
                                     "Arista Networks " ),
   "AristaQsfpCmis400gAr4Upper00": ( 'qsfpCmis',
                                     QsfpCmis.QsfpCmis400gAr4UpperPage00,
                                     "Arista Networks " ),
   "AristaQsfpCmis400gAr4Upper01": ( 'qsfpCmis',
                                     QsfpCmis.QsfpCmis400gAr4UpperPage01,
                                     "Arista Networks " ),

   "AristaQsfpDd400gSr8Cmis3": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gSr8Cmis3Lower00": ( 'qsfpDd',
                                         QsfpDd.QsfpDd400gSr8Cmis3LowerPage00,
                                         "Arista Networks " ),
   "AristaQsfpDd800gXdr8": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800gXdr8Lower00": ( 'qsfpDd',
                                     QsfpDd.QsfpDd800gXdr8LowerPage00,
                                     "Arista Networks " ),
   "AristaQsfpDd800gXdr8Upper00": ( 'qsfpDd',
                                     QsfpDd.QsfpDd800gXdr8UpperPage00,
                                     "Arista Networks " ),
   "AristaQsfpDd800gXdr8Upper01": ( 'qsfpDd',
                                     QsfpDd.QsfpDd800gXdr8UpperPage01,
                                     "Arista Networks " ),
   "AristaQsfpDd800gXdr8Upper02": ( 'qsfpDd',
                                     QsfpDd.QsfpDd800gXdr8UpperPage02,
                                     "Arista Networks " ),
   "AristaOsfp800g2lr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800g2lr4Lower00": ( 'osfp', Osfp.Osfp800g2lr4LowerPage00,
                                   'Arista Networks ' ),
   "AristaOsfp800g2lr4Upper00": ( 'osfp', Osfp.Osfp800g2lr4UpperPage00,
                                   'Arista Netwroks ' ),
   "AristaOsfp800g2lr4Upper01": ( 'osfp', Osfp.Osfp800g2lr4UpperPage01,
                                   'Arista Networks ' ),
   "AristaOsfp800g2lr4Upper02": ( 'osfp', Osfp.Osfp800g2lr4UpperPage02,
                                   'Arista Networks ' ),
   "AristaQsfpDd800g2lr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800g2lr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800g2lr4LowerPage00,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2lr4Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800g2lr4UpperPage00,
                                     'Arista Netwroks ' ),
   "AristaQsfpDd800g2lr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800g2lr4UpperPage01,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2lr4Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800g2lr4UpperPage02,
                                     'Arista Networks ' ),
   "AristaQsfpDd800gPlr8": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800gPlr8Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800gPlr8LowerPage00,
                                     'Arista Networks ' ),
   "AristaQsfpDd800gPlr8Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800gPlr8UpperPage00,
                                     'Arista Netwroks ' ),
   "AristaQsfpDd800gPlr8Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800gPlr8UpperPage01,
                                     'Arista Networks ' ),
   "AristaQsfpDd800gPlr8Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800gPlr8UpperPage02,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Xdr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800g2Xdr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4LowerPage00,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Xdr4Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4UpperPage00,
                                     'Arista Netwroks ' ),
   "AristaQsfpDd800g2Xdr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4UpperPage01,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Xdr4Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4UpperPage02,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Dr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800g2Dr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Dr4LowerPage00,
                                    'Arista Networks ' ),
   "AristaQsfpDd800g2Dr4Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Dr4UpperPage00,
                                    'Arista Netwroks ' ),
   "AristaQsfpDd800g2Dr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800g2Dr4UpperPage01,
                                    'Arista Networks ' ),
   "AristaQsfpDd800g2Dr4Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4UpperPage02,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Plr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800g2Plr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Plr4LowerPage00,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Plr4Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Plr4UpperPage00,
                                     'Arista Netwroks ' ),
   "AristaQsfpDd800g2Plr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800g2Plr4UpperPage01,
                                     'Arista Networks ' ),
   "AristaQsfpDd800g2Plr4Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800g2Plr4UpperPage02,
                                     'Arista Networks ' ),
   "AristaQsfpDd800gSr8": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800gSr8Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800gSr8LowerPage00,
                                    'Arista Networks ' ),
   "AristaQsfpDd800gSr8Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800gSr8UpperPage00,
                                    'Arista Networks ' ),
   "AristaQsfpDd800gSr8Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800gSr8UpperPage01,
                                    'Arista Networks ' ),
   "AristaQsfpDd800gSr8Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800gSr8UpperPage02,
                                    'Arista Networks ' ),
   # QSFP-DD LPO 800GBASE-2XDR4
   "AristaQsfpDdLpo800g2Xdr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDdLpo800g2Xdr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4LowerPage00,
                                     'Arista Networks ' ),
   "AristaQsfpDdLpo800g2Xdr4Upper00": ( 'qsfpDd',
                                     QsfpDd.QsfpDdLpo800g2Xdr4UpperPage00,
                                     'Arista Netwroks ' ),
   "AristaQsfpDdLpo800g2Xdr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4UpperPage01,
                                     'Arista Networks ' ),
   "AristaQsfpDdLpo800g2Xdr4Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800g2Xdr4UpperPage02,
                                     'Arista Networks ' ),

   "AristaOsfp800gSr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800gSr8Lower00": ( 'osfp', Osfp.Osfp800gSr8LowerPage00,
                                    'Arista Networks ' ),
   "AristaOsfp800gSr8Upper00": ( 'osfp', Osfp.Osfp800gSr8UpperPage00,
                                    'Arista Networks ' ),
   "AristaOsfp800gSr8Upper01": ( 'osfp', Osfp.Osfp800gSr8UpperPage01,
                                    'Arista Networks ' ),
   "AristaOsfp800gSr8Upper02": ( 'osfp', Osfp.Osfp800gSr8UpperPage02,
                                    'Arista Networks ' ),
   "AristaQsfpDd800g2Sr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd800g2Sr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Sr4LowerPage00,
                                    'Arista Networks ' ),
   "AristaQsfpDd800g2Sr4Upper00": ( 'qsfpDd', QsfpDd.QsfpDd800g2Sr4UpperPage00,
                                    'Arista Networks ' ),
   "AristaQsfpDd800g2Sr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd800g2Sr4UpperPage01,
                                    'Arista Networks ' ),
   "AristaQsfpDd800g2Sr4Upper02": ( 'qsfpDd', QsfpDd.QsfpDd800g2Sr4UpperPage02,
                                    'Arista Networks ' ),
   "AristaOsfp800g2Sr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800g2Sr4Lower00": ( 'osfp', Osfp.Osfp800g2Sr4LowerPage00,
                                    'Arista Networks ' ),
   "AristaOsfp800g2Sr4Upper00": ( 'osfp', Osfp.Osfp800g2Sr4UpperPage00,
                                    'Arista Networks ' ),
   "AristaOsfp800g2Sr4Upper01": ( 'osfp', Osfp.Osfp800g2Sr4UpperPage01,
                                    'Arista Networks ' ),
   "AristaOsfp800g2Sr4Upper02": ( 'osfp', Osfp.Osfp800g2Sr4UpperPage02,
                                    'Arista Networks ' ),
   "AristaQsfpCmis200gAr4": ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis200gAr4Lower00": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gAr4LowerPage00,
                                      "Arista Networks " ),
   "AristaQsfpCmis200gAr4Upper01": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis200gAr4UpperPage01,
                                      "Arista Networks " ),

   "AristaQsfpDd400gAr4": ( 'qsfpDd', None, "Arista Networks " ),
   "AristaQsfpDd400gAr4Lower00": ( 'qsfpDd', QsfpDd.QsfpDd400gAr4LowerPage00,
                                   "Arista Networks " ),
   "AristaQsfpDd400gAr4Upper00": ( 'qsfpDd', QsfpDd.QsfpDd400gAr4UpperPage00,
                                   "Arista Networks " ),
   "AristaQsfpDd400gAr4Upper01": ( 'qsfpDd', QsfpDd.QsfpDd400gAr4UpperPage01,
                                   "Arista Networks " ),

   "AristaOsfp400gAr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp400gAr4Lower00": ( 'osfp', Osfp.Osfp400gAr4LowerPage00,
                                   "Arista Networks " ),
   "AristaOsfp400gAr4Upper00": ( 'osfp', Osfp.Osfp400gAr4UpperPage00,
                                   "Arista Networks " ),
   "AristaOsfp400gAr4Upper01": ( 'osfp', Osfp.Osfp400gAr4UpperPage01,
                                   "Arista Networks " ),

   # OSFP 800G-CR8 ( base entry + Id EEPROM pages lowerPage00 and upperPage00 )
   "VeridianOsfp800gCr8": ( 'osfp', None, "VeridianDynamics" ),
   "AristaOsfp800gCr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800gCr8Lower00": ( 'osfp', Osfp.Cmis4Osfp800gCr8LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp800gCr8Upper00": ( 'osfp', Osfp.Cmis4Osfp800gCr8UpperPage00,
                                  "Arista Networks " ),
   # OSFP 800GBASE-2FR4
   "AristaOsfp800g2Fr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800g2Fr4Lower00": ( 'osfp', Osfp.Osfp800g2Fr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Fr4Upper00": ( 'osfp', Osfp.Osfp800g2Fr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Fr4Upper01": ( 'osfp', Osfp.Osfp800g2Fr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp800g2Fr4Upper02": ( 'osfp', Osfp.Osfp800g2Fr4UpperPage02,
                                    "Arista Networks " ),
   # 3rd party OSFP 800GBASE-2FR4
   "Osfp800g2Fr43rdParty": ( 'osfp', None, "Unknown Corp " ),
   "Osfp800g2Fr43rdPartyLower00": ( 'osfp', Osfp.Osfp800g2Fr4LowerPage003rdParty,
                                    "Unknown Corp " ),
   "Osfp800g2Fr43rdPartyUpper00": ( 'osfp', Osfp.Osfp800g2Fr4UpperPage00,
                                    "Unknown Corp " ),
   "Osfp800g2Fr43rdPartyUpper01": ( 'osfp', Osfp.Osfp800g2Fr4UpperPage01,
                                    "Unknown Corp " ),
   "Osfp800g2Fr43rdPartyUpper02": ( 'osfp', Osfp.Osfp800g2Fr4UpperPage02,
                                    "Unknown Corp " ),

   # Finisar OSFP 800GBASE-2FR4
   "Osfp800g2Fr4Finisar": ( 'osfp', None, "Finisar Corp " ),
   "Osfp800g2Fr4FinisarLower00": ( 'osfp', Osfp.Osfp800g2Fr4FinisarLowerPage00,
                                    "Finisar Corp " ),

   # OSFP 800GBASE-DR8
   "AristaOsfp800gDr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800gDr8Lower00": ( 'osfp', Osfp.Osfp800gDr8LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp800gDr8Upper00": ( 'osfp', Osfp.Osfp800gDr8UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp800gDr8Upper01": ( 'osfp', Osfp.Osfp800gDr8UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp800gDr8Upper02": ( 'osfp', Osfp.Osfp800gDr8UpperPage02,
                                    "Arista Networks " ),

   # 3rd party OSFP 800GBASE-DR8
   "WtdOsfp800gDr8": ( 'osfp', None, "WTD             " ),
   "WtdOsfp800gDr8Lower00": ( 'osfp', Osfp.WtdOsfp800gDr8LowerPage00,
                                    "WTD             " ),
   "WtdOsfp800gDr8Upper00": ( 'osfp', Osfp.WtdOsfp800gDr8UpperPage00,
                                    "WTD             " ),
   "WtdOsfp800gDr8Upper01": ( 'osfp', Osfp.WtdOsfp800gDr8UpperPage01,
                                    "WTD             " ),
   "WtdOsfp800gDr8Upper02": ( 'osfp', Osfp.WtdOsfp800gDr8UpperPage02,
                                    "WTD             " ),

   # 3rd party OSFP 800GBASE-DR8-2
   "InnolightOsfp800gXdr8": ( 'osfp', None, "INNOLIGHT       " ),
   "InnolightOsfp800gXdr8Lower00": ( 'osfp', Osfp.InnolightOsfp800gXdr8LowerPage00,
                                      "INNOLIGHT       " ),
   "InnolightOsfp800gXdr8Upper00": ( 'osfp', Osfp.InnolightOsfp800gXdr8UpperPage00,
                                      "INNOLIGHT       " ),
   "InnolightOsfp800gXdr8Upper01": ( 'osfp', Osfp.InnolightOsfp800gXdr8UpperPage01,
                                      "INNOLIGHT       " ),
   "InnolightOsfp800gXdr8Upper02": ( 'osfp', Osfp.InnolightOsfp800gXdr8UpperPage02,
                                      "INNOLIGHT       " ),

   # OSFP 800GBASE-XDR8
   "AristaOsfp800gXdr8": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800gXdr8Lower00": ( 'osfp', Osfp.Osfp800gXdr8LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp800gXdr8Upper00": ( 'osfp', Osfp.Osfp800gXdr8UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp800gXdr8Upper01": ( 'osfp', Osfp.Osfp800gXdr8UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp800gXdr8Upper02": ( 'osfp', Osfp.Osfp800gXdr8UpperPage02,
                                    "Arista Networks " ),
   # OSFP 800GBASE-2XDR4
   "AristaOsfp800g2Xdr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800g2Xdr4Lower00": ( 'osfp', Osfp.Osfp800g2Xdr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Xdr4Upper00": ( 'osfp', Osfp.Osfp800g2Xdr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Xdr4Upper01": ( 'osfp', Osfp.Osfp800g2Xdr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp800g2Xdr4Upper02": ( 'osfp', Osfp.Osfp800g2Xdr4UpperPage02,
                                    "Arista Networks " ),
   # OSFP 800GBASE-2DR4
   "AristaOsfp800g2Dr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800g2Dr4Lower00": ( 'osfp', Osfp.Osfp800g2Dr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Dr4Upper00": ( 'osfp', Osfp.Osfp800g2Dr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Dr4Upper01": ( 'osfp', Osfp.Osfp800g2Dr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp800g2Dr4Upper02": ( 'osfp', Osfp.Osfp800g2Xdr4UpperPage02,
                                    "Arista Networks " ),

   # OSFP 800GBASE-2DR4 based LPO module
   "AristaOsfpLpo800g2Dr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4Lower00": ( 'osfp', Osfp.OsfpLpo800g2Dr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4Upper00": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4Upper01": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4Upper02": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPage02,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4Upper10": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPage10,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4Upper11": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPage11,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4UpperC1": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPageC1,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Dr4UpperC2": ( 'osfp', Osfp.OsfpLpo800g2Dr4UpperPageC2,
                                    "Arista Networks " ),

   # OSFP 800GBASE-2PLR4
   "AristaOsfp800g2Plr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp800g2Plr4Lower00": ( 'osfp', Osfp.Osfp800g2Plr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Plr4Upper00": ( 'osfp', Osfp.Osfp800g2Plr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfp800g2Plr4Upper01": ( 'osfp', Osfp.Osfp800g2Plr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfp800g2Plr4Upper02": ( 'osfp', Osfp.Osfp800g2Plr4UpperPage02,
                                    "Arista Networks " ),
   # OSFP LPO 800GBASE-2XDR4
   "AristaOsfpLpo800g2Xdr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfpLpo800g2Xdr4Lower00": ( 'osfp', Osfp.Osfp800g2Xdr4LowerPage00,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Xdr4Upper00": ( 'osfp', Osfp.OsfpLpo800g2Xdr4UpperPage00,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Xdr4Upper01": ( 'osfp', Osfp.Osfp800g2Xdr4UpperPage01,
                                    "Arista Networks " ),
   "AristaOsfpLpo800g2Xdr4Upper02": ( 'osfp', Osfp.Osfp800g2Xdr4UpperPage02,
                                    "Arista Networks " ),
   # QsfpCmis 400G-CR4 ( base entry + Id EEPROM pages lowerPage00 and upperPage00 )
   "VeridianQsfpCmis400gCr4": ( 'qsfpCmis', None, "VeridianDynamics" ),
   "AristaQsfpCmis400gCr4": ( 'qsfpCmis', None, "Arista Networks " ),
   "AristaQsfpCmis400gCr4Lower00": ( 'qsfpCmis',
                                      QsfpCmis.QsfpCmis400gCr4LowerPage00,
                                      "Arista Networks " ),
   "AristaQsfpCmis400gCr4Upper00": ( 'qsfpCmis', Osfp.Cmis4Osfp400gCr8UpperPage00,
                                      "Arista Networks " ),
   # OSFP 2x400G-CR4 runs on lane 0 and lane 4, identify as 800G-CR8
   "AristaOsfp2x400gCr4": ( 'osfp', None, "Arista Networks " ),
   "AristaOsfp2x400gCr4Lower00": ( 'osfp', Osfp.Osfp2x400gCr4LowerPage00,
                                  "Arista Networks " ),
   "AristaOsfp2x400gCr4Upper00": ( 'osfp', Osfp.Cmis4Osfp800gCr8UpperPage00,
                                  "Arista Networks " ),
   # Page 13 (PRBS) Template for all xcvrs that support it
   'AristaCmisUpper13Bank0': ( 'osfp', Osfp.CmisUpperPage13Bank0,
                                "Arista Networks " ),

   # Blanco.  Special 200G Bypass module
   'AristaBlanco200GBypass': ( 'qsfpDd', None,
                                "Arista Networks " ),
   'AristaBlanco200GBypassLower00h': ( 'qsfpDd', Blanco.blancoLowerPage00h,
                                        "Arista Networks " ),
   'AristaBlanco200GBypassUpper00h': ( 'qsfpDd', Blanco.blancoUpperPage00h,
                                        "Arista Networks " ),

   # QSFP-LS; SFF8636-based POLS module
   'AristaQsfpAmpZr': ( 'qsfp', Qsfp.QsfpLsUpperPage0, "Arista Networks " ),
   'AristaQsfpAmpZrLower00h': ( 'qsfp', Qsfp.QsfpLsLowerPage0, "Arista Networks " ),
   'AristaQsfpAmpZrUpper02h': ( 'qsfp', Qsfp.QsfpLsUpperPage2, "Arista Networks " ),
   'AristaQsfpAmpZrUpper03h': ( 'qsfp', Qsfp.QsfpLsUpperPage3, "Arista Networks " ),

   # Dsfp types.
   "VeridianDsfp100gCr2": ( 'dsfp', None, "VeridianDynamics" ),
   "Arista100GCR2Dsfp": ( 'dsfp', None, "Arista Networks " ),
   "Arista100GCR2DsfpLower00": ( 'dsfp', Dsfp.Dsfp100gCr2LowerPage00,
                                  "Arista Networks " ),
   "Arista100GCR2DsfpUpper00": ( 'dsfp', Dsfp.Dsfp100gCr2UpperPage00,
                                  "Arista Networks " ),

   "Arista100GSR2Dsfp": ( 'dsfp', None, "Arista Networks " ),
   "Arista100GSR2DsfpLower00": ( 'dsfp', Dsfp.Dsfp100gSr2LowerPage00,
                                  "Arista Networks " ),
   "Arista100GSR2DsfpUpper00": ( 'dsfp', Dsfp.Dsfp100gSr2UpperPage00,
                                  "Arista Networks " ),
   "Arista100GSR2DsfpUpper01": ( 'dsfp', Dsfp.Dsfp100gSr2UpperPage01,
                                  "Arista Networks " ),
   "Arista100GSR2DsfpUpper02": ( 'dsfp', Dsfp.Dsfp100gSr2UpperPage02,
                                  "Arista Networks " ),
   "Arista100GSR2DsfpUpper10": ( 'dsfp', Dsfp.Dsfp100gSr2UpperPage10,
                                  "Arista Networks " ),
   "Arista100GSR2DsfpUpper11": ( 'dsfp', Dsfp.Dsfp100gSr2UpperPage11,
                                  "Arista Networks " ),
   "Arista100GAr2Dsfp": ( "dsfp", None, "Arista Networks " ),
   "Arista100GAr2DsfpLower00": ( "dsfp", Dsfp.Dsfp100gAr2LowerPage00,
                                  "Arista Networks " ),
   "Arista100GAr2DsfpUpper00": ( "dsfp", Dsfp.Dsfp100gAr2UpperPage00,
                                  "Arista Networks " ),
   "Arista100GAr2DsfpUpper01": ( "dsfp", Dsfp.Dsfp100gAr2UpperPage01,
                                  "Arista Networks " ),
   "Arista100GAr2DsfpUpper02": ( "dsfp", Dsfp.Dsfp100gAr2UpperPage02,
                                  "Arista Networks " ),
   "Arista100GAr2DsfpUpper10": ( "dsfp", Dsfp.Dsfp100gAr2UpperPage10,
                                  "Arista Networks " ),
   "Arista100GAr2DsfpUpper11": ( "dsfp", Dsfp.Dsfp100gAr2UpperPage11,
                                  "Arista Networks " ),
   # SfpDd types.
   "Arista100GCR2SfpDd": ( "sfpDd", None, "Arista Networks " ),
   "Arista100GCR2SfpDdLower00": ( "sfpDd", SfpDd.SfpDd100gCr2LowerPage00,
                                   "Arista Networks " ),
   "Arista100GCR2SfpDdUpper00": ( "sfpDd", SfpDd.SfpDd100gCr2UpperPage00,
                                   "Arista Networks " ),
   "Arista100GSR2SfpDd": ( "sfpDd", None, "Arista Networks " ),
   "Arista100GSR2SfpDdLower00": ( "sfpDd", SfpDd.SfpDd100gSr2LowerPage00,
                                   "Arista Networks " ),
   "Arista100GSR2SfpDdUpper00": ( "sfpDd", SfpDd.SfpDd100gSr2UpperPage00,
                                   "Arista Networks " ),
   "Arista100GSR2SfpDdUpper01": ( "sfpDd", SfpDd.SfpDd100gSr2UpperPage01,
                                   "Arista Networks " ),
   # QSFP112
   "AristaQsfpCmis400gCra4": ( "qsfpCmis", None, "Arista Networks " ),
   "AristaQsfpCmis400gCra4Lower00": ( "qsfpCmis",
                                       QsfpCmis.QsfpCmis400gCra4LowerPage00,
                                       "Arista Networks " ),
   "AristaQsfpCmis400gCra4Upper00": ( "qsfpCmis",
                                       QsfpCmis.QsfpCmis400gCra4UpperPage00,
                                       "Arista Networks " ),
   "AristaQsfpCmis400gCra4Upper01": ( "qsfpCmis",
                                       QsfpCmis.QsfpCmis400gCra4UpperPage01,
                                       "Arista Networks " ),
   # Finisar QSFP-DD 200G to 2 x QSFP100 AOC
   "FinisarQsfpCmis100g": ( "qsfpCmis", None, "Arista Networks " ),
   "FinisarQsfpCmis100gLower00": ( "qsfpCmis",
                                   QsfpCmis.FinisarQsfpCmis100gLowerPage00,
                                   "Arista Networks " ),
   "FinisarQsfpCmis100gUpper00": ( "qsfpCmis",
                                   QsfpCmis.FinisarQsfpCmis100gUpperPage00,
                                   "Arista Networks " ),
   "FinisarQsfpCmis100gUpper01": ( "qsfpCmis",
                                   QsfpCmis.FinisarQsfpCmis100gUpperPage01,
                                   "Arista Networks " ),
   "FinisarQsfpCmis100gUpper02": ( "qsfpCmis",
                                   QsfpCmis.FinisarQsfpCmis100gUpperPage02,
                                   "Arista Networks " ),
   "FinisarQsfpCmis100gUpper10": ( "qsfpCmis",
                                   QsfpCmis.FinisarQsfpCmis100gUpperPage10,
                                   "Arista Networks " ),
   "FinisarQsfpCmis100gUpper11": ( "qsfpCmis",
                                   QsfpCmis.FinisarQsfpCmis100gUpperPage11,
                                   "Arista Networks " ),
   "FinisarQsfpDd200g": ( "qsfpDd", None, "Arista Networks " ),
   "FinisarQsfpDd200gLower00": ( "qsfpDd", QsfpDd.FinisarQsfpDd200gLowerPage00,
                                "Arista Networks " ),
   "FinisarQsfpDd200gUpper00": ( "qsfpDd", QsfpDd.FinisarQsfpDd200gUpperPage00,
                                "Arista Networks " ),
   "FinisarQsfpDd200gUpper01": ( "qsfpDd", QsfpDd.FinisarQsfpDd200gUpperPage01,
                                "Arista Networks " ),
   "FinisarQsfpDd200gUpper02": ( "qsfpDd", QsfpDd.FinisarQsfpDd200gUpperPage02,
                                 "Arista Networks " ),
   "FinisarQsfpDd200gUpper10": ( "qsfpDd", QsfpDd.FinisarQsfpDd200gUpperPage10,
                                 "Arista Networks " ),
   "FinisarQsfpDd200gUpper11": ( "qsfpDd", QsfpDd.FinisarQsfpDd200gUpperPage11,
                                 "Arista Networks " ),
   'AristaQsfpDd800gZr': ( 'qsfpDd', None, "Arista Networks " ),
   'AristaQsfpDd800gZrLower00': ( 'qsfpDd', QsfpDd.QsfpDd800gZrLowerPage00,
                                   "Arista Networks " ),
   'AristaQsfpDd800gZrUpper00': ( 'qsfpDd', QsfpDd.QsfpDd800gZrUpperPage00,
                                   "Arista Networks " ),
   'AristaQsfpDd800gZrUpper01': ( 'qsfpDd', QsfpDd.QsfpDd800gZrUpperPage01,
                                   "Arista Networks " ),
   'AristaOsfp800gZr': ( 'osfp', None, "Arista Networks " ),
   'AristaOsfp800gZrLower00': ( 'osfp',
                                QsfpDd.Osfp800gZrLowerPage00,
                                "Arista Networks " ),
   'AristaOsfp800gZrB': ( 'osfp', None, "Arista Networks " ),
   'AristaOsfp800gZrBLower00': ( 'osfp',
                                 QsfpDd.Osfp800gZrBLowerPage00,
                                 "Arista Networks " ),
   'AristaOsfp800gZrC': ( 'osfp', None, "Arista Networks " ),
   'AristaOsfp800gZrCLower00': ( 'osfp',
                                 QsfpDd.Osfp800gZrCLowerPage00,
                                 "Arista Networks " ),
   }

qsfpEepromRevisions = {
   "AristaSR4Qsfp": 0,
   "AristaSR4Qsfp1002": 0,
   "AristaSR4Qsfp1003": 0,
   "AristaSR4Qsfp1004": 0,
   "AristaSR4QsfpNSRB1003": 0,
   "AristaSR4DQsfp": 0,
   "AristaSRBDQsfp": 0,
   "3rdPartySRBDQsfp": 0,
   "AristaXSR4Qsfp": 0,
   "BoboSR4Qsfp": 0,
   "AristaAR4Qsfp": 0,
   "AristaCR4Qsfp": 0,
   "Qsfp400gCr4": 0,
   "Qsfp400gCr4NoCompCode": 0,
   "3rdPartyCR4Qsfp": 0,
   "AristaLR4Qsfp": 0,
   "AristaLR4Qsfp1002": 0,
   "AristaLRL4Qsfp": 0,
   "AristaPLR4Qsfp": 0,
   "AristaPLRL4Qsfp": 0,
   "3rdPartyPLR4Qsfp": 0,
   "Psm4Qsfp": 0,
   "AristaUNIVQsfp": sff8636rev15,
   "Arista100GSR4Qsfp": sff8636rev15,
   "Arista100GSR4EQsfp": sff8636rev15,
   "Arista100GXSR4Qsfp": sff8636rev15,
   "3rdParty100G40GSR4Qsfp": sff8636rev15,
   "Arista100GLR4Qsfp": sff8636rev15,
   "Arista100GLRL4Qsfp": sff8636rev15,
   "3rdParty100G40GLR4Qsfp": sff8636rev20,
   "Arista100GER4Qsfp": sff8636rev15,
   "AristaER4Qsfp": sff8636rev15,
   "Arista100GAR4Qsfp": sff8636rev15,
   "Arista100GAR4NQsfp": sff8636rev15,
   "3rdParty100G40GAR4Qsfp": sff8636rev15,
   "3rdParty100G40GAR4NQsfp": sff8636rev15,
   "Arista100GCR4Qsfp": sff8636rev15,
   "Arista100GCR4SQsfp": sff8636rev15,
   "Arista100GCR4NQsfp": sff8636rev15,
   "Arista100GCRA4Qsfp": sff8636rev15,
   "Arista50GCR2LQsfp": sff8636rev15,
   "Arista50GCR2SQsfp": sff8636rev15,
   "Arista50GCR2NQsfp": sff8636rev15,
   "3rdParty100GCR4Qsfp": sff8636rev15,
   "3rdParty100G40GCR4Qsfp": sff8636rev15,
   "Arista100GPSM4Qsfp": sff8636rev15,
   "Arista100GPLRL4Qsfp": sff8636rev15,
   "Arista100GPLR4Qsfp": sff8636rev15,
   "Arista100GCLR4Qsfp": sff8636rev15,
   "Arista100GCWDM4Qsfp": sff8636rev15,
   "Arista100GEDWDM2Qsfp": sff8636rev15,
   "Arista100GSWDM4Qsfp": sff8636rev15,
   "Arista100GSRBDQsfp": sff8636rev15,
   "Arista100GERL4Qsfp": sff8636rev15,
   "Arista100GDRQsfp": sff8636rev15,
   "Arista100GFRQsfp": sff8636rev15,
   "Arista100GFRQsfpEnhanceDom": sff8636rev15,
   "Arista100GLRQsfp": sff8636rev15,
   "Arista100GFRFecQsfp": sff8636rev15,
   "Arista100GLRFecQsfp": sff8636rev15,
   "Arista100GERQsfp": sff8636rev15,
   "Arista100GERFecQsfp": sff8636rev15,
   "Arista100GXCWDM4Qsfp": sff8636rev15,
   "Arista100GZR4Qsfp": sff8636rev15,
   "AristaTxPowerQsfp": 0,
   "Arista400g2QqsfpSide": sff8636rev15,
   "Arista400g2QqsfpSide40G": sff8636rev15,
   "Arista400g2QqsfpSideDepop": sff8636rev15,
   "UnknownQsfp": 0,
   "Unknown100GQsfp": 0,
   "Unknown100GQsfpOptics": 0,
   "FaultyQsfp": 0,
   "AristaQsfp56Sff8636Cu200G": sff8636rev28,
   "3pxQsfp56Sff8636Cu200G": sff8636rev28,
   "AristaQsfpAmpZr": sff8636rev28,
   "AristaQsfp56Sff8636Sr4": sff8636rev28,
   "AristaQsfp56Sff8636Fr4": sff8636rev28,
   "AristaQsfp56Sff8636Ar4": sff8636rev28,
   "AristaQsfp56Sff8636Sr1_2": sff8636rev20,
   }

def getXcvrType( mediaType ):
   ( xcvrType, _templBytes, _vendorName ) = eepromTemplates[ mediaType ]
   return xcvrType

def getVendorName( mediaType ):
   ( _xcvrType, _templBytes, vendorName ) = eepromTemplates[ mediaType ]
   return vendorName

def getBytes( mediaType ):
   ( _xcvrType, templBytes, _vendorName ) = eepromTemplates[ mediaType ]
   return templBytes

def convertBytesToBinStr( bytez ):
   return bytes( bytearray( bytez ) )

def zeroOutStrEntries( byteList ):
   bytez = []
   for byte in byteList:
      i = byte
      if isinstance( i, str ):
         i = 0
      bytez.append( i )
   return bytez

def convertSff8472EepromTemplateToBytes( mediaType ):
   idEeprom = Tac.Value( "Xcvr::XcvrSff8472IdEepromContents" )
   templBytes = getBytes( mediaType )
   # gyorgym: This makes no sense.  Why is this 128 bytes when SFP pages are 256
   # bytes in length...
   #
   # I've added 128 bytes of 0s to get our page to 256 bytes in length.
   assert len( templBytes ) == 128, ( 'EEPROM template %s has %d '
          'bytes instead of the expected 128' % ( mediaType, len( templBytes ) ) )
   padding = bytes( bytearray( [ 0 ] * 128 ) )  # 128 bytes of padding
   bytez = zeroOutStrEntries( templBytes )
   idEeprom.buf = convertBytesToBinStr( bytez ) + padding
   vendorName = getVendorName( mediaType )
   if vendorName:
      idEeprom.vendorName = vendorName

   return idEeprom

def convertSff8436EepromTemplateToBytes( mediaType ):
   idEeprom = Tac.Value( "Xcvr::Sff8436IdEepromContents" )
   templBytes = getBytes( mediaType )
   assert len( templBytes ) == 128, ( 'EEPROM template %s has %d '
          'bytes instead of the expected 128' % ( mediaType, len( templBytes ) ) )
   bytez = zeroOutStrEntries( templBytes )
   idEeprom.eepromPage00UpperData = convertBytesToBinStr( bytez )

   return idEeprom

def cfp2Tables( mediaType ):
   if mediaType == 'AristaLR4Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaLR4Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaLR4Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaLR4Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaLR4Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaLR4Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaLR4Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaLR4Cfp2_tableA400' ),
             ]
   elif mediaType == 'BoboLR4Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'BoboLR4Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'BoboLR4Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'BoboLR4Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'BoboLR4Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'BoboLR4Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'BoboLR4Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'BoboLR4Cfp2_tableA400' ),
             ]
   elif mediaType == 'AristaSR10Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaSR10Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaSR10Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaSR10Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaSR10Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaSR10Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaSR10Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaSR10Cfp2_tableA400' ),
             ]
   elif mediaType == 'AristaXSR10Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaXSR10Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaXSR10Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaXSR10Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaXSR10Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaXSR10Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaXSR10Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaXSR10Cfp2_tableA400' ),
             ]
   elif mediaType == 'AristaER4Cfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaER4Cfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaER4Cfp2_table8080' ),
               ( 'table8420', 0x8420, 8, 'AristaER4Cfp2_table8420' ),
               ( 'tableA000', 0xA000, 16, 'AristaER4Cfp2_tableA000' ),
               ( 'tableA200', 0xA200, 16, 'AristaER4Cfp2_tableA200' ),
               ( 'tableA280', 0xA280, 16, 'AristaER4Cfp2_tableA280' ),
               ( 'tableA400', 0xA400, 16, 'AristaER4Cfp2_tableA400' ),
             ]
   elif mediaType == 'BoboDwdmCoherentCfp2':
      return [ ( 'table8000', 0x8000, 8, 'BoboDwdmCoherentCfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'BoboDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'BoboDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'BoboDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'BoboDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'BoboDwdmCoherentCfp2_tableB180' ),
               ( 'tableB300', 0xB300, 16, 'BoboDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'BoboDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'BoboDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'BoboDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'BoboDwdmCoherentCfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'BoboDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaDwdmCoherentCfp2':
      return [ ( 'table8000', 0x8000, 8, 'AristaDwdmCoherentCfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentCfp2_tableB180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentCfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaDwdmCoherentECfp2': # Oclaro
      return [ ( 'table8000', 0x8000, 8, 'AristaDwdmCoherentCfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_tableB180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaDwdmCoherentECfp2V2': # Acacia
      return [ ( 'table8000', 0x8000, 8, 'AristaDwdmCoherentECfp2V2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'OclaroDwdmCoherentECfp2':
      return [ ( 'table8000', 0x8000, 8, 'OclaroDwdmCoherentECfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AcaciaDwdmCoherentECfp2':
      return [ ( 'table8000', 0x8000, 8, 'AcaciaDwdmCoherentECfp2_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaDwdmCoherentCfp2_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaDwdmCoherentCfp2_table8180' ),
               ( 'table8420', 0x8420, 8, 'AristaDwdmCoherentCfp2_table8420' ),
               ( 'tableB000', 0xB000, 16, 'AristaDwdmCoherentCfp2_tableB000' ),
               ( 'tableB180', 0xB180, 16, 'AristaDwdmCoherentECfp2_table8180' ),
               ( 'tableB300', 0xB300, 16, 'AristaDwdmCoherentCfp2_tableB300' ),
               ( 'tableB400', 0xB400, 16, 'AristaDwdmCoherentCfp2_tableB400' ),
               ( 'tableB480', 0xB480, 16, 'AristaDwdmCoherentCfp2_tableB480' ),
               ( 'tableBA80', 0xBA80, 16, 'AristaDwdmCoherentCfp2_tableBA80' ),
               ( 'tableBB00', 0xBB00, 16, 'AristaDwdmCoherentECfp2_tableBB00' ),
               ( 'tableBB80', 0xBB80, 16, 'AristaDwdmCoherentCfp2_tableBB80' ),
             ]
   elif mediaType == 'AristaCfp2DwdmDco':
      return [ ( 'table8000', 0x8000, 8, 'AristaCfp2DwdmDco_table8000' ),
               ( 'table8080', 0x8080, 8, 'AristaCfp2DwdmDco_table8080' ),
               ( 'table8180', 0x8180, 8, 'AristaCfp2DwdmDco_table8180' ),
               ( 'table9000', 0x9000, 16, 'AristaCfp2DwdmDco_table9000' ),
               ( 'table9100', 0x9100, 16, 'AristaCfp2DwdmDco_table9100' ),
               ( 'tableB000', 0xB000, 16, 'AristaCfp2DwdmDco_tableB000' ),
               ( 'tableB480', 0xB480, 16, 'AristaCfp2DwdmDco_tableB480' ),
             ]
   elif mediaType == 'AcaciaCfp2DwdmDco':
      return [ ( 'table8000', 0x8000, 8, 'AcaciaCfp2DwdmDco_table8000' ),
               ( 'table8080', 0x8080, 8, 'AcaciaCfp2DwdmDco_table8080' ),
               ( 'table8180', 0x8180, 8, 'AcaciaCfp2DwdmDco_table8180' ),
               ( 'table9000', 0x9000, 16, 'AcaciaCfp2DwdmDco_table9000' ),
               ( 'tableB000', 0xB000, 16, 'AcaciaCfp2DwdmDco_tableB000' ),
               ( 'tableB480', 0xB480, 16, 'AcaciaCfp2DwdmDco_tableB480' ),
             ]
   elif mediaType == 'AcaciaCfp2DlDwdmDco':
      return [ ( 'table8000', 0x8000, 8, 'AcaciaCfp2DlDwdmDco_table8000' ),
               ( 'table8080', 0x8080, 8, 'AcaciaCfp2DlDwdmDco_table8080' ),
               ( 'table8180', 0x8180, 8, 'AcaciaCfp2DlDwdmDco_table8180' ),
               ( 'table9000', 0x9000, 16, 'AcaciaCfp2DlDwdmDco_table9000' ),
               ( 'tableB000', 0xB000, 16, 'AcaciaCfp2DlDwdmDco_tableB000' ),
               ( 'tableB480', 0xB480, 16, 'AcaciaCfp2DlDwdmDco_tableB480' ),
             ]
   else:
      assert 0, "New mediaType is not implemented in cfp2Tables()"
      return None

def convertCfp2EepromTemplateToBytes( mediaType ):
   cfp2 = Tac.newInstance( "Xcvr::CfpMsaMisIdEepromContents", "idprom" )
   for ( tableId, addr, dataWidth, templateStr ) in cfp2Tables( mediaType ):
      templBytes = getBytes( templateStr )
      dataBytes = zeroOutStrEntries( templBytes )
      memory = cfp2.newMemory( tableId, addr, dataWidth )
      memory.contents = convertBytesToBinStr( dataBytes )
   return cfp2

# Given a media type, return a list of tuples of the form
# ( CmisBankAndPage, pageTemplateString ) such that pageTemplateString
# indexes into the eepromTemplates table.  Accepts CMIS transceiver
# types i.e. OSFP and QSFP-DD
def cmisPages( mediaType ):
   if not XcvrLib.isCmisTypeStr( getXcvrType( mediaType ) ):
      assert False, "%s does not map to a CMIS xcvr type" % mediaType

   osfpBank = Tac.Type( "Xcvr::CmisBank" )
   osfpPage = Tac.Type( "Xcvr::CmisPage" )
   lowerPage00 = Tac.Value( "Xcvr::CmisBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.lowerPage00 )
   upperPage00 = Tac.Value( "Xcvr::CmisBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage00 )
   upperPage01 = Tac.Value( "Xcvr::CmisBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage01 )
   upperPage02 = Tac.Value( "Xcvr::CmisBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage02 )
   upperPage04 = Tac.Value( "Xcvr::CmisBankAndPage",
                            osfpBank.bankNa,
                            osfpPage.upperPage04 )
   upperPage10Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage10 )
   upperPage11Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage11 )
   upperPage12Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage12 )
   upperPage13Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage13 )
   upperPage20Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage20 )
   upperPage21Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage21 )
   upperPage22Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage22 )
   upperPage23Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage23 )
   upperPage28Bank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage28 )
   upperPage2FBank0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bank0,
                                 osfpPage.upperPage2F )
   upperPageC0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageC0 )
   upperPageC1 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageC1 )
   upperPageC2 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageC2 )
   upperPageD0 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD0 )
   upperPageD1 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD1 )
   upperPageD8 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD8 )
   upperPageD9 = Tac.Value( "Xcvr::CmisBankAndPage",
                                 osfpBank.bankNa,
                                 osfpPage.upperPageD9 )

   if mediaType in ( 'AristaOsfp400gCr8', 'VeridianOsfp400gCr8', ):
      return [ ( lowerPage00, 'AristaOsfp400gCr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gCr8Upper00' ) ]
   elif mediaType in ( 'AristaCmis4Osfp400gCr8', ):
      return [ ( lowerPage00, 'AristaCmis4Osfp400gCr8Lower00' ),
               ( upperPage00, 'AristaCmis4Osfp400gCr8Upper00' ) ]
   elif mediaType in ( 'AristaQsfpDd400gCr8', 'VeridianQsfpDd400gCr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gCr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gCr8Upper00' ) ]
   elif mediaType in ( 'AristaQsfpCmis200gCr4', 'VeridianQsfpCmis200gCr4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis200gCr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis200gCr4Upper00' ) ]
   elif mediaType in ( 'AristaOsfp400gSr8', 'VeridianOsfp400gSr8', ):
      return [ ( lowerPage00, 'AristaOsfp400gSr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gSr8Cmis3', ):
      return [ ( lowerPage00, 'AristaOsfp400gSr8Cmis3Lower00' ),
               ( upperPage00, 'AristaOsfp400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gAr8', 'VeridianOsfp400gAr8', ):
      return [ ( lowerPage00, 'AristaOsfp400gAr8Lower00' ),
               ( upperPage00, 'AristaOsfp400gAr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gAr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gAr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gAr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gAr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gCra8', ):
      return [ ( lowerPage00, 'AristaOsfp400gCra8Lower00' ),
               ( upperPage00, 'AristaOsfp400gCra8Upper00' ),
               ( upperPage01, 'AristaOsfp400gCra8Upper01' ),
               ( upperPage02, 'AristaOsfp400gAr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gAr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gAr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400g2Fr4', ):
      return [ ( lowerPage00, 'AristaOsfp400g2Fr4Lower00' ),
               ( upperPage00, 'AristaOsfp400g2Fr4Upper00' ),
               ( upperPage01, 'AristaOsfp400g2Fr4Upper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gZr', 'AcaciaQsfpDd400gZr' ):
      # No diag page 13h implemented in this module
      return [ ( lowerPage00, 'AristaQsfpDd400gZrLower00' ),
               ( upperPage00, 'AristaQsfpDd400gZrUpper00' ),
               ( upperPage01, 'AristaQsfpDd400gZrUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrUpper12Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrUpper20Bank0' ),
               ( upperPage21Bank0, 'AristaQsfpDd400gZrUpper21Bank0' ),
               ( upperPage22Bank0, 'AristaQsfpDd400gZrUpper22Bank0' ),
               ( upperPage23Bank0, 'AristaQsfpDd400gZrUpper23Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrUpper2FBank0' ) ]
   elif mediaType in ( 'AristaOsfp400gZr', ):
      return [ ( lowerPage00, 'AristaOsfp400gZrLower00' ),
               ( upperPage00, 'AristaOsfp400gZrUpper00' ),
               ( upperPage01, 'AristaOsfp400gZrUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrUpper12Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrUpper20Bank0' ),
               ( upperPage21Bank0, 'AristaQsfpDd400gZrUpper21Bank0' ),
               ( upperPage22Bank0, 'AristaQsfpDd400gZrUpper22Bank0' ),
               ( upperPage23Bank0, 'AristaQsfpDd400gZrUpper23Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrUpper2FBank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gZrP', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gZrPLower00' ),
               ( upperPage00, 'AristaQsfpDd400gZrPUpper00' ),
               ( upperPage01, 'AristaQsfpDd400gZrPUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrPUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrPUpper12Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrPUpper20Bank0' ),
               ( upperPage21Bank0, 'AristaQsfpDd400gZrPUpper21Bank0' ),
               ( upperPage22Bank0, 'AristaQsfpDd400gZrPUpper22Bank0' ),
               ( upperPage23Bank0, 'AristaQsfpDd400gZrPUpper23Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrPUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrPUpper2FBank0' ) ]
   elif mediaType in ( 'AristaOsfp400gZrP', ):
      return [ ( lowerPage00, 'AristaOsfp400gZrPLower00' ),
               ( upperPage00, 'AristaOsfp400gZrPUpper00' ),
               ( upperPage01, 'AristaQsfpDd400gZrPUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrPUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrPUpper12Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrPUpper20Bank0' ),
               ( upperPage21Bank0, 'AristaQsfpDd400gZrPUpper21Bank0' ),
               ( upperPage22Bank0, 'AristaQsfpDd400gZrPUpper22Bank0' ),
               ( upperPage23Bank0, 'AristaQsfpDd400gZrPUpper23Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrPUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrPUpper2FBank0' ) ]
   elif mediaType in ( 'AristaOsfpAmpZr', ):
      # No diag page 13h implemented in this module
      return [ ( lowerPage00, 'AristaOsfpAmpZrLower00' ),
               ( upperPage00, 'AristaOsfpAmpZrUpper00' ),
               ( upperPage01, 'AristaOsfpAmpZrUpper01' ),
               ( upperPage02, 'AristaOsfpAmpZrUpper02' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfpAmpZrUpper11Bank0' ),
               ( upperPageC0, 'AristaOsfpAmpZrUpperC0' ),
               ( upperPageC1, 'AristaOsfpAmpZrUpperC1' ),
               ( upperPageD0, 'AristaOsfpAmpZrUpperD0' ),
               ( upperPageD1, 'AristaOsfpAmpZrUpperD1' ),
               ( upperPageD8, 'AristaOsfpAmpZrUpperD8' ),
               ( upperPageD9, 'AristaOsfpAmpZrUpperD9' ) ]
   elif mediaType in ( 'GoogleOM8FNT', ):
      return [ ( lowerPage00, 'AristaOsfp400gSr8Lower00' ),
               ( upperPage00, 'GoogleOM8FNTUpper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'GoogleOsfp8x10gLr', ):
      return [ ( lowerPage00, 'GoogleOsfp8x10gLrLower00' ),
               ( upperPage00, 'GoogleOsfp8x10gLrUpper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd8x25gSr', ):
      return [ ( lowerPage00, 'AristaQsfpDd8x25gSrLower00' ),
               ( upperPage00, 'AristaQsfpDd8x25gSrUpper00' ),
               ( upperPage01, 'AristaQsfpDd8x25gSrUpper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gSr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gSr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gSrbd', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gSrbdLower00' ),
               ( upperPage00, 'AristaQsfpDd400gSrbdUpper00' ),
               ( upperPage01, 'AristaQsfpDd400gSrbdUpper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gSr8Cmis3', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gSr8Cmis3Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp400gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800gXdr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd800gXdr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd800gXdr8Upper00' ),
               ( upperPage01, 'AristaQsfpDd800gXdr8Upper01' ),
               ( upperPage02, 'AristaQsfpDd800gXdr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gDr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gDr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gDr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gXdr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gDr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gXdr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp4x100gDr', ):
      return [ ( lowerPage00, 'AristaOsfp4x100gDrLower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp4x100gDrUpper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gFr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gFr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gDr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gPlr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gPlr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gDr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gDr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gDr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gDr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gDr4Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gSrbd', ):
      return [ ( lowerPage00, 'AristaOsfp400gSrbdLower00' ),
               ( upperPage00, 'AristaOsfp400gSrbdUpper00' ),
               ( upperPage01, 'AristaOsfp400gSrbdUpper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpCmis200gFr4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis200gFr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis200gFr4Upper00' ),
               ( upperPage01, 'AristaQsfpCmis200gFr4Upper01' ),
               ( upperPage02, 'AristaQsfpCmis200gFr4Upper02' ),
               ( upperPage10Bank0, 'AristaQsfpCmis200gFr4Upper10' ),
               ( upperPage11Bank0, 'AristaQsfpCmis200gFr4Upper11' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpCmis200gAr4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis200gAr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis200gFr4Upper00' ),
               ( upperPage01, 'AristaQsfpCmis200gAr4Upper01' ),
               ( upperPage02, 'AristaQsfpCmis200gFr4Upper02' ),
               ( upperPage10Bank0, 'AristaQsfpCmis200gFr4Upper10' ),
               ( upperPage11Bank0, 'AristaQsfpCmis200gFr4Upper11' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800gCr8', 'VeridianOsfp800gCr8' ):
      return [ ( lowerPage00, 'AristaOsfp800gCr8Lower00' ),
               ( upperPage00, 'AristaOsfp800gCr8Upper00' ) ]
   elif mediaType in ( 'AristaOsfp800g2Fr4', ):
      return [ ( lowerPage00, 'AristaOsfp800g2Fr4Lower00' ),
               ( upperPage00, 'AristaOsfp800g2Fr4Upper00' ),
               ( upperPage01, 'AristaOsfp800g2Fr4Upper01' ),
               ( upperPage02, 'AristaOsfp800g2Fr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'Osfp800g2Fr43rdParty', ):
      return [ ( lowerPage00, 'Osfp800g2Fr43rdPartyLower00' ),
               ( upperPage00, 'Osfp800g2Fr43rdPartyUpper00' ),
               ( upperPage01, 'Osfp800g2Fr43rdPartyUpper01' ),
               ( upperPage02, 'Osfp800g2Fr43rdPartyUpper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'Osfp800g2Fr4Finisar', ):
      return [ ( lowerPage00, 'Osfp800g2Fr4FinisarLower00' ),
               ( upperPage00, 'Osfp800g2Fr43rdPartyUpper00' ),
               ( upperPage01, 'Osfp800g2Fr43rdPartyUpper01' ),
               ( upperPage02, 'Osfp800g2Fr43rdPartyUpper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800gDr8', ):
      return [ ( lowerPage00, 'AristaOsfp800gDr8Lower00' ),
               ( upperPage00, 'AristaOsfp800gDr8Upper00' ),
               ( upperPage01, 'AristaOsfp800gDr8Upper01' ),
               ( upperPage02, 'AristaOsfp800gDr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'WtdOsfp800gDr8', ):
      return [ ( lowerPage00, 'WtdOsfp800gDr8Lower00' ),
               ( upperPage00, 'WtdOsfp800gDr8Upper00' ),
               ( upperPage01, 'WtdOsfp800gDr8Upper01' ),
               ( upperPage02, 'WtdOsfp800gDr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'InnolightOsfp800gXdr8', ):
      return [ ( lowerPage00, 'InnolightOsfp800gXdr8Lower00' ),
               ( upperPage00, 'InnolightOsfp800gXdr8Upper00' ),
               ( upperPage01, 'InnolightOsfp800gXdr8Upper01' ),
               ( upperPage02, 'InnolightOsfp800gXdr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800gXdr8', ):
      return [ ( lowerPage00, 'AristaOsfp800gXdr8Lower00' ),
               ( upperPage00, 'AristaOsfp800gXdr8Upper00' ),
               ( upperPage01, 'AristaOsfp800gXdr8Upper01' ),
               ( upperPage02, 'AristaOsfp800gXdr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800g2lr4', ):
      return [ ( lowerPage00, 'AristaOsfp800g2lr4Lower00' ),
               ( upperPage00, 'AristaOsfp800g2lr4Upper00' ),
               ( upperPage01, 'AristaOsfp800g2lr4Upper01' ),
               ( upperPage02, 'AristaOsfp800g2lr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800g2lr4', ):
      return [ ( lowerPage00, 'AristaQsfpDd800g2lr4Lower00' ),
               ( upperPage00, 'AristaQsfpDd800g2lr4Upper00' ),
               ( upperPage01, 'AristaQsfpDd800g2lr4Upper01' ),
               ( upperPage02, 'AristaQsfpDd800g2lr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800gPlr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd800gPlr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd800gPlr8Upper00' ),
               ( upperPage01, 'AristaQsfpDd800gPlr8Upper01' ),
               ( upperPage02, 'AristaQsfpDd800gPlr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800g2Xdr4', ):
      return [ ( lowerPage00, 'AristaQsfpDd800g2Xdr4Lower00' ),
               ( upperPage00, 'AristaQsfpDd800g2Xdr4Upper00' ),
               ( upperPage01, 'AristaQsfpDd800g2Xdr4Upper01' ),
               ( upperPage02, 'AristaQsfpDd800g2Xdr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800g2Dr4', ):
      return [ ( lowerPage00, 'AristaQsfpDd800g2Dr4Lower00' ),
               ( upperPage00, 'AristaQsfpDd800g2Dr4Upper00' ),
               ( upperPage01, 'AristaQsfpDd800g2Dr4Upper01' ),
               ( upperPage02, 'AristaQsfpDd800g2Xdr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDdLpo800g2Xdr4', ):
      return [ ( lowerPage00, 'AristaQsfpDdLpo800g2Xdr4Lower00' ),
               ( upperPage00, 'AristaQsfpDdLpo800g2Xdr4Upper00' ),
               ( upperPage01, 'AristaQsfpDdLpo800g2Xdr4Upper01' ),
               ( upperPage02, 'AristaQsfpDdLpo800g2Xdr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800g2Plr4', ):
      return [ ( lowerPage00, 'AristaQsfpDd800g2Plr4Lower00' ),
               ( upperPage00, 'AristaQsfpDd800g2Plr4Upper00' ),
               ( upperPage01, 'AristaQsfpDd800g2Plr4Upper01' ),
               ( upperPage02, 'AristaQsfpDd800g2Plr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800g2Xdr4', ):
      return [ ( lowerPage00, 'AristaOsfp800g2Xdr4Lower00' ),
               ( upperPage00, 'AristaOsfp800g2Xdr4Upper00' ),
               ( upperPage01, 'AristaOsfp800g2Xdr4Upper01' ),
               ( upperPage02, 'AristaOsfp800g2Xdr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800g2Dr4', ):
      return [ ( lowerPage00, 'AristaOsfp800g2Dr4Lower00' ),
               ( upperPage00, 'AristaOsfp800g2Dr4Upper00' ),
               ( upperPage01, 'AristaOsfp800g2Dr4Upper01' ),
               ( upperPage02, 'AristaOsfp800g2Xdr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfpLpo800g2Dr4', ):
      return [ ( lowerPage00, 'AristaOsfpLpo800g2Dr4Lower00' ),
               ( upperPage00, 'AristaOsfpLpo800g2Dr4Upper00' ),
               ( upperPage01, 'AristaOsfpLpo800g2Dr4Upper01' ),
               ( upperPage02, 'AristaOsfpLpo800g2Dr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfpLpo800g2Dr4Upper10' ),
               ( upperPage11Bank0, 'AristaOsfpLpo800g2Dr4Upper11' ),
               ( upperPageC1, 'AristaOsfpLpo800g2Dr4UpperC1' ),
               ( upperPageC2, 'AristaOsfpLpo800g2Dr4UpperC2' ), ]
   elif mediaType in ( 'AristaOsfp800g2Plr4', ):
      return [ ( lowerPage00, 'AristaOsfp800g2Plr4Lower00' ),
               ( upperPage00, 'AristaOsfp800g2Plr4Upper00' ),
               ( upperPage01, 'AristaOsfp800g2Plr4Upper01' ),
               ( upperPage02, 'AristaOsfp800g2Plr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfpLpo800g2Xdr4', ):
      return [ ( lowerPage00, 'AristaOsfpLpo800g2Xdr4Lower00' ),
               ( upperPage00, 'AristaOsfpLpo800g2Xdr4Upper00' ),
               ( upperPage01, 'AristaOsfpLpo800g2Xdr4Upper01' ),
               ( upperPage02, 'AristaOsfpLpo800g2Xdr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800gSr8', ):
      return [ ( lowerPage00, 'AristaOsfp800gSr8Lower00' ),
               ( upperPage00, 'AristaOsfp800gSr8Upper00' ),
               ( upperPage01, 'AristaOsfp800gSr8Upper01' ),
               ( upperPage02, 'AristaOsfp800gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800gSr8', ):
      return [ ( lowerPage00, 'AristaQsfpDd800gSr8Lower00' ),
               ( upperPage00, 'AristaQsfpDd800gSr8Upper00' ),
               ( upperPage01, 'AristaQsfpDd800gSr8Upper01' ),
               ( upperPage02, 'AristaQsfpDd800gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaOsfp800g2Sr4', ):
      return [ ( lowerPage00, 'AristaOsfp800g2Sr4Lower00' ),
               ( upperPage00, 'AristaOsfp800g2Sr4Upper00' ),
               ( upperPage01, 'AristaOsfp800g2Sr4Upper01' ),
               ( upperPage02, 'AristaOsfp800g2Sr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpDd800g2Sr4', ):
      return [ ( lowerPage00, 'AristaQsfpDd800g2Sr4Lower00' ),
               ( upperPage00, 'AristaQsfpDd800g2Sr4Upper00' ),
               ( upperPage01, 'AristaQsfpDd800g2Sr4Upper01' ),
               ( upperPage02, 'AristaQsfpDd800g2Sr4Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ) ]
   elif mediaType in ( 'AristaQsfpCmis400gCr4', 'VeridianQsfpCmis400gCr4' ):
      return [ ( lowerPage00, 'AristaQsfpCmis400gCr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis400gCr4Upper00' ) ]
   elif mediaType in ( 'AristaOsfp2x400gCr4', ):
      return [ ( lowerPage00, 'AristaOsfp2x400gCr4Lower00' ),
               ( upperPage00, 'AristaOsfp2x400gCr4Upper00' ) ]
   elif mediaType in ( 'Arista100GCR2Dsfp', 'VeridianDsfp100gCr2' ):
      return [ ( lowerPage00, 'Arista100GCR2DsfpLower00' ),
               ( upperPage00, 'Arista100GCR2DsfpUpper00' ) ]
   elif mediaType in ( 'Arista100GSR2Dsfp', ):
      # No diag page 13h implemented for XcvrCtrl/test/CmisEepromManagerTest.py
      return [ ( lowerPage00, 'Arista100GSR2DsfpLower00' ),
               ( upperPage00, 'Arista100GSR2DsfpUpper00' ),
               ( upperPage01, 'Arista100GSR2DsfpUpper01' ),
               ( upperPage02, 'Arista100GSR2DsfpUpper02' ),
               ( upperPage10Bank0, 'Arista100GSR2DsfpUpper10' ),
               ( upperPage11Bank0, 'Arista100GSR2DsfpUpper11' ), ]
   elif mediaType in ( 'Arista100GAr2Dsfp', ):
      return [ ( lowerPage00, 'Arista100GAr2DsfpLower00' ),
               ( upperPage00, 'Arista100GAr2DsfpUpper00' ),
               ( upperPage01, 'Arista100GAr2DsfpUpper01' ),
               ( upperPage02, 'Arista100GAr2DsfpUpper02' ),
               ( upperPage10Bank0, 'Arista100GAr2DsfpUpper10' ),
               ( upperPage11Bank0, 'Arista100GAr2DsfpUpper11' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'Arista100GCR2SfpDd', ):
      return [ ( lowerPage00, 'Arista100GCR2SfpDdLower00' ),
               ( upperPage00, 'Arista100GCR2SfpDdUpper00' ), ]
   elif mediaType in ( 'Arista100GSR2SfpDd', ):
      return [ ( lowerPage00, 'Arista100GSR2SfpDdLower00' ),
               ( upperPage00, 'Arista100GSR2SfpDdUpper00' ),
               ( upperPage01, 'Arista100GSR2SfpDdUpper01' ), ]
   elif mediaType in ( 'AristaOsfp400gTo4Qsfp', ):
      return [ ( lowerPage00, 'AristaOsfp400gTo4QsfpLower00' ),
               ( upperPage00, 'AristaOsfp400gCr8Upper00' ) ]
   elif mediaType == 'AristaBlanco200GBypass':
      return [ ( lowerPage00, 'AristaBlanco200GBypassLower00h' ),
               ( upperPage00, 'AristaBlanco200GBypassUpper00h' ) ]
   elif mediaType in ( 'AristaQsfpCmis400gCra4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis400gCra4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis400gCra4Upper00' ),
               ( upperPage01, 'AristaQsfpCmis400gCra4Upper01' ),
               ( upperPage02, 'AristaQsfpCmis200gFr4Upper02' ),
               ( upperPage10Bank0, 'AristaQsfpCmis200gFr4Upper10' ),
               ( upperPage11Bank0, 'AristaQsfpCmis200gFr4Upper11' ) ]
   elif mediaType in ( 'FinisarQsfpCmis100g', ):
      return [ ( lowerPage00, 'FinisarQsfpCmis100gLower00' ),
               ( upperPage00, 'FinisarQsfpCmis100gUpper00' ),
               ( upperPage01, 'FinisarQsfpCmis100gUpper01' ),
               ( upperPage02, 'FinisarQsfpCmis100gUpper02' ),
               ( upperPage10Bank0, 'FinisarQsfpCmis100gUpper10' ),
               ( upperPage11Bank0, 'FinisarQsfpCmis100gUpper11' ), ]
   elif mediaType in ( 'FinisarQsfpDd200g', ):
      return [ ( lowerPage00, 'FinisarQsfpDd200gLower00' ),
               ( upperPage00, 'FinisarQsfpDd200gUpper00' ),
               ( upperPage01, 'FinisarQsfpDd200gUpper01' ),
               ( upperPage02, 'FinisarQsfpDd200gUpper02' ),
               ( upperPage10Bank0, 'FinisarQsfpDd200gUpper10' ),
               ( upperPage11Bank0, 'FinisarQsfpDd200gUpper11' ), ]

   elif mediaType in ( 'AristaQsfpDd800gZr',
                       'AristaOsfp800gZr',
                       'AristaOsfp800gZrB',
                       'AristaOsfp800gZrC' ):
      lowerPage00Str = \
      { 'AristaOsfp800gZr' : 'AristaOsfp800gZrLower00',
        'AristaOsfp800gZrB' : 'AristaOsfp800gZrBLower00',
        'AristaOsfp800gZrC' : 'AristaOsfp800gZrCLower00',
        'AristaQsfpDd800gZr' : 'AristaQsfpDd800gZrLower00'}.get( mediaType )

      return [ ( lowerPage00, lowerPage00Str ),
               ( upperPage00, 'AristaQsfpDd800gZrUpper00' ),
               ( upperPage01, 'AristaQsfpDd800gZrUpper01' ),
               ( upperPage02, 'AristaOsfp400g2Fr4Upper02' ),
               ( upperPage04, 'AristaQsfpDd400gZrUpper04' ),
               ( upperPage10Bank0, 'AristaOsfp400g2Fr4Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400g2Fr4Upper11Bank0' ),
               ( upperPage12Bank0, 'AristaQsfpDd400gZrUpper12Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ),
               ( upperPage20Bank0, 'AristaQsfpDd400gZrUpper20Bank0' ),
               ( upperPage21Bank0, 'AristaQsfpDd400gZrUpper21Bank0' ),
               ( upperPage22Bank0, 'AristaQsfpDd400gZrUpper22Bank0' ),
               ( upperPage23Bank0, 'AristaQsfpDd400gZrUpper23Bank0' ),
               ( upperPage28Bank0, 'AristaQsfpDd400gZrUpper28Bank0' ),
               ( upperPage2FBank0, 'AristaQsfpDd400gZrUpper2FBank0' ) ]
   elif mediaType in ( 'AristaQsfpDd400gAr4', ):
      return [ ( lowerPage00, 'AristaQsfpDd400gAr4Lower00' ),
               ( upperPage00, 'AristaQsfpDd400gAr4Upper00' ),
               ( upperPage01, 'AristaQsfpDd400gAr4Upper01' ),
               ( upperPage02, 'AristaQsfpDd800gXdr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaOsfp400gAr4', ):
      return [ ( lowerPage00, 'AristaOsfp400gAr4Lower00' ),
               ( upperPage00, 'AristaOsfp400gAr4Upper00' ),
               ( upperPage01, 'AristaOsfp400gAr4Upper01' ),
               ( upperPage02, 'AristaOsfp400gSr8Upper02' ),
               ( upperPage10Bank0, 'AristaOsfp400gSr8Upper10Bank0' ),
               ( upperPage11Bank0, 'AristaOsfp400gSr8Upper11Bank0' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]
   elif mediaType in ( 'AristaQsfpCmis400gAr4', ):
      return [ ( lowerPage00, 'AristaQsfpCmis400gAr4Lower00' ),
               ( upperPage00, 'AristaQsfpCmis400gAr4Upper00' ),
               ( upperPage01, 'AristaQsfpCmis400gAr4Upper01' ),
               ( upperPage02, 'AristaQsfpCmis200gFr4Upper02' ),
               ( upperPage10Bank0, 'AristaQsfpCmis200gFr4Upper10' ),
               ( upperPage11Bank0, 'AristaQsfpCmis200gFr4Upper11' ),
               ( upperPage13Bank0, 'AristaCmisUpper13Bank0' ) ]

   else:
      assert 0, "CMIS mediaType %s does not exist in IdpromLib" % mediaType
      return None

def cmisDynamicPageFactory( bank, page ):
   return Tac.Value( "Xcvr::CmisBankAndPage", bank, page )

def getCmisDynamicPages( mediaType ):
   if not XcvrLib.isCmisTypeStr( getXcvrType( mediaType ) ):
      assert False, "%s does not map to a CMIS xcvr type" % mediaType

   osfpBank = Tac.Type( "Xcvr::CmisBank" )
   osfpPage = Tac.Type( "Xcvr::CmisPage" )
   dynamicPages = []
   if mediaType in ( 'AristaQsfpDd800g2Xdr4', 'AristaOsfp800g2Xdr4' ):
      dynamicPages += \
            [ cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage9F ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA0 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA1 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA2 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA3 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA4 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA5 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA6 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA7 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA8 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA9 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAA ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAB ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAC ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAD ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAE ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAF ),
            ]
      return dynamicPages
   if mediaType in ( 'AristaQsfpDd400gZr', 'AcaciaQsfpDd400gZr',
                     'AristaOsfp400gZr', 'AristaQsfpDd400gZrP',
                     'AristaOsfp400gZrP',
                     'AristaQsfpDd800gZr', 'AristaOsfp800gZr',
                     'AristaOsfp800gZrB', 'AristaOsfp800gZrC' ):
      dynamicPages += \
            [ cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage24 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage25 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage26 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage27 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage29 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2A ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2B ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2C ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage2D ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage30 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage31 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage32 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage33 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage34 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage35 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage38 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage3A ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage3B ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage40 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage41 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage42 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage43 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage44 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPage9F ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA0 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA1 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA2 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA3 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA4 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA5 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA6 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA7 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA8 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageA9 ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAA ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAB ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAC ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAD ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAE ),
              cmisDynamicPageFactory( osfpBank.bank0, osfpPage.upperPageAF ),
            ]
      if mediaType in [ 'AcaciaQsfpDd400gZr', 'AristaQsfpDd400gZr',
                        'AristaQsfpDd400gZrP', 'AristaOsfp400gZrP',
                        'AristaQsfpDd800gZr', 'AristaOsfp800gZr',
                        'AristaOsfp800gZrB', 'AristaOsfp800gZrC' ]:
         dynamicPages += \
            [ cmisDynamicPageFactory( osfpBank.bankNa, osfpPage.upperPageB8 ) ]
      return dynamicPages
   else:
      return []

def convertCmisEepromTemplateToBytes( mediaType ):
   cmisRaw = Tac.newInstance( "Xcvr::CmisRawEepromContents", "idprom" )
   for ( bankAndPage, templateStr ) in cmisPages( mediaType ):
      templBytes = getBytes( templateStr )
      assert len( templBytes ) == cmisRaw.eepromPageSize, ( 'CMIS page %s has %d '
             'bytes instead of the expected %d' % ( templateStr, len( templBytes ),
             cmisRaw.eepromPageSize ) )
      bytez = zeroOutStrEntries( templBytes )
      pageContent = convertBytesToBinStr( bytez )
      cmisRaw.memory[ bankAndPage ] = pageContent

   if mediaType in ( 'AristaQsfpDd400gZr', 'AcaciaQsfpDd400gZr',
                     'AristaOsfp400gZr', 'AristaQsfpDd400gZrP',
                     'AristaOsfp400gZrP',
                     'AristaQsfpDd800gZr', 'AristaOsfp800gZr',
                     'AristaOsfp800gZrB', 'AristaOsfp800gZrC',
                     'AristaQsfpDd800g2Xdr4', 'AristaOsfp800g2Xdr4' ):
      for bankAndPage in getCmisDynamicPages( mediaType ):
         bytez = [ 0 ] * 128
         pageContent = convertBytesToBinStr( bytez )
         cmisRaw.memory[ bankAndPage ] = pageContent
   return cmisRaw

# Utility function to edit a single register of a raw CMIS EEPROM object.
# Parameters:
#  idprom - a CmisRawEepromContents object
#  reg - a CmisRegister object
#  bytez - a list of characters
def editCmisEeprom( idprom, reg, bytez ):
   length = len( bytez )
   assert length == reg.size, "Number of new bytes must equal register size"

   bankAndPage = Tac.Value( "Xcvr::CmisBankAndPage", reg.bank, reg.page )
   if reg.bank == Tac.Type( "Xcvr::CmisPage" ).lowerPage00:
      regOffset = reg.addr
   else:
      regOffset = reg.addr - idprom.eepromUpperPageOffset

   pageStr = bytearray( idprom.memory[ bankAndPage ] )

   pageStr[ regOffset : regOffset + reg.size ] = bytez
   idprom.memory[ bankAndPage ] = bytes( pageStr )
   assert len( idprom.memory[ bankAndPage ] ) == idprom.eepromPageSize

# Function to recompute the checksum value of a raw CMIS EEPROM page
def recomputeCmisEepromPageChecksum( idprom, page ):
   osfpReg = Tac.Type( "Xcvr::CmisRegister" )
   osfpPage = Tac.Type( "Xcvr::CmisPage" )
   pageToReg = { osfpPage.upperPage00: osfpReg.checksumUpPg00,
                 osfpPage.upperPage01: osfpReg.checksumUpPg01,
                 osfpPage.upperPage02: osfpReg.checksumUpPg02 }

   checksumReg = pageToReg.get( page )
   if checksumReg:
      byteList = list( idprom.memory[ Tac.Value( "Xcvr::CmisBankAndPage",
                                                 checksumReg.bank,
                                                 checksumReg.page ) ] )

      # UpperPage01 is weird. Its checksum range starts at byte 130.
      up01hStart = 130 - idprom.eepromUpperPageOffset
      offsetStart = up01hStart if page == osfpPage.upperPage01 else 0
      offsetEnd = checksumReg.addr - idprom.eepromUpperPageOffset

      checksum = 0
      for indx in range( offsetStart, offsetEnd ):
         checksum += byteList[ indx ]
      checksum &= 0xff
      editCmisEeprom( idprom, checksumReg, [ checksum ] )
   else:
      assert False, "Page: %s does not have a checksum" % page

def generateVendorInfo( mediaType, specificVn='', specificSn='', specificDc='' ):
   vendorName = specificVn
   if not vendorName:
      vendorName = getVendorName( mediaType )

   today = datetime.datetime.today()
   dy = today.day
   mo = today.month
   # A bit of a brutal hack to just get the last two digits of the
   # year, i.e., '08' for '2008'.
   yr = str( today.year )[ -2 : ]
   ( _d1, wk, _d2 ) = today.isocalendar()

   vendorSerialNum = specificSn
   if not vendorSerialNum:
      # Serial nos start with "X", then date (MMYYWW), and 5-digit number
      vendorSerialNum = "X" + "%02d%2s%02d%05d" % ( mo, yr, wk,
                                                    random.randint( 0, 1E5 ) )

   vendorDc = specificDc
   if not vendorDc:
      # Datecode = date (YYMMDD) + lot code (LL)
      vendorDc = "%02s%02d%02d%02d" % ( yr, mo, dy, random.randint( 0, 99 ) )

   return vendorName, vendorSerialNum, vendorDc

def deepCopyCfp2OverrideIdEepromContents( cfp2DiagConfig, idEepromContents ):
   '''
   This function copies the memory object corresponding to the simMediaType
   from IdpromLib into xc.diagConfig.overrideIdEepromContents.memory. This
   is needed for cfp2 because cfp2's diagConfig is an entity. Finally, after
   copying, the diagConfig's generationId is incremented.

   Parameters
   ----------
   cfp2DiagConfig : Xcvr::Cfp2DiagConfig
      Attribute belonging to Cfp2DiagConfig to assign Idprom to

   idEepromContents : CfpMsaMisIdEepromContents
      Idprom to assign to overrideIdEepromContents

   Returns
   -------
   None
   '''
   assert isinstance( cfp2DiagConfig, Tac.Type( "Xcvr::Cfp2DiagConfig" ) )
   assert isinstance( idEepromContents,
                      Tac.Type( "Xcvr::CfpMsaMisIdEepromContents" ) )

   cfp2DiagConfig.overrideIdEepromContents = ( '', )

   for ti, table in idEepromContents.memory.items():
      cfp2DiagConfig.overrideIdEepromContents.memory.newMember(
         table.tableId,
         table.address,
         table.registerWidth )

      cfp2DiagConfig.overrideIdEepromContents.memory[ ti ].contents = \
         table.contents

   # Increment the generationId to indicate that overrideIdEepromContents has
   # finished updating. Doing this is necessary since Cfp2's overrideIdEepromContents
   # is an entity with a nested memory collection and should not react to individual
   # pages changing.
   cfp2DiagConfig.generationId += 1

# Inputs are Xcvr::CmisRawEepromContents and Xcvr::VendorInfo.
# This function replaces the vendor info bytes in raw EEPROM
# with the data in vendorInfo, recomputes the checksum, and
# re-writes the checksum byte
def overwriteCmisEepromVendorInfo( idprom, vendorInfo ):
   assert vendorInfo.vendorName
   assert vendorInfo.vendorSn
   assert vendorInfo.vendorDateCode

   # Helper function which turns a vendor info string into a list of
   # characters while padding the list with whitespace up to 'size' length.
   def padVendorInfo( size, infoStr ):
      charList = bytearray( infoStr )
      charList.extend( ( size - len( charList ) ) * [ 32 ] )
      return charList

   osfpReg = Tac.Type( "Xcvr::CmisRegister" )
   # Map from register to the vendor info string stored in that register
   regToInfo = { osfpReg.vendorName: vendorInfo.vendorName,
                 osfpReg.vendorSn: vendorInfo.vendorSn,
                 osfpReg.dateCode: vendorInfo.vendorDateCode }
   if vendorInfo.vendorPn:
      regToInfo[ osfpReg.vendorPn ] = vendorInfo.vendorPn
   if vendorInfo.vendorRev:
      regToInfo[ osfpReg.vendorRev ] = vendorInfo.vendorRev

   # Loop over our mapping of vendor info registers, editing our raw EEPROM to
   # contain the new vendor info.
   for reg, info in regToInfo.items():
      paddedInfo = padVendorInfo( reg.size, info.encode() )
      editCmisEeprom( idprom, reg, paddedInfo )

   # Since we just overwrote EEPROM, we must recompute the checksum
   # for upperPage00 and overwrite the previous one.
   recomputeCmisEepromPageChecksum( idprom, osfpReg.vendorName.page )

def updateCmisVendorInfo( eepromTemplate, rawEeprom, vendorName='', serialNum='',
                          vendorRev='', dateCode='' ):
   '''
   Updates the vendor information in the raw eeprom with the given strings. If the
   empty string is provided for an argument, then a generic value will be generated.
   Parameters
   ----------
   eepromTemplate : str
      This is one of the entries from Idpromlib.eepromTemplates
   rawEeprom : Xcvr::CmisRawEepromContents
      A generated idprom
   vendorName : str
   serialNum : str
   dateCode : str
   '''
   vi = Tac.Value( "Xcvr::VendorInfo" )
   vn, sn, dc = generateVendorInfo( eepromTemplate, specificVn=vendorName,
                                    specificSn=serialNum, specificDc=dateCode )
   vi.vendorName = vn
   vi.vendorSn = sn
   vi.vendorDateCode = dc
   vi.vendorRev = vendorRev
   overwriteCmisEepromVendorInfo( rawEeprom, vi )

def decodeCmisIdprom( idEeprom ):
   # Invoke the CmisEepromParser to build a decoded EEPROM object
   decodedIdEeprom = Tac.newInstance( "Xcvr::CmisDecodedEepromContents",
                                      "idprom" )
   parser = Tac.newInstance( "Xcvr::CmisEepromParser",
                             idEeprom,
                             decodedIdEeprom )
   parser.parseLowerPage00()
   parser.parseUpperPage00()
   if not decodedIdEeprom.flatMem:
      # keep parsing banked and paged memory
      parser.parseUpperPage01()
      parser.parseUpperPage04()
      implPagesAndBanks = Tac.newInstance( "Xcvr::CmisImplementedPagesAndBanks",
                                           decodedIdEeprom.implPagesAndBanks )
      if implPagesAndBanks.vdmImpl:
         VdmGroup = Tac.Type( "Xcvr::CmisVdmGroup" )
         parser.parseVdmConfigPage( VdmGroup.vdmGroup1 )
         parser.parseVdmConfigPage( VdmGroup.vdmGroup2 )
         parser.parseVdmConfigPage( VdmGroup.vdmGroup3 )
         parser.parseVdmConfigPage( VdmGroup.vdmGroup4 )
         parser.parseUpperPage2F()

   return decodedIdEeprom

def addSff8436VendorInfoField( vendorInfoField, vendorInfoString, eList ):
   # The trick to adding a vendor name to the Sff8436 data is to
   # have the string put in the right place and have it be the
   # correct length.
   # Begin with an empty string of the right length.
   ( fieldOffset, fieldLength ) = sff8436VendorInfoDefs[ vendorInfoField ]
   newVI = b' ' * fieldLength
   # Make sure the length isn't too big
   vendorInfoString = vendorInfoString.encode()
   length = len( vendorInfoString )
   if length > fieldLength: # pylint: disable=consider-using-min-builtin
      length = fieldLength

   # Use slices to insert the vendor name into a list.
   viList = list( vendorInfoString )
   newVIList = list( newVI )
   newVIList[ : length ] = viList[ : length ]

   # Use slices to insert the string into our eList data.
   eList[ fieldOffset : fieldOffset + fieldLength ] = newVIList

   return eList

def genIdprom( mediaType, vendorInfo=None ):
   # Start by figuring out which type of xcvr we are dealing with.
   # The specifics of the creation depend on the type.
   xcvrType = getXcvrType( mediaType )
   if xcvrType == "sfp":
      idEeprom = convertSff8472EepromTemplateToBytes( mediaType )
   elif xcvrType == "qsfp":
      idEeprom = convertSff8436EepromTemplateToBytes( mediaType )
   elif xcvrType == "cfp2":
      idEeprom = convertCfp2EepromTemplateToBytes( mediaType )
   elif XcvrLib.isCmisTypeStr( xcvrType ):
      idEeprom = convertCmisEepromTemplateToBytes( mediaType )
   else:
      assert False, "Unknown Xcvr type"

   if mediaType in ( "empty", "WncLoopback" ):
      return idEeprom

   # Get some date/time info for IdEeprom fields
   # The values we create here are used directly in sfp.
   vn = None
   pn = None
   sn = None
   rev = None
   dc = None

   if vendorInfo:
      ( vn, pn, sn, rev, dc ) = ( vendorInfo.vendorName,
                                vendorInfo.vendorPn,
                                vendorInfo.vendorSn,
                                vendorInfo.vendorRev,
                                vendorInfo.vendorDateCode )
      assert vn and pn and sn and rev and dc

   else:
      vn, sn, dc = generateVendorInfo( mediaType )

   if xcvrType == "sfp":
      # Now doctor up the bits that aren't filled in in the template
      # Make a copy of the string as a list.
      if vn:
         idEeprom.vendorName = vn

      if pn:
         idEeprom.vendorPn = pn

      assert sn
      idEeprom.vendorSn = sn

      if rev:
         idEeprom.vendorRev = rev

      assert dc
      idEeprom.dateCode = dc

      idEeprom.storedCcBase = idEeprom.computedCcBase
      idEeprom.storedCcExt = idEeprom.computedCcExt

      # Corrupt the checksum for this mediaType ( used in a test )
      if mediaType == 'AristaSRv1001BadCc':
         idEeprom.storedCcBase = idEeprom.storedCcBase + 1

      # NOTE: we are *NOT* updating the CRC (bytes 124-127)!!!
      #
      # There's code to do that in Sol/manufacturing/xcvrIdprom.py

   elif xcvrType == "qsfp":
      # Now doctor up the bits that aren't filled in in the template. Since the
      # thing we are modifying is a string and they are hard to mutate in Python
      # we convert the eeprom data to a list of bytes.
      eList = bytearray( idEeprom.eepromPage00UpperData )

      # Add the Vendor name.
      if not vn:
         vn = getVendorName( mediaType )

      eList = addSff8436VendorInfoField( 'vendorName', vn, eList )

      # Vendor SN
      eList = addSff8436VendorInfoField( 'vendorSN', sn, eList )

      # Vendor Datecode
      eList = addSff8436VendorInfoField( 'vendorDateCode', dc, eList )

      # Vendor PN
      if pn:
         eList = addSff8436VendorInfoField( 'vendorPN', pn, eList )

      # Vendor Rev
      if rev:
         eList = addSff8436VendorInfoField( 'vendorRev', rev, eList )

      # Put the modified data back into the idEeprom so the CRC
      # calculations will be on the corrected data.
      idEeprom.eepromPage00UpperData = bytes( eList )

      # Re-list the data so we can add the CRC fields.
      eList = bytearray( idEeprom.eepromPage00UpperData )

      eList[ qsfpBaseChecksumOffset ] = idEeprom.baseChecksumCalculated[ 1 ] & 0xff

      eList[ qsfpExtChecksumOffset ] = idEeprom.extendedChecksumCalculated & 0xff

      # Revision Compliance
      idEeprom.sff8636Revision = qsfpEepromRevisions[ mediaType ]

      # Put the completed list back into the eeprom data.
      idEeprom.eepromPage00UpperData = bytes( eList )
      # Populate lower page A0 with DOM data.
      idEeprom.eepromPage00LowerData = genSff8436Page00LowerData( mediaType )
      # Populate the upper page2 for security data
      idEeprom.eepromPage02UpperData = genSff8436Page02UpperData()
      # And lastly, populate the upper page 3 for DOM thresholds
      idEeprom.eepromPage03UpperData = genSff8436Page03UpperData( mediaType )

      if mediaType == 'Arista100GEDWDM2Qsfp':
         idEeprom.eepromPage20hUpperData = genSff8436Page20hUpperData( )
         idEeprom.eepromPage21hUpperData = genSff8436Page21hUpperData( )
      # We only implement the enhanced dom for 100GBASE-FR/LR/DR, the enhanced
      # dom thresholds is not implemented, no need to save page 21h.
      elif mediaType == 'Arista100GFRQsfpEnhanceDom':
         idEeprom.eepromPage20hUpperData = genSff8436Page20hUpperFRData( )

   elif xcvrType == "cfp2":
      pass
   elif XcvrLib.isCmisTypeStr( xcvrType ):
      # Rewrite vendor information into the raw EEPROM object
      if vendorInfo:
         vi = vendorInfo
      else:
         vi = Tac.Value( "Xcvr::VendorInfo" )
         vi.vendorName = vn
         vi.vendorSn = sn
         vi.vendorDateCode = dc

      overwriteCmisEepromVendorInfo( idEeprom, vi )
   else:
      assert False, "Unknown xcvr type"

   return idEeprom

def changeQsfpSff8636Revision( idEeprom, revision ):
   '''
   Change the SFF-8636 revision compliance in the idEeprom specified.

   Parameters
   ----------
   idEeprom : Sff8436IdEepromContents
      Original EEPROM contents

   revision : Enum member from Xcvr::Sff8636Revisions

   Returns
   -------
   newEeprom : Sff8436IdEepromContents
      Modified EEPROM contents
   '''
   newEeprom = idEeprom
   eLowerList = bytearray( newEeprom.eepromPage00LowerData )
   eLowerList[ 1 ] = Tac.enumValue( Sff8636RevEnum, revision )
   newEeprom.eepromPage00LowerData = eLowerList
   return newEeprom

def changeQsfpSff8636( idEeprom, pageNumAttr, regNum, data ):
   '''
   Change the SFF-8636 data in the idEeprom, page number attribute,
   and register number specified.

   Parameters
   ----------
   idEeprom : Sff8436IdEepromContents
      Original EEPROM contents

   pageNumAttr: str
      The U8 : 128 eepromPageXXUpper/LowerData attribute for Sff8436IdEepromContents

   regNum : int (range 0 ... 127)
      Register number

   data : U8
      The byte value to be written to the given register at the given page number

   lowerData : bool
      True to modify page 00 lower data

      Returns
   -------
   newEeprom : Sff8436IdEepromContents
      Modified EEPROM contents
   '''
   newEeprom = idEeprom
   pageData = getattr( newEeprom, pageNumAttr )
   eLowerList = bytearray( pageData )
   eLowerList[ regNum ] = data
   setattr( newEeprom, pageNumAttr, eLowerList )
   return newEeprom

def changeQsfpPowerClass( idEeprom, powerClass, powerClass8=False,
                          powerDissipated=None ):
   '''
   Change the Power Class advertised in the idEeprom specified.

   Parameters
   ----------
   idEeprom : Sff8436IdEepromContents
      Original EEPROM contents

   powerClass : int (range 1 .. 8)
      Power class (between 1 and 8) to be advertised in EEPROM

   powerClass8 : bool
      Whether power class 8 should be advertised in the EEPROM

   powerDissipated : int (watts)
      Maximum power dissipated to be set in the EEPROM

   Returns
   -------
   newEeprom : Sff8436IdEepromContents
      Modified EEPROM contents
   '''
   newEeprom = idEeprom

   eUpperList = bytearray( newEeprom.eepromPage00UpperData )

   # Set up power advertise register 00:129
   pcReg = 0
   if powerClass >= 1 and powerClass <= 4: # pylint: disable=chained-comparison
      pcReg = ( powerClass - 1 ) << 6
   elif powerClass >= 5 and powerClass <= 7: # pylint: disable=chained-comparison
      pcReg = ( powerClass - 4 )
   # For pc8, set the same high power bits as pc7
   elif powerClass == 8:
      pcReg = 3

   if powerClass8:
      assert powerDissipated is not None, \
             "powerDissipated should be provided to set powerClass8"
      pcReg |= 0x20

   if ( powerClass < 1 or powerClass > 7 ) and not powerClass8:
      assert False, "Unknown Power Class"

   eUpperList[ 1 ] = pcReg
   newEeprom.eepromPage00UpperData = bytes( eUpperList )

   eUpperList[ qsfpBaseChecksumOffset ] = \
      newEeprom.baseChecksumCalculated[ 1 ] & 0xff

   eUpperList[ qsfpExtChecksumOffset ] = \
      newEeprom.extendedChecksumCalculated & 0xff

   # Put the completed list back into the eeprom data.
   newEeprom.eepromPage00UpperData = bytes( eUpperList )

   # Set up max power dissipation register 00:107
   if powerDissipated is not None:
      powerDissipated = int( powerDissipated * 10 )
      eLowerList = bytearray( newEeprom.eepromPage00LowerData )
      eLowerList[ 107 ] = powerDissipated
      newEeprom.eepromPage00LowerData = eLowerList
   return newEeprom

def changeQsfpPowerControl( idEeprom, powerClass ):
   '''
   Configure High Power Control register (00:93) according to the Power Class

   Parameters
   ----------
   idEeprom : Sff8436IdEepromContents
      Original EEPROM contents

   powerClass : int (range 1 .. 8)
      Power class (between 1 and 8) to be configured

   Returns
   -------
   newEeprom : Sff8436IdEepromContents
      Modified EEPROM contents
   '''
   newEeprom = idEeprom
   # enable Software Power Control
   pc = Tac.Value( "Xcvr::Sff8436PowerControl" )
   pc.powerOverride = True
   # Enable PC8 config
   if powerClass == 8:
      pc.powerClass8Enable = True
   # Enable High Power config
   if powerClass >= 5:
      pc.highPowerClassEnable = True
   eLowerList = bytearray( newEeprom.eepromPage00LowerData )
   eLowerList[ pc.powerClassEnableOffset ] = pc.reg
   newEeprom.eepromPage00LowerData = eLowerList
   return newEeprom

def changeSfpPowerClass( idEeprom, powerClass ):
   newEeprom = idEeprom
   eList = bytearray( newEeprom.buf )
   optionsValueBaseAddress = 64
   sfpExtChecksumOffset = 95
   existingFlag = eList[ optionsValueBaseAddress ]
   if powerClass == 3:
      eList[ optionsValueBaseAddress ] = ( 1 << 5 ) | existingFlag
   else:
      eList[ optionsValueBaseAddress ] = ~( 1 << 5 ) & existingFlag
      existingFlag = eList[ optionsValueBaseAddress ]
      if powerClass == 1:
         eList[ optionsValueBaseAddress ] = ~( 1 << 1 ) & existingFlag
      elif powerClass == 2:
         eList[ optionsValueBaseAddress ] = ( 1 << 1 ) | existingFlag

   newEeprom.buf = bytes( eList )
   eList[ sfpExtChecksumOffset ] = newEeprom.computedCcExt

   newEeprom.buf = bytes( eList )
   return newEeprom

# Make tuning parameters supported or not
def changeTuningSupport( idEeprom, enable ):
   newEeprom = idEeprom
   eList = bytearray( newEeprom.eepromPage00UpperData )

   suppByte = 0
   if enable:
      suppByte |= 0x07

   eList[ 65 ] = suppByte
   newEeprom.eepromPage00UpperData = bytes( eList )
   eList[ qsfpBaseChecksumOffset ] = newEeprom.baseChecksumCalculated[ 1 ] & 0xff
   eList[ qsfpExtChecksumOffset ] = newEeprom.extendedChecksumCalculated & 0xff
   # Put the completed list back into the eeprom data.
   newEeprom.eepromPage00UpperData = bytes( eList )
   return newEeprom

# Generating the Sff-8436 A0 lower page. Primarily for DOM data.

sff8436Page00LowerData = [
   0x0D, 0x00 ] + [ # QSFP+ Identifier
   0 for _i in range( 2, 22 ) ] + [
   # Module monitors:
   0x7F, # Byte 22 Temperature MSB
   0xFF, # Temperature LSB
   0x00, # Reserved
   0x00, # Reserved
   0xFF, # Supply Voltage MSB
   0xFF, # Supply Voltage LSB
   0x00, # Reserved
   0x00, # Reserved
   0x00, # Byte 30 Vendor Specific
   0x00, # Vendor Specific
   0x00, # Vendor Specific
   0x00, # Vendor Specific
   # Channel Monitors:
   0xFF, # Byte 34 Rx1 Power MSB
   0xFF, # Rx1 Power LSB
   0xFF, # Rx2 Power MSB
   0xFF, # Rx2 Power LSB
   0xFF, # Rx3 Power MSB
   0xFF, # Rx3 Power LSB
   0xFF, # Byte 40 Rx4 Power MSB
   0xFF, # Rx4 Power LSB
   0xFF, # Tx1 Bias MSB
   0xFF, # Tx1 Bias LSB
   0xFF, # Tx2 Bias MSB
   0xFF, # Tx2 Bias LSB
   0xFF, # Tx3 Bias MSB
   0xFF, # Tx3 Bias LSB
   0xFF, # Tx4 Bias MSB
   0xFF ] + [ # Tx4 Bias LSB
   # Bytes 50-127 not applicable to test, all zeroed
   0 for _i in range( 50, 128 ) ]

# Generating the Sff-8636 A0 lower page for data updated after Revision 1.5

sff8636Page00LowerDataAfterRev15 = sff8436Page00LowerData[ 0 : 50 ] + [
   0xFF, # Byte 50 Tx1 PWR MSB
   0xFF, # Tx1 PWR LSB
   0xFF, # Tx2 PWR MSB
   0xFF, # Tx2 PWR LSB
   0xFF, # Tx3 PWR MSB
   0xFF, # Tx3 PWR LSB
   0xFF, # Tx4 PWR MSB
   0xFF  # Tx4 PWR LSB
   ] + sff8436Page00LowerData[ 58 : 128 ]

def genSff8436Page00LowerData( mediaType ):
   revision = qsfpEepromRevisions[ mediaType ]
   # The TX power is only defined after Revision 1.5 in SFF8636 or PSM4
   if mediaType == 'Psm4Qsfp' or revision >= sff8636rev15 or \
          mediaType == 'AristaTxPowerQsfp':
      thisLowerPage = sff8636Page00LowerDataAfterRev15[ : ]
      if mediaType in ( 'Arista50GCR2LQsfp', 'Arista50GCR2SQsfp',
            'Arista50GCR2NQsfp', 'Arista400g2QqsfpSideDepop' ):
         # Set only channels 0 and 1 implemented in the near-end config
         thisLowerPage[ 113 ] = 0xC
      if mediaType == 'Arista100GAR4Qsfp':
         thisLowerPage[ 50 : 57 ] = [ 0x00,
                                      0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00 ]
   else:
      thisLowerPage = sff8436Page00LowerData[ : ]
   # Mark the revision compliance byte
   thisLowerPage[ 1 ] = revision
   # Update the identifier to match upper page
   thisLowerPage[ 0 ] = getBytes( mediaType )[ 0 ]
   return convertBytesToBinStr( thisLowerPage )

# Generating the SFF-8436 upper page 20h and 21h. Primarily for Enhanced DOM.

Sff8436Page20hUpper = [
   0x01, 0x00, 0x00, 0x20, 0x11, 0x00, 0x02, 0xa2,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x13, 0x64, 0x12, 0x59, 0xff, 0xfe, 0xff, 0xfe,
   0x00, 0x4b, 0x01, 0xa7, 0x8b, 0x13, 0x00, 0x00,
   0xfb, 0x3d, 0xfd, 0x1a, 0x00, 0x18, 0x00, 0x18,
   0x00, 0x00, 0x8b, 0x11, 0x90, 0x92, 0x8b, 0x13,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8b, 0x13,
   0x00, 0x36, 0x80, 0x05, 0x00, 0x00, 0x00, 0x00,
   0x04, 0x01, 0x05, 0x01, 0x14, 0x02, 0x15, 0x02,
   0x24, 0x03, 0x25, 0x03, 0x30, 0x04, 0x40, 0x05,
   0x54, 0x06, 0x55, 0x06, 0x64, 0x07, 0x65, 0x07,
   0x80, 0x0b, 0x90, 0x09, 0xa0, 0x0a, 0xb0, 0x0c,
   0xc0, 0x0f, 0xd0, 0x0d, 0xe0, 0x0e, 0xf0, 0x10,
   0x74, 0x08, 0x75, 0x08, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ]

Sff8436Page21hUpper = [
   0x28, 0x00, 0x12, 0x00, 0x27, 0x00, 0x13, 0x00,
   0x10, 0x00, 0xf0, 0x00, 0x05, 0x00, 0xfb, 0x00,
   0xe0, 0x00, 0x00, 0x00, 0xd0, 0x00, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x10, 0x00, 0xf0, 0x00, 0x05, 0x00, 0xfb, 0x00,
   0x02, 0x00, 0xfe, 0x00, 0x01, 0x00, 0xff, 0x00,
   0x00, 0x40, 0xff, 0xc0, 0x00, 0x33, 0xff, 0xcd,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x90, 0x64, 0x00, 0x00, 0x88, 0x64, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ]

# Generating the SFF-8436 upper page 20h for 100GBASE-FR with enhanced
# dom suported.
Sff8436Page20hUpperFR = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x89, 0x79, 0x8f, 0xef, 0x8c, 0x0c, 0x8c, 0xb8,
   0xc0, 0x00, 0x75, 0x7a, 0x7e, 0x05, 0x79, 0xed,
   0x79, 0x47, 0xc0, 0x00, 0x10, 0xdd, 0x00, 0x00,
   0xfe, 0xfb, 0x00, 0xfa, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x04, 0x09, 0x04, 0x0a, 0x04, 0x04, 0x04, 0x0c,
   0x04, 0x0b, 0x04, 0x0d, 0x04, 0x0e, 0x04, 0x05,
   0x04, 0x10, 0x04, 0x0f, 0x04, 0x01, 0x00, 0x00,
   0x00, 0x06, 0x00, 0x08, 0x00, 0x11, 0x00, 0xC0,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, ]

# Generating the SFF-8436 upper page 201. Primarily for Enhanced DOM thresholds.
# Generating the SFF-8436 upper page 3. Primarily for DOM thresholds.

Sff8436Page03Upper = [
   0x01, # Temp high alarm MSB -128-
   0x02, # Temp high alarm LSB
   0x03, # Temp low alarm MSB
   0x04, # Temp low alarm LSB
   0x05, # Temp high warn MSB
   0x06, # Temp high warn LSB
   0x07, # Temp low warn MSB
   0x08 ] + [ # Temp low warn LSB
   # Reserved 136 to 143
   0x00 for _i in range( 8 ) ] + [
   0x09, # Vcc high alarm MSB
   0x0A, # Vcc high alarm LSB
   0x0B, # Vcc low alarm MSB
   0x0C, # Vcc low alarm LSB
   0x0D, # Vcc high warn MSB -148-
   0x0E, # Vcc high warn LSB
   0x0F, # Vcc low warn MSB
   0x10 ] + [ # Vcc low warn LSB
   # Reserved 152-168, Vendor-specific 160-175
   0x00 for _i in range( 24 ) ] + [
   0x21, # Rx power high alarm MSB
   0x22, # Rx power high alarm LSB
   0x23, # Rx power low alarm MSB -178-
   0x24, # Rx power low alarm LSB
   0x25, # Rx power high warn MSB
   0x26, # Rx power high warn LSB
   0x27, # Rx power low warn MSB
   0x28, # Rx power low warn LSB
   0x11, # Tx bias high alarm MSB
   0x12, # Tx bias high alarm LSB
   0x13, # Tx bias low alarm MSB
   0x14, # Tx bias low alarm LSB
   0x15, # Tx bias high warn MSB -188-
   0x16, # Tx bias high warn LSB
   0x17, # Tx bias low warn MSB
   0x18 ] + [ # Tx bias low warn LSB
   # We are not interested in the rest of this page
   0x00 for _i in range( 64 ) ]

# The following values are for testing tuning values
offset = 128
# values from sff8436 specification
# Sff8436InputEqualization
Sff8436Page03Upper[ 234 - offset ] = Sff8436Page03Upper[ 235 - offset ] = 0x11
# Sff8436OutputEmphasisControl
Sff8436Page03Upper[ 236 - offset ] = Sff8436Page03Upper[ 237 - offset ] = 0x22
# Sff8436OutputDiffAmpControl
Sff8436Page03Upper[ 238 - offset ] = Sff8436Page03Upper[ 239 - offset ] = 0x33

# Generating the SFF-8636 upper page 3 for data updated after Revision 1.5

Sff8436Page02Upper = [ 0x00 ] * 128
Sff8436Page02Upper[ 0xce - 128 : ( 0xce - 128 + 22 ) ] = list( range( 22 ) )

def genSff8436Page02UpperData():
   return convertBytesToBinStr( Sff8436Page02Upper )

Sff8636Page03UpperAfterRev15 = Sff8436Page03Upper[ 0 : 64 ] + [
   0x31, # Byte 192 Tx PWR High Alarm MSB
   0x32, # Tx PWR High Alarm LSB
   0x33, # Tx PWR Low Alarm MSB
   0x34, # Tx PWR Low Alarm LSB
   0x35, # Tx PWR High Warning MSB
   0x36, # Tx PWR High Warning LSB
   0x37, # Tx PWR Low Warning MSB
   0x38  # Tx PWR Low Warning LSB
   ] + Sff8436Page03Upper[ 72 : 128 ]

def genSff8436Page03UpperData( mediaType ):
   revision = qsfpEepromRevisions[ mediaType ]
   # The TX power is only defined after Revision 1.5 in SFF8636 or PSM4
   if mediaType == 'Psm4Qsfp' or revision >= sff8636rev15:
      thisUpperPage = Sff8636Page03UpperAfterRev15[ : ]
   else:
      thisUpperPage = Sff8436Page03Upper[ : ]
   return convertBytesToBinStr( thisUpperPage )

def genSff8436Page20hUpperData( ):
   thisUpperPage = Sff8436Page20hUpper[ : ]
   return convertBytesToBinStr( thisUpperPage )

def genSff8436Page20hUpperFRData( ):
   thisUpperPage = Sff8436Page20hUpperFR[ : ]
   return convertBytesToBinStr( thisUpperPage )

def genSff8436Page21hUpperData( ):
   thisUpperPage = Sff8436Page21hUpper[ : ]
   return convertBytesToBinStr( thisUpperPage )

# Generating the Sff-8472 A2 page. Primarily for DOM data.

sff8472A2Page = [
   0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
   0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
   0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
   0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
   0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
   0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x3F, 0x80, 0x00, 0x00, 0x3F, 0x80, 0x00, 0x00,
   0x3F, 0x80, 0x00, 0x00, 0x3F, 0x80, 0x00, 0x00,
   0x3F, 0x80, 0x00, 0x00, 0x01, 0x00, 0x7F, 0xFF,
   0x01, 0x00, 0x00, 0xFF, 0x01, 0x00, 0x01, 0x00,
   0x01, 0x00, 0x27, 0x10, 0x00, 0x00, 0x00, 0x00,
   0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
   0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
   0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x02, 0x03, 0x04, 0x00, 0xbf, 0x03, 0xe8,
   0x00, 0xc4, 0x05, 0xdc, 0x01, 0xf4, 0x0F, 0x10,
   0x00, 0x01, 0x13, 0x14, 0x15, 0x16, 0x01, 0x02,
   0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
   0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12,
   0x13, 0x14, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF,
   0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
   0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00,
   0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

def genSff8472A2Page():
   return convertBytesToBinStr( sff8472A2Page )

def genRandomSff8472Page():
   size = Tac.Type( "Xcvr::XcvrSff8472IdEepromContents" ).eepromSize
   maxByte = 0xff
   pageData = [ random.randint( 0, maxByte ) for _ in range( size ) ]
   return convertBytesToBinStr( pageData )

# Mappings - fieldName : ( offset, sizeInBytes )
# Note: this mapping is according to the specs, some Arista specific information
# might be missing or incomplete!
sfpIdpromPageA0Mapping = {
   'identifier': ( 0, 1 ),
   'identifierExtended': ( 1, 1 ),
   'connector': ( 2, 1 ),
   'complianceEth10GAndInfiniBand': ( 3, 1 ),
   'complianceEsconAndSonet': ( 4, 1 ),
   'complianceSonet': ( 5, 1 ),
   'complianceEth': ( 6, 1 ),
   'complianceFibreChannelLinkLengthAndTech': ( 7, 1 ),
   'complianceFibreChannelTech': ( 8, 1 ),
   'complianceFibreChannelTxMedia': ( 9, 1 ),
   'complianceFibreChannelSpeed': ( 10, 1 ),
   'encoding': ( 11, 1 ),
   'bitrateNominal': ( 12, 1 ),
   'rateIdentifier': ( 13, 1 ),
   'lengthSmfKm': ( 14, 1 ),
   'lengthSmf100m': ( 15, 1 ),
   'lengthOm2': ( 16, 1 ),
   'lengthOm1': ( 17, 1 ),
   'lengthDirectAttach': ( 18, 1 ),
   'lengthOm3': ( 19, 1 ),
   'vendorName': ( 20, 16 ),
   'vendorOui': ( 37, 3 ),
   'vendorPn': ( 40, 16 ),
   'vendorRev': ( 56, 4 ),
   'wavelength': ( 60, 2 ),
   'wavelengthFraction': ( 62, 1 ),
   'ccBase': ( 63, 1 ),
   'options': ( 64, 2 ),
   'bitrateMax': ( 66, 1 ),
   'bitrateMin': ( 67, 1 ),
   'vendorSn': ( 68, 16 ),
   'dateCode': ( 84, 8 ),
   'diagsMonitoringType': ( 92, 1 ),
   'enhancedOptions': ( 93, 1 ),
   'complianceSff8472': ( 94, 1 ),
   'ccExt': ( 95, 1 ),
   'vendorSpecific': ( 96, 21 ),
}

qsfpIdpromPage00UpperMapping = {
   'identifier': ( 0, 1 ),
   'identifierExtended': ( 1, 1 ),
   'connector': ( 2, 1 ),
   'complianceEth10And40G': ( 3, 1 ),
   'complianceSonet': ( 4, 1 ),
   'complianceSasSata': ( 5, 1 ),
   'complianceEth1G': ( 6, 1 ),
   'complianceFibreChannelLinkLengthAndTxTech': ( 7, 2 ),
   'complianceFibreChannelTxMedia': ( 9, 1 ),
   'complianceFibreChannelSpeed': ( 10, 1 ),
   'encoding': ( 11, 1 ),
   'bitrateNominal': ( 12, 1 ),
   'extendedRateSelectCompliance': ( 13, 1 ),
   'lengthSmf': ( 14, 1 ),
   'lengthOm3': ( 15, 1 ),
   'lengthOm2': ( 16, 1 ),
   'lengthOm1': ( 17, 1 ),
   'lengthCableOrOm4': ( 18, 1 ),
   'deviceTech': ( 19, 1 ),
   'vendorName': ( 20, 16 ),
   'vendorOui': ( 37, 3 ),
   'vendorPn': ( 40, 16 ),
   'vendorRev': ( 56, 2 ),
   'wavelength': ( 58, 2 ),
   'wavelengthTolerance': ( 60, 2 ),
   'maxCaseTemp': ( 62, 1 ),
   'ccBase': ( 63, 1 ),
   'extendedCompliance': ( 64, 1 ),
   'vendorSn': ( 68, 16 ),
   'dateCode': ( 84, 8 ),
   'diagsMonitoringType': ( 92, 1 ),
   'enhancedOptions': ( 93, 1 ),
   'ccExt': ( 95, 1 )
}

QSA28 = [
   0xff, 0x51, 0x53, 0x41, 0x00, 0x01, 0x00, 0x00,
   0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x01, 0x01, 0x01, 0x19, 0x46, 0x00, 0x00, 0x00,
   0x03, 0x03, 0x06, 0x07, 0xff, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x02, 0xc9, 0x00, 0x4d, 0x65, 0x6c, 0x6c,
   0x61, 0x6e, 0x6f, 0x78, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x4d, 0x54, 0x31, 0x37,
   0x33, 0x35, 0x55, 0x4b, 0x30, 0x30, 0x32, 0x31,
   0x33, 0x20, 0x20, 0x20, 0x4d, 0x41, 0x4d, 0x31,
   0x51, 0x30, 0x30, 0x41, 0x2d, 0x51, 0x53, 0x41,
   0x32, 0x38, 0x20, 0x20, 0x41, 0x34, 0x31, 0x37,
   0x30, 0x38, 0x32, 0x39, 0xff, 0xff, 0x00, 0x32 ]

def genQsaEepromData( ):
   return convertBytesToBinStr( QSA28 )

def genQsfpLsEepromData():
   eeprom = genIdprom( "AristaQsfpAmpZr" )
   # Overwrite generic lower/upper pages
   _pages = [ ( "eepromPage00LowerData", Qsfp.QsfpLsLowerPage0 ),
              ( "eepromPage02UpperData", Qsfp.QsfpLsUpperPage2 ),
              ( "eepromPage03UpperData", Qsfp.QsfpLsUpperPage3 ), ]
   for pageAttr, pageData in _pages:
      setattr( eeprom, pageAttr, convertBytesToBinStr( pageData ) )
   return eeprom
