# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#################################################
# 100GBASE-LR4 CFP2 module
# All tables have 128 entries. Tables A000 and on
# do not contain reserved entries, as XcvrCtrl
# does not read them.
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GLR4Table8000 = [
   0x11, 0xe4, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x08, 0x44, 0x11, 0x81, 0x81, 0x0a, 0x00, 0x00,
   0x01, 0x04, 0xca, 0x6e, 0xcc, 0x8d, 0x00, 0x64,
   0x21, 0x44, 0x40, 0x70, 0x1c, 0x3c, 0x64, 0x46,
   0x00, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0xf6
]

# Table 8080 has data width of 8 bits.
Cfp2MsaMis100GLR4Table8080 = [
   0x41, 0x00, 0x3c, 0x00, 0x28, 0x00, 0x23, 0x00, # Temp thresholds
   0x8c, 0xa0, 0x88, 0xb8, 0x79, 0x18, 0x75, 0x30,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xaf, 0xc8, 0x92, 0x7c, 0x44, 0x5c, 0x30, 0xd4,
   0x8a, 0x99, 0x7b, 0x86, 0x0c, 0xef, 0x0b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x8a, 0x99, 0x7b, 0x86, 0x02, 0xe5, 0x02, 0x4d,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x3c, 0x01, 0x3c, 0x01, 0x3c, 0x01, 0x3c,
   0x8a, 0x99, 0x8a, 0x99, 0x8a, 0x99, 0x8a, 0x99,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e
]

# Table 8420 has data width of 8 bits.
Cfp2MsaMis100GLR4Table8420 = [ 0x00 ] * 128 # Version 0 = 01

# Table A000 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA000 = [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xf0, 0x20, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf0, 0xa0, 0x62, 0x00, 0xf0, 0x0f,
   0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0xff, 0x7f, # Temp A02f
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage A030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x99, 0x8a, 0x86, 0x7b, 0xe5, 0x02, 0x4d, 0x02
]

# Table A200 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA200 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xd8, 0x80, 0xd8, 0x80, 0xd8, 0x80, 0xd8, 0x80,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# Table A280 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA280 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # txBias A2A0 up
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # txPower A2B0 up
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # rxPower A2D0 up
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

# Table A400 has data width of 16 bits.
Cfp2MsaMis100GLR4TableA400 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

   0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

#################################################

# Bobo module
# This module is only used for the XcvrUnlockTest
# It does not match any real transceiver
#
# All tables have 128 entries.
#################################################
Cfp2MsaMisBobo100GLR4Table8000 = [
   0x11, 0xe4, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x08, 0x44, 0x11, 0x81, 0x81, 0x0a, 0x00, 0x00,
   0x01, 0x04, 0xca, 0x6e, 0xcc, 0x8d, 0x00, 0x64,
   0x21, 0x44, 0x40, 0x70, 0x1c, 0x3c, 0x64, 0x46,
   0x00, 0x42, 0x6f, 0x62, 0x6f, 0x43, 0x6f, 0x72,
   0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0x0b
]

Cfp2MsaMisBobo100GLR4TableA000 = [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xf0, 0x20, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf0, 0xa0, 0x62, 0x00, 0xf0, 0x0f,
   0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0xff, 0x0f, # Temp A02f
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage A030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x99, 0x8a, 0x86, 0x7b, 0xe5, 0x02, 0x4d, 0x02
]

Cfp2MsaMisBobo100GLR4Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMisBobo100GLR4Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMisBobo100GLR4TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMisBobo100GLR4TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMisBobo100GLR4TableA400 = Cfp2MsaMis100GLR4TableA400
#################################################

# 100GBASE-SR10 CFP2 module
# This module is only supported for module
# detection in simulation at this time.
#
# All tables have 128 entries. Tables A000 and on
# do not contain reserved entries, as XcvrCtrl
# does not read them.
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GSR10Table8000 = [
   0x11, 0xa0, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00,
   0x00, 0xaa, 0x4a, 0x39, 0x39, 0x00, 0x0a, 0x00,

   0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x04, 0x40, 0x11, 0x11, 0x2d, 0x64, 0x46,
   0x00, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0xf3
]

Cfp2MsaMis100GSR10Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMis100GSR10Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMis100GSR10TableA000 = Cfp2MsaMis100GLR4TableA000
Cfp2MsaMis100GSR10TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMis100GSR10TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMis100GSR10TableA400 = Cfp2MsaMis100GLR4TableA400
#################################################

# 100GBASE-ER4 CFP2 module
# This module is only supported for module
# detection in simulation at this time.
#################################################
# Table 8000 has data width of 8 bits.
# Cfp2MsaMis100GER4Table8000 = [
#   Cfp2MsaMis100GLR4Table8000[ 0:2 ] +
#   [ 0x02 ] +                           # 8003 = EthAppCode ER4
#   Cfp2MsaMis100GLR4Table8000[ 4:0xC ] +
#   [ 0x28 ] +                           # 800D = SMF length 40km
#   Cfp2MsaMis100GLR4Table8000[ 0xE:0x7F ]
# ]
Cfp2MsaMis100GER4Table8000 = Cfp2MsaMis100GLR4Table8000[ : ]
Cfp2MsaMis100GER4Table8000[ 3 ] = 0x02   # 8003 = EthAppCode ER4
Cfp2MsaMis100GER4Table8000[ 0xD ] = 0x28 # 800D = SMF length 40km
Cfp2MsaMis100GER4Table8000[ 0x7f ] = 0x15 # 807F = Checksum

Cfp2MsaMis100GER4Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMis100GER4Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMis100GER4TableA000 = Cfp2MsaMis100GLR4TableA000
Cfp2MsaMis100GER4TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMis100GER4TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMis100GER4TableA400 = Cfp2MsaMis100GLR4TableA400

# 100GDwdmCoherent CFP2 module
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GDwdmCoherentTable8000 = [
   0x14, 0xca, 0x07, 0x11, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x11, 0x01, 0x00, 0xa0, 0xff, 0x00, 0x00,
   0x01, 0x01, 0xee, 0xde, 0xf5, 0x1e, 0x01, 0x90,
   0x32, 0xf4, 0x14, 0x19, 0x00, 0x3c, 0x64, 0x46,
   0xfb, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x58,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x44, 0x57,
   0x44, 0x4d, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x35, 0x31, 0x30, 0x30, 0x30, 0x31, 0x32, 0x36,
   0x20, 0x20, 0x20, 0x20, 0x32, 0x30, 0x31, 0x35,
   0x30, 0x33, 0x30, 0x35, 0x00, 0x00, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x0a, 0x16, 0x01, 0x01, 0x34, 0x01, 0x0c, 0x03,
   0x0e, 0x00, 0x3c, 0x3c, 0x00, 0x00, 0x64, 0x05,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfa
]

Cfp2MsaMis100GDwdmCoherentEV2Table8000 = Cfp2MsaMis100GDwdmCoherentTable8000[ : ]
Cfp2MsaMis100GDwdmCoherentEV2Table8000[ 0x45 ] = ord( 'T' )
Cfp2MsaMis100GDwdmCoherentEV2Table8000[ 0x46 ] = ord( 'C' )
Cfp2MsaMis100GOclaroDwdmCoherentETable8000 = Cfp2MsaMis100GDwdmCoherentTable8000[ : ]
vendor = "OCLARO          "
vendorPn = "TRB100BR-01     "
vendorSn = "SMD5407.1       "
for idx in range( 16 ):
   Cfp2MsaMis100GOclaroDwdmCoherentETable8000[ 0x21 + idx ] = ord( vendor[ idx ] )
   Cfp2MsaMis100GOclaroDwdmCoherentETable8000[ 0x34 + idx ] = ord( vendorPn[ idx ] )
   Cfp2MsaMis100GOclaroDwdmCoherentETable8000[ 0x44 + idx ] = ord( vendorSn[ idx ] )
Cfp2MsaMis100GAcaciaDwdmCoherentETable8000 = Cfp2MsaMis100GDwdmCoherentTable8000[ : ]
vendor = "Acacia Comm Inc."
vendorPn = "AC200-A20-00E   "
vendorSn = "163107143       "
for idx in range( 16 ):
   Cfp2MsaMis100GAcaciaDwdmCoherentETable8000[ 0x21 + idx ] = ord( vendor[ idx ] )
   Cfp2MsaMis100GAcaciaDwdmCoherentETable8000[ 0x34 + idx ] = ord( vendorPn[ idx ] )
   Cfp2MsaMis100GAcaciaDwdmCoherentETable8000[ 0x44 + idx ] = ord( vendorSn[ idx ] )

# Table 8080 has data width of 8 bits.
Cfp2MsaMis100GDwdmCoherentTable8080 = [
   0x46, 0x00, 0x41, 0x00, 0x00, 0x00, 0xfb, 0x00, # Temp thresholds
   0x8c, 0xa0, 0x88, 0xb8, 0x79, 0x18, 0x75, 0x30,
   0xfd, 0xe8, 0xfd, 0xe8, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xfd, 0xe8, 0xfd, 0xe8, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x90, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x94,
   0x32, 0x00, 0x31, 0xcc, 0x28, 0x33, 0x28, 0x00,
   0x62, 0x1f, 0x4d, 0xf1, 0x00, 0x20, 0x00, 0x14,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a
]

# Table 8180 has data width of 16 bits. ( 0x8180 and 0x8181 are 8-bits width)
Cfp2MsaMis100GDwdmCoherentTable8180 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xbf, 0x17, 0x70, 0x00, 0xc4,
   0x07, 0xd0, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, # Tuning capabilities
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xed
]

# Table 8420 has data width of 8 bits.
Cfp2MsaMis100GDwdmCoherentTable8420 = Cfp2MsaMis100GLR4Table8420

def padTable( table, size, padding=0x00 ):
   return ( table + size * [ padding ] )[ : size ]


# Table B000 has data width of 16 bits.
Cfp2MsaMis100GDwdmCoherentTableB000 = padTable( [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # Password is not readable
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, # Module state
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Module FAWs
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf8, 0xa7, 0x62, 0x00, 0xff, 0x0f,
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x7f, # Temp B02f
   0x99, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage B030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x90, 0x01, 0x2c, 0x01, 0x3c, 0xf6, 0x74, 0xf5,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit enabled
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
], 256 )




# Table B180 has data width of 16 bits.
Cfp2MsaMis100GDwdmCoherentTableB180 = [
   # 0xB180 Network lane alarm and warning 1
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB190 Network lane alarm and warning 2
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1A0 Network lane fualt and status
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1B0
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1C0
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1D0
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB1E0
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   # 0xB1F0
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
]

# Table B300 has data width of 16 bits.
Cfp2MsaMis100GDwdmCoherentTableB300 = [
   # 0xB300
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB310
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB320 Network Lane TX Laser Bias Current
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB330 Network Lane TX Laser Output Power
   0xff, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB340 Network Lane TX Laser Temp
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB350 Network Lane Receiver Input Power
   0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB360
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   # 0xB370
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
]

Cfp2MsaMis100GDwdmCoherentTableB400 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]
Cfp2MsaMis100GDwdmCoherentTableB480 = Cfp2MsaMis100GDwdmCoherentTableB400[ : ]
Cfp2MsaMis100GDwdmCoherentTableB480[ 64 ] = 0xff
Cfp2MsaMis100GDwdmCoherentTableB480[ 65 ] = 0x7f
Cfp2MsaMis100GDwdmCoherentTableB480[ 192 ] = 0xb3
Cfp2MsaMis100GDwdmCoherentTableB480[ 193 ] = 0x15
Cfp2MsaMis100GDwdmCoherentTableBA80 = Cfp2MsaMis100GDwdmCoherentTableB400
Cfp2MsaMis100GDwdmCoherentTableBB00 = Cfp2MsaMis100GDwdmCoherentTableB400[ : ]
Cfp2MsaMis100GDwdmCoherentTableBB00[ 1 ] = 0x02
Cfp2MsaMis100GDwdmCoherentTableBB00[ 0x55 * 2 + 1 ] = 0xF0
Cfp2MsaMis100GDwdmCoherentETable8180 = Cfp2MsaMis100GDwdmCoherentTable8180[ : ]
Cfp2MsaMis100GDwdmCoherentETableB180 = padTable( Cfp2MsaMis100GDwdmCoherentTableB180,
                                                 3 * 128 * 2 )[ : ]  # from the spec.
Cfp2MsaMis100GDwdmCoherentETable8180[ 18 ] = 0x27
Cfp2MsaMis100GDwdmCoherentETable8180[ 19 ] = 0x10
Cfp2MsaMis100GDwdmCoherentETable8180[ 20 ] = 0x27
Cfp2MsaMis100GDwdmCoherentETable8180[ 21 ] = 0x10
Cfp2MsaMis100GDwdmCoherentETable8180[ 127 ] = 0xcb
Cfp2MsaMis100GDwdmCoherentETableBB00 = Cfp2MsaMis100GDwdmCoherentTableB400[ : ]
Cfp2MsaMis100GDwdmCoherentETableBB00[ 0x9 ] = 0x40
Cfp2MsaMis100GDwdmCoherentETableBB00[ 1 ] = 0x02
Cfp2MsaMis100GDwdmCoherentETableBB00[ 0x55 * 2 + 1 ] = 0xF0
Cfp2MsaMis100GDwdmCoherentTableBB80 = Cfp2MsaMis100GDwdmCoherentTableB400[ : ]
Cfp2MsaMis100GDwdmCoherentTableBB80[ 21 ] = 0x02
Cfp2MsaMis100GDwdmCoherentTableBB80[ 42 ] = 0x40
Cfp2MsaMis100GDwdmCoherentTableBB80[ 43 ] = 0x06

#################################################
#
# Bobo DWDM Coherent module
# This module is only used for the XcvrUnlockTest
# It does not match any real transceiver
#
#################################################

Cfp2MsaMis100GBoboDwdmCoherentTable8000 = [
   0x14, 0xe4, 0x07, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x08, 0x44, 0x11, 0x81, 0x81, 0x0a, 0x00, 0x00,
   0x01, 0x04, 0xca, 0x6e, 0xcc, 0x8d, 0x00, 0x64,
   0x21, 0x44, 0x40, 0x70, 0x1c, 0x3c, 0x64, 0x46,
   0x00, 0x42, 0x6f, 0x62, 0x6f, 0x43, 0x6f, 0x72,
   0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x31, 0x30, 0x30, 0x47, 0x2d, 0x4c, 0x52,
   0x34, 0x20, 0x20, 0x20, 0x58, 0x41, 0x50, 0x31,
   0x33, 0x34, 0x39, 0x4a, 0x38, 0x30, 0x30, 0x35,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x31, 0x32,
   0x31, 0x32, 0x30, 0x30, 0x30, 0x30, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   0x00, 0x16, 0x01, 0x00, 0x01, 0x00, 0x0c, 0x03,
   0x0f, 0x68, 0x0a, 0x01, 0x0c, 0x00, 0x01, 0x01,
   0x20, 0x07, 0x07, 0x00, 0x00, 0x00, 0x20, 0x0e
]

# Table B000 has data width of 16 bits.
Cfp2MsaMis100GBoboDwdmCoherentTableB000 = padTable( [
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, # Password is not readable
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, # Module state
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Module FAWs
   0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x6a, 0x00, 0xf8, 0xa7, 0x62, 0x00, 0xff, 0x0f,
   0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x0f, # Temp B02f
   0x99, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # Voltage B030
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x90, 0x01, 0x2c, 0x01, 0x3c, 0xf6, 0x74, 0xf5,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, # MDIO write ready bit enabled
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
], 256 )

Cfp2MsaMis100GBoboDwdmCoherentTable8080 = Cfp2MsaMis100GDwdmCoherentTable8080
Cfp2MsaMis100GBoboDwdmCoherentTable8180 = Cfp2MsaMis100GDwdmCoherentTable8180
Cfp2MsaMis100GBoboDwdmCoherentTable8420 = Cfp2MsaMis100GDwdmCoherentTable8420
Cfp2MsaMis100GBoboDwdmCoherentTableB180 = Cfp2MsaMis100GDwdmCoherentTableB180
Cfp2MsaMis100GBoboDwdmCoherentTableB300 = Cfp2MsaMis100GDwdmCoherentTableB300
Cfp2MsaMis100GBoboDwdmCoherentTableB400 = Cfp2MsaMis100GDwdmCoherentTableB400
Cfp2MsaMis100GBoboDwdmCoherentTableB480 = Cfp2MsaMis100GDwdmCoherentTableB480
Cfp2MsaMis100GBoboDwdmCoherentTableBA80 = Cfp2MsaMis100GDwdmCoherentTableBA80
Cfp2MsaMis100GBoboDwdmCoherentTableBB00 = Cfp2MsaMis100GDwdmCoherentTableBB00
Cfp2MsaMis100GBoboDwdmCoherentTableBB80 = Cfp2MsaMis100GDwdmCoherentTableBB80

#################################################

# 100GBASE-XSR10 CFP2 module
# This module is only supported for module
# detection in simulation at this time.
#
# All tables have 128 entries. Tables A000 and on
# do not contain reserved entries, as XcvrCtrl
# does not read them.
#################################################
# Table 8000 has data width of 8 bits.
Cfp2MsaMis100GXSR10Table8000 = Cfp2MsaMis100GSR10Table8000[ : ]
# EthAppCode 0x8003 remains SR10
Cfp2MsaMis100GXSR10Table8000[ 0xe ] = 0x1e # 800E = MMF length 300m
Cfp2MsaMis100GXSR10Table8000[ 0x7f ] = 0x07 # 807F = Checksum

Cfp2MsaMis100GXSR10Table8080 = Cfp2MsaMis100GLR4Table8080
Cfp2MsaMis100GXSR10Table8420 = Cfp2MsaMis100GLR4Table8420
Cfp2MsaMis100GXSR10TableA000 = Cfp2MsaMis100GLR4TableA000
Cfp2MsaMis100GXSR10TableA200 = Cfp2MsaMis100GLR4TableA200
Cfp2MsaMis100GXSR10TableA280 = Cfp2MsaMis100GLR4TableA280
Cfp2MsaMis100GXSR10TableA400 = Cfp2MsaMis100GLR4TableA400
#################################################

# AC200-DCFP2 CFP2-DCO module
# All tables have 128 entries.
#################################################
# Page 8000 has data width of 8 bits.
Cfp2MsaMisCfp2DwdmDcoTable8000 = [
   0x19, 0xca, 0x07, 0x11, 0x00, 0x00, 0x00, 0x00,
   # 0x8009 - num of host/network lanes supported
   0x50, 0x18, 0x01, 0x00, 0x8c, 0x00, 0x00, 0x00,
   0x01, 0x01, 0xee, 0xdf, 0xf4, 0xee, 0x01, 0x90,
   # 0x801D - max power consumption
   0x35, 0xe2, 0x6c, 0x0a, 0x64, 0x00, 0x64, 0x46,
   # 0x8021 - 0x8030 Vendor Name
   0xfb, 0x41, 0x72, 0x69, 0x73, 0x74, 0x61, 0x20,
   0x4e, 0x65, 0x74, 0x77, 0x6f, 0x72, 0x6b, 0x73,
   0x20, 0x00, 0x1c, 0x73, 0x43, 0x46, 0x50, 0x32,
   0x2d, 0x32, 0x30, 0x30, 0x47, 0x2d, 0x44, 0x43,
   0x4f, 0x2d, 0x41, 0x20, 0x58, 0x54, 0x43, 0x31,
   0x37, 0x33, 0x30, 0x35, 0x33, 0x33, 0x32, 0x30,
   0x20, 0x20, 0x20, 0x20, 0x32, 0x30, 0x31, 0x37,
   0x30, 0x38, 0x30, 0x34, 0xff, 0xff, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
   # 0x806a-806d - hardware and firmware versions
   0x0e, 0x18, 0x30, 0x31, 0x25, 0x00, 0x0c, 0x03,
   0x0d, 0xf8, 0xb4, 0x01, 0xff, 0x00, 0x0a, 0x02,
   0x25, 0x21, 0x21, 0x25, 0x00, 0x14, 0xe0, 0x05
]

# CFP NVR2 Alarm/Warning Threshold Registers
Cfp2MsaMisCfp2DwdmDcoTable8080 = [
   0x5a, 0x00, 0x55, 0x00, 0x00, 0x00, 0xfd, 0x80,
   0x84, 0xd0, 0x83, 0xa4, 0x7d, 0x00, 0x7a, 0xa8,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x9b, 0x82, 0x00, 0x3f, 0x00, 0x32,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf5, 0x54, 0xee, 0xee, 0x0a, 0xaa, 0x04, 0x44,
   0x03, 0x1a, 0x03, 0x08, 0x00, 0x0a, 0x00, 0x01,
   0xf6, 0x78, 0xc3, 0xc3, 0x9b, 0x80, 0x7b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x37,
]

# CFP NVR4: MSA-100GLH Extended Identifiers
Cfp2MsaMisCfp2DwdmDcoTable8180 = [
   0xa0, 0x00, 0x00, 0x00, 0x09, 0xce, 0x07, 0xd0,
   0x00, 0x00, 0x00, 0xbf, 0x13, 0x88, 0x00, 0xc4,
   0x07, 0xd0, 0x17, 0x70, 0x17, 0x70, 0xc8, 0x62,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x08,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xea,
]

# The only necessary value is the module id (0x9015=0x0050)
Cfp2MsaMisCfp2DwdmDcoTable9000 = [ 0x00 ] * 256
Cfp2MsaMisCfp2DwdmDcoTable9000[ 43 ] = 0x50 # LSB of 0x9015

Cfp2MsaMisCfp2DwdmDcoTable9100 = [ 0x00 ] * 256

# CFP VR1:
Cfp2MsaMisCfp2DwdmDcoTableB000 = [
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x03, 0x00, 0x02, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x0e, 0x10, 0x01, 0x10, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x10, 0x00,
   0xe1, 0xa0, 0x00, 0x00, 0x00, 0x01, 0x00, 0xff,
   0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x01, 0xbe, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x6a, 0xa6, 0xfc, 0x01, 0xe2, 0x0f, 0xf0,
   0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x7f,
   0x96, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
   0x00, 0x80, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x80, 0x00, 0x25, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
]

Cfp2MsaMisCfp2DwdmDcoTableB480 = Cfp2MsaMis100GDwdmCoherentTableB480[ : ]
Cfp2MsaMisCfp2DwdmDcoTableB480[ 64 ] = 0xfd
Cfp2MsaMisCfp2DwdmDcoTableB480[ 65 ] = 0x44
Cfp2MsaMisCfp2DwdmDcoTableB480[ 192 ] = 0x00
Cfp2MsaMisCfp2DwdmDcoTableB480[ 193 ] = 0xf0

# 3rd party CFP2-DCO module
# Ethernet Application code (0x8003) is not populated
Cfp2MsaMisCfp2DwdmDco3rdPartyTable8000 = [
   0x19, 0x0a, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x50, 0x12, 0x01, 0x00, 0x8c, 0x00, 0x00, 0x00,
   0x01, 0x01, 0xee, 0xdf, 0xf4, 0xee, 0x01, 0x90,
   0x35, 0xe2, 0x2c, 0x0a, 0x64, 0x00, 0x64, 0x46,
   0xfb, 0x41, 0x63, 0x61, 0x63, 0x69, 0x61, 0x20,
   0x43, 0x6f, 0x6d, 0x6d, 0x20, 0x49, 0x6e, 0x63,
   0x2e, 0x00, 0x00, 0x00, 0x41, 0x43, 0x32, 0x30,
   0x30, 0x2d, 0x44, 0x32, 0x33, 0x2d, 0x30, 0x30,
   0x35, 0x20, 0x20, 0x20, 0x31, 0x38, 0x31, 0x38,
   0x35, 0x33, 0x34, 0x38, 0x38, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x32, 0x30, 0x31, 0x38,
   0x30, 0x35, 0x30, 0x38, 0xff, 0xff, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x18, 0x42, 0x20, 0x26, 0x02, 0x0c, 0x03,
   0x0d, 0xf8, 0x3c, 0x01, 0xff, 0x00, 0x0a, 0x02,
   0x25, 0x21, 0x21, 0x8a, 0x69, 0x14, 0xe0, 0xb3
]

Cfp2MsaMisCfp2DwdmDco3rdPartyTable8080 = [
   0x4b, 0x00, 0x46, 0x00, 0x00, 0x00, 0xfb, 0x00,
   0x84, 0xd0, 0x83, 0xa4, 0x7e, 0x2c, 0x7d, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x9b, 0x82, 0x00, 0x3f, 0x00, 0x32,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf5, 0x54, 0xee, 0xee, 0x0a, 0xaa, 0x04, 0x44,
   0x03, 0x1a, 0x03, 0x08, 0x00, 0x0a, 0x00, 0x01,
   0xf6, 0x78, 0xc3, 0xc3, 0x9b, 0x80, 0x7b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
]

Cfp2MsaMisCfp2DwdmDco3rdPartyTable8180 = Cfp2MsaMisCfp2DwdmDcoTable8180[ : ]
Cfp2MsaMisCfp2DwdmDco3rdPartyTable9000 = Cfp2MsaMisCfp2DwdmDcoTable9000[ : ]
Cfp2MsaMisCfp2DwdmDco3rdPartyTableB000 = Cfp2MsaMisCfp2DwdmDcoTableB000[ : ]
Cfp2MsaMisCfp2DwdmDco3rdPartyTableB480 = Cfp2MsaMisCfp2DwdmDcoTableB480[ : ]

# CFP2 Dual Laser DCO
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable8000 = [
   0x19, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x18, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x01, 0x01, 0xf0, 0xa6, 0xf3, 0xc8, 0x00, 0x00,
   0x35, 0xf2, 0x00, 0x64, 0xc7, 0x00, 0x00, 0x4b,
   0xfb, 0x41, 0x63, 0x61, 0x63, 0x69, 0x61, 0x20,
   0x43, 0x6f, 0x6d, 0x6d, 0x20, 0x49, 0x6e, 0x63,
   0x00, 0x00, 0x00, 0x00, 0x44, 0x50, 0x30, 0x34,
   0x53, 0x46, 0x50, 0x38, 0x2d, 0x45, 0x33, 0x30,
   0x00, 0xff, 0xff, 0xff, 0x32, 0x30, 0x33, 0x35,
   0x35, 0x34, 0x31, 0x37, 0x36, 0x20, 0x20, 0x20,
   0x20, 0x20, 0x20, 0x20, 0x31, 0x39, 0x31, 0x30,
   0x30, 0x39, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x0e, 0x1a, 0x31, 0x00, 0xa6, 0x00, 0x0c, 0x03,
   0x0d, 0xf8, 0x3c, 0xe8, 0xff, 0x00, 0x00, 0x0a,
   0xa7, 0x00, 0x00, 0xa6, 0x00, 0x19, 0xa0, 0x4f,
]
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable8080 = [
   0x4b, 0x00, 0x46, 0x00, 0x00, 0x00, 0xfb, 0x00,
   0x87, 0x5a, 0x86, 0x10, 0x7e, 0x54, 0x7d, 0xae,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0x9b, 0x82, 0x00, 0x3f, 0x00, 0x32,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xe6, 0x65, 0xd9, 0x98, 0x26, 0x66, 0x0c, 0xcc,
   0x03, 0x1a, 0x03, 0x08, 0x00, 0x0a, 0x00, 0x01,
   0xf6, 0x78, 0xc3, 0xc3, 0x9b, 0x80, 0x7b, 0x87,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0,
]

# The only necessary value to modify for table 9000 is the module id.
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9000 = Cfp2MsaMisCfp2DwdmDcoTable9000[ : ]
# The module id for the dual laser dco is 0x9009, so set that to 0x51 )
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9000[ 19 ] = 0x51 # LSB of 0x9009
# The module id for the single laser dco is 0x9015, so set that to 0
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9000[ 43 ] = 0 # LSB of 0x9015
# The dual laser dco has current registers in page 9B80
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9B80 = [ 0x00 ] * 256
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9B80[ 0x61 * 2 ] = 0x55 # LSB of 9BE1
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9B80[ 0x61 * 2 + 1 ] = 0xaa # MSB of 9BE1
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9B80[ 0x71 * 2 ] = 0xaa # LSB of 9FE1
Cfp2MsaMisCfp2DlDwdmDco3rdPartyTable9B80[ 0x71 * 2 + 1 ] = 0x55 # MSB of 9BF1
