# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This module contains a utility function that handles the rules
for syncing mtu configuration between intfStatus and intfConfig
"""

import Tac
import Tracing
import Logging
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs
# Set trace flags to TRACE=Intf::SyncMtu/*
t0 = Tracing.trace0        # Basic tracing
t1 = Tracing.trace1        # Details

globalIntfConfig = Tac.root.newEntity( 'Interface::GlobalIntfConfig',
                                       'globalIntfConfig' )

def syncMtuV2( intfStatus, intfStatusLocal, effectiveL3Mtu, verifyKernelIntf=False ):
   """Check the intfStatus mtu value against the forwardingModel and
   configured value, if any, and update the mtu value as needed"""
   t1( "syncMtu:", intfStatus.intfId )

   if intfStatus.maxMtu == 0:
      # This is the case for a Port-Channel with no members, for example.
      intfStatus.mtu = 0
      return True    # No point in having caller retry, as it will keep failing

   if intfStatus.forwardingModel == 'intfForwardingModelRouted':
      if effectiveL3Mtu:
         configuredMtu = effectiveL3Mtu
      else:
         configuredMtu = globalIntfConfig.l3MtuDefault

      # Make sure the configured mtu is valid.
      # pylint: disable-msg=E1102
      if configuredMtu > intfStatus.maxMtu:
         INTF_MAX_MTU_EXCEEDED( configuredMtu,
                                intfStatus.maxMtu,
                                intfStatus.intfId )
         # Clip to maxMtu
         newMtu = intfStatus.maxMtu
         t1( "syncMtu:", intfStatus.intfId, "mtu clipped to max", newMtu )
      else:
         # Normal case - using configured mtu
         newMtu = configuredMtu
         t1( "syncMtu:", intfStatus.intfId, "mtu configured as", newMtu )
   else:
      # Bridged interfaces (non-routed eth and lag)
      # set intfStatus.mtu to max
      newMtu = intfStatus.maxMtu

   if intfStatus.mtu != newMtu and not verifyKernelIntf:
      # Set intfStatus mtu to our new value
      t0( "syncMtu:", intfStatus.intfId,
            "changing intfStatus.mtu from", intfStatus.mtu,
            "to", newMtu )
      intfStatus.mtu = newMtu

   deviceName = intfStatus.deviceName
   if not deviceName:
      t0( " syncMtu: deviceName is none" )
      return False

   ns = DEFAULT_NS
   if intfStatusLocal and intfStatusLocal.netNsName != "":
      ns = intfStatusLocal.netNsName

   # Run the linux ip command to update the actual interface
   # Note: We need to run "ip link set ..." even if intfStatus.mtu
   # is unchanged, as the device may not have existed from the kernel
   # perspective (no deviceName) when we last updated intfStatus.mtu
   strMtu = str( newMtu )
   t0( "syncMtu: ip link set", deviceName, "mtu", newMtu )
   t0( "syncMtu: NS = ", ns )

   # The MTU between the default namespace and namespace specific versions of
   # the interface must be maintained such that the specific namespace MTU can
   # not be larger than the default namespace interface. This code keeps the
   # two versions synchronized.
   #
   def runIpLinkSetMtu( ns, deviceName, strMtu ):
      try:
         runMaybeInNetNs( ns,
                          [ 'ip', 'link', 'set', deviceName, 'mtu', strMtu ],
                          stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      except Tac.SystemCommandError:
         t0( "syncMtu: Error running for namespace", ns,
             ": ip link set", deviceName, "mtu", newMtu )
         return False
      return True
   kernelIntfUpdated = True
   if not runIpLinkSetMtu( DEFAULT_NS, deviceName, strMtu ):
      t0( "syncMtu: runIpLinkSetMtu = false" )
      kernelIntfUpdated = False
   if ns != DEFAULT_NS and not runIpLinkSetMtu( ns, deviceName, strMtu ):
      t0( "syncMtu: non DEFAULT_NS runIpLinkSetMtu = false" )
      kernelIntfUpdated = False
   t0( "syncMtu: runIpLinkSetMtu =", kernelIntfUpdated )
   if kernelIntfUpdated and verifyKernelIntf:
      # Set intfStatus mtu to our new value
      t0( "syncMtu:", intfStatus.intfId,
          "changing intfStatus.mtu from", intfStatus.mtu,
          "to", newMtu )
      intfStatus.mtu = newMtu
   return kernelIntfUpdated

def syncMtu( intfStatus, intfStatusLocal, intfConfigOrEffectiveL3Mtu,
             verifyKernelIntf=False ):
   return syncMtuV2( intfStatus, intfStatusLocal,
                     intfConfigOrEffectiveL3Mtu
                     if isinstance( intfConfigOrEffectiveL3Mtu, int )
                     else intfConfigOrEffectiveL3Mtu.mtu,
                     verifyKernelIntf=verifyKernelIntf )

def verifyMtu( intfStatus, intfConfig ):
   if intfStatus.forwardingModel == 'intfForwardingModelRouted':
      if intfConfig:
         configuredMtu = intfConfig.mtu
      else:
         configuredMtu = globalIntfConfig.l3MtuDefault
      expectedMtu = min( configuredMtu, intfStatus.maxMtu )
   else:
      expectedMtu = intfStatus.maxMtu

   assert intfStatus.mtu == expectedMtu

INTF_MAX_MTU_EXCEEDED = Logging.LogHandle(
   "INTF_MAX_MTU_EXCEEDED",
   severity=Logging.logWarning,
   fmt="Configured mtu (%d) exceeds the maximum mtu (%d) for interface %s",
   explanation=
   "This interface was configured with an mtu value that exceeds the maximum "
   "mtu supported for this type of interface",
   recommendedAction="Change the interface configuration to be consistent with "
   "the maximum supported mtu for this interface type." )
