# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
import Tracing

t0 = Tracing.trace0

def getNonProfileCmds( currentConfig, profileCmds ):
   """ Process the currentConfig so profile commands don't show up in the 
   resulting config, and config that overrides profile configs are explicit.

   currentConfig - list of running-config commands
   profileCmds - list of profile commands
   """
   lines = list( currentConfig )
   assert lines[ 0 ].startswith( 'profile ' ), \
      "'profile' command must be first command in mode"
   overridenLines = [] # profile commands that need to be overriden
   for pCmd in profileCmds:
      if pCmd in lines:
         lines.remove( pCmd )
      elif pCmd.startswith( 'default ' ):
         # not sure why we have default commands in profile, but ignore
         continue
      else:
         # This line in profile does not appear in running-config, so
         # it's overriden. We run a "default" command to remove it as
         # there may not be an overriding non-profile command.
         if pCmd.startswith( 'no ' ):
            pCmd = pCmd[ 3: ]
         overridenLines.append( "default " + pCmd )

   # first command is "profile ..."
   return overridenLines + lines[ 1 : ]

def filterIntfConfig( config, intfNames ):
   """ Returns a dictionary mapping { intfName : config under that interface }
   If intfNames is None, all interfaces are included.
   """
   intfConfig = {}
   for line in re.split( '^interface ', config, flags=re.MULTILINE ):
      line = line.splitlines()
      intf = line[ 0 ]
      if intfNames is None or intf in intfNames:
         intfCmds = []
         for cmd in line[ 1: ]:
            if cmd.startswith( '   ' ):
               # strip the 3 leading spaces from intf commands
               intfCmds.append( cmd[ 3 : ] )
            else:
               # The rest of these aren't cmds under the intf mode, save and move on
               intfConfig[ intf ] = '\n'.join( intfCmds ) + '\n' if intfCmds else ''
               break
   return intfConfig

class AppliedProfileConfig:
   """ Functions to generate the config text that will be used to apply a profile 
   to an interface.

   newProfileConfig() - generates the rollback config and the profile commands
                        config, call this one first
   nonProfileConfig() - generate the non-profile config, this goes after the
                        profile config
   """

   def __init__( self, intfName, config, oldProfCmds, newProfCmds,
                 overrideCmds=None, defaultCmds=None ):
      """
      intfName - name of interface to apply profile
      config - the raw unprocessed interface config text
      oldProfileCmds - list of previously applied profile commands
      newProfileCmds - list of profile commands to apply
      defaultCmds - default commands for the interface
      overrideCmds - list of overriden commands if available
      """
      self.intfName = intfName

      self.comments, self.config = self.splitComments( config )
      self.oldProfileCmds = oldProfCmds
      self.newProfileCmds = newProfCmds
      self.overrideCmds = overrideCmds
      self.defaultCmds = defaultCmds

      t0( "AppliedProfileConfig: intf", intfName )
      if self.comments:
         t0( "comments:", self.comments )
      t0( "config:", self.config )
      t0( "oldProfileCmds:", oldProfCmds )
      t0( "newProfileCmds:", newProfCmds )
      t0( "defaultCmds:", defaultCmds )
      t0( "overrideCmds:", overrideCmds )

   def splitComments( self, config ):
      # the config may have comments, i.e.:
      #
      # !! comment 1
      # !! comment 2
      # command 1
      # command2
      #
      # Let's split it into "comments" and "commands"
      lines = config.splitlines()
      comments = []
      for cmd in lines:
         if cmd.startswith( '!!' ):
            comments.append( cmd )
         else:
            break
      return comments, lines[ len( comments ) : ]

   def _nothingToApply( self ):
      return not self.oldProfileCmds and not self.newProfileCmds

   def getOverrideCmds( self, nonProfileCmds ):
      # add overriden commands in front of the nonProfileCmds
      if self.overrideCmds:
         cmds = sorted( x for x in self.overrideCmds if x not in nonProfileCmds )
         return cmds + nonProfileCmds
      else:
         return nonProfileCmds

   def newProfileConfig( self ):
      if self._nothingToApply():
         return []
      return [ f'default interface {self.intfName}',
               f'interface {self.intfName}' ] + self.newProfileCmds

   def nonProfileConfig( self ):
      if self._nothingToApply():
         return []
      config = [ f'interface {self.intfName}' ] + self.comments
      if self.oldProfileCmds:
         nonProfileCmds = getNonProfileCmds( self.config,
                                             self.oldProfileCmds )
      else:
         if self.config and self.config[ 0 ].startswith( 'profile ' ):
            # Remove the "profile" line
            nonProfileCmds = self.config[ 1: ]
         else:
            nonProfileCmds = self.config

      # add overriden commands in front of the nonProfileCmds
      if self.defaultCmds:
         t0( "filtering nonProfileCmds", nonProfileCmds, "by defaultCmds" )
         nonProfileCmds = [ x for x in nonProfileCmds
                            if x not in self.defaultCmds ]

      return config + self.getOverrideCmds( nonProfileCmds )
