# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools

import CliSession
import ConfigMount
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import Tac
import Tracing

t0 = Tracing.trace0

_em = None
ethPhyIntfConfigSliceDir = None

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as able to participate in interface ranges.
#-------------------------------------------------------------------------------

class EthPhyRangeIntfTypeBase( IntfRange.AutoIntfType ):
   def __init__( self, name, shortName, desc, subSupported, subIntfGuard,
                 helpdesc=None, **kwargs ):
      IntfRange.AutoIntfType.__init__(
         self, name, shortName, desc, helpdesc=helpdesc,
         subSupported=subSupported, subIntfGuard=subIntfGuard, **kwargs )
      # List of class, function pairs.
      self.clazz = None
      self.extraIntfs_ = set()

   def getCliIntf( self, mode, name ):
      if self.subSupported_ and '.' in name and self.cliSubIntfClazz_:
         return self.cliSubIntfClazz_( name, mode )
      else:
         return self.clazz( name, mode )

   def registerEthPhyIntfClass( self, clazz ):
      """
      clazz is the Cli Intf class that should be instantiated if
      func( name, mode ) is true
      """
      self.clazz = clazz

   def collection( self ):
      # This is a generator of two collections
      allIntfConfigs = [ iter( ethPhyDir.intfConfig )
                         for ethPhyDir in ethPhyIntfConfigSliceDir.values() ]
      if self.extraIntfs_:
         allIntfConfigs.append( iter( self.extraIntfs_ ) )
      return itertools.chain.from_iterable( allIntfConfigs )

   def collectionVersion( self ):
      # This function implements versioning. Using this method the IntfRange
      # infrastructure code can figure out if the collection returned by
      # ethPhyIntfCollection() has changed
      # This method concatenates the version of the intfConfig collection of each
      # EthPhyIntfConfigDir. The version attribute is incremented every time the
      # collection is updated
      versionList = [ ethPhyIntfConfigSliceDir.entityPtr.version() ]
      for ethPhyDir in sorted( ethPhyIntfConfigSliceDir.values(),
                               key=lambda d: d.name ):
         versionList.append( ethPhyDir.intfConfig.version() )
      versionList.append( len( self.extraIntfs_ ) )

      return versionList

   def collectionSessionName( self ):
      return CliSession.currentSession( _em )

# Ethernet interfaces
class EthPhyRangeIntfType( EthPhyRangeIntfTypeBase ):
   def __init__( self, helpdesc=None, **kwargs ):
      EthPhyRangeIntfTypeBase.__init__( self, "Ethernet", "Et",
                                        "hardware Ethernet interface",
                                        subSupported=True,
                                        subIntfGuard=IntfRange.subintfSupportedGuard,
                                        **kwargs )

SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
SubExtendedMin = SubIntfRangeDetails.subIntfExtendedIdMin
SubExtendedMax = SubIntfRangeDetails.subIntfExtendedIdMax
EthPhyAutoIntfType = EthPhyRangeIntfType( intfSubLowerBound=SubExtendedMin,
                                          intfSubUpperBound=SubExtendedMax )
intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=True,
                                     perSessionSupport=True )
IntfRange.registerIntfType( EthPhyAutoIntfType, intfOptions )

class ManagementRangeIntfType( EthPhyRangeIntfTypeBase ):
   def __init__( self, helpdesc=None ):
      EthPhyRangeIntfTypeBase.__init__(
         self, "Management", "Ma", "Management interface",
         subSupported=False, subIntfGuard=None, helpdesc=helpdesc )
      # List of class, function pairs.
      self.cliIntfClasses_ = []

   def getCliIntf( self, mode, name ):
      for clazz, func in self.cliIntfClasses_:
         if func( name, mode ):
            return clazz( name, mode )
      return self.cliIntfClazz_( name, mode )

   def registerManagementIntfClass( self, clazz, func, extraIntfs ):
      """
      clazz is the Cli Intf class that should be instantiated if
      func( name, mode ) is true
      """
      self.cliIntfClasses_.append( ( clazz, func ) )
      self.extraIntfs_.update( extraIntfs )

MgmtAutoIntfType = ManagementRangeIntfType( )
intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=False,
                                     perSessionSupport=True )
IntfRange.registerIntfType( MgmtAutoIntfType, intfOptions )

# This is the physical mgmt collection (nobody should call 
# registerManagementIntfClass on this)
MgmtPhyAutoIntfType = ManagementRangeIntfType( helpdesc=[ 'Port number' ] )

# Unconnected Ethernet Interfaces
class UnconnectedEthPhyRangeIntfType( EthPhyRangeIntfTypeBase ):
   def __init__( self, helpdesc=None ):
      EthPhyRangeIntfTypeBase.__init__( self, "UnconnectedEthernet", "Ue",
                                        "Unconnected internal Ethernet interface",
                                        subSupported=False, subIntfGuard=None )

UnconnectedEthPhyAutoIntfType = UnconnectedEthPhyRangeIntfType()
intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=False,
                                     perSessionSupport=True )
IntfRange.registerIntfType( UnconnectedEthPhyAutoIntfType, intfOptions )

# there are more than one type of interface types inside interface/config/eth/phy,
# so let's allow other plugins to register new types.
configuredEthPhyIntfTypes_ = { 'Ethernet',
                               'UnconnectedEthernet',
                               'Management' }

def registerConfiguredEthPhyIntfType( intfType ):
   configuredEthPhyIntfTypes_.add( intfType )

def configuredEthPhyCallback( requireMounts, explicitIntfTypes ):
   if explicitIntfTypes:
      myTypes = configuredEthPhyIntfTypes_ & set( explicitIntfTypes )
   else:
      myTypes = configuredEthPhyIntfTypes_

   if myTypes:
      ethPhyDir = requireMounts.getValue( 'interface/config/eth/phy/slice' )
      return itertools.chain.from_iterable(
         filter( lambda x: any( x.startswith( t )
                                          for t in myTypes ),
                           intfConfig.intfConfig )
         for intfConfig in ethPhyDir.values() )
   else:
      return []

IntfRange.registerConfiguredIntfCallback( configuredEthPhyCallback )

def Plugin( em ):
   global ethPhyIntfConfigSliceDir, _em

   if _em == em:
      return

   _em = em
   ethPhyIntfConfigSliceDir = ConfigMount.mount(
                                         em,
                                         "interface/config/eth/phy/slice", 
                                         "Tac::Dir",
                                         "wi" )
