# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import RoutingIntfUtils
import Tac
import socket
from IpUtils import IpAddress
from collections import namedtuple
from functools import total_ordering

leaseTupleFields = ( 'ip', 'mac', 'source', 'intf', 'installed', 'expirationTime',
                     'action' )
leaseTupleDefaults = ( 'None', ) * ( len( leaseTupleFields ) - 1 )
leaseTupleDefaults += ( 'permit', )
LeaseTuple = namedtuple( "Lease", leaseTupleFields, defaults=leaseTupleDefaults )
SubIntfIdType = Tac.Type( "Arnet::SubIntfId" )

def intfIsEthAndSupportsIpLocking( intfName ):
   return ( not RoutingIntfUtils.isManagement( intfName ) and
            intfName.startswith( 'Ethernet' ) and
            not SubIntfIdType.isSubIntfId( intfName )
          )

def intfIsLag( intfName ):
   return ( not RoutingIntfUtils.isManagement( intfName ) and
            intfName.startswith( 'Port-Channel' ) and
            not SubIntfIdType.isSubIntfId( intfName )
          )

def convertToIpGenAddr( ipAddrs ):
   return [ Arnet.IpGenAddr( ip ) for ip in ipAddrs ]

def getVlanStr( delim, vlan, mode, intfEnforcementDisabled=False ):
   if mode == "invalid":
      return ""
   vlanStr = delim + str( vlan )
   if mode == "enforcementDisabled" or intfEnforcementDisabled:
      vlanStr += "*"
   return vlanStr

def getNextVlanAsString( offset, vlan, assignedVlanModes, attr,
                         intfEnforcementDisabled=False ):
   vlanStr = ""
   assignedVlanMode = getattr( assignedVlanModes[ vlan ], attr )
   if offset > 1:
      vlanStr += getVlanStr( '-', vlan, assignedVlanMode,
                             intfEnforcementDisabled=intfEnforcementDisabled )
   elif offset == 1:
      vlanStr += getVlanStr( ',', vlan, assignedVlanMode,
                             intfEnforcementDisabled=intfEnforcementDisabled )
   return vlanStr

# This function takes as input the list of address locking enabled VLANs to which
# an enforcement with locked-address enforcement disabled is assigned and returns
# a simplified string made up of comma separated VLAN ranges.
def getAssignedVlansAsStringsUnenforced( assignedVlanModes ):
   vlanStr = ""
   if not assignedVlanModes:
      return vlanStr
   i = 0
   vlanList = sorted( assignedVlanModes.keys() )
   while i < len( vlanList ):
      delim = ',' if i > 0 and vlanStr else ''
      currentVlan = vlanList[ i ]
      vlanStr += getVlanStr( delim, currentVlan,
                             assignedVlanModes[ currentVlan ].ipv4Mode,
                             intfEnforcementDisabled=True )
      currIdx = i
      # Keep incrementing index while vlans are consecutive and have the same mode.
      # These will be grouped together in the display
      while ( ( i < len( vlanList ) - 1 ) and
              ( vlanList[ i + 1 ] == vlanList[ i ] + 1 ) and
              assignedVlanModes[ vlanList[ i + 1 ] ].ipv4Mode != 'invalid' ):
         i = i + 1
      offset = i - currIdx
      vlanStr += getNextVlanAsString( offset, vlanList[ i ],
                                      assignedVlanModes, 'ipv4Mode',
                                      intfEnforcementDisabled=True )
      i = i + 1
   return vlanStr

def getAssignedVlansAsStringsRaw( assignedVlanModes, includeEnforcementEnabled,
                                  attr ):
   vlanStr = ""
   if not assignedVlanModes:
      return vlanStr
   i = 0
   vlanList = sorted( assignedVlanModes.keys() )
   while i < len( vlanList ):
      currentVlan = vlanList[ i ]
      currentAssignedVlanMode = getattr( assignedVlanModes[ currentVlan ], attr )
      if ( currentAssignedVlanMode == "enforcementEnabled" and not
           includeEnforcementEnabled ):
         i = i + 1
         continue
      delim = ',' if i > 0 and vlanStr else ''
      vlanStr += getVlanStr( delim, currentVlan, currentAssignedVlanMode )
      currIdx = i
      # Keep incrementing index while vlans are consecutive and have the same mode.
      # These will be grouped together in the display
      while ( ( i < len( vlanList ) - 1 ) and
              ( vlanList[ i + 1 ] == vlanList[ i ] + 1 ) and
              ( currentAssignedVlanMode ==
              getattr( assignedVlanModes[ vlanList[ i + 1 ] ], attr ) ) ):
         i = i + 1
      offset = i - currIdx
      vlanStr += getNextVlanAsString( offset, vlanList[ i ],
                                      assignedVlanModes, attr )
      i = i + 1
   return vlanStr

# This function takes as input the list of address locking enabled VLANs to which
# an interface is assigned and returns a simplified string made up of comma
# separated VLAN ranges interweaved with '*' to indicate that a VLAN range has
# locked-address enforcement disabled on it
def getAssignedVlansAsStrings( assignedVlanModes, includeEnforcementEnabled, attr ):
   vlanStr = getAssignedVlansAsStringsRaw(
         assignedVlanModes, includeEnforcementEnabled, attr )
   if not includeEnforcementEnabled and vlanStr:
      vlanStr = f' ({vlanStr})'
   return vlanStr

def getLeaseQueryStr( retryInterval, timeout ):
   leaseQueryStr = (
         f'Lease query retry interval/timeout: '
         f'{retryInterval}/{timeout} seconds'
   )
   return leaseQueryStr

@total_ordering
class LeaseTupleForCompare:
   """Represent a lease object for comparison. Only IP address and source are
   used."""

   def __init__( self, lease ):
      """lease is a LeaseTuple defined above"""
      self.ip = lease.ip
      self.source = lease.source
      self.action = lease.action

   def actionPriority( self, action ):
      if action == 'deny':
         return 2
      if action == 'permit*':
         return 1
      return 0

   def ipAddress( self, ip ):
      """ip can either be a string or Arnet::IpGenAddr"""
      if not isinstance( ip, str ):
         ip = str( ip )
      if ':' in ip:
         return IpAddress( ip, addrFamily=socket.AF_INET6 )
      else:
         return IpAddress( ip )

   def __eq__( self, other ):
      return ( self.source == other.source and
               self.ipAddress( self.ip ) == self.ipAddress( other.ip ) )

   def __ne__( self, other ):
      return not self == other

   def __lt__( self, other ):
      """Denied entries should be display first
         followed by leases that came from config"""
      if self.action == other.action:
         if self.source == other.source:
            return self.ipAddress( self.ip ) < self.ipAddress( other.ip )
         return self.source == 'config'
      return ( self.actionPriority( self.action ) >
               self.actionPriority( other.action ) )

   def __hash__( self ):
      return hash( ( self.ipAddress( self.ip ), self.source ) )
