#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import sys
from functools import partial

# The ones with "pilint" are to satisfy "a4 submit" that barfs about invalid option
# replace back with "pylint" when checking in workspace (I guess py2 vs py3 pylint)
# pylint: disable=anomalous-backslash-in-string
# pylint: disable=cell-var-from-loop
# pilint: disable=chained-comparison
# pilint: disable=consider-using-from-import
# pilint: disable=consider-using-f-string
# pilint: disable=consider-using-in
# pylint: disable=inconsistent-return-statements
# pylint: disable=singleton-comparison
# pylint: disable=ungrouped-imports
# pylint: disable=unused-import
# pilint: disable=use-a-generator
# pilint: disable=useless-object-inheritance
# This one is required by "a4 submit" only!
# pylint: disable=len-as-condition

import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import

from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME

from IsisCliLib import LVL_KW_TO_TAC_TYPE_MAP

from CliModel import cliPrinted
# pylint: disable=cyclic-import

from RibCapiLib import EmptyResponseException
from RibCapiLib import AmiResponseException
from CliPrint import CliPrint
from TypeFuture import TacLazyType

from CliPlugin.RoutingIsisCli import (
   isisConfig,
   isisStatusDir,
   vrfInfoDir,
   vrf6InfoDir,
)

cprinter = CliPrint().lib

# Exception not raised when try fails. Suppress pylint warning
# pylint: disable-msg=bare-except
# pylint: disable-msg=protected-access
# pylint: disable-msg=broad-except

IsisAddressFamily = TacLazyType( 'Routing::Isis::AddressFamily' )

def showIsisCommand( *args, **kwargs ):
   '''
   args - Contains CLI mode and callback function
   kwargs - Contains the cmdVrfModel and instDictModelType for the Capi compatible
          commands.

   instDictModelType - Instance models for CAPI compatible commands
   cmdVrfModel - cmd VRF model for CAPI compatible commands

   If vrf name is specified show the information for the Isis instance in that VRF.
   If it doesn't exist report error.
   We do not allow both instance name and vrf name to be specified
   If neither instance name nor vrf name are specified use the VRF name from the
   routing context if it is set. If routing context is not set and there is an
   Isis instance configured in the default VRF, then show information for that
   instance. If there is no Isis instance in the default VRF but there is/are Isis
   instance(s) in non-default VRF(s), show information for one of those instances.
   Sort them by instance name and show the status for first one.
   Otherwise report error.
   '''

   mode = args[ 0 ]

   # callback will be the last entry in *args
   callback = args[ -1 ]
   args = args[ : -1 ]

   # RibCapiLib will not accept these extra arguments - Need to remove them
   for keyword in ( 'show', 'isis', 'vrf' ):
      kwargs.pop( keyword, None )

   # Some callback functions take 'level' as keyword argument
   # and not as 'level-1' or 'level-2'
   level = kwargs.pop( 'LEVEL', None )
   if level is not None:
      kwargs[ 'level' ] = level

   instDictModelType = kwargs.pop( 'instDictModelType', None )
   cmdVrfModel = kwargs.pop( 'cmdVrfModel', None )
   instanceName = kwargs.pop( 'INSTANCE', None )
   vrfName = kwargs.pop( 'VRFNAME', None )

   useCliPrint = kwargs.pop( 'useCliPrint', False )
   if useCliPrint:
      showVrfModelRoutine = isisShowVrfModelUsingCliPrint
   else:
      showVrfModelRoutine = isisShowVrfModel

   sortedInstanceList = [ isisConfig.instanceConfig[ iName ] for iName in 
                         sorted( isisConfig.instanceConfig ) ]

   vrfs = { inst.vrfName for inst in  sortedInstanceList }

   instanceList = sortedInstanceList
   vrfSet = vrfs

   if instanceName:
      instance = isisConfig.instanceConfig[ instanceName ]
      instanceList = [ instance ]
      vrfSet = { instance.vrfName }
   elif vrfName:
      # vrf name is specified
      if vrfName != ALL_VRF_NAME:
         if vrfName not in vrfs:
            mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
            return
         else:
            vrfSet = { vrfName }
   else:
      # If vrf name and instance name isn't specified
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName not in vrfs:
            mode.addError( "No IS-IS instance is configured in routing-context "
                           "VRF: %s" % vrfName )
            return
         vrfSet = { vrfName }
      elif DEFAULT_VRF in vrfs:
         vrfSet = { DEFAULT_VRF }
      else:
         if not isisConfig.instanceConfig:
            return cmdVrfModel() if cmdVrfModel else None
         vrfSet = { sortedInstanceList[0].vrfName }
   # Some vrf name is always set.

   if cmdVrfModel:
      return showVrfModelRoutine( vrfSet, mode, cmdVrfModel, instDictModelType,
                                  callback, instanceList, *args, **kwargs)
   else:
      for instance in instanceList:
         if instance.vrfName in vrfSet:
            invokeCallback( instance, callback, *args, **kwargs )

def getIsisConfigAndStatus( mode, instanceId, instanceName ):
   if not instanceName:
      mode.addError( "Invalid IS-IS instance name" )
      return None, None
   
   instConfig = isisConfig.instanceConfig[ instanceName ]
   
   if not instConfig:
      mode.addError( "IS-IS (%s) Can not find configuration" % instanceName )
      return None, None
  
   isisStatus = isisStatusDir.get( instConfig.vrfName )
   instStatus = isisStatus.instanceStatus[ instanceName ]
   if not instStatus:
      mode.addError( "IS-IS (%s) Can not find status data" % instanceName )
      return None, None
   
   if instConfig.shutdown:
      return None, None
   
   return instConfig, instStatus

#-------------------------------------------------------------------------------
# Show commands
#-------------------------------------------------------------------------------
def invokeCallback( instance, callback, *args, **kwargs ):
   cmdPrefix = 'show isis'
   cmd = cmdPrefix
   mode = args[0]

   isisStatus = isisStatusDir.get( instance.vrfName )
   warnings, statusValid = checkAndShowIsisConfigMismatch( mode, instance,
                                                           isisStatus )
   if warnings:
      for warning in warnings:
         mode.addWarning( warning )
   if statusValid:
      # statusValid is True only if isisStatus is not None and
      # isisStatus.instanceStatus[instance.instanceName] is not None
      instanceId = isisStatus.instanceStatus[
            instance.instanceName ].instanceId
      # The actual show command handlers don't need/accept vrfName as
      # keyword arguments.
      if 'vrfName' in kwargs:
         del kwargs[ 'vrfName' ]
      kwargs[ 'cmdPrefix' ] = cmd
      kwargs[ 'instanceId' ] = instanceId
      kwargs[ 'instanceName' ] = instance.instanceName

      return callback( *args, **kwargs )

def isisShowVrfModelUsingCliPrint( vrfSet, mode, cmdVrfModel, instDictModelType,
                                   callback, instanceList, *args, **kwargs ):
   vrfInstDict = {}
   # Loop for prepopulating warnings for all instances
   for vrf in vrfSet:
      vrfInstDict[ vrf ] = []
   for instance in instanceList:
      isisStatus = isisStatusDir.get( instance.vrfName )
      if instance.vrfName in vrfSet:
         warnings, statusValid = checkAndShowIsisConfigMismatch( mode,
                                                       instance, isisStatus )
         if warnings:
            for warning in warnings:
               mode.addWarning( warning )
         vrfInstDict[ instance.vrfName ] += \
                                    [ { 'instanceName' : instance.instanceName,
                                        'statusValid' :statusValid } ]

   fd = sys.stdout.fileno()
   outputFormat = cprinter.stringToOutputFormat( mode.session_.outputFormat_ )
   p = cprinter.initPrinter( fd, outputFormat, True )
   cprinter.startRender( p )
   cprinter.startDict( p, "vrfs" )
   for vrf in vrfSet:
      cprinter.startDict( p, vrf )
      isisStatus = isisStatusDir.get( vrf )
      if isisStatus is None:
         return
      cprinter.startDict( p, "isisInstances" )
      for inst in vrfInstDict[ vrf ]:
         if inst[ 'statusValid' ]:
            cprinter.startDict( p, inst[ 'instanceName' ] )
            kwargs[ 'vrfName' ] = vrf
            kwargs[ 'instanceId' ] = isisStatus. \
                                     instanceStatus[ inst[ 'instanceName' ] ]. \
                                     instanceId
            kwargs[ 'instanceName' ] = inst[ 'instanceName' ]
            _ = callback( *args, **kwargs )
            cprinter.endDict( p ) # this instance
      cprinter.endDict( p ) # instances
      cprinter.endDict( p ) # this vrf
   cprinter.endDict( p ) # vrfs
   cprinter.endRender( p )
   cprinter.deinitPrinter( p )
   return cliPrinted( cmdVrfModel )

def isisShowVrfModel( vrfSet, mode, cmdVrfModel, instDictModelType, callback,
                      instanceList, *args, **kwargs ):
   def instDictFunc( vrf ):
      isisStatus = isisStatusDir.get( vrf )
      if isisStatus is None:
         return
      for inst in vrfInstDict[ vrf ]:
         instModel = None
         if inst[ 'statusValid' ]:
            kwargs[ 'vrfName' ] = vrf
            kwargs[ 'instanceId' ] = isisStatus. \
                                   instanceStatus[inst[ 'instanceName' ]]. \
                                   instanceId
            kwargs[ 'instanceName' ] = inst[ 'instanceName' ]

            # handling EmptyResponseException raised from RibCapiLib
            try:
               instModel = callback( *args, **kwargs )
            except EmptyResponseException:
               instModel = None
            except AmiResponseException:
               instModel = None
            
         if instModel is None:
            continue
         yield inst[ 'instanceName' ], instModel 
   def vrfListFunc( vrfs ):
      for vrf in vrfs:
         instDictModel = instDictModelType()
         instDictModel.isisInstances = instDictFunc( vrf )
         yield vrf, instDictModel

   vrfInstDict = {}
   #Loop for prepopulating warnings for all instances
   for vrf in vrfSet:
      vrfInstDict[ vrf ] = []

   for instance in instanceList:
      isisStatus = isisStatusDir.get( instance.vrfName )
      if instance.vrfName in vrfSet:
         warnings, statusValid = checkAndShowIsisConfigMismatch( mode,
                                                       instance, isisStatus )
         if warnings:
            for warning in warnings:
               mode.addWarning( warning )
         vrfInstDict[ instance.vrfName ] += \
                                       [ { 'instanceName' : instance.instanceName,
                                           'statusValid' :statusValid } ]

   model = cmdVrfModel()
   model.vrfs = vrfListFunc( vrfSet )
   return model

def _isisConfigVrfIsNotActiveReason( config ):
   return "Vrf %s is not active" % config.vrfName

def _isisConfigInvalidReason( config, status, vrfStatus, vrf6Status ):
   msgs = []
   if not status.netValid:
      msgs.append( "IS-IS Network Entity Title (NET) configuration is not present" )

   v4RoutingStatus = vrfStatus and vrfStatus.routing
   v6RoutingStatus = vrf6Status and vrf6Status.routing
   vrfMessage = ''
   if config.vrfName != DEFAULT_VRF:
      vrfMessage = ' in vrf %s' % config.vrfName

   if status.activeAddressFamily == IsisAddressFamily.addressFamilyNone:
      if config.addressFamily == IsisAddressFamily.addressFamilyNone:
         msgs.append( "IS-IS address family configuration is not present" )
      if config.addressFamily == IsisAddressFamily.addressFamilyIPv4 \
             and not v4RoutingStatus:
         msgs.append( "IPv4 unicast routing is not enabled%s" % vrfMessage )
      elif config.addressFamily == IsisAddressFamily.addressFamilyIPv6 \
             and not v6RoutingStatus:
         msgs.append( "IPv6 unicast routing is not enabled%s" % vrfMessage )
      elif config.addressFamily == IsisAddressFamily.addressFamilyBoth \
             and not v4RoutingStatus and not v6RoutingStatus:
         msgs.append( "Routing is not enabled%s" % vrfMessage )
   return msgs

def _isisBothAfConfigMismatchReason( config, status, vrfStatus, vrf6Status ):
   msg = None
   v4RoutingStatus = vrfStatus and vrfStatus.routing
   v6RoutingStatus = vrf6Status and vrf6Status.routing
   if not v4RoutingStatus:
      msg = 'IS-IS IPv4 and IPv6 address family are configured but IPv4 unicast '
      msg += 'routing is not enabled for instance %s' % status.instanceName 
   if not v6RoutingStatus:
      msg = "IS-IS IPv4 and IPv6 address family are configured but IPv6 unicast "
      msg += "routing is not enabled for instance %s" % status.instanceName
   return msg

def checkAndShowIsisConfigMismatch( mode, config, isisStatus ):
   '''
   Wrapper function that needs to be used to invoke a show command handler, to show
   reason for ISIS to be disabled / not running as configured due to a config
   mismatch.
   If the config is valid it calls the show command handler function.
   '''
   # When this function is called, an instance exists. The status, however, may
   # or may not exist. The status doesn't exist in the case of the isis instance
   # living in a non-default VRF which was either not created or was deleted.
   # If the status is None, we let users know about it printing
   # Vrf <vrfName> is not active
   instanceName = config.instanceName
   warnings = []
   configValid = None

   if isisStatus is not None and instanceName in isisStatus.instanceStatus:
      status = isisStatus.instanceStatus[ instanceName ]
      configValid = status.configValid
      vrfStatus = vrfInfoDir.get( config.vrfName )
      vrf6Status = vrf6InfoDir.get( config.vrfName )
      if not configValid:
         reasons = _isisConfigInvalidReason( config, status, vrfStatus, vrf6Status )
         for r in reasons:
            mode.addError( "IS-IS (%s) is disabled because: %s"
                           % ( instanceName, r ) )
      else:
         if config.addressFamily == IsisAddressFamily.addressFamilyBoth:
            reason = _isisBothAfConfigMismatchReason( config, status, vrfStatus,
                                                      vrf6Status )
            if reason is not None:
               warnings.append( "IS-IS ({}) has incorrect configuration: {}".format(
                  instanceName, reason ) )

   elif isisStatus is None and config.vrfName != DEFAULT_VRF:
      configValid = False
      reason = _isisConfigVrfIsNotActiveReason( config )
      mode.addError( "IS-IS (%s) is disabled because: %s"
                     % ( instanceName, reason ) )

   if config.shutdown:
      warnings.append( "IS-IS (%s) is shutdown" % instanceName )

   return warnings, configValid

# This function allows us to reverse-lookup from the LVL_KW_TO_TAC_TYPE dict.
getLevel = partial( LVL_KW_TO_TAC_TYPE_MAP.reverse().__getitem__ )

def isisSrv6IsActive( mode, instanceName ): 
   instConfig = isisConfig.instanceConfig[ instanceName ]

   if instConfig.srv6Configured == instConfig.srv6ConfiguredDefault:
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing IPv6 " \
            "not configured" % instanceName )
      return False

   if instConfig.srv6Enabled != instConfig.srv6EnabledDefault:
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing IPv6 has been " \
            "administratively shutdown" % instanceName )
      return False

   return True
