# Copyright (c) 2022, Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from itertools import (
   chain,
   starmap,
)
import json
import jsonschema

import ArTogglesPyAgent
from MultiRangeRule import decodeStrRanges
from TypeFuture import TacLazyType
from YamlLoaderLibrary.Exceptions import ParsingError
from YamlLoaderLibrary.Parser import ParserBase, getValidationError
from YamlLoaderLibrary.Versions import Version

# This regex simulates an arbitray numeric range. In essence, it is a comma delimited
# list of items. Each item can be a single number, or a range of numbers denoted by a
# start value and an end value joined by a dash. Note that regex does NOT help with
# verifying that any range is nonempty or negative, so we will need the actual decode
# to do that for us. Also, note that \d+ by nature rejects any negative numbers
SINGLE_OR_RANGE = r'(\d+)(-(\d+))?'
INTF_SLOT_GROUP_REGEX = fr'^({SINGLE_OR_RANGE})(,({SINGLE_OR_RANGE}))*$'

CARD_PROFILE_NAME_REGEX = r'^[A-Za-z0-9/_-]+$'

CardProfile = TacLazyType( 'L1Profile::CardProfile' )
CardProfileDescriptor = TacLazyType( 'L1Profile::CardProfileDescriptor' )
CardProfileSource = TacLazyType(
         'L1Profile::CardProfileSource::CardProfileSource' )
InterfaceSlotDescriptor = TacLazyType( 'L1Profile::InterfaceSlotDescriptor' )
InterfaceSlotProfileDescriptor = TacLazyType(
   'L1Profile::InterfaceSlotProfileDescriptor' )
InterfaceSlotProfileSource = TacLazyType(
   'L1Profile::InterfaceSlotProfileSource::InterfaceSlotProfileSource' )

class ParserV1( ParserBase ):
   version = Version( 1, 3 )

   schema = {
      '$schema': 'https://json-schema.org/draft/2020-12/schema',
      '$id': 'cardProfileBuiltinV1',
      'title': 'Card profile YAML definition V1',
      'description': ( 'The root object containing the definition of a card'
                       'profile' ),
      'patternProperties': {
         CARD_PROFILE_NAME_REGEX: {
            'type': 'object',
            'properties': {
               'display-name': {
                  'type': 'string',
                  'minLength': 1,
               },
               'visible': {
                  'type': 'boolean',
               },
               'description': {
                  'type': 'string',
                  'minLength': 1,
               },
               'applicability': {
                  'type': 'array',
                  'items': {
                     'anyOf': [
                        {
                           'type': 'object',
                           'patternProperties': {
                              r'^\S+$': {
                                 'type': 'string',
                              }
                           }
                        },
                        {
                           'type': 'string'
                        },
                     ]
                  },
               },
               'interface-slots': {
                  'type': 'object',
                  'properties': {
                     'eth': {
                        'patternProperties': {
                           INTF_SLOT_GROUP_REGEX: {
                              'type': 'string',
                           },
                        },
                        'additionalProperties': False,
                        'minProperties': 1,
                     },
                  },
                  'additionalProperties': False,
               },
            },
            'additionalProperties': False,
            'required': [ 'description', 'interface-slots', 'applicability' ],
         },
      },
      'additionalProperties': False,
      'maxProperties': 1,
   }

   def validateDocument( self, documentSource, document ):
      try:
         # This is where the limitations of using a JSON schema with a YAML input
         # become painfully clear. YAML is sane and allows keys to be integers, JSON
         # is not and does not. The solution: dump the YAML dict to string and reload
         # it using the JSON infra which will automatically resolve these type
         # issues...
         jsonschema.validate( json.loads( json.dumps( document ) ), self.schema )
      except jsonschema.exceptions.ValidationError as error:
         raise getValidationError( f'{error}', document ) from error

   def parseDocument( self, document ):
      profileName, profileDefinition = next( iter( document.items() ) )
      cardProfile = CardProfile(
         CardProfileDescriptor( CardProfileSource.builtin,
                                             profileName ) )
      cardProfile.displayName = profileDefinition.get( 'display-name', profileName )
      cardProfile.visible = profileDefinition.get( 'visible', True )
      cardProfile.description = profileDefinition[ 'description' ]

      toggles = { toggle[ 'name' ]: toggle[ 'enabled' ]
                  for toggle in ArTogglesPyAgent.pyGetAllToggles() }

      for applicability in profileDefinition[ 'applicability' ]:
         if isinstance( applicability, dict ):
            skuBaseName, featureToggle = next( iter( applicability.items() ) )
         else:
            skuBaseName, featureToggle = ( applicability, None )

         if featureToggle and featureToggle not in toggles:
            raise ParsingError( f'Invalid featureToggle for {skuBaseName}: '
                                f'{featureToggle}' )

         if featureToggle and not toggles[ featureToggle ]:
            continue

         if skuBaseName in cardProfile.applicability:
            raise ParsingError( f'Multiple {skuBaseName} found in applicability' )
         cardProfile.applicability.add( skuBaseName )

      assignedSlots = set()

      def writeIntfSlotProfile( prefix, intfSlotId, name ):
         if intfSlotId in assignedSlots:
            raise ParsingError(
               f'Interface Slot {intfSlotId} already has a profile' )
         assignedSlots.add( intfSlotId )
         intfSlotDesc = InterfaceSlotDescriptor( prefix, intfSlotId )
         intfSlotProfileDesc = InterfaceSlotProfileDescriptor(
            InterfaceSlotProfileSource.builtin, name )
         cardProfile.intfSlotProfile[ intfSlotDesc ] = intfSlotProfileDesc

      ethIntfSlots = profileDefinition[ 'interface-slots' ][ 'eth' ]
      for intfSlotGroups, intfSlotProfileName in ethIntfSlots.items():
         try:
            intfSlotRanges = decodeStrRanges( str( intfSlotGroups ),
                                              noSingletons=True )
         except ValueError as e:
            raise ParsingError( 'Invalid Interface Slot ranges encountered: '
                                f'{intfSlotGroups}' ) from e

         for intfSlotId in chain.from_iterable( starmap(
               lambda start, end: range( start, end + 1 ), intfSlotRanges ) ):
            writeIntfSlotProfile( 'Ethernet', intfSlotId, intfSlotProfileName )

      return cardProfile
