# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import re

__defaultTraceHandle__ = Tracing.Handle( "LagAsu" )

t0 = Tracing.trace0
t1 = Tracing.trace1

# NOTE the last three pattern is repeated. Maybe there is a better way to
# define the re
portIdRe = \
   re.compile( r"([0-9a-fA-F]+),([0-9a-fA-F\-]+)," +
               "([0-9a-fA-F]+),([0-9a-fA-F]+),([0-9a-fA-F]+)" )

class LagId:

   @staticmethod
   def deserialize( repstr ):
      # LagId is consists of two LagPortId concated together. I want to reuse
      # portIdRe. If there are better ways to do it. I will be all ears.

      slist = portIdRe.findall( repstr )
      if len( slist ) != 2:
         # LagId string is malformed
         return None
      localStr = ",".join( slist[0] )
      remoteStr = ",".join( slist[1] )
      lagId = Tac.Value( 'Lacp::LagId' )
      lagId.localId = LagPortId.deserialize( localStr )
      lagId.remoteId = LagPortId.deserialize( remoteStr )
      return lagId

   def __init__( self, lagId ):
      self.localId = LagPortId( lagId.localId )
      self.remoteId = LagPortId( lagId.remoteId )

   def __hash__( self ):
      return hash( ( self.localId, self.remoteId ) )

   def __eq__( self, other ):
      return ( self.localId, self.remoteId ) == ( other.localId, other.remoteId )

   def __repr__( self ):
      return "[" + repr( self.localId ) + "," + repr( self.remoteId ) + "]"

   def __str__( self ):
      return self.__repr__()

def dot43sysid( priority, macAddr ):
   addrStr = macAddr.replace( ":", "-" )
   # pylint: disable-next=consider-using-f-string,superfluous-parens
   return( "%04X,%s"% ( priority, addrStr ) )

class LagPortId:

   @staticmethod
   def deserialize( repstr ):
      portId = Tac.Value( 'Lacp::LagPortId' )
      m = portIdRe.search( repstr )
      if m:
         portId.sysPriority = int( m.group( 1 ), 16 )
         portId.sysId = m.group( 2 ).replace( "-", ":" )
         portId.key = int( m.group( 3 ), 16 )
         portId.portPriority = int( m.group( 4 ), 16 )
         portId.portId = int( m.group( 5 ), 16 )
      else:
         # Either repstr is empty or malformatted. In both cases, return
         # default value to caller.
         t0( "Unable to deserialize LagPortId", repstr )

      return portId

   def __init__( self, lagPortId ):
      self.sysPriority = lagPortId.sysPriority
      self.sysId = lagPortId.sysId
      self.key = lagPortId.key
      self.portPriority = lagPortId.portPriority
      self.portId = lagPortId.portId

   def __hash__( self ):
      return hash( ( self.sysPriority, self.sysId, self.key, self.portPriority,
                     self.portId ) )

   def __eq__( self, other ):
      return( ( self.sysPriority, self.sysId, self.key, 
                self.portPriority, self.portId )
              ==
              ( other.sysPriority, other.sysId, other.key,
                other.portPriority, other.portId ) )

   def __repr__( self ):
      s = "(" + dot43sysid( self.sysPriority, self.sysId ) + ","
      s += f"{self.key:04X},{self.portPriority:04X},{self.portId:04X})"
      return( s ) # pylint: disable=superfluous-parens

   def __str__( self ):
      return self.__repr__()
