#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac

LanzEnabled = Tac.Type( "Lanz::LanzEnabled" )

class LanzLocked( Exception ):
   pass

def configType():
   return 'Lanz::Config'

def hwConfigType():
   return 'Lanz::Hardware::Platform'

def lockLanz( lanzConfig, lanzHwStatus, feature_name ):
   """Prevent lanz from being enabled

   Return True if the lock has been successfully set.
   The CLI will show `feature_name` in its error message.
   The exact same `feature_name` string must be used to remove the lock"""
   if lanzEnabled( lanzConfig, lanzHwStatus ):
      return False
   lanzConfig.lockedBy[ feature_name ] = True
   # protection against a concurrent call to lanzEnabledIs
   if lanzEnabled( lanzConfig, lanzHwStatus ):
      lanzConfig.lockedBy.remove( feature_name )
      return False
   return True

def unlockLanz( lanzConfig, feature_name ):
   """Remove a lock previously set by lockLanz"""
   if feature_name in lanzConfig.lockedBy:
      del lanzConfig.lockedBy[ feature_name ]


def lanzEnabledIs( lanzConfig, enabled ):
   if lanzConfig.lockedBy and enabled is not False:
      raise LanzLocked(
            # pylint: disable-next=consider-using-f-string
            "lanz is not compatible with the following enabled features: %s" %
            ', '.join( lanzConfig.lockedBy ) )
   if enabled is None:
      lanzConfig.enabled = LanzEnabled.platformDefault
   elif enabled:
      lanzConfig.enabled = LanzEnabled.enabledByCli
   else:
      lanzConfig.enabled = LanzEnabled.disabledByCli

def lanzEnabled( lanzConfig, lanzHwStatus ):
   return ( lanzConfig.enabled == LanzEnabled.enabledByCli
            or ( lanzConfig.enabled == LanzEnabled.platformDefault
                  and lanzHwStatus.lanzEnabledByDefault ) )

def lanzSupported( lanzHwStatus ):
   return lanzHwStatus.supported

def lanzTxLatencyEnabled( lanzConfig, lanzHwStatus, lanzLatencyStatus ):
   latencyEnablingSupported = lanzHwStatus.txLatencyEnablingSupported
   latencySupported = lanzHwStatus.txLatencySupported
   latencyEnabled = lanzConfig.txLatencyEnabled
   ptpSupported = lanzLatencyStatus.ptpTimeSyncSupported
   return latencySupported and lanzEnabled( lanzConfig, lanzHwStatus ) and \
         ( latencyEnabled and ptpSupported if latencyEnablingSupported else True )

def getDefaultHighThreshold( lanzConfig, lanzHwStatus ):
   if lanzConfig.defaultThresholdsConfig.isActive:
      return lanzConfig.defaultThresholdsConfig.defHighThreshold
   else:
      return lanzHwStatus.defHighThreshold

def getDefaultLowThreshold( lanzConfig, lanzHwStatus ):
   if lanzConfig.defaultThresholdsConfig.isActive:
      return lanzConfig.defaultThresholdsConfig.defLowThreshold
   else:
      return lanzHwStatus.defLowThreshold
