#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Import used to generate dependency on Launcher
# pkgdeps : import LauncherContext

from LauncherPlugin.Maintenance import getMmodeRunnability

def Plugin( context ):
   # ArBgp depends on qualPath maintenancemode/ready whenever MaintenanceMode 
   # agent is runnable and ArBgp hasn't already come up. If the MaintenanceMode
   # agent is runnable, this qualPath is set after it comes up and is done with 
   # its work. Considering the MaintenanceMode agent is runnable, the various 
   # stages when the qualPath is set in different cases is:
   # 1. In case of normalBoot, normalBootXp and supervisorColdBoot,
   #    maintenancemode/ready is set once all the plugins are loaded,
   #    maintenance/mapping is setup and onBoot processing is done.
   # 2. In case of ASU reboot( Hitless/Fastboot ), MaintenanceMode agent
   #    participates in early stages ( CriticalAgent stage in hitless bootMode and
   #    PrimaryAgent stage in fastboot boot mode ) and sets this qualPath at
   #    the end of doMountsComplete.
   # 3. In case of RPR standby, MaintenanceMode agent sets this qualPath and
   #    prevents all the SMs from starting. It reacts to redundancyStatus()->mode()
   #    and start all SMs if it is running on active supe.
   # 4. In case of SSO standby, MaintenanceMode agent does not run and
   #    maintenancemode/ready qualPath is synced between active and standby.
   #    So when switchover happens, path is already set and ArBgp will start.
   
   # MaintenanceMode agent is now conditionally runnable on configuration of 
   # "maintenance" command.
   # So, ArBgp should not wait on MaintenanceMode agent to set "ready" qualPath
   # if the agent is not runnable.

   roleName = context.allSupervisorsRoleName
   agentName = "Bgp"
   exeName = "/usr/bin/Bgp"
   baseRunnability = { "qualPath" :
                       "cell/%cellId/routing/defaultVrfProtocolLaunch/Bgp" }

   # getMmodeRunnability returns the runnability criteria for MaintenanceMode
   # agent.
   mmodeRunnable = getMmodeRunnability()

   # If MaintenanceMode is runnable, Bgp should wait for "ready" qualPath
   mmodeReady = { "qualPath" : "maintenance/agent/maintenancemode/ready" }
   
   # getMmodeRunnability with negate as True returns the conditions which
   # evaluate to True when MaintenanceMode agent is not runnable. 
   mmodeNotRunnable = getMmodeRunnability( negate=True ) 

   # There could be a case where Bgp is already running and MaintenanceMode
   # comes up later. In that case, we need to ensure that Bgp does not depend 
   # on MaintenanceMode agent setting the ready qualPath. To ensure this, we 
   # add another qualPath, which specifies whether Bgp is already running
   bgpRanOnce = ( { "qualPath" : "cell/%cellId/agent/config/Bgp" } )

   # BGP agent runs on the CVX device (controller) or on the EOS device having
   # forwarding capability either in hardware or software.

   # CVX node can be either in standalone mode or be a part of a cluster. When
   # the CVX node is part of cluster, it participates in the leader election,has
   # a result of the leader election it is either assigned a 'leader' or
   # 'follower' role.

   # The presence of controllerdbStatus indicates that the EOS is running on a
   # CVX node . When controllerdbStatus is not enabled it indicates that it is
   # on a switch/router node.

   # Hence we base our runnability condition on presence of controllerdbStatus.

   # While executing on the CVX it should only run when the node is either in
   # 'standalone' mode or it is  leader of the cluster. While in the role of a
   # switch/router it should be based on BGP runnability condition.

   # The logic that is implemented is

   # if ( ( not controllerdbStatus ) or
   #      ( ( controllerdbStatus and
   #          defaultClusterStatus == isStandaloneOrLeader ) and
   #        bgpRunnabilityCondition ) )

   controllerdbStatus = context.lookup( 'controller/status' )
   defaultClusterStatus = context.lookup(
      'controller/cluster/statusDir.status/default' )

   mastership = [
      { 'entity' : controllerdbStatus,
        'qualAttr' : 'enabled',
        'negate' : True },
      { 'entity' : defaultClusterStatus,
        'qualAttr' : 'isStandaloneOrLeader' } ]

   # So, the final runnability is:
   # mastership AND baseRunnability AND
   # ((mmodeRunnable AND mmodeReady) OR mmodeNotRunnable OR bgpRanOnce)
   runnability = ( mastership,
                   baseRunnability,
                   [ ( mmodeRunnable, mmodeReady ),
                      mmodeNotRunnable,
                      bgpRanOnce ] )
   
   agentCfg = { "name": agentName,
                "exe": exeName,
                "argv" : [],
                "heartbeatPeriod": 30,
                "oomScoreAdj" : -300, # see AID3426
                "coredumpTimeout": 600,
                "runnability" : runnability }

   context.agentConfigIs( roleName, agentCfg )
