#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Dummy import used to generate dependency on Launcher
# pkgdeps: import LauncherContext
from Toggles.SfeToggleLib import toggleSfeIpv6Enabled
import os
import sys
import Tac

def Plugin( context ):
   agentName = "Sfe"

   launcherPath = 'hardware/sfe/launcherConfig/Sfe'
   FruConfigPath = context.lookup( 'hardware/sfe/veosConfig' )
   routeCacheConfig = context.lookup( 'routecache/config' )
   routingHwStatus = context.lookup( 'routing/hardware/status' )
   openFlowStatus = context.lookup( 'openflow/status' )
   natHwCapabilities = context.lookup( 'ip/nat/hwCapabilities' )
   natConfig = context.lookup( 'ip/nat/config' )
   natStatus = context.lookup( 'ip/nat/status' )
   mapCapabilities = context.lookup( 'nat/map/capabilities' )
   nacSwitchStatus = context.lookup(
         'sfe/nacies/nacAgentStatus' )
   fphHwCapabilities = context.lookup( 'flowPinHole/hwcapabilities' )

   sfeLauncherPath = ( { 'qualPath' : launcherPath } )

   fruConfigUpdated = ( { 'entity' : FruConfigPath, 'qualAttr' : 'configUpdated' } )

   routeCacheSupported = ( { 'entity' : routingHwStatus,
                             'qualAttr' : 'routeCacheSupported' } )
   notRouteCacheSupported = ( { 'entity' : routingHwStatus,
                                'qualAttr' : 'routeCacheSupported',
                                'negate' : True } )
   routeCacheEnabled = ( { 'entity' : routeCacheConfig,
                           'qualAttr' : 'cachingEnabled' } )

   natDpdkSupported = ( { 'entity' : natHwCapabilities,
                          'qualAttr' : 'natDpdkProcessingSupported' } )

   fphSupported = ( { 'entity' : fphHwCapabilities,
                      'qualAttr' : 'flowPinHoleIpv6Supported' } )

   # these are the same conditions like at NAT PI agent
   natRunnability = [ { 'entity' : natConfig,
                        'qualAttr' : 'intfConfig' },
                      { 'entity' : natConfig,
                        'qualAttr' : 'vrfConfig' },
                      { 'entity' : natConfig,
                        'qualAttr' : 'natFlow' },
                      { 'entity' : natStatus,
                        'qualAttr' : 'intfStatus' } ]

   swOpenFlowStatus = ( { 'entity' : openFlowStatus,
                          'qualAttr' : 'softwareForwarding' } )

   mapTSupported = ( { 'entity' : mapCapabilities, 'qualAttr' : 'mapTSupported' } )

   notCaravanPlatform = ( { 'entity' : FruConfigPath,
                            'qualAttr' : 'platformCaravan',
                            'negate' : True } )

   # Independence and Councilbluffs need NacIes to be running
   caravanPhySwitchUp = ( { 'entity' : FruConfigPath,
                            'qualAttr' : 'platformCaravan' },
                          { 'entity' : FruConfigPath,
                            'qualAttr' : 'platformHasNac' },
                          { 'entity' : nacSwitchStatus,
                             'qualAttr' : 'switchStateUp' } )

   # Willamette systems do not have a NAC and will not run NacIes
   caravanNacNotPresent = ( { 'entity' : FruConfigPath,
                            'qualAttr' : 'platformCaravan' },
                          { 'entity' : FruConfigPath,
                            'qualAttr' : 'platformHasNac',
                            'negate' : True } )

   def populateRunnability( forCaravanPlatform=False, forNacPresent=False ):

      if forCaravanPlatform:
         # for Caravan platforms check the switch state up before
         # starting Sfe agent
         if forNacPresent:
            caravanRunnability = caravanNacNotPresent
         else:
            caravanRunnability = caravanPhySwitchUp
      else:
         caravanRunnability = notCaravanPlatform

      runnability = [
         # runnability is:

         # launcherPath present AND FruConfigPath.configUpdated true AND
         # routingHwStatus.routeCacheSupported = False AND
         # caravan runnability defined above
         ( sfeLauncherPath, fruConfigUpdated,
            notRouteCacheSupported, caravanRunnability ),

         # OR
         # launcherPath present AND routingHwStatus.routeCacheSupported AND
         #     routeCacheConfig.cachingEnabled AND
         # caravan runnability defined above
         ( sfeLauncherPath, routeCacheSupported,
            routeCacheEnabled, caravanRunnability ),

         # OR
         # openFlowStatus.softwareForwarding = True AND
         # caravan runnability defined above
         ( swOpenFlowStatus, caravanRunnability ),

         # OR
         # launcherPath present AND
         # natHwCapabilities.natDpdkProcessingSupported = True AND
         # nat pi runnability is fulfilled AND
         # caravan runnability defined above
         ( sfeLauncherPath, natDpdkSupported, natRunnability,
            caravanRunnability ),

         # OR
         # launcherPath present AND
         # fphHwCapabilities.flowPinHoleIpv6Supported = True AND
         # there is a flow pin hole configuration AND
         # caravan runnability defined above
         ( sfeLauncherPath, fphSupported,
            caravanRunnability ),
      ]

      if toggleSfeIpv6Enabled():
         runnability += [
            # OR
            # launcherPath present AND
            # mapCapabilities.mapTSupported (for MAP-T offload) AND
            # caravan runnability defined above
            ( sfeLauncherPath, mapTSupported,
               caravanRunnability ),
         ]

      return runnability

   runnability = populateRunnability( forCaravanPlatform=True )
   runnability += populateRunnability( forCaravanPlatform=False )
   runnability += populateRunnability( forCaravanPlatform=True, forNacPresent=True )
   runnability += populateRunnability( forCaravanPlatform=True, forNacPresent=False )

   # Sfe/bess requires allocating a large contigious virtual address
   # space, if the ArSlab slab allocator also grabs a significant
   # chunk of virtual address space bess may fail to initialize, so
   # limit the number of 4kB pages that ArSlab will grab.
   is64bit = sys.maxsize >= 0xFFFFFFFF
   if 'RUNNING_IN_NMDUT' not in os.environ and not is64bit:
      # Until BUG964038 is implemented, use ProcMgrModifyCmdline to
      # set the env variable.
      cmd = ( '/usr/bin/ProcMgrModifyCmdline',
              'add',
              'Sfe',
              'environ',
              'ARSLAB_MMAP_PAGE_COUNT=4096', # 16 MB
             )
      output = Tac.run( cmd,
                        stdout=Tac.CAPTURE,
                        stderr=Tac.CAPTURE )
      assert not output or output in ( 'ProcMgr has loaded changes.\n',
                                       'Changes queued for ProcMgr start\n' ), \
         f'Unexpected output from {" ".join( cmd )}: {output}'

   agentCfg = { "name" : agentName,
                "exe" : "/usr/bin/Sfe",
                "argv" : [],
                "heartbeatPeriod" : 180,
                "coredumpTimeout" : 200,
                "runnability" : runnability
              }

   roleName = context.activeSupervisorRoleName
   context.agentConfigIs( roleName, agentCfg )

