#!/usr/bin/env python3
# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tracing
import pwd
import spwd

# pkgdeps: rpm AaaPam-lib

t0 = Tracing.Handle( "Aaa" ).trace0

nssAaaDisableFunc_ = None

def getNssAaaDisableFunc( ):
   global nssAaaDisableFunc_
   if not nssAaaDisableFunc_:
      from ctypes import CDLL, c_uint
      LIBPAM = CDLL( "libAaaPamCommon.so" )
      nssAaaDisableFunc_ = LIBPAM.aaa_nss_disable
      nssAaaDisableFunc_.restype = c_uint
      nssAaaDisableFunc_.argstypes = [ c_uint ]
   return nssAaaDisableFunc_

########## Regular expressions for account management ###########

# Regular expression that matches a valid username. Valid usernames
# consist of 1 letter or digit followed by letters or digits or special characters.
# Special characters allowed by Console/SSH/Telnet are #$&%*+,-.;<>=@^_`|~()[]{}
# Console/SSH/Telnet do not allow '" special characters.
# Console login do not allow !: special characters.
# Console/SSH/Telnet allows \ but after login prints the following
# error: "Could not chdir to home directory".
# pylint: disable=anomalous-backslash-in-string
usernameRe = r'[a-zA-Z0-9][a-zA-Z0-9#$&%*+,\-.;<>=@^_|~]*'

######### Account management helper functions ##########

# There might be a number of reasons why a given account name is
# invalid. So far, we know of one reason: the account name is already
# in use by some other package.
def isValidAccountName( username ):
   if username is None:
      return False
   return not isSystemAccount( username )

def getPasswdEntry( username ):
   return getUserEntry( username, readShadow=False )

def getShadowPasswdEntry( username ):
   return getUserEntry( username, readShadow=True )

def getUserEntry( username, readShadow=False ):
   userEntry = None
   disableFunc = getNssAaaDisableFunc( )
   nssAaaDisabled = disableFunc( 1 )
   try:
      try:
         if readShadow:
            userEntry = spwd.getspnam( username )
         else:
            userEntry = pwd.getpwnam( username )
      except KeyError:
         # The account doesn't exist -- we could create and manage it if
         # we wanted.
         #
         # Note that this doesn't cover the case where an account is
         # created by the administrator now (say 'foo'), and then package
         # "Foo" is installed, creating account 'foo' as part of its
         # installation scripts. I'm not sure we can cover that case
         # without being prescient...
         pass
   finally:
      disableFunc( nssAaaDisabled )
   return userEntry

def isSystemAccount( username ):
   """Returns True if an account with the given username exists, and
   it is a 'system' account, e.g. one created by some Linux package.
   Cuurently, any account in /etc/passwd is a system account, since
   Eos doesn't create any account.  """

   if getPasswdEntry( username ) is None:
      return False
   return True
