#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import collections
import json

import ArPyUtils
import EntityManager
import PyClient
import Tac

mcscvplibpath = 'mcs/status/cvp'
MSC = Tac.Type( "McsCvpLib::McsStatusCode" )
RR = Tac.Type( "McsCvpLib::MulticastActiveRouteReferrar" )
MKey = collections.namedtuple( 'MKey', 'source group' )

# Examples of values for message codes
HAClients = {
   "clients": [
      "gts498.sjc.aristanetworks.com",
      "gts449.sjc.aristanetworks.com",
      "yo657.sjc.aristanetworks.com"
   ]
}

RedisStatus = {
   "restartCount": 1,
   "stopCount": 0,
   "startCount": 0,
   "enabled": True
}

ApiStatus = {
   'enabled': True,
   'sysnStatus': 'finished'
}

ArPyUtils.runMeAsRoot()

class McsCvpTest:
   def __init__( self, namespace="ar" ):
      pc = PyClient.PyClient( namespace, 'Sysdb' )
      sock = pc.root()[ 'Connection' ][ 'NboAttrLog' ].connDir.listenDomainSock
      em = EntityManager.Sysdb( namespace, sysdbsockname=sock )
      mg = em.mountGroup()
      cvpMount = mg.mount( mcscvplibpath, 'McsCvpLib::McsAgentStatus', 'wi' )
      mg.close( blocking=True )
      self.cvp = cvpMount

      # Devices in topology
      self.mcsDevices = {}
      self.devices = {
         'spine1': 'aa:bb:cc:dd:ee:01',
         'spine2': 'aa:bb:cc:dd:ee:02',
         'leaf1': 'aa:bb:cc:dd:ee:03',
         'leaf2': 'aa:bb:cc:dd:ee:04',
         'leaf3': 'aa:bb:cc:dd:ee:05',
      }
      for dk, dv in self.devices.items():
         self.mcsDevices[ dk ] = self.tacMcsDevice( dv, devName=dk )

   def tacMcastKey( self, s, g ):
      return Tac.Value( "McsCvpLib::McastKey", s, g )

   def tacMcsDevice( self, ethAddr, devName=None ):
      mcsDevice = Tac.Value( "McsCvpLib::McsDevice", ethAddr )
      mcsDevice.sysName = devName
      return mcsDevice

   def tacMcsSender( self, source, group ):
      sg = self.tacMcastKey( source, group )
      mcsSender = Tac.Value( "McsCvpLib::McsSender", sg )
      mcsSender.trackingId = 100
      mcsSender.bwInKbps = 100
      mcsSender.label = 'My Sender'
      return mcsSender

   def tacMcsFlow( self, s, g ):
      mcastKey = self.tacMcastKey( s, g )
      return Tac.Value( "McsCvpLib::McsFlow", mcastKey )

   def tacMcsRoute( self, sg, iif, oifs ):
      mcsRoute = Tac.Value( "McsCvpLib::McsRoute", sg )
      mcsRoute.iif = iif
      for oif in oifs:
         mcsRoute.oif.add( oif )
      return mcsRoute

   def tacBandwidthInfo( self, totalInKbps, usedInKbps=0, availableInKbps=0 ):
      bwInfo = Tac.Value( "McsCvpLib::BandwidthInfo", totalInKbps )
      bwInfo.usedInKbps = usedInKbps
      bwInfo.availableInKbps = availableInKbps
      return bwInfo

   def tacMcsEndpoint( self, mcsDevice, intfId, direction='rx' ):
      return Tac.Value( "McsCvpLib::McsEndpoint",
                        mcsDevice, intfId, direction )

   def tacMcsRcvKey( self, sg, device, intfId ):
      return Tac.Value( "McsCvpLib::McsReceiverKey", sg, device, intfId )

   def tacMcsReceiver( self, s, g, device, intfId ):
      sg = self.tacMcastKey( s, g )
      dev = self.mcsDevices.get( device, '' )
      mcsRcvKey = self.tacMcsRcvKey( sg, dev, intfId )
      return Tac.Value( "McsCvpLib::McsReceiver", mcsRcvKey )

   def tacMcsDeviceFlowKey( self, device, s, g, ref=RR.mcs ):
      mcsDevice = self.mcsDevices.get( device, '' )
      sg = self.tacMcastKey( s, g )
      return Tac.Value( "McsCvpLib::McsDeviceFlowKey", mcsDevice, sg, ref )

   def tacMulticastActiveRoute( self, flowKey, iif, oifs ):
      mActiveRoute = Tac.Value( "McsCvpLib::MulticastActiveRoute", flowKey )
      mActiveRoute.iif = iif
      for oif in oifs:
         mActiveRoute.oif.add( oif )
      mActiveRoute.inHw = True
      return mActiveRoute

   def tacIgmpSnoopingMember( self, vlanId, intfs ):
      igmp = Tac.Value( "McsCvpLib::IgmpSnoopingMember", vlanId )
      for intf in intfs:
         igmp.interface.add( intf )
      return igmp

   def linkDevice( self, device1, intfId1, device2, intfId2 ):
      dev1 = self.mcsDevices.get( device1, '' )
      dev2 = self.mcsDevices.get( device2, '' )
      endpointA = self.tacMcsEndpoint( dev1, intfId1, 'rx' )
      endpointB = self.tacMcsEndpoint( dev2, intfId2, 'tx' )
      mcsNetworkLinkA = Tac.Value( "McsCvpLib::McsNetworkLink",
                                   endpointA, endpointB )
      endpointC = self.tacMcsEndpoint( dev1, intfId1, 'tx' )
      endpointD = self.tacMcsEndpoint( dev2, intfId2, 'rx' )
      mcsNetworkLinkB = Tac.Value( "McsCvpLib::McsNetworkLink",
                                   endpointC, endpointD )
      return [ mcsNetworkLinkA, mcsNetworkLinkB ]

   def addClientDevices( self ):
      for device in self.mcsDevices.values():
         self.cvp.mcsClientDevice.addMember( device )
      Tac.runActivities( 1 )

   def addHaStatus( self, data=None ):
      haData = data if data else json.dumps( HAClients )
      self.cvp.msgCode[ MSC.haStatus ] = haData
      Tac.runActivities( 1 )

   def addAgentStatus( self, data=None ):
      agentData = data if data else "MCS AGENT IS ALIVE"
      self.cvp.msgCode[ MSC.agentStatus ] = agentData
      Tac.runActivities( 1 )

   def addApiStatus( self, data=None ):
      agentData = data if data else json.dumps( ApiStatus )
      self.cvp.msgCode[ MSC.apiStatus ] = agentData
      Tac.runActivities( 1 )

   def addRedisStatus( self, data=None ):
      redisData = data if data else json.dumps( RedisStatus )
      self.cvp.msgCode[ MSC.redisStatus ] = redisData
      Tac.runActivities( 1 )

   def addActiveSenders( self ):
      for i in range( 1, 21 ):
         f = MKey( f'2.1.1.{i}', '225.1.1.1' )
         self.cvp.activeSender.addMember( self.tacMcsSender( f.source, f.group ) )
      Tac.runActivities( 1 )

   def addInactiveSenders( self ):
      for i in range( 1, 11 ):
         f = MKey( f'2.1.1.{i}', '225.1.1.1' )
         self.cvp.inactiveSender.addMember( self.tacMcsSender( f.source, f.group ) )
      Tac.runActivities( 1 )

   def addActiveRecievers( self ):
      f = MKey( '1.1.1.1', '225.1.1.1' )
      self.cvp.activeReceiver.addMember(
            self.tacMcsReceiver( f.source, f.group, 'leaf1', 'Ethernet1' ) )
      self.cvp.activeReceiver.addMember(
            self.tacMcsReceiver( f.source, f.group, 'spine1', 'Ethernet1' ) )
      Tac.runActivities( 1 )

   def addInactiveReceivers( self ):
      f = MKey( '1.1.1.10', '225.1.1.1' )
      self.cvp.inactiveReceiver.addMember(
            self.tacMcsReceiver( f.source, f.group, 'leaf2', 'Ethernet1' ) )
      Tac.runActivities( 1 )

   def addImpactedReceivers( self ):
      f = MKey( '1.1.1.11', '225.1.1.1' )
      self.cvp.impactedReceiver.addMember(
            self.tacMcsReceiver( f.source, f.group, 'leaf2', 'Ethernet1' ) )
      Tac.runActivities( 1 )

   def addFailedReceivers( self ):
      f = MKey( '1.1.1.12', '225.1.1.1' )
      self.cvp.failedReceiver.addMember(
            self.tacMcsReceiver( f.source, f.group, 'leaf2', 'Ethernet1' ) )
      Tac.runActivities( 1 )

   def addFlowProgrammed( self ):
      f = MKey( '1.1.1.1', '225.1.1.1' )
      self.cvp.flowProgrammed.addMember( self.tacMcsFlow( f.source, f.group ) )
      Tac.runActivities( 1 )

   def addMaintenanceDevice( self ):
      spine2 = self.mcsDevices.get( 'spine2', '' )
      self.cvp.maintenanceDevice.addMember( spine2 )
      Tac.runActivities( 1 )

   def addEndpoint( self ):
      for v in self.cvp.mcsClientDevice.values():
         for i in range( 1, 6 ):
            self.cvp.endPoint.add( self.tacMcsEndpoint( v, f"Ethernet{i}", 'rx' ) )
            self.cvp.endPoint.add( self.tacMcsEndpoint( v, f"Ethernet{i}1", 'tx' ) )
      Tac.runActivities( 1 )

   def addNetworkLink( self ):
      for l in self.linkDevice( 'spine1', 'Ethernet1', 'leaf1', 'Ethernet1' ):
         self.cvp.networkLink.add( l )
      for l in self.linkDevice( 'spine1', 'Ethernet2', 'leaf2', 'Ethernet1' ):
         self.cvp.networkLink.add( l )
      for l in self.linkDevice( 'spine1', 'Ethernet3', 'leaf3', 'Ethernet1' ):
         self.cvp.networkLink.add( l )
      for l in self.linkDevice( 'spine2', 'Ethernet1', 'leaf1', 'Ethernet2' ):
         self.cvp.networkLink.add( l )
      for l in self.linkDevice( 'spine2', 'Ethernet2', 'leaf2', 'Ethernet2' ):
         self.cvp.networkLink.add( l )
      for l in self.linkDevice( 'spine2', 'Ethernet3', 'leaf3', 'Ethernet2' ):
         self.cvp.networkLink.add( l )
      Tac.runActivities( 1 )

   def addActiveFlows( self ):
      for device in self.devices:
         for i in range( 1, 100 ):
            flow = MKey( f'1.1.1.{i}', '225.1.1.1' )
            fk = self.tacMcsDeviceFlowKey( device, flow.source, flow.group )
            mRoute = self.tacMulticastActiveRoute( fk, 'Ethernet1',
                                                   [ 'Ethernet10', 'Ethernet12' ] )
            for vlan in [ 100, 200, 300 ]:
               igmp = self.tacIgmpSnoopingMember( vlan,
                                                  [ 'Ethernet10', 'Ethernet12' ] )
               mRoute.igmpSnooping.add( igmp )
            self.cvp.activeFlows.addMember( mRoute )
      Tac.runActivities( 1 )

def updateMcsAgentStatus( namespace='ar' ):
   mcsCvp = McsCvpTest( namespace=namespace )
   mcsCvp.addHaStatus()
   mcsCvp.addAgentStatus()
   mcsCvp.addRedisStatus()
   mcsCvp.addApiStatus()
   mcsCvp.addClientDevices()
   mcsCvp.addActiveSenders()
   mcsCvp.addInactiveSenders()
   mcsCvp.addActiveRecievers()
   mcsCvp.addInactiveReceivers()
   mcsCvp.addImpactedReceivers()
   mcsCvp.addFailedReceivers()
   mcsCvp.addFlowProgrammed()
   mcsCvp.addMaintenanceDevice()
   mcsCvp.addEndpoint()
   mcsCvp.addNetworkLink()
   mcsCvp.addActiveFlows()
   print( "... updateMcsAgentStatus DONE ..." )

if __name__ == '__main__':
   parser = argparse.ArgumentParser(
      description="MCS Agent Status for CVP" )
   parser.add_argument( "-n", "--sysName", default="ar",
                        help="Namepace name" )
   args = parser.parse_args()
   updateMcsAgentStatus( namespace=args.sysName )
