#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import random
import time
import ArPyUtils
import EntityManager
import PyClient
import Tac

mcscvplibpath = 'mcs/status/cvp'
BWTYPE = Tac.Type( "McsCvpSysdb::BandwidthType" )
DEVROLE = Tac.Type( "McsCvpSysdb::DeviceRole" )

maxDevAddr = 0x1000000000000

ArPyUtils.runMeAsRoot()

class McsCvpSysdbTest:
   def __init__( self, sysName="ar", spineDev=1, spineIntf=1, leafDev=1, leafIntf=1,
                 recvDev=1, recvIntf=1, groupNum=1, sourceNum=1 ):
      # number of available leaves must be equal to or more than devices per flow
      assert leafDev >= recvDev + 1
      # number of spine interfaces must be equal to or more than devices per flow
      assert spineIntf >= recvDev + 1


      # name for system
      self.sysName = sysName
      # number of spine devices
      self.spineDev = spineDev
      # number of interfaces for each spine device
      self.spineIntf = spineIntf
      # number of leaf devices (to choose receivers/sender from)
      self.leafDev = leafDev
      # number of interfaces for each leaf device
      self.leafIntf = leafIntf
      # number of receiver devices per flow
      self.recvDev = recvDev
      # number of output interfaces per receiver device in any flow
      self.recvIntf = recvIntf
      # number of groups
      self.groupNum = groupNum
      # number of sources
      self.sourceNum = sourceNum

      # mount model in sysdb
      if self.sysName == 'ar':
         pc = PyClient.PyClient( sysName, 'Sysdb' )
         sock = pc.root()[ 'Connection' ][ 'NboAttrLog' ].connDir.listenDomainSock
         em = EntityManager.Sysdb( sysName, sysdbsockname=sock )
         mg = em.mountGroup()
         cvpMount = mg.mount( mcscvplibpath, 'McsCvpSysdb::Status', 'wi' )
         mg.close( blocking=True )
         self.status = cvpMount
      else:
         self.status = Tac.newInstance( 'McsCvpSysdb::Status', 'cvp' )

      # set up device info
      curDevHex = 0x0000000000000

      self.spines = [ None ] * spineDev
      for i in range( spineDev ):
         devAddr = self.getDevAddr( curDevHex )
         curDevHex += 15
         hostname = f'spine{ i }'
         self.spines[ i ] = ( devAddr, hostname )


      self.leaves = [ None ] * leafDev
      for i in range( leafDev ):
         devAddr = self.getDevAddr( curDevHex )
         curDevHex += 15
         hostname = f'leaf{ i + spineDev }'
         self.leaves[ i ] = ( devAddr, hostname )

   def waitForReaders( self ):
      if self.sysName == 'ar':
         Tac.runActivities( 1 )

   def getDevAddr( self, devHex ):
      assert devHex < maxDevAddr

      hexAddr = format( devHex, '012x' )
      return ":".join( hexAddr[ j : j + 2 ] for j in range( 0, 12, 2 ) )

   def sampleFromList( self, n, s ):
      return random.sample( range( n ), s )

   def tacBandwidth( self, val, bwType ):
      return Tac.Value( "McsCvpSysdb::Bandwidth", val, bwType )

   def tacMcastSender( self, key ):
      return Tac.Value( "McsCvpSysdb::McastSender", key )

   def tacMcastHopDevice( self, key ):
      return Tac.Value( "McsCvpSysdb::McastHopDevice", key )

   def tacMcastReceiver( self, key ):
      return Tac.Value( "McsCvpSysdb::McastReceiver", key )

   def tacIntfEdge( self, key, hostname, label, transactionId, trackingId ):
      return Tac.Value( "McsCvpSysdb::McastIntfEdge",
                        key, hostname, label, transactionId, trackingId )

   def tacIntfLink( self, key, neighborDevId, neighborIntfId, hostname ):
      return Tac.Value( "McsCvpSysdb::McastIntfLink",
                        key, neighborDevId, neighborIntfId, hostname )

   def addFlow( self, source, trackingId ):
      # choosing leaf devices to use in current flow
      flowIdLst = random.sample( range( self.leafDev ), self.recvDev + 1 )
      random.shuffle(flowIdLst)
      senderId = flowIdLst[ 0 ]
      receiverIdLst = flowIdLst[ 1: ]

      # choosing spine device/interfaces for current flow
      spineId = random.choice( range( self.spineDev ) )
      spineIntfLst = random.sample( range( 1, self.spineIntf + 1 ),
                                    self.recvDev + 1 )
      random.shuffle(spineIntfLst)

      # initialize spine tac value
      spineAddr = self.spines[ spineId ][ 0 ]
      spineName = self.spines[ spineId ][ 1 ]
      spine = self.tacMcastHopDevice( spineAddr )
      spine.devRole = DEVROLE.spine

      # initialize sender tac value
      senderAddr = self.leaves[ senderId ][ 0 ]
      senderName = self.leaves[ senderId ][ 1 ]
      sender = self.tacMcastSender( senderAddr )
      sender.devRole = DEVROLE.sender

      # set up sender's iif and oif (connected to spine), add sender to flow
      senderIntfLst = random.sample( range( 1, self.leafIntf + 1 ), 2 )
      random.shuffle( senderIntfLst )
      sender.iif = self.tacIntfEdge( f'Ethernet{ senderIntfLst[ 0 ] }',
                                     f'Input { trackingId }',
                                     f'Camera { trackingId }',
                                     f' txnId for { trackingId }',
                                     trackingId )
      oif = self.tacIntfLink( f'Ethernet{ senderIntfLst[ 1 ] }', spineAddr,
                              f'Ethernet{ spineIntfLst[ 0 ] }', spineName )
      sender.oifs.add( oif )
      source.sender = sender

      # add sender info to spine iif
      spine.iif = self.tacIntfLink( f'Ethernet{ spineIntfLst[ 0 ] }', senderAddr,
                                    f'Ethernet{ senderIntfLst[ 1 ] }', senderName )

      # initialize receivers, add receivers to flow, and connect each to spine
      for i in range( self.recvDev ):
         receiverId = receiverIdLst[ i ]
         receiverAddr = self.leaves[ receiverId ][ 0 ]
         receiverName = self.leaves[ receiverId ][ 1 ]
         receiver = self.tacMcastReceiver( receiverAddr )
         receiver.devRole = DEVROLE.receiver

         receiverIntfLst = random.sample( range( 1, self.leafIntf + 1 ),
                                          self.recvIntf + 1 )
         random.shuffle( receiverIntfLst )
         receiver.iif = self.tacIntfLink( f'Ethernet{ receiverIntfLst[ 0 ] }',
                                          spineAddr,
                                          f'Ethernet{ spineIntfLst[ i + 1 ] }',
                                          spineName)

         spineOif = self.tacIntfLink( f'Ethernet{ spineIntfLst[ i + 1 ] }',
                                      receiverAddr,
                                      f'Ethernet{ receiverIntfLst[ 0 ] }',
                                      receiverName )
         spine.oifs.add( spineOif )

         for j in range( self.recvIntf):
            receiverOif = self.tacIntfEdge( f'Ethernet{ receiverIntfLst[ j + 1 ] }',
                                          f'Input { trackingId }',
                                          f'Cluster { trackingId }',
                                          f' txnId for { trackingId }',
                                          trackingId )
            receiver.oifs.add( receiverOif )
         source.receiver.addMember( receiver )

      source.hopDevice.addMember( spine )

   def addSources( self, group ):
      counter = self.sourceNum

      for a in range( 128, 192 ):
         for b in range( 0, 256 ):
            for c in range( 0, 256 ):
               for d in range( 0, 256 ):
                  if counter <= 0:
                     return
                  counter -= 1

                  s = f'{ a }.{ b }.{ c }.{ d }'
                  group.sources.newMember( s )
                  group.sources[ s ].status = 0
                  group.sources[ s ].flowBw = self.tacBandwidth( 0, BWTYPE.kilobps )
                  self.addFlow( group.sources[ s ], counter )

   def addGroups( self ):
      counter = self.groupNum
      for a in range( 224, 240 ):
         for b in range( 0, 256 ):
            for c in range( 0, 256 ):
               for d in range( 0, 256 ):
                  if counter <= 0:
                     return
                  counter -= 1

                  g = f'{ a }.{ b }.{ c }.{ d }'
                  self.status.groups.newMember( g )
                  self.addSources( self.status.groups[ g ] )

   def cleanup( self ):
      if self.status is not None:
         self.status.groups.clear()

   def printCount( self ):
      print( )


def updateStatus( sysName, spineDev, spineIntf, leafDev, leafIntf,
                  recvDev, recvIntf, groupNum, sourceNum ):
   print( "!!! Started updatedStatus !!!" )

   mcsCvp = McsCvpSysdbTest(
                              sysName,
                              spineDev,
                              spineIntf,
                              leafDev,
                              leafIntf,
                              recvDev,
                              recvIntf,
                              groupNum,
                              sourceNum,
                           )
   mcsCvp.cleanup()

   startTime = time.time()
   mcsCvp.addGroups()
   finishTime = time.time()

   mcsCvp.waitForReaders()
   exitTime = time.time()

   print( "!!! Finished updatedStatus !!!" )

   print( f"Time for populating model: { finishTime - startTime }" )
   print( f"Time for waiting for reader: { exitTime - finishTime }" )

if __name__ == '__main__':
   parser = argparse.ArgumentParser(
      description="MCS Status for CVP" )
   parser.add_argument( "-n", "--sysName", default="ar",
                       help="Namepace name" )
   parser.add_argument( "-sd", "--spineDev", default=1,
                       help="Number of spine devices" )
   parser.add_argument( "-si", "--spineIntf", default=50,
                       help="Number of available interfaces per spine" )
   parser.add_argument( "-ld", "--leafDev", default=200,
                       help="Number of available leaf devices" )
   parser.add_argument( "-li", "--leafIntf", default=120,
                       help="Number of available interfaces per leaf" )
   parser.add_argument( "-rd", "--recvDev", default=20,
                       help="Number of receiver devices per flow" )
   parser.add_argument( "-ri", "--recvIntf", default=1,
                       help="Number of receiver oifs per device in a flow" )
   parser.add_argument( "-g", "--groupNum", default=103000,
                       help="Number of groups" )
   parser.add_argument( "-s", "--sourceNum", default=1,
                       help="Number of sources per group" )

   args = parser.parse_args()

   updateStatus(
               args.sysName,
               int( args.spineDev ),
               int( args.spineIntf ),
               int( args.leafDev ),
               int( args.leafIntf ),
               int( args.recvDev ),
               int( args.recvIntf ),
               int( args.groupNum ),
               int( args.sourceNum ),
               )

   # python3 test/McsCvpSysdbStatusUpdate.py -n test
