# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ApiBaseModels import ModelJsonSerializer
from McsHttpErrorCodes import errorCodes
from GenericReactor import GenericReactor
from UwsgiRequestContext import ( HttpNotFound,
                                 UwsgiRequestContext )
from ControllerdbEntityManager import Controllerdb
from CliPlugin.IntfModel import intfOperStatusToEnum
from CliPlugin.IntfCli import _forwardingModelMap

import Agent
import AaaApiClient
import Tac
import Tracing
import BothTrace
import UwsgiConstants
import UrlMap
import McsApiModels
import json
import Arnet
import os
import McsHttpAgentLib
from McsHttpAgentLib import MCS_BOUNDARY_INTF
import McsStateHAReplicate
import http.client
import copy
import Toggles.McsToggleLib
import threading

info = BothTrace.tracef0
warn = BothTrace.tracef1
error = BothTrace.tracef2
log = BothTrace.tracef3

bv = BothTrace.Var
__defaultTraceHandle__ = Tracing.Handle( 'McsHttpAgent' )

mcastKeyType = Tac.Type( "Mcs::McastKey" )
apiSyncStatus = Tac.Type( "Mcs::ApiSyncStatus" )
deviceIntfId = McsHttpAgentLib.DevAndIntfType
boundaryIntf = Tac.Type( "Mcs::McsBoundaryIntf" )
chStateType = McsHttpAgentLib.ChStateType

class McsControllerMounts:
   def __init__( self, cdbMg ):

      self.cdbTopology = cdbMg.mount( 'topology/version3/global/status',
                                      'NetworkTopologyAggregatorV3::Status', 'r' )
      self.switchStatusDir = cdbMg.mount( 'mcs/v1/fromSwitch/status',
                                       'Tac::Dir', 'riS' )
      self.cdbActiveFlows = cdbMg.mount( 'mcs/v1/fromSwitch/activeflows',
                                 'Tac::Dir', 'riS' )
      self.mcsStatusHA = cdbMg.mount( "mcs/v1/apiCfgRedStatus/switch",
                                      "Tac::Dir", "riS" )

class McsApiBase:
   """All classes that handle a particular URL should be derived from this class"""
   def __init__( self ):
      # The decorator UrlMap.urlHandler won't register the actual object. So, this is
      # a hack to replace the preregistered method for the url with a tuple of the
      # object and the method name. This helps McsApiHandler in calling the method on
      # the real object
      apis = self.api if isinstance( self.api, list ) else [ self.api ]
      for api in apis:
         uri = UrlMap.UrlToRegex()( api )
         for httpType, reqFunc in UrlMap.urlMap.items():
            if uri in reqFunc:
               # pylint: disable-next=unnecessary-dict-index-lookup
               f = UrlMap.urlMap[ httpType ][ uri ]
               # pylint: disable-next=unnecessary-dict-index-lookup
               UrlMap.urlMap[ httpType ][ uri ] = ( self, f.__name__ )

   def get_resp_template( self ):
      return { 'messages': [],
               'success': False
               }

   def get_mcast_resp_template( self ):
      return { 'messages': [],
               'success': False,
               'status': {} }

   def get_mcast_sender_template( self ):
      return { 'messages': [],
               'success': False,
               'trackingID': 0,
               'transactionID': 'AA',
               'receivers': { },
               'senders': { 'action': '',
                           'failedSenders': [],
                           'validSenders': []
                           }
                  }

   def get_mcast_recv_template( self ):
      return { 'messages': [],
               'success': False,
               'trackingID': 0,
               'transactionID': 'AA',
               'receivers': {
                           'messages': [],
                           'failedReceivers': [],
                           'validReceivers': []
                           },
               'senders': { }
                  }

   def get_boundary_intf_resp_template( self ):
      return { 'action': '',
               'data': [],
               'messages': [],
               'success': False
            }

   def get_boundary_intf_template( self ):
      return { 'name': '',
               'interfaces': [],
               'messages': [],
               'success': False
            }

   @property
   def api( self ):
      raise NotImplementedError

class ApiStatus( McsApiBase ):
   api = "/mcs/apiStatus[/]"

   def __init__( self, apiStatus ):
      McsApiBase.__init__( self )
      self.apiStatus = apiStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getApiStatus( self, requestContext ):
      model = McsApiModels.ApiStatusModel()
      model.fromSysdb( self.apiStatus )
      result = json.loads( json.dumps( model, cls=ModelJsonSerializer ) )
      log( "Getting API status ", bv( result ) )
      if result:
         return http.client.OK, result
      else:
         return http.client.NO_CONTENT, None


class PostMcastSender( McsApiBase ):
   api = "/mcs/multicast/senders[/]"

   def __init__( self, apiConfig, apiNotify, apiStatus ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.apiNotify = apiNotify
      self.apiStatus = apiStatus
      self.devIntfType = deviceIntfId
      self.MCAST_FLOW_KEYS = { 'flow-action',
                               'trackingID',
                               'data',
                               'transactionID' }
      self.SENDER_FIELDS = { 'destinationIP',
                             'sourceIP',
                             'bandwidth',
                             'inIntfID',
                             'bwType' }
      self.bwTypeMap = { 'k': 1,
            'm': 1000,
            'g': 1000000 }
      self.senderActionMap = { 'addSenders': 'set',
                            'delSenders': 'del',
                            'modBw': 'modBw' }
      # Building pairFlow collection of apiStatus after agent restart
      self.apiStatus.pairFlow.clear()
      for key, value in self.apiConfig.mcastSender.items():
         if value.pairFlow != mcastKeyType():
            self.apiStatus.pairFlow[ key ] = value.pairFlow
            self.apiStatus.pairFlow[ value.pairFlow ] = key

   def sendNotify( self, postData, errorCode=None, sender=None, isFailure=True ):
      """Send failure or delete notification messages for a sender requests"""

      action = postData.get( 'flow-action' )
      action = self.senderActionMap.get( action ) if action else 'U/A'
      messages = ( errorCode if isinstance( errorCode, list )
                   else [ errorCode ] if errorCode else [] )
      data = ( sender if isinstance( sender, list )
                         else [ sender ] if sender else [] )
      if isFailure:
         # Adding FailedSenders in post error code
         messages.extend( [ '185' ] )

      # Add the deleted senders, valid and failed, to the message queue
      notifyData = {
         'action': action,
         'trackingID': postData.get( 'trackingID', 0 ),
         'messages': messages,
         'data': data
      }
      notifyType = Tac.Type( "Mcs::ApiNotifyMsgType" )
      notify = Tac.Value( "Mcs::ApiNotifyMsg", notifyType.delSender,
                             json.dumps( notifyData ) )
      self.apiNotify.message = notify

   def addSenderPopulateSysdb( self, data ):
      for sender in data[ 'data' ]:
         postedBwType = sender[ 'bwType' ]
         source = sender[ 'sourceIP' ]
         group = sender[ 'destinationIP' ]
         mcastKey = mcastKeyType( source, group )
         senderApiConfig = self.apiConfig.mcastSender.newMember( mcastKey )
         if ( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
             sender[ 'inIntfID' ].startswith( MCS_BOUNDARY_INTF ) ):
            self.apiConfig.boundarySender[ mcastKey ] = sender[ 'inIntfID' ]
         else:
            device, intfId = sender[ 'inIntfID' ].split( '-' )
            senderApiConfig.senderId = self.devIntfType( device, intfId )
         label = sender[ 'label' ] if 'label' in sender else None
         if 'pairSourceIP' in sender and 'pairDestinationIP' in sender:
            pairSource = sender[ 'pairSourceIP' ]
            pairGroup = sender[ 'pairDestinationIP' ]
            pairFlow = Tac.Value( 'Mcs::McastKey', pairSource, pairGroup )
            senderApiConfig.pairFlow = pairFlow
            self.apiStatus.pairFlow[ mcastKey ] = pairFlow
            self.apiStatus.pairFlow[ pairFlow ] = mcastKey
            pairSender = self.apiConfig.mcastSender.get( pairFlow )
            if pairSender:
               pairSender.pairFlow = mcastKey
         bandwidth = self.bwTypeMap[ postedBwType ] * sender[ 'bandwidth' ]
         senderApiConfig.bw = ( bandwidth, 'kilobps' )
         if label is not None:
            senderApiConfig.label = label
         if sender.get( 'dscp' ) and sender.get( 'tc' ):
            senderApiConfig.dscp = sender.get( 'dscp' )
            senderApiConfig.tc = sender.get( 'tc' )
         if isinstance( sender.get( 'applyPolicy' ), bool ):
            senderApiConfig.applyPolicy = sender.get( 'applyPolicy' )
         senderApiConfig.transactionId = data[ 'transactionID' ]
         senderApiConfig.trackingId = data[ 'trackingID' ]
         # The visitCount MUST be the last attribute to be updated since this
         # dictates when this mcastSender will be synced over to the standby.
         senderApiConfig.visitCount += 1

   def validateSenders( self, postData ):
      # returns senderUpdate, debugMsgs

      # Maintaing a local collection of pairFlow per request
      # object. This is needed to handle case where a single
      # add request has more than one flows paired two a flow
      localPairedFlow = {}
      senderUpdate = self.get_mcast_resp_template()
      senderUpdate[ 'status' ] = self.get_mcast_sender_template()
      ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.MCAST_FLOW_KEYS,
                                                         postData )
      # Validate Sender Post Fields
      if ec:
         senderUpdate[ 'status' ][ 'messages' ].append( ec )
         if 'data' in missingKeys:
            failedSender = []
         else:
            failedSender = postData[ 'data' ]
         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] = failedSender
         if 'flow-action' in missingKeys:
            senderUpdate[ 'status' ][ 'senders' ][ 'action' ] = 'U/A'

         # Send notification
         self.sendNotify( postData, errorCode=ec, sender=failedSender )
         return senderUpdate

      flowAction = postData[ 'flow-action' ]
      if flowAction in self.senderActionMap:

         senderUpdate[ 'status' ][ 'senders' ][ 'action' ] = \
                                    self.senderActionMap[ flowAction ]
      else:
         senderUpdate[ 'status' ][ 'senders' ][ 'action' ] = 'U/A'
         ec = '113'
         error( errorCodes[ ec ].replace( '<postedAction>',
                                             flowAction ) )

         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] += \
                                                                  postData[ 'data' ]
         senderUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData, errorCode=ec, sender=postData[ 'data' ] )
         return senderUpdate

      if not isinstance( postData[ 'data' ], list ):
         ec = '121'
         error( errorCodes[ ec ] )
         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] += \
                                                                  postData[ 'data' ]
         senderUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData, errorCode=ec, sender=postData[ 'data' ] )
         return senderUpdate

      valid, ec = McsHttpAgentLib.validateTrackingId( postData[ 'trackingID' ] )
      if not valid:
         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] += postData[
               'data' ]
         senderUpdate[ 'status' ][ 'messages' ].append( ec )
         senderUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
         senderUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
         self.sendNotify( postData, errorCode=ec, sender=postData.get( 'data', [] ) )
         return senderUpdate

      valid, ec = McsHttpAgentLib.validateTransactionId( postData[ 'transactionID'
         ] )
      if not valid:
         senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] += postData[
               'data' ]
         senderUpdate[ 'status' ][ 'messages' ].append( ec )
         senderUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
         senderUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
         self.sendNotify( postData, errorCode=ec, sender=postData.get( 'data', [] ) )
         return senderUpdate

      for sender in postData[ 'data' ]:
         if 'messages' not in sender:
            sender[ 'messages' ] = []

         pairFlowFields = {}
         if 'pairSourceIP' in sender or 'pairDestinationIP' in sender:
            pairFlowFields = { 'pairSourceIP', 'pairDestinationIP' }
         senderFields = self.SENDER_FIELDS.union( pairFlowFields )
         ec, missingKeys = McsHttpAgentLib.findMissingKeys( senderFields, sender )
         if ec:
            sender[ 'messages' ].append( ec )
            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )

            # Send notification
            self.sendNotify( postData, sender=sender )
            continue
         sourceIP = sender[ 'sourceIP' ]
         destinationIP = sender[ 'destinationIP' ]
         valid, m = McsHttpAgentLib.validateMcastGroup( destinationIP )
         if not valid:
            if m:
               sender[ 'messages' ].append( m )

            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         valid, m = McsHttpAgentLib.validateSourceIP( sourceIP )
         if not valid:
            if m:
               sender[ 'messages' ].append( m )

            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         if ( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
             isinstance( sender[ 'inIntfID' ], str ) and
             sender[ 'inIntfID' ].isalnum() ):
            valid, ec = McsHttpAgentLib.validateBoundaryIntfName(
                                                              sender[ 'inIntfID' ] )
            if ( valid and
                 sender[ 'inIntfID' ] not in self.apiConfig.boundaryInterface ):
               valid = False
               ec = '252'
               error( errorCodes[ ec ].replace( '<bIntf>', sender[ 'inIntfID' ] ) )

         else:
            valid, ec, inDevice, inIntf = McsHttpAgentLib.validateDeviceID(
                                                              self.apiConfig,
                                                              sender[ 'inIntfID' ] )
         if not valid:
            sender[ 'messages' ].append( ec )
            senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append( sender )
            # Send notification
            self.sendNotify( postData, sender=sender )
            continue

         mcastKey = mcastKeyType( sourceIP, destinationIP )
         if flowAction != 'delSenders':
            if 'label' in sender:
               valid, ec = McsHttpAgentLib.validateLabel( sender[ 'label' ] )
               if not valid:
                  error( errorCodes[ ec ] )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] \
                     .append( sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue

            valid = McsHttpAgentLib.validateBandwidthType( sender[ 'bwType' ] )
            if not valid:
               ec = '124'
               error( errorCodes[ ec ].replace( '<bwType>', sender[ 'bwType' ] ) )
               sender[ 'messages' ].append( ec )

               senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                     sender )
               # Send notification
               self.sendNotify( postData, sender=sender )
               continue

            if 'dscp' in sender:
               valid, ec = McsHttpAgentLib.validateDscp( sender[ 'dscp' ] )
               if not valid:
                  msg = errorCodes[ ec ].replace( '<dscp>', str( sender[ 'dscp' ] ) )
                  error( msg )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] \
                     .append( sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue
            if 'tc' in sender:
               valid, ec = McsHttpAgentLib.validateTc( sender[ 'tc' ] )
               if not valid:
                  msg = errorCodes[ ec ].replace( '<tc>', str( sender[ 'tc' ] ) )
                  error( msg )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] \
                     .append( sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue
            if 'applyPolicy' in sender:
               valid, ec = McsHttpAgentLib.validateApplyPolicy(
                     sender[ 'applyPolicy' ] )
               if not valid:
                  msg = errorCodes[ ec ].replace( '<applyPolicy>', str( sender[
                     'applyPolicy' ] ) )
                  error( msg )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ] \
                     .append( sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue

            parsedBw = sender[ 'bandwidth' ] * self.bwTypeMap[ sender[ 'bwType' ] ]

            valid = McsHttpAgentLib.validateBandwidth( parsedBw )
            if not valid:
               ec = '123'
               error( errorCodes[ ec ] )
               sender[ 'messages' ].append( ec )
               senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                     sender )
               # Send notification
               self.sendNotify( postData, sender=sender )
               continue
            sender[ 'bandwidth' ] = parsedBw
            sender[ 'bwType' ] = 'k'
            # Verify validity of pair flow
            pairSourceIP = sender.get( "pairSourceIP" )
            pairDestinationIP = sender.get( "pairDestinationIP" )
            if pairSourceIP or pairDestinationIP:
               valid, m = McsHttpAgentLib.validateMcastGroup( pairDestinationIP )
               if not valid:
                  if m:
                     sender[ 'messages' ].append( m )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                                                                            sender )
                  self.sendNotify( postData, sender=sender )
                  continue

               valid, m = McsHttpAgentLib.validateSourceIP( pairSourceIP )
               if not valid:
                  if m:
                     sender[ 'messages' ].append( m )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                                                                            sender )
                  self.sendNotify( postData, sender=sender )
                  continue

               # Handling case when flow and paired flow is already programmed
               # new request is trying to pair them
               mcastKeyPairedFlow = mcastKeyType( pairSourceIP, pairDestinationIP )
               if ( mcastKey in self.apiConfig.mcastSender and mcastKeyPairedFlow in
                     self.apiConfig.mcastSender ):
                  ec = '253'
                  msg = errorCodes[ ec ].replace( '<flow1>', sourceIP +
                       ',' + destinationIP )
                  msg = msg.replace( '<flow2>', pairSourceIP + ',' +
                        pairDestinationIP )
                  error( msg )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders'
                        ].append( sender )
                  # send notification
                  self.sendNotify( postData, sender=sender )
                  continue

               srcIpDestIpTuple = ( sourceIP, destinationIP )
               pairSrcIPDestIpTuple = ( pairSourceIP, pairDestinationIP )

               if ( srcIpDestIpTuple in localPairedFlow and localPairedFlow[
                     srcIpDestIpTuple ] != pairSrcIPDestIpTuple ) or ( mcastKey in
                     self.apiStatus.pairFlow and self.apiStatus.pairFlow[ mcastKey ]
                     != mcastKeyPairedFlow ):
                  ec = '247'
                  msg = errorCodes[ ec ].replace( '<flow1>', sourceIP + ',' +
                     destinationIP )
                  if srcIpDestIpTuple in localPairedFlow:
                     msg = msg.replace( '<flow2>', localPairedFlow[ srcIpDestIpTuple
                        ][ 0 ] + ',' + localPairedFlow[ srcIpDestIpTuple ][ 1 ] )
                  else:
                     msg = msg.replace( '<flow2>', self.apiStatus.pairFlow[ mcastKey
                        ].source + ',' + self.apiStatus.pairFlow[ mcastKey ].group )
                  error( msg )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                     sender )
                  # send notification
                  self.sendNotify( postData, sender=sender )
                  continue

               if ( pairSrcIPDestIpTuple in localPairedFlow and localPairedFlow[
                     pairSrcIPDestIpTuple ] != srcIpDestIpTuple ) or (
                  mcastKeyPairedFlow in self.apiStatus.pairFlow and
                     self.apiStatus.pairFlow[ mcastKeyPairedFlow ] != mcastKey ):
                  ec = '247'
                  msg = errorCodes[ ec ].replace( '<flow1>', pairSourceIP +
                     ',' + pairDestinationIP )
                  if pairSrcIPDestIpTuple in localPairedFlow:
                     msg = msg.replace( '<flow2>', localPairedFlow[
                        pairSrcIPDestIpTuple ][ 0 ] + ',' + localPairedFlow[
                           pairSrcIPDestIpTuple ][ 1 ] )
                  else:
                     msg = msg.replace( '<flow2>', self.apiStatus.pairFlow[
                        mcastKeyPairedFlow ].source + ',' + self.apiStatus.pairFlow[
                           mcastKeyPairedFlow ].group )

                  error( msg )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                     sender )
                  # send notification
                  self.sendNotify( postData, sender=sender )
                  continue
               if flowAction == 'addSenders':
                  localPairedFlow[ srcIpDestIpTuple ] = pairSrcIPDestIpTuple
                  localPairedFlow[ pairSrcIPDestIpTuple ] = srcIpDestIpTuple

         storedBw = 0
         if mcastKey in self.apiConfig.mcastSender:
            mcSender = self.apiConfig.mcastSender[ mcastKey ]
            storedBw = mcSender.bw.value if mcSender.bw else 0
            currSenderId = mcSender.senderId
            if ( currSenderId != deviceIntfId() and mcastKey not in
                  self.apiConfig.boundarySender ):
               # Existing sender is non-boundary sender
               currentDevice = currSenderId.device
               currentIntfId = currSenderId.intfId
               if not sender[ 'inIntfID' ].startswith( MCS_BOUNDARY_INTF ):
                  if currentDevice != inDevice or currentIntfId != inIntf:
                     ec = '122'
                     wordMap = { '<deviceId>': inDevice,
                                 '<port>': inIntf }
                     error( McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap ) )
                     sender[ 'messages' ].append( ec )
                     senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                           sender )
                     self.sendNotify( postData, sender=sender )
                     continue
               else:
                  ec = '122'
                  error( errorCodes[ ec ].replace( '<deviceId>-<port>', sender[
                     'inIntfID' ] ) )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                        sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue
            else:
               # Existing sender is a boundary sender
               if( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
                     mcastKey in self.apiConfig.boundarySender ):
                  currSenderId = self.apiConfig.boundarySender[ mcastKey ]
                  if currSenderId != sender[ 'inIntfID' ]:
                     ec = '122'
                     error( errorCodes[ ec ].replace( '<deviceId>-<port>', sender[
                        'inIntfID' ] ) )
                     sender[ 'messages' ].append( ec )
                     senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                           sender )
                     # Send notification
                     self.sendNotify( postData, sender=sender )
                     continue
            if flowAction == 'addSenders':
               if storedBw != parsedBw:
                  ec = '120'
                  wordMap = { '<sourceIP>': sourceIP,
                              '<destinationIP>': destinationIP,
                              '<actualBandwidth>': storedBw }
                  error( McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap ) )
                  sender[ 'messages' ].append( ec )
                  senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ].append(
                        sender )
                  # Send notification
                  self.sendNotify( postData, sender=sender )
                  continue
         senderUpdate[ 'status' ][ 'senders' ][ 'validSenders' ].append( sender )

      senderUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
      senderUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
      senderUpdate[ 'success' ] = True

      return senderUpdate

   @UrlMap.urlHandler( ( 'POST', ), api )
   def postMcastSenders( self, requestContext ):
      """
      Sample post:
      {
           "flow-action": "addSenders",
            "transactionID": "test",
            "trackingID": 69,
            "data": [{
                       "destinationIP": "224.2.0.11",
                       "sourceIP": "10.1.1.100",
                       "bandwidth": 500, "bwType": "m",
                       "inIntfID": "2899.3a8f.998d-Ethernet6",
                       "pairSourceIP": "10.1.1.1",
                       "pairDestinationIP": "224.1.1.1" }
                   }]}
      """
      info( "Configuring Multicast Sender" )
      # XXX TODO: Log trackingID to track posts
      data = requestContext.getRequestContent()
      data = json.loads( data )
      info( 'Received data:', bv( data ) )

      return self.handleSenders( data )

   def handleSenders( self, data ):
      # pylint: disable=too-many-nested-blocks
      log( 'handleSender started' )
      def addSender( d ):
         if d:
            self.addSenderPopulateSysdb( d )
            return True
         else:
            return False
      senderUpdate = self.validateSenders( data )
      validSenders = senderUpdate[ 'status' ][ 'senders' ][ 'validSenders' ]
      if validSenders:
         validData = {
               'data': validSenders,
               'flow-action': data[ 'flow-action' ],
               'trackingID': data[ 'trackingID' ],
               'transactionID': data[ 'transactionID' ]
            }
         action = senderUpdate[ 'status' ][ 'senders' ][ 'action' ]

         if action == 'set':
            addSenderStatus = addSender( validData )
            if not addSenderStatus:
               senderUpdate[ 'success' ] = False
               return http.client.OK, senderUpdate
         else:
            for sender in validSenders:
               if 'messages' not in sender:
                  sender[ 'messages' ] = []
               source = sender[ 'sourceIP' ]
               group = sender[ 'destinationIP' ]

               mcastKey = mcastKeyType( source, group )
               if mcastKey in self.apiConfig.mcastSender:
                  if action == 'del':
                     mcastSender = self.apiConfig.mcastSender[ mcastKey ]
                     senderCopy = copy.copy( sender )
                     senderCopy[ 'label' ] = mcastSender.label
                     senderCopy[ 'dscp' ] = mcastSender.dscp
                     senderCopy[ 'tc' ] = mcastSender.tc
                     senderCopy[ 'applyPolicy' ] = mcastSender.applyPolicy
                     if sender[ 'inIntfID' ].startswith( MCS_BOUNDARY_INTF ):
                        senderCopy[ 'inIntfID' ] = (
                           f"{mcastSender.senderId.device}-"
                           f"{mcastSender.senderId.intfId}" )
                        senderCopy[ 'boundaryInterface' ] = sender[ 'inIntfID' ]
                     pairedFlow = mcastSender.pairFlow
                     del self.apiConfig.mcastSender[ mcastKey ]
                     if mcastKey in self.apiConfig.boundarySender:
                        del self.apiConfig.boundarySender[ mcastKey ]
                     if ( mcastKey in self.apiStatus.pairFlow and pairedFlow not in
                           self.apiConfig.mcastSender ):
                        del self.apiStatus.pairFlow[ mcastKey ]
                        del self.apiStatus.pairFlow[ pairedFlow ]
                     # Add the deleted senders to the message queue
                     self.sendNotify( data, sender=senderCopy, isFailure=False )
                  if action == 'modBw':
                     modSender = self.apiConfig.mcastSender[ mcastKey ]
                     modSender.bw = ( sender[ 'bandwidth' ], 'kilobps' )
                     if 'label' in sender:
                        modSender.label = sender[ 'label' ]
                     if 'dscp' in sender:
                        modSender.dscp = sender[ 'dscp' ]
                     if 'tc' in sender:
                        modSender.tc = sender[ 'tc' ]
                     if 'applyPolicy' in sender:
                        modSender.applyPolicy = sender[ 'applyPolicy' ]
                     modSender.transactionId = validData[ 'transactionID' ]
                     modSender.trackingId = validData[ 'trackingID' ]
                     # The visitCount MUST be the last attribute to be updated since
                     # this dictates when this mcastSender will be synced over to the
                     # standby.
                     modSender.visitCount += 1
               else:
                  if action == 'del':
                     ec = '105'
                     msg = errorCodes[ ec ].replace( '<sender post>', str( sender ) )
                     warn( msg )
                     senderUpdate[ 'status' ][ 'messages' ].append( ec )
                     self.sendNotify( data, errorCode=ec, sender=sender )
                     # FIXME for this error code the failureSenders is not set!
                  if action == 'modBw':
                     # New behavior is to create the sender if its missing
                     # This section is only there when sender is missing
                     # so we treat it just like addSender
                     addSenderStatus = addSender( validData )
                     if not addSenderStatus:
                        senderUpdate[ 'success' ] = False
                        return http.client.OK, senderUpdate
      # pylint: enable=too-many-nested-blocks
      if not senderUpdate[ 'status' ][ 'senders' ][ 'failedSenders' ]:
         senderUpdate[ 'status' ][ 'success' ] = True
      else:
         senderUpdate[ 'status' ][ 'messages' ].append( '185' )
      log( 'handleSender completed' )
      return http.client.OK, senderUpdate

class ClearRoutes( McsApiBase ):
   api = "/mcs/clearRoutes[/]"

   def __init__( self, apiConfig, apiNotify, apiStatus ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.apiNotify = apiNotify
      self.apiStatus = apiStatus

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext ):
      """ Sample post :
      {
         "ipRoutes":{
            "selected": [
               {
                  "destinationIP": "224.1.0.1",
                  "sourceIP": "10.1.5.100"
               } ],
            "all" : false
         }
      } """
      data = requestContext.getRequestContent()
      data = json.loads( data )
      log( 'Received data:', bv( data ) )
      isValid, allClear, routesToClear, errorCode = \
            McsHttpAgentLib.validateClearRoutes( data )

      # Clear routes
      if not isValid:
         response = { 'success': False,
                     'messages': [ errorCode ] }
         return http.client.OK, response
      if allClear:
         self.apiConfig.mcastSender.clear()
         self.apiConfig.mcastReceiver.clear()
         self.apiStatus.pairFlow.clear()
         if Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled():
            self.apiConfig.boundarySender.clear()
            self.apiConfig.boundaryReceiver.clear()
            self.apiStatus.discardIntf.clear()
      else:
         for r in routesToClear:
            source = r[ 0 ]
            group = r[ 1 ]
            mcastKey = mcastKeyType( source, group )
            pFlow = self.apiConfig.mcastSender[ mcastKey ].pairFlow if \
                  mcastKey in self.apiConfig.mcastSender else mcastKeyType()
            if pFlow != mcastKeyType():
               if pFlow not in self.apiConfig.mcastSender:
                  del self.apiStatus.pairFlow[ mcastKey ]
                  del self.apiStatus.pairFlow[ pFlow ]
            del self.apiConfig.mcastSender[ mcastKey ]
            del self.apiConfig.mcastReceiver[ mcastKey ]
            if Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled():
               del self.apiConfig.boundarySender[ mcastKey ]
               del self.apiConfig.boundaryReceiver[ mcastKey ]
               del self.apiStatus.discardIntf[ mcastKey ]

      response = { 'success': True,
                  'messages': [] }
      return http.client.OK, response

class Oui( McsApiBase ):
   api = "/mcs/oui[/]"

   def __init__( self, apiConfig, agentStatus ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext ):

      response = { 'success': True,
                   'messages': [],
                   'oui': [] }

      for client in self.agentStatus.mcsDevice:
         parsedOui = client[ : 8 ]
         response[ 'oui' ].append( { parsedOui: McsHttpAgentLib.VENDOR_NAME } )

      return http.client.OK, response

class GetMcastSender( McsApiBase ):
   api = "/mcs/senders[/]"

   def __init__( self, apiConfig, cdbTopology ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.topoStatus = cdbTopology

   def getMcastSendersSysdb( self, config, mcastKey ):
      sender = {}
      sender[ 'sourceIP' ] = mcastKey.source
      sender[ 'destinationIP' ] = mcastKey.group
      mcSender = config.mcastSender[ mcastKey ]
      configBwValue = mcSender.bw.value if mcSender.bw else 0
      if ( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
          mcastKey in self.apiConfig.boundarySender ):
         sender[ 'boundaryInterface' ] = self.apiConfig.boundarySender[ mcastKey ]
      configInIntfID = f"{mcSender.senderId.device}-{mcSender.senderId.intfId}"
      sender[ 'bandwidth' ] = configBwValue
      sender[ 'bwType' ] = 'k'
      sender[ 'inIntfID' ] = configInIntfID
      iptype = Tac.Type( 'Arnet::IpAddr' )
      pairSource = mcSender.pairFlow.source
      pairGroup = mcSender.pairFlow.group
      if ( pairSource != iptype.ipAddrZero and
            pairGroup != iptype.ipAddrMulticastBase ):
         sender[ 'pairSourceIP' ] = pairSource
         sender[ 'pairDestinationIP' ] = pairGroup
      sender[ 'label' ] = mcSender.label
      sender[ 'dscp' ] = mcSender.dscp
      sender[ 'tc' ] = mcSender.tc
      sender[ 'applyPolicy' ] = mcSender.applyPolicy
      return sender

   def getSendersFromSysdb( self, filters ):
      data = []
      senders = {}
      messages = []

      # Validate the filters
      if not isinstance( filters, list ):
         error( f"Invalid request with {filters}" )
         messages.append( '187' )
         return senders, messages

      for filt in filters:
         if ec := McsHttpAgentLib.verifyKeysInFilters( filt ):
            messages.append( ec )
            continue

         srcFilter = filt.get( 'sourceIP' )
         if srcFilter:
            _, ec = McsHttpAgentLib.validateSourceIP( srcFilter )
            if ec:
               if ec not in messages:
                  messages.append( ec )
               return senders, messages

         grpFilter = filt.get( 'destinationIP' )
         if grpFilter:
            _, ec = McsHttpAgentLib.validateMcastGroup( grpFilter )
            if ec:
               if ec not in messages:
                  messages.append( ec )
               return senders, messages

         deviceFilter = None
         deviceId = filt.get( 'deviceId' )
         if deviceId:
            deviceFilter = McsHttpAgentLib.get_colon_mac( deviceId )
            if not deviceFilter:
               ec = '137'
               if ec not in messages:
                  messages.append( ec )
               return senders, messages

         log( f"{srcFilter=}, {grpFilter=}, {deviceFilter=}" )

         for mcastKey in self.apiConfig.mcastSender:
            if srcFilter and mcastKey.source != srcFilter:
               continue
            if grpFilter and mcastKey.group != grpFilter:
               continue
            devId = self.apiConfig.mcastSender[ mcastKey ].senderId.device
            if ( deviceFilter and
                not McsHttpAgentLib.isSameMac( devId, deviceFilter ) ):
               continue

            result = self.getMcastSendersSysdb( self.apiConfig, mcastKey )
            data.append( result )
            senders[ 'transactionID' ] = self.apiConfig.mcastSender[ mcastKey ].\
                  transactionId
            senders[ 'trackingID' ] = self.apiConfig.mcastSender[ mcastKey ].\
                  trackingId
      senders[ 'data' ] = data
      return senders, messages

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getMcastSenders( self, requestContext ):
      defaultFilters = """[
                  {
                     "sourceIP": null,
                     "destinationIP": null,
                     "deviceId": null
                  }]
               """
      senders = self.get_resp_template()
      senders[ 'senders' ] = []
      filterData = requestContext.getRequestContent()
      filters = None
      try:
         if filterData:
            filters = json.loads( filterData )
         if not filters:
            filters = json.loads( defaultFilters )
      except json.JSONDecodeError:
         error( f"Invalid request with {filterData}" )
         senders[ 'messages' ].append( '187' )
         senders[ 'success' ] = False
         return http.client.OK, senders

      info( 'Received data:', bv( filters ) )

      info( "Getting Multicast Senders" )
      result, errMessages = self.getSendersFromSysdb( filters )
      if not errMessages:
         senders[ 'success' ] = True
         for sender in result[ 'data' ]:

            mcKey = mcastKeyType( sender[ 'sourceIP' ], sender[ 'destinationIP' ] )
            trackingId = self.apiConfig.mcastSender[ mcKey ].trackingId
            if sender[ 'inIntfID' ].startswith( MCS_BOUNDARY_INTF ):
               senderPort = ''
               hostName = ''
               senderId = sender[ 'inIntfID' ]
            else:
               senderId, senderPort = sender[ 'inIntfID' ].split( '-' )
               hostName = McsHttpAgentLib.getDevName( senderId, self.topoStatus )
               senderId = McsHttpAgentLib.get_dotted_mac( senderId )

            data = {
               "bandwidth": sender[ 'bandwidth' ],
               "destinationIP": sender[ 'destinationIP' ],
               "messages": [],
               "senderId": senderId,
               "network-device-name": hostName,
               "senderPort": senderPort,
               "sourceIP": sender[ 'sourceIP' ],
               "success": True,
               "trackingID": trackingId,
               "label": sender[ 'label' ]
            }
            pairSource = sender.get( 'pairSourceIP' )
            pairDest = sender.get( 'pairDestinationIP' )
            if pairSource and pairDest:
               data.update( { 'pairSourceIP': pairSource,
                              'pairDestinationIP': pairDest } )
            if bIntf := sender.get( 'boundaryInterface' ):
               data.update( { 'boundaryInterface': bIntf } )

            # Update dscp and tc optionally
            dscp = sender.get( 'dscp' )
            tc = sender.get( 'tc' )
            if dscp and tc:
               data.update( { 'dscp': dscp,
                              'tc': tc } )
            applyPolicy = sender.get( 'applyPolicy' )
            if applyPolicy is not None:
               data.update( { 'applyPolicy': applyPolicy } )
            senders[ 'senders' ].append( data )
         return http.client.OK, senders
      else:
         senders[ 'success' ] = False
         senders[ 'messages' ] = errMessages
         return http.client.OK, senders

class PostMcastReceiver( McsApiBase ):
   api = "/mcs/multicast/receivers[/]"

   def __init__( self, apiConfig, apiNotify ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.apiNotify = apiNotify
      self.MCAST_FLOW_KEYS = { 'flow-action',
                               'trackingID',
                               'data',
                               'transactionID' }
      self.RECEIVER_FIELDS = { 'sourceIP',
                               'destinationIP',
                               'outIntfID' }
      self.receiverActionMap = { 'addReceivers': 'set',
                            'delReceivers': 'del' }

   def sendNotify( self, postData, errorCode=None, receiver=None, isFailure=True ):
      """Send failure or delete notification messages for a receiver requests"""

      action = postData.get( 'flow-action' )
      action = self.receiverActionMap.get( action ) if action else 'U/A'
      messages = ( errorCode if isinstance( errorCode, list )
                   else [ errorCode ] if errorCode else [] )
      data = ( receiver if isinstance( receiver, list )
                           else [ receiver ] if receiver else [] )
      if isFailure:
         messages.extend( [ "114" ] )

      # Add the deleted senders, valid and failed, to the message queue
      notifyData = {
         'action': action,
         'trackingID': postData.get( 'trackingID', 0 ),
         'messages': messages,
         'data': data
      }
      notifyType = Tac.Type( "Mcs::ApiNotifyMsgType" )
      notify = Tac.Value( "Mcs::ApiNotifyMsg", notifyType.delReceiver,
                           json.dumps( notifyData ) )
      self.apiNotify.message = notify

   def addReceiversPopulateSysdb( self, data ):
      intfType = Tac.Type( "Arnet::IntfId" )
      for receiver in data[ 'data' ]:
         source = receiver[ 'sourceIP' ]
         group = receiver[ 'destinationIP' ]
         mcastKey = mcastKeyType( source, group )
         if mcastKey in self.apiConfig.mcastReceiver:
            recv = self.apiConfig.mcastReceiver.get( mcastKey )
         else:
            recv = self.apiConfig.mcastReceiver.newMember( mcastKey )
         for outID in receiver[ 'outIntfID' ]:
            if ( Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled() and
                  outID.startswith( MCS_BOUNDARY_INTF ) ):
               if mcastKey not in self.apiConfig.boundaryReceiver:
                  bRcv = self.apiConfig.boundaryReceiver.newMember( mcastKey )
               else:
                  bRcv = self.apiConfig.boundaryReceiver.get( mcastKey )
               if outID not in bRcv.receiver:
                  bIntf = bRcv.receiver.newMember( outID )
               else:
                  bIntf = bRcv.receiver.get( outID )
               bIntf.transactionId = data[ 'transactionID' ]
               bIntf.trackingId = data[ 'trackingID' ]
            else:
               device, intfId = outID.split( '-' )
               arNetDevice = McsHttpAgentLib.get_colon_mac( device )
               if arNetDevice not in recv.receivers:
                  recv.newReceivers( arNetDevice )
               intfKey = intfType( intfId )
               if intfKey not in recv.receivers[ arNetDevice ].recvIntfs:
                  tacIntfId = recv.receivers[ arNetDevice ].recvIntfs.\
                        newMember( intfKey )
               else:
                  tacIntfId = recv.receivers[ arNetDevice ].recvIntfs[ intfKey ]
               tacIntfId.trackingId = data[ 'trackingID' ]
               tacIntfId.transactionId = data[ 'transactionID' ]
         # The visitCount MUST be the last attribute to be updated since this
         # dictates when this mcastReceiver will be synced over to the standby.
         recv.visitCount += 1

   def validateReceivers( self, postData ):
      receiverUpdate = self.get_mcast_resp_template()
      receiverUpdate[ 'status' ] = self.get_mcast_recv_template()
      ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.MCAST_FLOW_KEYS,
                                                         postData )
      # Validate Receiver Post Fields
      if ec:
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )
         if 'data' in missingKeys:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] = []
         else:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += \
                                                                  postData[ 'data' ]
         if 'flow-action' in missingKeys:
            receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ] = 'U/A'

         # Send notification
         self.sendNotify( postData, errorCode=ec,
                          receiver=postData.get( 'data', [] ) )
         return receiverUpdate

      flowAction = postData[ 'flow-action' ]
      if flowAction in self.receiverActionMap:

         receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ] = \
                                    self.receiverActionMap[ flowAction ]
      else:
         receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ] = 'U/A'
         ec = '113'
         error( errorCodes[ ec ].replace( '<postedAction>',
                                             flowAction ) )

         receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += \
                                                                  postData[ 'data' ]
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData,
                          errorCode=ec,
                          receiver=postData.get( 'data', [] ) )
         return receiverUpdate

      try:
         assert isinstance( postData[ 'data' ], list )
      except AssertionError:
         ec = '121'
         error( errorCodes[ '121' ] )
         receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += \
                                                                  postData[ 'data' ]
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )

         # Send notification
         self.sendNotify( postData,
                          errorCode=ec,
                          receiver=postData.get( 'data', [] ) )
         return receiverUpdate

      valid, ec = McsHttpAgentLib.validateTransactionId( postData[ 'transactionID'
         ] )
      if not valid:
         receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += postData[
               'data' ]
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )
         receiverUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
         receiverUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
         self.sendNotify( postData, errorCode=ec, receiver=postData.get(
            'data', [] ) )
         return receiverUpdate

      valid, ec = McsHttpAgentLib.validateTrackingId( postData[ 'trackingID' ] )
      if not valid:
         receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ] += postData[
               'data' ]
         receiverUpdate[ 'status' ][ 'messages' ].append( ec )
         receiverUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
         receiverUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
         self.sendNotify( postData, errorCode=ec, receiver=postData.get(
            'data', [] ) )
         return receiverUpdate

      for receiver in postData[ 'data' ]:
         if 'messages' not in receiver:
            receiver[ 'messages' ] = []

         ec, missingKeys = McsHttpAgentLib.findMissingKeys( self.RECEIVER_FIELDS,
                                                            receiver )
         if ec:
            receiver[ 'messages' ].append( ec )
            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].append(
                                                                           receiver )
            # Send notification
            self.sendNotify( postData, receiver=receiver )
            continue
         sourceIP = receiver[ 'sourceIP' ]
         destinationIP = receiver[ 'destinationIP' ]
         valid, m = McsHttpAgentLib.validateMcastGroup( destinationIP )
         if not valid:
            if m:
               receiver[ 'messages' ].append( m )

            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].append(
                                                                           receiver )
            # Send notification
            self.sendNotify( postData, receiver=receiver )
            continue

         valid, m = McsHttpAgentLib.validateSourceIP( sourceIP )
         if not valid:
            if m:
               receiver[ 'messages' ].append( m )

            receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].append(
                                                                           receiver )
            # Send notification
            self.sendNotify( postData, receiver=receiver )
            continue
         currentKey = mcastKeyType( sourceIP, destinationIP )
         ec = ''

         def getRespReceiver( receiver, outIntfID, ec=None ):
            respReceiver = {}
            respReceiver[ 'destinationIP' ] = receiver[ 'destinationIP' ]
            respReceiver[ 'sourceIP' ] = receiver[ 'sourceIP' ]
            respReceiver[ 'outIntfID' ] = [ outIntfID ]
            if ec:
               respReceiver[ 'messages' ] = [ ec ]
            return respReceiver

         for outIntfID in receiver[ 'outIntfID' ]:
            valid, ec, outDevice, outIntf = McsHttpAgentLib.validateDeviceID(
                                                                     self.apiConfig,
                                                                        outIntfID )
            if not valid:
               failedReceiver = getRespReceiver( receiver, outIntfID, ec )
               receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].\
                                                         append( failedReceiver )
               # Send notification
               self.sendNotify( postData, receiver=failedReceiver )
               continue

            if currentKey in self.apiConfig.mcastSender:
               if outIntfID.startswith( MCS_BOUNDARY_INTF ):
                  # Receiver is in boundary interface
                  if currentKey in self.apiConfig.boundarySender:
                     # Sender is also in boundary interface
                     bIntfSender = self.apiConfig.boundarySender[ currentKey ]
                     if outIntfID == bIntfSender:
                        ec = '159'
                        receiverId = outIntfID
                        senderId = bIntfSender
                     # Sender in non-boundary ensure sender and receiver will not be
                     # in same intf because while adding non boundary we have check
                     # to verify non boundary flow shouldnot take boundary intf
                     # physical intf
               else:
                  # Receiver is non-boundary
                  value = self.apiConfig.mcastSender[ currentKey ]
                  senderDev = value.senderId.device
                  senderIntf = value.senderId.intfId
                  if senderDev == outDevice and senderIntf == outIntf:
                     ec = "159"
                     receiverId = outDevice + '-' + outIntf
                     senderId = senderDev + '-' + senderIntf
               if ec:
                  sg = sourceIP + ':' + destinationIP
                  wordMap = { '<sourceIp:destinationIp> ': sg,
                              'sender=<deviceId:port>': 'sender=' + senderId,
                              'receiver=<deviceId:port>': ' receiver=' + receiverId
                              }
                  error( McsHttpAgentLib.replaceAll( errorCodes[ ec ], wordMap ) )

                  failedReceiver = getRespReceiver( receiver, outIntfID, ec )
                  receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ].\
                                                            append( failedReceiver )
                  # Send Notification
                  self.sendNotify( postData, receiver=failedReceiver )
                  continue

            receiverUpdate[ 'status' ][ 'receivers' ][ 'validReceivers' ].append(
                                    getRespReceiver( receiver, outIntfID ) )

      receiverUpdate[ 'status' ][ 'trackingID' ] = postData[ 'trackingID' ]
      receiverUpdate[ 'status' ][ 'transactionID' ] = postData[ 'transactionID' ]
      receiverUpdate[ 'success' ] = True
      return receiverUpdate

   @UrlMap.urlHandler( ( 'POST', ), api )
   def postMcastReceivers( self, requestContext ):
      """
      Sample post:
      {
         "flow-action": "addReceivers",
         "transactionID": "test", "trackingID": 76,
         "data": [
               {
                  "destinationIP": "224.2.0.11",
                  "sourceIP": "10.1.1.100",
                  "outIntfID": [
                  "001c.738d.1569-Ethernet8"]
               }]
      }
      """
      info( "Configuring Multicast Receiver" )
      # XXX TODO: Log trackingID to track posts
      data = requestContext.getRequestContent()
      data = json.loads( data )
      log( 'Received data:', bv( data ) )

      return self.handleReceivers( data )

   def handleReceivers( self, data ):
      log( 'handleReceiver started' )
      receiverUpdate = self.validateReceivers( data )
      validReceivers = receiverUpdate[ 'status' ][ 'receivers' ][ 'validReceivers' ]
      # pylint: disable-msg=R1702
      if validReceivers:
         action = receiverUpdate[ 'status' ][ 'receivers' ][ 'action' ]
         validatedData = {
                     'data': validReceivers,
                     'flow-action': data[ 'flow-action' ],
                     'trackingID': data[ 'trackingID' ],
                     'transactionID': data[ 'transactionID' ]
                  }
         # Also check if this is an existing receiver- In this case, we can avoid
         # update and return null op.
         if action == 'set':
            if validatedData[ 'data' ]:
               self.addReceiversPopulateSysdb( validatedData )
            else:
               receiverUpdate[ 'success' ] = False
               return http.client.OK, receiverUpdate
         else:
            for receiver in validReceivers:
               if 'messages' not in receiver:
                  receiver[ 'messages' ] = []
               mcastKey = mcastKeyType( receiver.get( 'sourceIP' ),
                                        receiver.get( 'destinationIP' ) )
               if mcastKey not in self.apiConfig.mcastReceiver:
                  warn( "Deleteing receiver not configured" )
                  continue
               receivers = self.apiConfig.mcastReceiver[ mcastKey ].receivers
               if mcastKey in self.apiConfig.boundaryReceiver:
                  bReceivers = self.apiConfig.boundaryReceiver[ mcastKey ].receiver
               recCopy = copy.copy( receiver )
               recCopy[ 'outIntfID' ] = []
               for outIntfID in receiver[ 'outIntfID' ]:
                  if ( outIntfID.startswith( MCS_BOUNDARY_INTF ) and
                       outIntfID in bReceivers ):
                     del bReceivers[ outIntfID ]
                     for devIntf in (
                           self.apiConfig.boundaryInterface[ outIntfID ].interface ):
                        device = devIntf.device
                        intfId = devIntf.intfId
                        if ( device in receivers and
                             intfId in receivers[ device ].recvIntfs ):
                           del receivers[ device ].recvIntfs[ intfId ]
                           if not receivers[ device ].recvIntfs:
                              del receivers[ device ]
                           recCopy[ 'boundaryInterface' ] = outIntfID
                           recCopy[ 'outIntfID' ].extend( [ f"{device}-{intfId}" ] )
                           break
                     self.sendNotify( data, receiver=recCopy, isFailure=False )
                  else:
                     device, _, intfId = outIntfID.partition( '-' )
                     if ( device in receivers and
                         intfId in receivers[ device ].recvIntfs ):
                        del receivers[ device ].recvIntfs[ intfId ]
                        self.sendNotify( data, receiver=receiver, isFailure=False )
                        if not receivers[ device ].recvIntfs:
                           del receivers[ device ]

               if mcastKey in self.apiConfig.boundaryReceiver and not bReceivers:
                  del self.apiConfig.boundaryReceiver[ mcastKey ]
               if not receivers and mcastKey not in self.apiConfig.boundaryReceiver:
                  # Second check currently is required because currently for boundary
                  # receiver there is no selected intf hence can lead to removal
                  # of mcastKey from mcastReceiver even though boundary intf exist
                  # to avoid that introduced this check
                  del self.apiConfig.mcastReceiver[ mcastKey ]

               # The visitCount MUST be the last attribute to be updated since this
               # dictates when this mcastReceiver will be synced over to the standby.
               if mcastKey in self.apiConfig.mcastReceiver:
                  self.apiConfig.mcastReceiver[ mcastKey ].visitCount += 1

      if not receiverUpdate[ 'status' ][ 'receivers' ][ 'failedReceivers' ]:
         receiverUpdate[ 'status' ][ 'success' ] = True
      else:
         receiverUpdate[ 'status' ][ 'messages' ].append( '114' )

      if not receiverUpdate[ 'status' ][ 'receivers' ][ 'validReceivers' ]:
         receiverUpdate[ 'status' ][ 'messages' ].append( '117' )

      log( 'handleReceiver completed' )
      return http.client.OK, receiverUpdate

class PostMcastProgramFlows( McsApiBase ):
   api = "/mcs/multicast/programFlows[/]"

   def __init__( self, amcastSender, amcastReceiver ):
      McsApiBase.__init__( self )
      self.programFlow = {
         'addSenders': amcastSender.handleSenders,
         'delSenders': amcastSender.handleSenders,
         'addReceivers': amcastReceiver.handleReceivers,
         'delReceivers': amcastReceiver.handleReceivers,
         'modBw': amcastSender.handleSenders,
      }

   def handleInvalidFlowAction( self, data ):
      resp = self.get_mcast_resp_template()
      errCode = '113'
      resp[ 'messages' ].append( errCode )
      error( errorCodes[ errCode ].replace( '<postedAction>',
                                             data.get( 'flow-action' ) ) )
      resp[ 'status' ].update( data )
      return http.client.OK, resp

   @UrlMap.urlHandler( ( 'POST', ), api )
   def postMcastProgramFlows( self, requestContext ):
      """ Handler for batch request of senders and receivers

      This handler supports addSender, addReceiver, delSender, delReceiver and modBw
      in flow-action
      """
      info( "Configuring Multicast Program Flow" )
      data = requestContext.getRequestContent()
      reqData = json.loads( data )
      log( 'Received data:', bv( reqData ) )
      resp = []
      for req in reqData:
         _, rd = self.programFlow.get( req.get( 'flow-action' ),
                                       self.handleInvalidFlowAction )( req )
         resp.extend( [ rd ] )

      return http.client.OK, resp

class GetMcastReceiver( McsApiBase ):
   receiversApi = "/mcs/receivers[/]"
   impactedApi = "/mcs/receivers/impacted[/]"
   maintenanceApi = "/mcs/receivers/maintenance/{maintDev}[/]"
   api = [ receiversApi, impactedApi, maintenanceApi ]

   def __init__( self, apiConfig, agentStatus, cdbMounts ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus
      self.cdbMounts = cdbMounts

   @UrlMap.urlHandler( ( 'GET', ), impactedApi )
   def getMcastReceiversImpacted( self, requestContext ):
      receivers = self.handleGetReceivers( requestContext, impacted=True )
      return http.client.OK, receivers

   @UrlMap.urlHandler( ( 'GET', ), receiversApi )
   def getMcastReceivers( self, requestContext ):
      receivers = self.handleGetReceivers( requestContext )
      return http.client.OK, receivers

   @UrlMap.urlHandler( ( 'GET', ), maintenanceApi )
   def getMcastReceiversMaintenance( self, requestContext, maintDev ):
      assert Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled()
      receivers = self.handleGetReceivers( requestContext, maintDev=maintDev )
      return http.client.OK, receivers

   def handleGetReceivers( self, requestContext, **kwargs ):
      defaultFilters = """[
                  {
                     "sourceIP": null,
                     "destinationIP": null,
                     "deviceId": null
                  }]
               """
      receivers = self.get_resp_template()
      receivers[ 'receivers' ] = []
      receivers[ 'messages' ] = []
      data = requestContext.getRequestContent()
      filters = None
      try:
         if data:
            filters = json.loads( data )
         if not filters:
            filters = json.loads( defaultFilters )
      except json.JSONDecodeError:
         error( f"Invalid request with {data}" )
         receivers[ 'messages' ].append( '187' )
         receivers[ 'success' ] = False
         return receivers

      for f in filters:
         f.update( { 'impacted': kwargs.get( 'impacted' ) } )
         if Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled():
            f.update( { 'maintenance': kwargs.get( 'maintDev' ) } )

      info( 'Received data:', bv( filters ) )
      storedReceivers, errMessages = McsHttpAgentLib.getStoredReceivers(
         self.apiConfig, self.agentStatus, filters )
      receivers[ 'success' ] = not errMessages
      receivers[ 'messages' ] = errMessages

      # pylint: disable=too-many-nested-blocks
      for flow in storedReceivers.values():
         for srcDetail in flow.values():
            for grpDetail in srcDetail.values():
               for tDetail in grpDetail.values():
                  for devDetail in tDetail.values():
                     devDetail[ 'deviceId' ] = McsHttpAgentLib.get_dotted_mac(
                                                         devDetail[ 'deviceId' ] )
                     hostName = McsHttpAgentLib.getDevName(
                        devDetail[ 'deviceId' ], self.cdbMounts.cdbTopology )
                     devDetail[ 'network-device-name' ] = hostName
                     receivers[ 'receivers' ].append( devDetail )
      return receivers

class GetFlowsActive( McsApiBase ):
   api = "/mcs/flows-active[/]"

   def __init__( self, mcsControllerMounts ):
      McsApiBase.__init__( self )
      self.mcsControllerMounts = mcsControllerMounts

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getFlowsActive( self, requestContext ):
      info( "Getting active flows" )
      activeFlows = self.handleGetFlowsActive( requestContext )
      return http.client.OK, activeFlows

   def handleGetFlowsActive( self, requestContext, **kwargs ):
      defaultFilters = """[
                  {
                     "sourceIP": null,
                     "destinationIP": null,
                     "deviceId": null
                  }]
               """
      activeFlows = { 'active-flows': [], 'messages': [], 'success': True }
      data = requestContext.getRequestContent()
      filters = None
      try:
         if data:
            filters = json.loads( data )
         if not filters:
            filters = json.loads( defaultFilters )
      except json.JSONDecodeError:
         error( f"Invalid request with {data}" )
         activeFlows[ 'messages' ].append( '187' )
         activeFlows[ 'success' ] = False
         return activeFlows

      info( 'Received data: ', bv( filters ) )
      storedActiveFlows, errMessages = McsHttpAgentLib.getActiveFlows(
         self.mcsControllerMounts, filters )
      activeFlows[ 'success' ] = not errMessages
      activeFlows[ 'messages' ] = errMessages
      if errMessages:
         activeFlows[ 'active-flows' ] = []
      else:
         activeFlows[ 'active-flows' ] = storedActiveFlows
      return activeFlows

class GetApiErrorCodes( McsApiBase ):
   api = "/mcs/apiErrorCodes[/]"

   def __init__( self ):
      McsApiBase.__init__( self )

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getApiErrorCodes( self, requestContext ):
      info( "Getting MCS Api Error Codes " )
      return http.client.OK, errorCodes

linkStatusStringsFormat2 = { 'linkUp': 'connected',
                                'linkDown': 'notconnect',
                                'linkUnknown': 'unknown' }

def getDevices( agentStatus, mcsMounts, apiConfig, deviceId=None,
                network=False, endpoint=False ):

   netDevs = set()
   endEdges = set()
   netDevInfo = []
   endDevInfo = []
   topoStatus = mcsMounts.cdbTopology
   topologyEdges = mcsMounts.cdbTopology.edge

   for value in topologyEdges.values():
      fromHostMac = value.fromPort.host().name
      netFrom = fromHostMac in agentStatus.mcsDevice.members()
      for toPort in value.toPort:
         toHostMac = toPort.host().name
         netTo = toHostMac in agentStatus.mcsDevice.members()
         if netTo and netFrom:
            if toHostMac in agentStatus.mcsDevice.members():
               netDevs.add( toHostMac.replace( ':', '-' ) )
            if fromHostMac in agentStatus.mcsDevice.members():
               netDevs.add( fromHostMac.replace( ':', '-' ) )
         else:
            if not netTo:
               endDevId = toHostMac
               netDevId = fromHostMac
               endPort = toPort.name
            if not netFrom:
               endDevId = fromHostMac
               netDevId = toHostMac
               endPort = value.fromPort.name
            if netDevId in agentStatus.mcsDevice.members():
               endEdge = ( netDevId, endDevId, endPort )
               endEdges.add( endEdge )
               netDevs.add( netDevId.replace( ':', '-' ) )

   # pylint: disable-msg=R1702
   if network:
      for dev in netDevs:
         parsedMac = McsHttpAgentLib.get_dotted_mac( dev )

         if deviceId and not McsHttpAgentLib.isSameMac( deviceId, parsedMac ):
            continue
         netDev = {
                  'id': parsedMac,
                  'interfaces': [],
                  'mac-address': parsedMac,
                  'messages': [],
                  'mgmt-ip': '',
                  'name': '',
                  'product-description': '',
                  'success': True,
                  'vendor': 'Arista Networks' }


         # From Lldp local-info
         client = mcsMounts.switchStatusDir.get( dev )
         topologyHost = topoStatus.host.get(
               McsHttpAgentLib.get_colon_mac( dev ) )
         mgmtIpAddress = ''
         if topologyHost and topologyHost.mgmtIp:
            mgmtIpAddress = next( iter( topologyHost.mgmtIp ) )
            netDev[ 'mgmt-ip' ] = str( mgmtIpAddress )
         if client:
            netDev[ 'name' ] = client.sysName
            netDev[ 'product-description' ] = client.sysDesc
         apiIntfs = []

         if client:
            for intf, inSt in client.intfSpeed.items():
               intfInfo = { 'MTU': '',
                        'admin-status': '',
                        'built-in-mac-address': '',
                        'description': '',
                        'down-reason': '',
                        'if-name': '',
                        'ip-address': '',
                        'mac-address': '',
                        'mode': '',
                        'oper-status': '',
                        'operating-speed': '' }
               intfInfo[ 'operating-speed' ] = McsHttpAgentLib.get_bandwidth_num(
                                                                        inSt )
               intfInfo[ 'built-in-mac-address' ] = \
                     McsHttpAgentLib.get_dotted_mac(
                           client.intfBuiltinMacAddr.get( intf ) )
               intfInfo[ 'mac-address' ] = \
                     McsHttpAgentLib.get_dotted_mac(
                           client.intfMacAddr.get( intf ) )
               intfInfo[ 'if-name' ] = intf
               intfInfo[ 'description' ] = client.intfDesc.get( intf )
               if client.intfLinkStatus.get( intf ):
                  intfInfo[ 'admin-status' ] = linkStatusStringsFormat2[
                        client.intfLinkStatus.get( intf ) ]
               else:
                  intfInfo[ 'admin-status' ] = 'N/A'
               if client.intfOperStatus.get( intf ):
                  intfInfo[ 'oper-status' ] = intfOperStatusToEnum(
                                               client.intfOperStatus.get( intf ) )

               intfInfo[ 'down-reason' ] = ''
               # ip address from L3/Intf/status
               ipAddr = ''
               if intf in client.intfIpAddr:
                  ipAddr = str( client.intfIpAddr.get( intf ) )
               intfInfo[ 'ip-address' ] = ipAddr
               intfInfo[ 'MTU' ] = client.intfMtu.get( intf )
               if client.intfMode.get( intf ):
                  intfInfo[ 'mode' ] = \
                        _forwardingModelMap[ client.intfMode.get( intf ) ]
               apiIntfs.append( intfInfo.copy() )
         netDev[ 'interfaces' ] = apiIntfs
         netDevInfo.append( netDev.copy() )

   if endpoint:
      for netDevId, endDevId, endPort in endEdges:
         if deviceId and not McsHttpAgentLib.isSameMac( deviceId, netDevId ):
            continue
         hostName = McsHttpAgentLib.getDevName( netDevId, topoStatus )
         endDev = {
                  "chassis-id": endDevId,
                  "messages": [],
                  "network-device-id": McsHttpAgentLib.get_dotted_mac( netDevId ),
                  "network-device-name": hostName,
                  "port-id": endPort,
                  "success": True
               }

         endDevInfo.append( endDev )
   return endDevInfo, netDevInfo

def getEdges( agentStatus, mcsMounts, apiConfig, deviceId=None,
                                                network=False, endpoint=False ):
   edges = []
   uniqueEndEdges = set()
   topoStatus = mcsMounts.cdbTopology
   topologyEdges = mcsMounts.cdbTopology.edge

   # pylint: disable-msg=R1702
   for value in topologyEdges.values():
      fromHostMac = value.fromPort.host().name
      netFrom = fromHostMac in agentStatus.mcsDevice.members()

      if deviceId and not McsHttpAgentLib.isSameMac( deviceId, fromHostMac ):
         continue

      for toPort in value.toPort:
         toHostMac = toPort.host().name
         netTo = toHostMac in agentStatus.mcsDevice.members()
         if netTo and netFrom:
            if network:
               if fromHostMac in agentStatus.mcsDevice.members():
                  cdbDev = fromHostMac.replace( ':', '-' )
                  edgeItem = {}
                  edgeItem[ "network-device-id" ] = McsHttpAgentLib.get_dotted_mac(
                                                                  fromHostMac )
                  hostName = McsHttpAgentLib.getDevName( fromHostMac, topoStatus )
                  peerHostName = McsHttpAgentLib.getDevName( toHostMac, topoStatus )
                  edgeItem[ "network-device-name" ] = hostName
                  edgeItem[ "network-interface" ] = value.fromPort.name
                  edgeItem[ "peer-network-interface" ] = toPort.name
                  edgeItem[ "peer-network-device-id" ] = \
                              McsHttpAgentLib.get_dotted_mac( toHostMac )
                  edgeItem[ "peer-network-device-name" ] = peerHostName
                  edgeItem[ "speed" ] = ""
                  mcsClientStatus = mcsMounts.switchStatusDir.get( cdbDev )
                  if mcsClientStatus:
                     edgeItem[ "speed" ] = mcsClientStatus.intfSpeed.get(
                           value.fromPort.name, "" )
                  edgeItem[ "success" ] = True
                  edgeItem[ "messages" ] = []
                  edgeItem[ "endpoint-link" ] = False
                  edges.append( edgeItem )
         else:
            if endpoint:
               netDev = toHostMac if netTo else fromHostMac
               if netDev in agentStatus.mcsDevice.members():
                  netPort = toPort.name if netTo else value.fromPort.name
                  endDev = toHostMac if not netTo else fromHostMac
                  endPort = toPort.name if not netTo else value.fromPort.name
                  # Use unique edges to not append same network-endpoint edge again
                  uniqueEndEdges.add( ( netDev, netPort, endDev, endPort ) )

   for netDev, netPort, endDev, endPort in uniqueEndEdges:
      edgeItem = {}
      edgeItem[ "speed" ] = ""
      dev = netDev.replace( ':', '-' )
      mcsClientStatus = mcsMounts.switchStatusDir.get( dev )
      if mcsClientStatus:
         edgeItem[ "speed" ] = mcsClientStatus.intfSpeed.get( netPort, "" )
      hostName = McsHttpAgentLib.getDevName( netDev, topoStatus )
      edgeItem[ "network-device-id" ] = McsHttpAgentLib.get_dotted_mac( netDev )
      edgeItem[ "network-device-name" ] = hostName
      edgeItem[ "network-interface" ] = netPort
      edgeItem[ "endpoint-id" ] = endDev
      edgeItem[ "endpoint-port-id" ] = endPort
      edgeItem[ "messages" ] = []
      edgeItem[ "endpoint-link" ] = True
      edgeItem[ "success" ] = True
      edges.append( edgeItem )

   return edges

class GetNetworkLinks( McsApiBase ):
   api = "/mcs/network-links[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkLinks( self, requestContext ):
      """
      [
         {
            "endpoint-link": false,
            "network-device-id": "2899.3abb.c1ee",
            "network-device-name": "arista.networks.ce21",
            "network-interface": "Ethernet2",
            "peer-network-device-id": "001c.7318.630e",
            "peer-network-device-name": "arista.networks.ce22",
            "peer-network-interface": "Ethernet23",
            "speed": "speed10Gbps",
            "messages": [],
            "success": true
         }
      ]
      """
      info( "Getting MCS Network Links" )
      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
                        network=True )
      return http.client.OK, edges

class GetNetworkLink( McsApiBase ):
   api = "/mcs/network-link/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkLink( self, requestContext, deviceId ):
      """
      [
         {
            "endpoint-link": false,
            "network-device-id": "2899.3abb.c1ee",
            "network-device-name": "arista.networks.ce21",
            "network-interface": "Ethernet2",
            "peer-network-device-id": "001c.7318.630e",
            "peer-network-device-name": "arista.networks.ce22",
            "peer-network-interface": "Ethernet23",
            "speed": "speed10Gbps",
            "messages": [],
            "success": true
         }
      ]
      """
      info( "Getting MCS Network Links for device", bv( deviceId ) )
      edges = []
      if not McsHttpAgentLib.get_dotted_mac( deviceId ):
         ec = '137'
         error( errorCodes[ ec ].replace( '<deviceId>', str( deviceId ) ) )
         edge = { "endpoint-link": False,
                  "network-device-id": deviceId,
                  "network-device-name": "",
                  "network-interface": "",
                  "peer-network-device-id": "",
                  "peer-network-device-name": "",
                  "peer-network-interface": "",
                  "speed": "",
                  "messages": [ ec ],
                  "success": False }
         edges.append( edge )
         return http.client.OK, edges

      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
               deviceId, network=True )
      return http.client.OK, edges

class GetNetworkDevices( McsApiBase ):
   api = "/mcs/network-devices[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkDevices( self, requestContext ):
      """
      [
         {
            "id": "74:83:ef:00:06:c9",
            "interfaces": [
                  {
                     "MTU": 1500,
                     "admin-status": "connected",
                     "built-in-mac-address": "74:83:ef:00:06:c9",
                     "description": "",
                     "down-reason": "",
                     "if-name": "Ethernet8",
                     "ip-address": "10.37.8.1",
                     "mac-address": "74:83:ef:00:06:c9",
                     "mode": "routed",
                     "oper-status": "up",
                     "operating-speed": 1000000000
                  }
            ],
            "mac-address": "74:83:ef:00:06:c9",
            "messages": [],
            "mgmt-ip": "172.20.0.37",
            "name": "cs-lf37.sjc.aristanetworks.com",
            "product-description": "Arista Networks EOS version",
            "serial-number": "SSJ17261113",
            "success": True,
            "vendor": "Arista Networks"
         }
      ]
      """
      info( "Getting MCS Network Devices" )
      _, netDevs = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              network=True )
      return http.client.OK, netDevs

class GetNetworkDevice( McsApiBase ):
   api = "/mcs/network-device/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getNetworkDevice( self, requestContext, deviceId ):
      """
      {
         "id": "28:99:3a:bb:c1:ee",
         "interfaces": [
            {
                  "MTU": 1500,
                  "admin-status": "connected",
                  "built-in-mac-address": "28:99:3a:bb:c1:ee",
                  "description": "NTNX-AHV-4",
                  "down-reason": "",
                  "if-name": "Ethernet8",
                  "ip-address": "10.23.8.1",
                  "mac-address": "28:99:3a:bb:c1:ee",
                  "mode": "routed",
                  "oper-status": "up",
                  "operating-speed": 1000000000
            }
         ],
         "mac-address": "28:99:3a:bb:c1:ee",
         "messages": [],
         "mgmt-ip": "172.20.0.23",
         "name": "cs-lf23.sjc.aristanetworks.com",
         "product-description": "Arista Networks EOS version 4.20.2.1F running on
                  an Arista Networks DCS-7160-48YC6",
         "serial-number": "SSJ17164906",
         "success": true,
         "vendor": "Arista Networks"
      }
      """

      info( "Getting MCS Network Device of id", bv( deviceId ) )
      netDev = { "mac-address": deviceId,
               "messages": [],
               "success": True }
      # Normalize deviceId to xx.xx.xx.xx format

      if not McsHttpAgentLib.get_dotted_mac( deviceId ):
         ec = '137'
         netDev[ 'messages' ].append( ec )
         netDev[ 'success' ] = False
         error( errorCodes[ ec ].replace( '<deviceId>', str( deviceId ) ) )
         return http.client.OK, netDev

      _, netDevs = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              deviceId, network=True )
      if netDevs:
         netDev.update( netDevs[ 0 ] )

      return http.client.OK, netDev

class GetEndpointDevices( McsApiBase ):
   api = "/mcs/endpoints[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )

      self.apiConfig = apiConfig
      self.mcsMounts = mcsMounts
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointDevices( self, requestContext ):
      """
      [
      {
         "chassis-id": "0050.56ba.32aa",
         "messages": [],
         "network-device-id": "2899.3abb.c1ee",
         "network-device-name": "arista.networks.ce21",
         "port-id": "00:50:56:ba:32:aa",
         "success": true
      }
      ]
      """
      info( "Getting MCS Endpoint Devices" )
      endDevs, _ = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              endpoint=True )
      return http.client.OK, endDevs

class GetEndpointDevice( McsApiBase ):
   api = "/mcs/endpoint/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointDevice( self, requestContext, deviceId ):
      """
      [
      {
         "chassis-id": "0050.56ba.32aa",
         "messages": [],
         "network-device-id": "2899.3abb.c1ee",
         "network-device-name": "arista.networks.ce21",
         "port-id": "00:50:56:ba:32:aa",
         "success": true
      }
      ]
      """
      info( "Getting MCS Endpoint Device %s", bv( deviceId ) )

      endDevs = []
      if not McsHttpAgentLib.get_dotted_mac( deviceId ):
         ec = '137'
         error( errorCodes[ ec ].replace( '<deviceId>', str( deviceId ) ) )
         endDev = { "chassis-id": "",
                     "messages": [ ec ],
                     "network-device-id": deviceId,
                     "network-device-name": "",
                     "port-id": "",
                     "success": False }
         endDevs.append( endDev )
         return http.client.OK, endDevs
      endDevs, _ = getDevices( self.agentStatus, self.mcsMounts, self.apiConfig,
                              deviceId, endpoint=True )
      return http.client.OK, endDevs

class GetEndpointLinks( McsApiBase ):
   api = "/mcs/endpoint-links[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointinks( self, requestContext ):
      """
      [
         {
            "endpoint-id": "0050.56ba.fb78",
            "endpoint-link": true,
            "endpoint-port-id": "00:50:56:ba:fb:78",
            "network-device-id": "2899.3abb.c1ee",
            "network-device-name": "arista.networks.ce21"
            "network-interface": "Ethernet8",
            "speed": "speed1Gbps",
            "success": true,
            "messages": []
         }
      ]
      """
      info( "Getting MCS Endpoint Links" )
      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
                        endpoint=True )
      return http.client.OK, edges

class GetEndpointLink( McsApiBase ):
   api = "/mcs/endpoint-link/{deviceId}[/]"

   def __init__( self, agentStatus, mcsMounts, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getEndpointink( self, requestContext, deviceId ):
      """
      [
         {
            "endpoint-id": "0050.56ba.fb78",
            "endpoint-link": true,
            "endpoint-port-id": "00:50:56:ba:fb:78",
            "network-device-id": "2899.3abb.c1ee",
            "network-device-name": "arista.networks.ce21",
            "network-interface": "Ethernet8",
            "speed": "speed1Gbps",
            "success": true,
            "messages": []
         }
      ]
      """
      info( "Getting MCS Endpoint Links" )
      edges = []
      if not McsHttpAgentLib.get_dotted_mac( deviceId ):
         ec = '137'
         error( errorCodes[ ec ].replace( '<deviceId>', str( deviceId ) ) )
         edge = {
                  "endpoint-id": "",
                  "endpoint-link": True,
                  "endpoint-port-id": "",
                  "network-device-id": deviceId,
                  "network-device-name": "",
                  "network-interface": "",
                  "speed": "",
                  "success": False,
                  "messages": [ ec ]
               }
         edges.append( edge )
         return http.client.OK, edges
      edges = getEdges( self.agentStatus, self.mcsMounts, self.apiConfig,
                                                deviceId, endpoint=True )
      return http.client.OK, edges

class GetInterfaces( McsApiBase ):
   api = "/mcs/interface-bandwidth[/]"

   def __init__( self, mcsMounts, agentStatus, apiConfig ):
      McsApiBase.__init__( self )
      self.mcsMounts = mcsMounts
      self.agentStatus = agentStatus
      self.apiConfig = apiConfig


   @UrlMap.urlHandler( ( 'GET', ), api )
   def getInterfaces( self, requestContext ):
      """
      [
         {
            "availableRxBw": 25000000,
            "availableTxBw": 25000000,
            "deviceID": "2899.3abb.c1ee",
            "network-device-name": "arista.networks.ce21",
            "interfaceId": "Ethernet35",
            "messages": [],
            "portSpeed": 25000000,
            "reservation-percentage": "1.0",
            "rxTotal": 0,
            "success": True,
            "totalMcastBw": 0,
            "txTotal": 0,
            "vlanId": 1,
            "vlanMode": "access"
         }
      ]
      """
      interfaces = []
      topoStatus = self.mcsMounts.cdbTopology
      info( "Getting MCS Interfaces" )
      _, bwProgrammed = McsHttpAgentLib.getFlowBwProgram( self.agentStatus )
      if self.mcsMounts.switchStatusDir:
         for mac, clientStatus in self.mcsMounts.switchStatusDir.entityPtr.items():
            for intf, speed in clientStatus.intfSpeed.items():
               intfDet = {}
               bw = McsHttpAgentLib.get_bandwidth_num( speed )
               linkStatus = clientStatus.intfLinkStatus.get( intf )
               swIntf = clientStatus.switchportModeAndNativeVlan.get( intf )
               intfDet[ 'vlanId' ] = 0
               intfDet[ 'vlanMode' ] = ""
               if swIntf:
                  intfDet[ 'vlanId' ] = swIntf.nativeVlan
                  intfDet[ 'vlanMode' ] = swIntf.switchportMode
               mac = McsHttpAgentLib.get_dotted_mac( mac )

               intfDet[ 'deviceID' ] = mac
               tmac = McsHttpAgentLib.get_colon_mac( mac )
               hostName = McsHttpAgentLib.getDevName( tmac, topoStatus )
               intfDet[ 'network-device-name' ] = hostName
               intfDet[ 'portSpeed' ] = bw
               intfDet[ 'interfaceId' ] = intf
               intfDet[ 'availableRxBw' ] = bw
               intfDet[ 'availableTxBw' ] = bw
               intfDet[ 'rxTotal' ] = 0
               intfDet[ 'txTotal' ] = 0
               intfDet[ 'totalMcastBw' ] = 0
               intfDet[ 'linkStatus' ] = linkStatus

               resKey = Tac.Value( "Mcs::DeviceAndIntfId",
                                       Arnet.EthAddr( mac ).stringValue, intf )
               resvPercent = 1.0
               resvPercentEntity = self.apiConfig.reservationPercentage.get( resKey )
               if resvPercentEntity:
                  resvPercent = resvPercentEntity.percentValue
               intfDet[ 'reservation-percentage' ] = f"{resvPercent:.2f}"
               if mac in bwProgrammed and intf in bwProgrammed[ mac ]:
                  intfDet[ 'rxTotal' ] = bwProgrammed[ mac ][ intf ][ 'rxTotal' ]
                  intfDet[ 'txTotal' ] = bwProgrammed[ mac ][ intf ][ 'txTotal' ]
                  intfDet[ 'totalMcastBw' ] = intfDet[ 'rxTotal' ] + \
                                             intfDet[ 'txTotal' ]
               availBw = round( float( resvPercent ), 2 ) * bw
               intfDet[ 'availableRxBw' ] = max(
                     int( availBw ) - intfDet[ 'rxTotal' ], 0 )
               intfDet[ 'availableTxBw' ] = max(
                     int( availBw ) - intfDet[ 'txTotal' ], 0 )

               intfDet[ 'success' ] = True
               intfDet[ 'messages' ] = []
               interfaces.append( intfDet )

      return http.client.OK, interfaces

class ReservationPercentage( McsApiBase ):
   api = "/mcs/reservation-percentage[/][{deviceId}]"

   def __init__( self, apiConfig, mcsMounts ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.mcsMounts = mcsMounts

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext, deviceId ):
      """
      Sample GET response
      [
         {
            "deviceId": "001c.738d.1569",
            "network-device-name": "arista.networks.ce21",
            "interface-name": "all",
            "messages": [],
            "reservation-percent": "1.0",
            "success": true
         }
         ]
      """
      resvPr = []
      topoStatus = self.mcsMounts.cdbTopology
      if not McsHttpAgentLib.get_dotted_mac( deviceId ):
         ec = '137'
         data = { 'success': False, 'messages': [ ec ] }
         error( errorCodes[ ec ].replace( '<deviceId>', str( deviceId ) ) )
         resvPr.append( data )
         return http.client.OK, resvPr
      devPresent = False
      mac = McsHttpAgentLib.get_colon_mac( deviceId )
      hostName = McsHttpAgentLib.getDevName( mac, topoStatus )
      for deviceAndIntfId, rpValue in self.apiConfig.reservationPercentage.items():
         if deviceAndIntfId.device == Arnet.EthAddr( deviceId ).stringValue:
            devPresent = True
            resvPr.append( { 'deviceId': deviceId,
                     'network-device-name': hostName,
                     'interface-name': deviceAndIntfId.intfId,
                     'success': True,
                     'messages': [],
                     'reservation-percent': str( round( rpValue.percentValue, 2 ) )
                     } )

      if not devPresent:
         resvPr.append( { 'deviceId': deviceId,
                     'network-device-name': hostName,
                     'interface-name': 'all',
                     'success': True,
                     'messages': [],
                     'reservation-percent': '1.0'
                     } )

      return http.client.OK, resvPr

   def populateRpModel( self, data ):
      model = McsApiModels.RPModel()
      model.populateModelFromJson( data )
      if model.deviceId is not None:
         model.toSysdb( self.apiConfig )
         return True, {
                  "deviceId": model.deviceId,
                  "interfaceId": model.intfName,
                  "reservation-percentage": str( model.rp ) }
      return False, {}

   def processPost( self, data, response ):
      valid, ec = McsHttpAgentLib.checkMissingKeyAndValidateRpValue( data )
      # handling missing key case
      if not valid:
         tmpData = {}
         if 'chassis-id' in data:
            tmpData[ 'deviceId' ] = data[ 'chassis-id' ]
         if 'interface-name' in data:
            tmpData[ 'interfaceId' ] = data[ 'interface-name' ]
         if 'reservation-percent' in data:
            tmpData[ 'reservation-percentage' ] = data[ 'reservation-percent' ]
         tmpData[ 'messages' ] = [ ec ]
         if not response[ 'messages' ]:
            response[ 'messages' ].append( ec )
         response[ 'response' ].append( tmpData )
         return False
      ecList, validDevices = McsHttpAgentLib.validateRpChassisId( data,
            self.mcsMounts )
      if not validDevices:
         tmpData = {}
         tmpData[ 'deviceId' ] = data[ 'chassis-id' ]
         tmpData[ 'interfaceId' ] = data[ 'interface-name' ]
         tmpData[ 'reservation-percentage' ] = data[ 'reservation-percent' ]
         tmpData[ 'messages' ] = ecList
         response[ 'response' ].append( tmpData )
         if not response[ 'messages' ]:
            response[ 'messages' ].extend( ecList )
         return False
      if ecList and not response[ 'messages' ]:
         response[ 'messages' ].append( '235' )
      for deviceId in validDevices:
         ec, intfList = McsHttpAgentLib.validateRpInterfaceId( data,
               self.mcsMounts, deviceId )
         ecList.extend( ec )
         if not intfList:
            if not response[ 'messages' ]:
               response[ 'messages' ].extend( ec )
         if intfList and ec and not response[ 'messages' ]:
            response[ 'messages' ].append( '235' )
         for intf in intfList:
            populateData = { 'chassis-id': deviceId,
                              'interface-name': intf,
                              'reservation-percent': data[ 'reservation-percent' ]
                              }
            self.populateRpModel( populateData )
      tData = {}
      # each error code need to be present once
      ecList = set( ecList )
      tData[ 'deviceId' ] = data[ 'chassis-id' ]
      tData[ 'interfaceId' ] = data[ 'interface-name' ]
      tData[ 'reservation-percentage' ] = data[ 'reservation-percent' ]
      if ecList:
         tData[ 'messages' ] = list( ecList )
      response[ 'response' ].append( tData )
      return not ecList

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext, deviceId=None ):
      """
      Sample for single POST:
      {"chassis-id":"00:1c:73:8d:15:69",
       "interface-name":"Ethernet1",
       "reservation-percent":"1.0"
      }
      Sample post for multiple POST:
      [ {"chassis-id":"00:1c:73:8d:15:69",
      "interface-name":"Ethernet1",
      "reservation-percent":"1.0"},
      { "chassis-id":"00:1c:73:8d:15:70",
      "interface-name":"Ethernet1-5,Ethernet6,Ethernet8-9,",
      "reservation-percent":"1.0"}
      ]

      Response:
      {
         "messages": [],
         "response": [
            {
               "deviceId": "001c.738d.1569",
               "interfaceId": "Ethernet1",
               "reservation-percentage": "1.0",
               "messages": []
            }
         ],
         "success": true
         }

      """
      info( "Posting Reservation percentages" )
      data = requestContext.getRequestContent()
      data = json.loads( data )
      info( 'Received data:', bv( data ) )
      response = { 'success': True,
               'messages': [],
               'response': [] }
      if not isinstance( data, list ):
         data = [ data ]
      for tData in data:
         status = self.processPost( tData, response )
         if not status:
            response[ 'success' ] = status
      return http.client.OK, response

class GetFailedFlows( McsApiBase ):
   api = "/mcs/failed-flows[/]"

   def __init__( self, agentStatus, cdbTopology ):
      McsApiBase.__init__( self )
      self.agentStatus = agentStatus
      self.topoStatus = cdbTopology

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getFailedFlows( self, requestContext ):
      """
      Sample request:
      [
         {
            "sourceIP": "10.1.1.100",
            "destinationIP": "224.1.0.1",
            "deviceId": "0000.0000.0001"
      } ]
      Sample response:
      [
         {
            "sourceIP": "10.1.1.100",
            "destinationIP": "224.1.0.1",
            "status": [ {
               "deviceId": "0000.0000.0001",
               "deviceName": "arista.networks.ce21",
               "interface": "Ethernet1",
               "errorCode": "198",
               "failReason": "No path diversity"
            } ],
            "messages": [],
            "success": true
      } ]"""
      def getFailedFlowTemplate():
         failedFlow = {
            'sourceIP': '',
            'destinationIP': '',
            'status': [],
            'messages': [],
            'success': True }
         return failedFlow

      def appendFailedFlow( sg, flow, response, d=None ):
         failedFlow = getFailedFlowTemplate()
         failedFlow[ 'sourceIP' ] = sg.source
         failedFlow[ 'destinationIP' ] = sg.group
         fpsType = Tac.Type( 'Mcs::FlowProgramStatus' )
         found = False
         if flow.status:
            for devIntf, fps in flow.status.items():
               status = {}
               devId = McsHttpAgentLib.get_dotted_mac( devIntf.device )
               if d and not McsHttpAgentLib.isSameMac( d, devId ):
                  continue
               status[ 'deviceId' ] = devId
               status[ 'deviceName' ] = McsHttpAgentLib.getDevName(
                  devIntf.device, self.topoStatus )
               status[ 'interface' ] = devIntf.intfId
               status[ 'errorCode' ] = str( Tac.enumValue( fpsType, fps ) )
               status[ 'failReason' ] = McsHttpAgentLib.getFpsMsg( fps )
               failedFlow[ 'status' ].append( status )
               found = True
         if found:
            response.append( failedFlow )
         return found

      def appendErrorFailedFlow( ec, failedFlow, response ):
         failedFlow[ 'messages' ].append( ec )
         failedFlow[ 'success' ] = False
         response.append( failedFlow )

      info( 'Getting MCS Failed Flows' )
      data = requestContext.getRequestContent()
      if data:
         data = json.loads( data )
      else:
         data = {}
      info( 'Received data:', bv( data ) )

      response = []
      if not data:
         for sg, flow in self.agentStatus.flowFailed.items():
            appendFailedFlow( sg, flow, response )
         return http.client.OK, response

      keys = { 'sourceIP', 'destinationIP', 'deviceId' }
      for query in data:
         found = False
         failedFlow = getFailedFlowTemplate()
         ec, missingKeys = McsHttpAgentLib.findMissingKeys( keys, query )
         if len( missingKeys ) == len( keys ):
            appendErrorFailedFlow( ec, failedFlow, response )
            continue
         s = query.get( 'sourceIP', '' )
         g = query.get( 'destinationIP', '' )
         devId = query.get( 'deviceId', '' )
         if s and g:
            sg = mcastKeyType( s, g )
            if sg in self.agentStatus.flowFailed:
               flow = self.agentStatus.flowFailed[ sg ]
               found = appendFailedFlow( sg, flow, response, d=devId )
         else:
            for sg, flow in self.agentStatus.flowFailed.items():
               if s and s != sg.source:
                  continue
               if g and g != sg.group:
                  continue
               found = appendFailedFlow( sg, flow, response, d=devId ) or found
         if not found:
            failedFlow[ 'sourceIP' ] = s
            failedFlow[ 'destinationIP' ] = g
            if devId:
               failedFlow[ 'deviceId' ] = devId
            appendErrorFailedFlow( '236', failedFlow, response )
      return http.client.OK, response

class ConsistentHashing( McsApiBase ):
   api = "/mcs/consistent-hashing[/]"

   def __init__( self, apiConfig ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext ):
      """
      Sample response:
      {
         "mode": "consistent",
         "weight": 200
      }
      """
      response = {}
      response[ 'mode' ] = self.apiConfig.chState
      response[ 'weight' ] = self.apiConfig.nodeWeight
      return http.client.OK, response

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext ):
      """
      Sample request:
      {
         "mode": "consistent",
         "weight": 200
      }
      Sample response:
      {
         "mode": "consistent",
         "weight": 200,
         "success": true,
         "messages": []
      }
      """
      data = requestContext.getRequestContent()
      data = json.loads( data ) if data else {}
      log( 'Received data:', bv( data ) )
      response = { 'success': True, 'messages': [] }
      if 'mode' in data:
         response[ 'mode' ] = data[ 'mode' ]
         if data[ 'mode' ] == chStateType.disabled:
            data[ 'weight' ] = 1
      if 'weight' in data:
         response[ 'weight' ] = data[ 'weight' ]
      valid, msgs = McsHttpAgentLib.validateConsistentHashingPost( data )
      if not valid:
         response[ 'success' ] = False
         response[ 'messages' ] = msgs
         return http.client.OK, response
      if 'mode' in data:
         self.apiConfig.chState = data[ 'mode' ]
      if 'weight' in data:
         self.apiConfig.nodeWeight = data[ 'weight' ]
      return http.client.OK, response

class DeviceMaintenance( McsApiBase ):
   api = "/mcs/device-maintenance[/]"

   def __init__( self, apiConfig, agentStatus, topoStatus ):
      assert Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled()
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.agentStatus = agentStatus
      self.topoStatus = topoStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext ):
      '''
      Sample response:
      [
         {
            "deviceId": "0000.0000.0007",
            "networkDeviceName": "arista.networks.ce7",
            "maintenanceState": "red",
         }
      ]
      '''
      response = []
      for d, m in self.agentStatus.maintenanceDeviceStatus.items():
         resp = {}
         resp[ 'deviceId' ] = McsHttpAgentLib.get_dotted_mac( d )
         resp[ 'networkDeviceName' ] = McsHttpAgentLib.getDevName(
            d, self.topoStatus )
         resp[ 'maintenanceState' ] = m.state
         response.append( resp )
      return http.client.OK, response

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext ):
      '''
      Sample request:
      [
         {
            "deviceId": "0000.0000.0001",
            "maintenance": true
         }
      ]
      Sample response:
      [
         {
            "deviceId": "0000.0000.0001",
            "networkDeviceName": "arista.networks.ce1",
            "maintenance": true,
            "success": true,
            "messages": []
         }
      ]
      '''
      request = requestContext.getRequestContent()
      request = json.loads( request ) if request else {}
      log( 'Received data:', bv( request ) )
      response = []
      for req in request:
         resp = req
         valid, msgs = McsHttpAgentLib.validateDeviceMaintenancePost( req )
         resp[ 'success' ] = valid
         resp[ 'messages' ] = msgs
         if valid:
            resp[ 'networkDeviceName' ] = McsHttpAgentLib.getDevName(
               req[ 'deviceId' ], self.topoStatus )
            if req[ 'maintenance' ]:
               self.apiConfig.maintenanceDevice.add( req[ 'deviceId' ] )
            else:
               del self.apiConfig.maintenanceDevice[ req[ 'deviceId' ] ]
         response.append( resp )
      return http.client.OK, response

class JsonSchema( McsApiBase ):
   api = "/mcs/schema/{schemaPathPrefix}/{schemaName}[/]"

   def __init__( self ):
      info( "Registering JsonSchema" )
      McsApiBase.__init__( self )

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getJsonSchema( self, requestContext, schemaPathPrefix, schemaName ):
      info( "Getting MCS Json Schema:", bv( schemaName ) )
      filePath = os.path.join( "/usr", "share", "Mcs", "jsonSchema",
            schemaPathPrefix, schemaName )
      with open( filePath ) as f:
         result = json.load( f )
         if result:
            return http.client.OK, result
         else:
            return http.client.NO_CONTENT, None

class AgentStatus( McsApiBase ):
   api = "/mcs/agentStatus[/]"

   def __init__( self, agentStatus ):
      info( "Registering AgentStatus" )
      McsApiBase.__init__( self )
      self.agentStatus = agentStatus

   @UrlMap.urlHandler( ( 'GET', ), api )
   def getAgentStatus( self, requestContext ):
      info( "Getting MCS Agent status " )
      model = McsApiModels.AgentStatusModel()
      model.fromSysdb( self.agentStatus )
      result = json.loads( json.dumps( model, cls=ModelJsonSerializer ) )
      if result:
         return http.client.OK, result
      else:
         return http.client.NO_CONTENT, None

class ShutdownReactor:
   def __init__( self, mcsConfig, clusterStatus, controllerStatus, httpConfig,
                 apiStatus ):
      self.mcsConfig = mcsConfig
      self.clusterStatus = clusterStatus
      self.controllerStatus = controllerStatus
      self.httpConfig = httpConfig
      self.apiStatus = apiStatus

      self.mcsCfgReactor = GenericReactor( self.mcsConfig,
                                           [ 'enabled' ],
                                           self.handleEnabled )
      self.clusterReactor = GenericReactor( self.clusterStatus,
                                            [ 'isStandaloneOrLeader' ],
                                            self.handleEnabled )
      self.controllerReactor = GenericReactor( self.controllerStatus,
                                               [ 'enabled' ],
                                              self.handleEnabled )
      self.httpReactor = GenericReactor( self.httpConfig,
                                         [ 'enabled' ],
                                         self.handleEnabled )
      self.handleEnabled()

   def handleEnabled( self, notifiee=None ):
      log( "Mcs enabled =", bv( self.mcsConfig.enabled ),
           ", CVX enabled =", bv( self.controllerStatus.enabled ),
           ", isStandaloneOrLeader =", bv( self.clusterStatus.isStandaloneOrLeader ),
           ", httpd =", bv( self.httpConfig.enabled ) )
      enabled = True
      if not ( self.mcsConfig.enabled and
               self.clusterStatus.isStandaloneOrLeader and
               self.controllerStatus.enabled ):
         # Clear agent's states when shutting down
         self.apiStatus.ready = False
         self.apiStatus.syncInProgress = apiSyncStatus.notStarted
         self.apiStatus.syncUuid = ""
         enabled = False
      elif not self.httpConfig.enabled:
         # If Httpd is shutdown, we'll shutdown McsHttpAgent but continue to run
         # Mcs service. So, cleanup shouldn't be done
         enabled = False
      self.apiStatus.enabled = enabled

      # Set syncUuid when Mcs is enabled and Controller is standalone or leader
      if self.mcsConfig.enabled and self.clusterStatus.isStandaloneOrLeader:
         log( "syncUuid:", bv( self.clusterStatus.controllerUUID ) )
         self.apiStatus.syncUuid = self.clusterStatus.controllerUUID

class GetHAClients( McsApiBase ):
   api = "/mcs/haClients[/]"

   def __init__( self, cdbMounts ):
      McsApiBase.__init__( self )
      self.cdbMounts = cdbMounts

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext ):
      """
      Sample response:
      {
         "sucess": true,
         "messages": []
         "clients": [
            "client1.domainname.com",
            "client2.domainname.com",
            "client3.domainname.com"
         ]
      }
      """
      clients = []
      for client, clientStatus in self.cdbMounts.switchStatusDir.items():
         if clientStatus.isSavingHAState:
            hostId = McsHttpAgentLib.get_colon_mac( client )
            if hostId in self.cdbMounts.cdbTopology.host:
               clients.append(
                  self.cdbMounts.cdbTopology.host.get( hostId ).hostname )

      response = { 'success': True,
                   'messages': [],
                   'clients': clients }

      log( f"GET /mcs/haClients {response}" )
      return http.client.OK, response

class McsBoundaryInterface( McsApiBase ):
   api = "/mcs/boundaryInterface[/]"

   def __init__( self, apiConfig, apiStatus ):
      McsApiBase.__init__( self )
      self.apiConfig = apiConfig
      self.apiStatus = apiStatus
      self.boundaryKeys = {
            'action',
            'data' }
      self.actions = {
            'del',
            'define' }

   @UrlMap.urlHandler( ( 'GET', ), api )
   def handleGet( self, requestContext ):
      """
      Sample response
      {
         "boundaryInterfaces":[
            {
               "name": "McsBoundaryIntf1",
               "interfaces": [ "001c.738d.1569-Ethernet6",
                               "001c.738d.1569-Ethernet7" ]
            },
            {
               "name": "McsBoundaryIntf2",
               "interfaces": [ "001c.738d.1570-Ethernet6",
                               "001c.738d.1570-Ethernet7" ]
            }
         ],
         "messages": [],
         "success": true
      }
      """

      defaultFilter = """
               {
                "boundaryInterface": null
               }
      """
      boundaryInterface = self.get_resp_template()
      boundaryInterface[ 'boundaryInterfaces' ] = []
      filterData = requestContext.getRequestContent()
      filters = None

      def handleInvalidReq( filterData ):
         error( f"Invalid request with {filterData}" )
         boundaryInterface[ 'messages' ].append( '187' )
         boundaryInterface[ 'success' ] = False

      try:
         if filterData:
            filters = json.loads( filterData )
         if not filters:
            filters = json.loads( defaultFilter )
      except json.JSONDecodeError:
         handleInvalidReq( filterData )
         return http.client.OK, boundaryInterface

      info( 'Received data:', bv( filters ) )
      if isinstance( filters, str ):
         handleInvalidReq( filterData )
         return http.client.OK, boundaryInterface

      if not filters.get( 'boundaryInterface' ):
         for name, intf in self.apiConfig.boundaryInterface.items():
            data = {
                  "name": name,
                  "interfaces": [ devIntf.device + '-' + devIntf.intfId for devIntf
                     in intf.interface ]
            }
            boundaryInterface[ 'boundaryInterfaces' ].append( data )
         boundaryInterface[ 'success' ] = True
      else:
         valid, ec = McsHttpAgentLib.validateBoundaryIntfName( filters[
            'boundaryInterface' ] )
         if not valid:
            boundaryInterface[ 'messages' ].append( ec )
            data = {
                  "name": filters[ 'boundaryInterface' ],
                  "interfaces": []
            }
            boundaryInterface[ 'boundaryInterfaces' ].append( data )
         else:
            if ( filters[ 'boundaryInterface' ] not in self.apiConfig.
                  boundaryInterface ):
               data = {
                     "name": filters[ 'boundaryInterface' ],
                     "interfaces": []
               }
            else:
               bIntf = self.apiConfig.boundaryInterface[
                     filters[ 'boundaryInterface' ] ]
               data = {
                     "name": filters[ 'boundaryInterface' ],
                     "interfaces": [ devIntfId.device + '-' + devIntfId.intfId
                        for devIntfId in bIntf.interface ]
               }
            boundaryInterface[ 'boundaryInterfaces' ].append( data )
            boundaryInterface[ 'success' ] = True
      return http.client.OK, boundaryInterface

   @UrlMap.urlHandler( ( 'POST', ), api )
   def handlePost( self, requestContext ):
      """
      Sample request:
      {
         "action": "define",
         "data": [ {
            "name": "McsBoundaryIntf1",
            "interfaces":  ["001c.738d.1569-Ethernet6", "001c.738d.1569-Ethernet7"]
          },
          {
            "name": "McsBoundaryIntf2",
            "interfaces": ["001c.738d.1569-Ethernet6"]
          } ]
      }
      """
      info( "Configuring boundary interface" )
      data = requestContext.getRequestContent()
      data = json.loads( data )
      log( 'Received data:', bv( data ) )

      return self.handleBoundaryInterface( data )

   def validateData( self, postData ):
      boundaryIntfUpdate = self.get_boundary_intf_resp_template()
      ecFailedIntfInPost = '251'
      def boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf,
            boundaryIntfUpdateOuter=False ):
         if boundaryIntfUpdateOuter:
            boundaryIntfUpdate[ 'messages' ].append( ec )
            boundaryIntfUpdate[ 'action' ] = postData.get( 'action', 'U/A' )
            boundaryIntfUpdate[ 'data' ] += postData.get( 'data', [] )
         else:
            intfUpdate[ 'messages' ].append( ec )
            intfUpdate[ 'interfaces' ] += bIntf.get( 'interfaces', [] )
            boundaryIntfUpdate[ 'data' ].append( intfUpdate )
            if ecFailedIntfInPost not in boundaryIntfUpdate[ 'messages' ]:
               boundaryIntfUpdate[ 'messages' ].append( ecFailedIntfInPost )

      ec, _ = McsHttpAgentLib.findMissingKeys( self.boundaryKeys, postData )
      if ec:
         boundaryIntfUpdateForFailedIntf( ec, None, None, True )
         return boundaryIntfUpdate
      action = postData[ 'action' ]
      if action not in self.actions:
         ec = '248'
         error( errorCodes[ ec ].replace( '<postedAction>', action ) )
         boundaryIntfUpdateForFailedIntf( ec, None, None, True )
         return boundaryIntfUpdate
      boundaryIntfUpdate[ 'action' ] = action
      if not isinstance( postData[ 'data' ], list ):
         ec = '121'
         error( errorCodes[ ec ] )
         boundaryIntfUpdateForFailedIntf( ec, None, None, True )
         return boundaryIntfUpdate

      boundaryIntfUpdate[ 'action' ] = action
      for bIntf in postData[ 'data' ]:
         intfUpdate = self.get_boundary_intf_template()
         boundaryInterfaceKeys = { 'name' }
         if action != 'del':
            boundaryInterfaceKeys.add( 'interfaces' )
         ec, _ = McsHttpAgentLib.findMissingKeys( boundaryInterfaceKeys, bIntf )
         if ec:
            intfUpdate[ 'name' ] = bIntf.get( 'name', 'U/A' )
            boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
            continue
         valid, ec = McsHttpAgentLib.validateBoundaryIntfName( bIntf[ 'name' ] )
         if not valid:
            # Converting name to str, to handle case when name data type is other
            # than str
            error( errorCodes[ ec ].replace( '<name>', str( bIntf[ 'name' ] ) ) )
            intfUpdate[ 'name' ] = bIntf[ 'name' ]
            boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
            continue
         intfUpdate[ 'name' ] = bIntf[ 'name' ]
         if ( bIntf[ 'name' ] not in self.apiConfig.boundaryInterface and
            action in [ 'del' ] ):
            ec = '246'
            error( errorCodes[ ec ].replace( '<BoundaryIntfName>', bIntf[ 'name'
               ] ) )
            boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
            if action == 'del':
               # Since boundaryIntfUpdateForFailedIntf function add interfaces
               # [] when not present.In del operation interfaces is not required
               intfUpdate.pop( 'interfaces' )
            continue
         if action in [ 'del', 'define' ]:
            for mSender in self.apiConfig.boundarySender.values():
               if mSender == bIntf[ 'name' ]:
                  ec = '256'
                  error( errorCodes[ '256' ] )
                  boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
                  break
            if ec:
               continue
            for mRcvr in self.apiConfig.boundaryReceiver.values():
               if bIntf[ 'name' ] in mRcvr.receiver:
                  ec = '256'
                  error( errorCodes[ '256' ] )
                  boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
                  break
               if ec:
                  break
            if ec:
               continue
         if action in [ 'define' ]:
            if not isinstance( bIntf[ 'interfaces' ], list ):
               ec = '121'
               error( errorCodes[ ec ] )
               boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
               continue
            if not bIntf[ 'interfaces' ] or len( bIntf[ 'interfaces' ] ) > 1000:
               ec = '250'
               error( errorCodes[ ec ].replace( '<boundaryIntf>', bIntf[ 'name'
                  ] ) )
               boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
               continue

            for intfId in bIntf[ 'interfaces' ]:
               valid, ec, _, _ = McsHttpAgentLib.validateDeviceID( self.apiConfig,
                     intfId, bIntf[ 'name' ] )
               if not valid:
                  boundaryIntfUpdateForFailedIntf( ec, intfUpdate, bIntf )
                  break
            # Ensuring modification or addition of new boundary interface to be have
            # non-overlapping physical interfaces list
            if not valid:
               continue

            intfUpdate[ 'interfaces' ] += bIntf[ 'interfaces' ]

         intfUpdate[ 'success' ] = True
         boundaryIntfUpdate[ 'data' ].append( intfUpdate )
      # When action is del and boundary interface has active
      # sender or receiver reject the delete request. That case
      # will be added later because currently we dont have
      # McastBoundarySender, McastBoundaryReceiver collection in
      # apiConfig
      boundaryIntfUpdate[ 'success' ] = True
      return boundaryIntfUpdate

   def handleBoundaryInterface( self, data ):
      log( 'handleBoundaryInterface started' )
      boundaryIntfUpdate = self.validateData( data )
      if not boundaryIntfUpdate[ 'success' ]:
         return http.client.OK, boundaryIntfUpdate

      if boundaryIntfUpdate[ 'action' ] in [ 'define' ]:
         for intf in boundaryIntfUpdate[ 'data' ]:
            if intf[ 'success' ]:
               bIntf = boundaryIntf()
               for intfId in intf[ 'interfaces' ]:
                  dev, interfaceId = intfId.split( '-' )
                  devIntfId = deviceIntfId( dev, interfaceId )
                  bIntf.interface.add( devIntfId )
                  self.apiStatus.devIntfToBintf[ devIntfId ] = intf[ 'name' ]
               self.apiConfig.boundaryInterface[ intf[ 'name' ] ] = bIntf
      elif boundaryIntfUpdate[ 'action' ] == 'del':
         for intf in boundaryIntfUpdate[ 'data' ]:
            if intf[ 'success' ]:
               bInterfaces = self.apiConfig.boundaryInterface[ intf[ 'name' ] ]
               for devIntfId in bInterfaces.interface:
                  del self.apiStatus.devIntfToBintf[ devIntfId ]
               del self.apiConfig.boundaryInterface[ intf[ 'name' ] ]
      log( 'handleBoundaryInterface finished' )
      return http.client.OK, boundaryIntfUpdate

class McsApiHandler( Agent.Agent ):
   # pylint: disable-msg=W0201
   def __init__( self, em, blocking=False ):
      Agent.Agent.__init__( self, em, agentName="McsHttpAgent" )
      self.warm_ = False
      self.sysname = em.sysname()
      self.controllerdbMountsComplete = False
      self.blocking = blocking
      self.sysdbMountsComplete = False
      self.apiConfig = None
      self.mcsControllerMounts = None
      self.agentStatus = None
      self.cliConfig = None
      self.apiStatus = None
      self.apiNotify = None
      self.readyReactor = None
      self.mcsNotify = None
      self.mcsHA = None
      self.httpConfig = None
      self.threadLocalData_ = threading.local()
      self.em = em
      self.agentInitializationDone = False
      self.boundaryPhysicalIntfSelSm = None
      # pylint: disable-next=consider-using-f-string
      qtfile = "{}{}.qt".format( self.agentName, "-%d" if "QUICKTRACEDIR"
                                 not in os.environ else "" )
      BothTrace.initialize( qtfile, "8, 128, 8, 8, 8, 128, 8, 8, 128, 8",
                            maxStringLen=240 )

      cdbSocket = os.environ.get(
         'CONTROLLERDBSOCKNAME',
         Tac.Value( 'Controller::Constants' ).controllerdbDefaultSockname )

      self.cdbEm = Controllerdb(
         em.sysname(), controllerdbSockname_=cdbSocket, mountRoot=False )

   @property
   def aaaApiClient_( self ):
      # AaaApiClient.AaaApiClient is not thread safe, so each thread should
      # create it's own client
      if not hasattr( self.threadLocalData_, 'aaaApiClient' ):
         self.threadLocalData_.aaaApiClient = AaaApiClient.AaaApiClient(
               self.sysname )
      return self.threadLocalData_.aaaApiClient

   def doMaybeFinishInit( self ):
      if self.sysdbMountsComplete and self.controllerdbMountsComplete:
         self.mountsComplete()
         self.warm_ = True
      else:
         log( "Sysdb and controllerdb mounts incomplete" )
         log( "controllerdbMountsComplete", bv( self.controllerdbMountsComplete ) )
         log( "sysdbMountComplete", bv( self.sysdbMountsComplete ) )

   # Parameters differ from overridden 'doInit' method
   # pylint: disable-msg=arguments-differ
   def doControllerdbMounts( self ):
      def _onControllerdbMountsComplete():
         info( 'controllerdb mounts complete' )
         self.controllerdbMountsComplete = True
         self.doMaybeFinishInit()
         self.finishAgentInitalization()

      cdbMg = self.cdbEm.mountGroup()
      cdbMg.mount( '', 'Tac::Dir', 'rt' )  # t=toplevel rootMount, must do this first
      self.mcsControllerMounts = McsControllerMounts( cdbMg )

      if self.blocking:
         info( 'doing Controllerdb mounts: blocking' )
         cdbMg.close( blocking=True )
      else:
         info( 'doing Controllerdb mounts: non-blocking' )
         cdbMg.close( callback=_onControllerdbMountsComplete )

   def mountsComplete( self ):
      info( 'mountsComplete' )
      self.syncUuidUpdate = GenericReactor( self.apiStatus,
                                            [ 'syncUuid' ],
                                            self.mcsStateHAReplicate )
      self.shutdownReactor = ShutdownReactor(
                                    self.cliConfig,
                                    self.clusterStatus.status[ 'default' ],
                                    self.controllerStatus,
                                    self.httpConfig.service[ 'mcs' ],
                                    self.apiStatus )

      if Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled():
         self.boundaryPhysicalIntfSelSm = Tac.newInstance(
               "Mcs::McsBoundaryIntfSelcPhysicalIntfSm",
              self.apiStatus, self.apiConfig, self.agentStatus )

   def doSysdbMounts( self, em ):
      def _onMountsComplete():
         info( 'sysdb mounts complete' )
         self.sysdbMountsComplete = True
         self.doMaybeFinishInit()
         self.finishAgentInitalization()

      mg = em.mountGroup()

      self.apiConfig = mg.mount( "mcs/config/api", "Mcs::ApiConfig", "w" )
      self.cliConfig = mg.mount( "mcs/config/cli", "Mcs::CliConfig", "r" )
      self.agentStatus = mg.mount( "mcs/status/agent", "Mcs::AgentStatus", "r" )
      self.apiStatus = mg.mount( "mcs/status/api", "Mcs::ApiStatus", "w" )
      self.apiNotify = mg.mount( "mcs/status/notify", "Mcs::ApiNotifyQueue", "w" )
      self.httpConfig = mg.mount( "mgmt/capi/config", "HttpService::Config", "r" )
      self.clusterStatus = mg.mount( "controller/cluster/statusDir",
                                     "ControllerCluster::ClusterStatusDir", "r" )
      self.controllerStatus = mg.mount( "controller/status",
                                        "Controllerdb::Status", "r" )

      if self.blocking:
         info( 'doing Sysdb mounts: blocking' )
         mg.close( blocking=True )  # synchronous mounting
         _onMountsComplete()
      else:
         info( 'doing Sysdb mounts: non-blocking' )
         mg.close( callback=_onMountsComplete )

   def mcsStateHAReplicate( self, notifiee=None ):
      # Initialize replication object
      if not self.apiStatus.syncUuid:
         return

      info( "Initializing McsStateHAReplicate" )
      self.mcsHA = McsStateHAReplicate.McsStateReplicate(
         self.apiConfig,
         self.clusterStatus,
         self.apiStatus,
         self.agentStatus,
         self.mcsControllerMounts.mcsStatusHA,
         self.em )
      self.mountReadyReactor = GenericReactor( self.agentStatus,
                                               [ 'mountDone' ],
                                               self.handleStateSync )
      self.syncStatusUpdate = GenericReactor( self.apiStatus,
                                              [ 'syncInProgress' ],
                                              self.handleSyncStatus )
      self.handleStateSync()

   def doInit( self, em ): # pylint: disable=arguments-renamed
      info( "doInit..." )
      self.doControllerdbMounts()
      self.doSysdbMounts( em )
      info( "doInit done" )

   def finishAgentInitalization( self ):
      if self.sysdbMountsComplete and self.controllerdbMountsComplete:
         info( "finishAgentInitalization started" )
         AgentStatus( self.agentStatus )
         ApiStatus( self.apiStatus )
         mcastSender = PostMcastSender( self.apiConfig, self.apiNotify,
                                        self.apiStatus )
         GetMcastSender( self.apiConfig, self.mcsControllerMounts.cdbTopology )
         mcastReceiver = PostMcastReceiver( self.apiConfig, self.apiNotify )
         Oui( self.apiConfig, self.agentStatus )
         GetMcastReceiver( self.apiConfig, self.agentStatus,
                           self.mcsControllerMounts )
         GetFlowsActive( self.mcsControllerMounts )
         GetNetworkLinks( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetEndpointLinks( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetNetworkLink( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetEndpointLink( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetInterfaces( self.mcsControllerMounts, self.agentStatus, self.apiConfig )
         ReservationPercentage( self.apiConfig, self.mcsControllerMounts )
         GetApiErrorCodes()

         GetNetworkDevices( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )

         GetEndpointDevices( self.agentStatus, self.mcsControllerMounts,
                              self.apiConfig )

         GetNetworkDevice( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )

         GetEndpointDevice( self.agentStatus, self.mcsControllerMounts,
                           self.apiConfig )
         GetFailedFlows( self.agentStatus,
                         self.mcsControllerMounts.cdbTopology )
         ConsistentHashing( self.apiConfig )
         if Toggles.McsToggleLib.toggleDeviceMaintenanceEnabled():
            DeviceMaintenance( self.apiConfig, self.agentStatus,
                               self.mcsControllerMounts.cdbTopology )
         if Toggles.McsToggleLib.toggleMcsBoundaryIntfEnabled():
            McsBoundaryInterface( self.apiConfig, self.apiStatus )
         JsonSchema()
         ClearRoutes( self.apiConfig, self.apiNotify, self.apiStatus )
         PostMcastProgramFlows( mcastSender, mcastReceiver )
         GetHAClients( self.mcsControllerMounts )

         self.agentInitializationDone = True
         self.apiStatus.ready = True
         info( "finishAgentInitalization done" )

   def handleStateSync( self, notifiee=None ):
      log( "Sync Status:", bv( self.apiStatus.syncInProgress ),
           ", Sync UUID:", bv( self.apiStatus.syncUuid ) )
      if self.mcsHA and self.apiStatus.syncUuid:
         self.mcsHA.syncApiConfig()

   def handleSyncStatus( self, notifiee=None ):
      log( "Sync Status:", bv( self.apiStatus.syncInProgress ),
           ", Sync UUID:", bv( self.apiStatus.syncUuid ) )
      if self.apiStatus.syncInProgress == apiSyncStatus.finished:
         log( "Done with the sync" )
         self.mcsHA = None

   def processRequest( self, request ):
      try:
         requestContext = UwsgiRequestContext( self.sysname, request,
               aaaApiClient=self.aaaApiClient_ )
         requestType = requestContext.getRequestType()
         parsedUrl = requestContext.getParsedUrl()
         log( 'processRequest(): request type:', bv( requestType ),
              ', parsed url:',
              bv( parsedUrl ), 'received data:',
              bv( requestContext.getRequestContent() ) )
         func, urlArgs = UrlMap.getHandler( requestType, parsedUrl )

         log( "Got", bv( func ), "and", bv( urlArgs ),
              "for url", bv( parsedUrl ) )
         if func is None:
            # pylint: disable-msg=no-value-for-parameter
            raise HttpNotFound( "Invalid endpoint requested" )
            # pylint: enable-msg=no-value-for-parameter
         obj, funcName = func
         status, result = getattr( obj, funcName )( requestContext, **urlArgs )
         result = json.dumps( result )
      except HttpNotFound as e:
         log( "processRequest HttpNotFound :", bv( str( e ) ) )
         ec = '187'
         errMsg = json.dumps( { 'messages': ec, 'success': False } )
         error( errorCodes[ ec ] )
         return ( e.code, "application/json",
                  e.additionalHeaders, errMsg.encode() )
      except OSError as e:
         log( "processRequest OSError :", bv( str( e ) ) )
         ec = '118'
         errMsg = json.dumps( { 'messages': ec, 'success': False } )
         error( errorCodes[ ec ].replace( '<traceback>', str( e ) ) )
         return ( 500, "application/json", [], errMsg.encode() )
      else:
         return ( status, 'application/json', None, result.encode() )

   def doCleanup( self ):
      pass

class McsHttpAgent:
   def __init__( self ):
      self.container_ = Agent.AgentContainer( [ McsApiHandler ],
                                              agentTitle="McsHttpAgent" )
      self.container_.startAgents()
      # Run ActivityLoop in a separate thread in order to serve HTTP requests
      # in the main thread
      Tac.activityThread().start( daemon=True )
      Tac.waitFor( self.mcsHttpAgentWarm,
                   maxDelay=1,
                   sleep=True, description="Mcs HTTP Agent warmup" )
      self.agent_ = self.container_.agents_[ 0 ]

   def mcsHttpAgentWarm( self ):
      return self.container_.agents_ and self.container_.agents_[ 0 ].warm()

   def __call__( self, request, start_response ):
      if not self.agent_.agentInitializationDone:
         return []

      ( responseCode, contentType, headers, body ) = \
            self.agent_.processRequest( request )
      # pylint: disable-next=consider-using-f-string
      responseStr = '%d %s' % ( responseCode, http.client.responses[ responseCode ] )
      headers = headers or []
      headers.append( ( 'Content-type', contentType ) )
      if body:
         headers.append( ( 'Content-length', str( len( body ) ) ) )
      start_response( responseStr, UwsgiConstants.DEFAULT_HEADERS + headers )
      return [ body ]
