#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys
import stat
import termios
import pexpect
import re
import subprocess
import time
import glob
import Tracing # pylint: disable-msg=import-error
import Pci
import ScdRegisters
import argparse
import MicrosemiLib

# pylint: disable-msg=consider-using-f-string

Tracing.traceSettingIs( f"{Tracing.defaultTraceHandle().name}/0" )

t0 = Tracing.trace0
t1 = Tracing.trace1
t9 = Tracing.trace9

#
# ConfigDesc is the parsed string descriptor of some resource
# object. Not a URL, but in similar spirit. Mediates between what we
# register locally, in MicrosemiLib, or provide/override on the
# command line.
#
# Example:
#   ConfigDesc.fromString( "ScdResetGpo,0x4000,pinId=7" ).resolve()
#   => ScdResetGpo.fromDesc( 0x4000, pinId=7 )
#
class ConfigDesc:

   def __init__( self, t, *args, **kwargs ):
      self.t = t
      self.args = args if args else []
      self.kwargs = kwargs if kwargs else {}

   @classmethod
   def fromString( cls, s ):
      typeName, argv = s.split( ',', 1 )
      args = []
      kwargs = {}
      for arg in argv.split( ',' ):
         if '=' in arg:
            k, v = arg.split( '=', 1 )
            kwargs[ k ] = v
         else:
            args.append( arg )

      t = globals().get( typeName )
      assert t is not None, f"No such type: '{typeName}'"

      return cls( t, *args, **kwargs )

   def resolve( self ):
      return self.t.fromDesc( *self.args, **self.kwargs )

   def __repr__( self ):
      return \
         "{}({},{})".format(
            self.__class__.__name__,
            self.t.__name__,
            ",".join( list( self.args ) +
                      [ f"{k}={v}"
                        for k, v in self.kwargs.items() ] ) )

   def __str__( self ):
      return "{},{}".format(
         self.t.__name__,
         ",".join( list( self.args ) +
                   [ f"{k}={v}"
                     for k, v in self.kwargs.items() ] ) )

#
# Supervisors keep their GPOs in the SCD reset register (0x4000)
# E.g. ScdResetGpo,9f:00.0,0x4000,pinId=7
#
class ScdResetGpo:

   def __init__( self, scdReg, pinId ):
      self.scdReg = scdReg
      self.pinId = pinId

   @classmethod
   def fromDesc( cls, scdPciAddr, offset, pinId ):
      scdPciAddr = Pci.Address( scdPciAddr )
      scdReg = ScdResetReg.lookup( scdPciAddr, int( offset, 0 ) )
      return cls( scdReg, int( pinId ) )

   @classmethod
   def gpoDesc( cls, scdPciAddr, offset, pinId ):
      return ConfigDesc( cls,
                         str( scdPciAddr ),
                         hex( offset ),
                         pinId=pinId )

   def write( self, v ):
      if bool( v ):
         self.scdReg.setBits( 1 << self.pinId )
      else:
         self.scdReg.clearBits( 1 << self.pinId )

   def __repr__( self ):
      return "{}({!r}, pinId={})".format(
         self.__class__.__name__, self.scdReg, self.pinId )

   def __str__( self ):
      return repr( self )

class ScdResetReg:

   OFFSET = 0x4000

   def __init__( self, name, regs, offset ):
      self.name = name
      self.regs = regs
      self.offset = offset

   SCDS = {}

   @classmethod
   def lookup( cls, scdPciAddr, offset ):
      name = str( scdPciAddr )
      regs = cls.SCDS.get( name )
      if not regs:
         regs = Pci.Device( scdPciAddr ).resource( 0 )
         cls.SCDS[ name ] = regs
      return cls( name, regs, offset )

   def read( self ):
      val = self.regs.read32( self.offset )
      t0( f"{self!r} rd {val:#010x}" )
      return val

   def _wr( self, offset, val ):
      t0( f"{self!r} wr {val:#010x}" )
      self.regs.write32( offset, val )

   def setBits( self, mask ):
      offset = self.offset + ScdRegisters.ResetGpo.setOffset
      self._wr( offset, mask )

   def clearBits( self, mask ):
      offset = self.offset + ScdRegisters.ResetGpo.clearOffset
      self._wr( offset, mask )

   def __repr__( self ):
      return "{}({!r}, {:#06x})".format(
         self.__class__.__name__, self.name, self.offset )

   def __str__( self ):
      return repr( self )

#
# Combine one reset-GPO and one recovery-GPO into a recovery-mode control.
# * GpoPair.enter() resets the switch into recovery mode.
# * GpoPair.leave() resets again, leaving recovery mode.
#
class GpoPair:

   def __init__( self, resetGpo, recoveryGpo ):
      self.resetGpo = resetGpo
      self.recoveryGpo = recoveryGpo

   @classmethod
   def fromDescStrings( cls, resetGpoStr, recoveryGpoStr ):
      resetGpo = \
         ConfigDesc.fromString( resetGpoStr ).resolve()

      recoveryGpo = \
         ConfigDesc.fromString( recoveryGpoStr ).resolve()

      return cls( resetGpo, recoveryGpo )

   DELAY = .1

   def enter( self ):
      self.resetGpo.write( 1 )
      self.recoveryGpo.write( 1 )
      time.sleep( self.DELAY )
      self.resetGpo.write( 0 )

   def leave( self ):
      self.resetGpo.write( 1 )
      self.recoveryGpo.write( 0 )
      time.sleep( self.DELAY )
      self.resetGpo.write( 0 )

   def __repr__( self ):
      return "{}(resetGpo={!r}, recoveryGpo={!r})".format(
         self.__class__.__name__,
         self.resetGpo, self.recoveryGpo )

   def __str__( self ):
      return repr( self )

#
# PM853x terminal IO control and file-like interface.
# * Speed is fixed, B230400,8n1
# * 1-byte MIN, 1.5s TIME seems to work okay.
# * We switch between ICANON on and off, depending on context.
#   This is just cosmetic.
#
class Tty:

   def __init__( self, fh, tca ):
      self.fh = fh
      self.tca = self.tcr = tca

   @classmethod
   def open( cls, path, **config ):
      fh = open( path, "r+b", buffering=0 ) # pylint: disable-msg=consider-using-with
      tc = termios.tcgetattr( fh.fileno() )

      tty = cls( fh, tc )
      tty.configure( when=termios.TCSANOW, **config )

      t0( "{}.open({}): {!r}".format(
         cls.__name__,
         ", ".join( [ f"{k}={v!r}"
                      for k, v in [
                            ( 'path', path ), *config.items() ] ] ),
         tty ) )

      return tty

   def configure( self,
                  when=termios.TCSANOW,
                  icanon=True,
                  speed=termios.B230400,
                  vmin=1,
                  vtime=15 ):
      # iflag
      self.tca[ 0 ] = 0
      # oflag
      self.tca[ 1 ] = 0
      # cflag
      self.tca[ 2 ] = termios.CREAD | termios.CLOCAL | termios.CS8
      # lflag
      if icanon is not None:
         self.tca[ 3 ] = termios.ICANON if icanon else 0

      if speed is not None:
         # ispeed
         self.tca[ 4 ] = speed
         # ospeed
         self.tca[ 5 ] = speed

      # cc
      if vmin is not None:
         self.tca[ 6 ][ termios.VMIN ] = vmin # chars
      if vtime is not None:
         self.tca[ 6 ][ termios.VTIME ] = vtime # 1s/10

      termios.tcsetattr( self.fileno(), when, self.tca )

   def icanon( self, enable=None, when=termios.TCSANOW ):
      def enabled():
         return bool( self.tca[ 3 ] & termios.ICANON )
      if enable is not None and bool( enable ) != enabled():
         if enable:
            self.tca[ 3 ] |= termios.ICANON
         else:
            self.tca[ 3 ] &= ~termios.ICANON
         termios.tcsetattr( self.fileno(), when, self.tca )
      return enabled()

   def close( self ):
      if self.fh:
         t0( f"{self!r}.close()" )
         self.tcflush()
         if self.tcr is not None:
            termios.tcsetattr( self.fileno(), termios.TCSANOW, self.tcr )
         self.fh.close()
         self.fh = None

   def fileno( self ):
      return self.fh.fileno() if self.fh else None

   def tcflush( self, what=termios.TCIFLUSH | termios.TCOFLUSH ):
      return termios.tcflush( self.fileno(), what )

   def tcdrain( self ):
      return termios.tcdrain( self.fileno() )

   def read( self, n ):
      s = self.fh.read( n )
      t0( f"read({n:d}): {s!r}" )
      assert len( s ) > 0
      return s

   def readline( self ):
      s = self.fh.readline()
      t0( f"readline(): {s}" )
      assert len( s ) > 0
      return s

   def write( self, s ):
      t0( f"write({s!r})" )
      self.fh.write( bytes( s, 'utf-8' ) )

   def __del__( self ):
      self.close()

   def __enter__( self ):
      return self

   def __exit__( self, t, v, tb ):
      self.close()

   def __repr__( self ):
      return "{}(fileno={}, icanon={})".format(
         self.__class__.__name__, self.fileno(), self.icanon() )

   def __str__( self ):
      return repr( self )

   @classmethod
   def lookup( cls, ttySysDev ):
      ttyDevPath = None
      if ttySysDev.startswith( "usb:" ):
         ttyDevPath = cls.lookupUsbDev( ttySysDev[ 4: ] )
      return ttyDevPath

   @classmethod
   def lookupUsbDev( cls, usbId ):
      ttyDevPath = None

      paths = glob.glob( f"/sys/bus/usb/devices/{usbId}/*/tty/*" )
      if paths:
         assert len( paths ) == 1
         path = paths[ 0 ]
         name = os.path.basename( path )
         ttyDevPath = f"/dev/{name}"

      return ttyDevPath

#
# Combine Tty and GpoPair into a session reprentative.
#
# This is for python `with` statements as try/finally
# placeholders (See __enter__, __exit__).
# Finally
#  - closing the tty (restoring tcattrs) and
#  - leaving recovery mode (resetting into normal mode).
#
class RecoverySession:

   def __init__( self, gpos, tty ):
      self.gpos = gpos
      self.tty = tty

   def __enter__( self ):
      if self.gpos:
         self.gpos.enter()
      return self

   def __exit__( self, t, v, tb ):
      if self.gpos:
         self.gpos.leave()
      if self.tty:
         self.tty.close()

   def __repr__( self ):
      return "{}(gpos={!r}, tty={!r})".format(
         self.__class__.__name__, self.gpos, self.tty )

   def __str__( self ):
      return repr( self )

#
# XMODEM transfers
# * /usr/bin/sx to transmit
# * /usr/bin/rx to receive,
#
class lrzsz:

   def __init__( self, args ):
      self.args = args

   @classmethod
   def rx( cls, path ):
      return cls( [ "/usr/bin/rx", path ] )

   @classmethod
   def sx( cls, path ):
      return cls( [ "/usr/bin/sx", path ] )

   def strace( self, path ):
      return type( self )( [ "strace", "-o", path, "--" ] + self.args )

   def launch( self, tty ):
      return subprocess.Popen( self.args, stdin=tty, stdout=tty )

   def __repr__( self ):
      return "{}({!r})".format( self.__class__.__name__,
                                self.args )

   def __str__( self ):
      return str( self.args )

#
# Pexpect-based dialog for the boot recovery menu
#
class ConsoleClient:

   def __init__( self, ts ):
      self.ts = ts

   @classmethod
   def fromTty( cls, tty ):
      ts = cls.TtySpawn( tty )
      return cls( ts )

   class TtySpawn ( pexpect.spawnbase.SpawnBase ):
      # Minimal Spawn for pexpect.
      # Worked better than fdpexpect, and has Tracing.

      def __init__( self, tty, *args, **kwargs ):
         super().__init__( *args, **kwargs )
         self.tty = tty

      def read_nonblocking( self, size=1, timeout=None ):
         t9( f"rnb({size:d}, timeout={timeout})" )
         return self.tty.read( size )

      def expect( self, *args, **kwargs ):
         t9( "expect({})".format(
            ",".join( [ f"{a!r}" for a in args ] +
                      [ f"{k}={v!r}"
                        for k, v in kwargs.items() ] ) ) )
         rv = super().expect( *args, **kwargs )
         t9( f"/expect: {rv}" )
         return rv

   class Options:
      # keep a sorted list of options, and make it iterable

      def __init__( self, **kwargs ):
         self._keys = [
            k for k, v in sorted( kwargs.items(),
                                  key=lambda item: item[ 1 ] )
         ]
         for k, v in kwargs.items():
            setattr( self, k, v )

      def __contains__( self, key ):
         return hasattr( self, key )

      def __getitem__( self, key ):
         return getattr( self, key )

      def __getattr__( self, key ):
         # just to fix pylint 'no-member' errors
         assert False

      def get( self, key ):
         return self[ key ] if key in self._keys else None

      def key( self, opt ):
         for k, v in self:
            if v == opt:
               return k
         return None

      def names( self ):
         return list( self._keys )

      def __iter__( self ):
         for k in self._keys:
            yield ( k, getattr( self, k ) )

   Option = Options(
      UpdateImage='1',
      DumpImage='2',
      UpdateFlash='3',
      DumpFlash='4',
   )

   Menu = {
      Option.UpdateImage : (
         re.escape( "1. Update Image Through Xmodem" ),
         lrzsz.sx ),
      Option.DumpImage : (
         re.escape( "2. Dump Image Through Xmodem" ),
         lrzsz.rx ),
      Option.UpdateFlash : (
         re.escape( "3. Update Raw Flash Through Xmodem" ),
         lrzsz.sx ),
      Option.DumpFlash : (
         re.escape( "4. Dump Raw Flash Through Xmodem" ),
         lrzsz.rx )
   }

   ImageOption = Options(
      Bootloader="0",
      PartMap0="1",
      PartMap1="2",
      Firmware0="3",
      Config0="4",
      Config1="5",
      NVLog="6",
      Firmware1="7",
      SEEPROM="8",
   )

   ImageMenu = {
      ImageOption.Bootloader : (
         re.escape( "0: Bootloader" ) ),
      ImageOption.PartMap0 : (
         re.escape( "1: Partition Map 0" ) ),
      ImageOption.PartMap1 : (
         re.escape( "2: Partition Map 1" ) ),
      ImageOption.Firmware0 : (
         re.escape( "3: Firmware Image 0" ) ),
      ImageOption.Config0 : (
         re.escape( "4: Config File 0" ) ),
      ImageOption.Config1 : (
         re.escape( "5: Config File 1" ) ),
      ImageOption.NVLog : (
         re.escape( "6: NVLog" ) ),
      ImageOption.Firmware1 : (
         re.escape( "7: Firmware Image 1" ) ),
      ImageOption.SEEPROM : (
         re.escape( "8: SEEPROM" ) ),
   }

   def transfer( self, option, path, imgoption=None, strace=None ):
      ts = self.ts
      tty = ts.tty

      def ctrl( c ):
         return chr( ord( c ) & 0x1f )

      # Start in canonical (i.e. line-oriented) mode.
      # If only for somewhat decorative reasons.
      tty.icanon( True, when=termios.TCSAFLUSH )
      time.sleep( .2 )
      tty.tcflush()
      tty.write( ctrl( 'c' ) )

      ts.expect(
         re.escape( "-----------[Boot Recovery Menu]------------" ) )
      ts.expect(
         re.escape( "-------------------------------------------" ) )
      tty.write( '\n' ) # this seems to speed things up up

      ts.expect(
         re.escape( "Press [Ctrl+B] to enter configuration menu" ) )
      tty.write( ctrl( 'b' ) )

      menuitem, cmdfn = self.Menu[ str( option ) ]
      ts.expect( menuitem )
      ts.expect(
         re.escape( "+-------------------------------------------+" ) )
      tty.write( str( option ) )

      # "Dump Image" -> ImageMenu
      if option == self.Option.DumpImage:
         menuitem = self.ImageMenu[ str( imgoption ) ]
         ts.expect( menuitem )
         tty.icanon( False )
         ts.expect( re.escape( "Other key to cancel" ) )
         tty.write( str( imgoption ) )
      else:
         tty.icanon( False )
      # Drop ICANON. The last console message from the chip, before
      # Xmodem, is not line-terminated. Be careful to read() up to
      # that point -- any other last words from the loader, if not
      # consumed before Xmodem, will drive sx ack/nak processing off
      # the road.
      msgs = [
         "press [CTRL+C] to cancel...",
         "Image is not valid! Give up image dump!",
      ]
      n = ts.expect( [ re.escape( msg ) for msg in msgs ] )
      if n != 0:
         img = self.ImageOption.key( imgoption )
         raise self.InvalidImage( img )

      # The receiving end repeats 'C' to request start of transfer for
      # a few rounds. Flush input, then wait for the first 'C' to
      # confirm.
      if cmdfn is lrzsz.sx:
         tty.tcflush()
         c = tty.read( 1 )
         assert c == b'C'

      # Fire up the transfer. If there is trouble, consider strace
      # and carefully check the io stream.
      cmd = cmdfn( path )
      if strace is not None:
         cmd = cmd.strace( strace )

      t0( cmd )

      xm = cmd.launch( tty )
      try:
         status = xm.wait()
         t0( f"status: {status:d}" )
      except KeyboardInterrupt as e:
         status = e

      tty.icanon( True )

      msgs = [
         "File {} successfully!".format( {
            lrzsz.sx : 'transfer', lrzsz.rx : 'receive' }[ cmdfn ] ),
         "Some timeout event occurs!",
         "Transfer was cancelled by Host!",
      ]
      n = ts.expect( [ re.escape( msg ) for msg in msgs ] )
      if n != 0 or status != 0:
         raise self.FileTransferFailure( cmd, status, msgs[ n ] )

      ts.expect(
         re.escape( "6. Exit and Reset" ) )
      ts.expect(
         re.escape( "+-------------------------------------------+" ) )
      tty.write( '6' )

   class FileTransferFailure ( Exception ):

      def __init__( self, cmd, status, msg ):
         super().__init__( msg )
         self.cmd = cmd
         self.status = status

      @property
      def msg( self ):
         return self.args[ 0 ]

      def __repr__( self ):
         return "{}(cmd={!r}, status={!r}, msg={!r})".format(
            self.__class__.__name__,
            self.cmd, self.status, self.msg )

   class InvalidImage ( Exception ):
      def __init__( self, img ):
         super().__init__(
            f"Image {img!r} is not valid." )
         self.img = img

      @property
      def msg( self ):
         return self.args[ 0 ]

      def __repr__( self ):
         return "{}(img={!r}, msg={!r})".format(
            self.__class__.__name__, self.img, self.msg )

   def updateImage( self, path ):
      self.transfer( self.Option.UpdateImage, path )

   def dumpImage( self, img, path ):
      self.transfer( self.Option.DumpImage, path, imgoption=img )

   def updateFlash( self, path ):
      self.transfer( self.Option.UpdateFlash, path )

   def dumpFlash( self, path ):
      self.transfer( self.Option.DumpFlash, path )

#
# Command-line flash/image recovery utility.
#
class RecoveryUtil:

   class ArgParser ( argparse.ArgumentParser ):

      class Exit ( Exception ):

         def __init__( self, status, msg ):
            super().__init__( msg )
            self.status = status

         @property
         def msg( self ):
            return self.args[ 0 ]

         def __repr__( self ):
            return "{}(status={}, msg={!r}".format(
               self.__class__.__name__, self.status, self.msg )

      # overriding exit() is according to docs
      def exit( self, status=0, message=None ):
         raise RecoveryUtil.ArgParser.Exit(
            status, message.rstrip() if message else None )

   epilog = """
Examples

  Unbrick local supervisor PCIe switch:

  # %(prog)s --supervisor updateImage --firmware --config

  Unbrick supervisor, with details:

  # %(prog)s \\
      --tty=/dev/ttyUSB0 \\
      --reset-gpo='ScdResetGpo,0000:06:00.0,0x4000,pinId=8' \\
      --recovery-gpo='ScdResetGpo,0000:06:00.0,0x4000,pinId=7' \\
      updateImage \\
        --firmware=/usr/share/Microsemi/firmware-1.B.8C.pmc \\
        --config=/usr/share/Microsemi/OtterLake.pmc

  Download all valid images from supervisor PCIe switch:

  # %(prog)s --supervisor dumpImage --archive -o /tmp

  Download raw flash image from local supervisor PCIe switch:

  # %(prog)s --supervisor dumpFlash --image=/tmp/flash.img

  Restore raw flash image on local supervisor PCIe switch:

  # %(prog)s --supervisor updateFlash --image=/tmp/flash.img

   """

   argParser = ArgParser(
      formatter_class=argparse.RawDescriptionHelpFormatter,
      epilog=epilog )

   argParser.add_argument( "--supervisor",
                           help="Recover local PCIe switch",
                           action="store_true" )
   argParser.add_argument( "--tty", metavar="TTY",
                           help="Path to local serial I/O device",
                           dest="ttyDevPath" )
   argParser.add_argument( "--reset-gpo", metavar="GPO",
                           help="Reset GPO descriptor",
                           dest="resetGpo" )
   argParser.add_argument( "--recovery-gpo", metavar="GPO",
                           help="Recovery GPO descriptor",
                           dest="recoveryGpo" )

   @classmethod
   def main( cls, *args, **kwargs ):

      config = argparse.Namespace( subcmd=None )
      cls.argParser.parse_args( *args, *kwargs, namespace=config )

      cmd = config.subcmd
      if cmd:
         t0( f"{cmd.name}: {cmd.help}" )
         cmd.fn( cls, config )
      else:
         cls.argParser.print_help()

   subParsers = argParser.add_subparsers( metavar="COMMAND" )

   class subcommand:
      # Just kidding.
      #
      #   @subcommand(option(), ...)
      #   cmdFn( ... ):
      #      pass
      #
      # => subcommand(cmdFn).__set_name__(RecoveryUtil, "cmdFn")
      # to gather RecoveryUtil.subParser/.add_parser()/.add_argument()
      # invocations.
      #
      # Also implies @classmethod.

      def __init__( self, *options, **spargs ):
         self.options = options
         self.spargs = spargs
         self.fn = None
         self.help = None
         self.name = None
         self.sp = None

      def __call__( self, fn ):
         self.fn = fn
         self.help = fn.__doc__
         return self

      def __set_name__( self, cls, name ):
         self.name = name

         sp = cls.subParsers.add_parser( self.name,
                                         help=self.help,
                                         **self.spargs )
         for args, kwargs in self.options:
            sp.add_argument( *args, **kwargs )

         self.sp = sp
         sp.set_defaults( subcmd=self )

         setattr( cls, name, classmethod( self.fn ) )

   def option( *args, **kwargs ): # pylint: disable-msg=no-method-argument
      return args, kwargs

   @subcommand(
      option( "--firmware",
              metavar="PMC", nargs="?", const=True,
              dest="firmware", action="store",
              help="Firmware PMC file" ),
      option( "--config",
              metavar="PMC", nargs="?", const=True,
              dest="firmwareConfig", action="store",
              help="Firmware configuration PMC file" ) )
   def updateImage( cls, config ): # pylint: disable-msg=no-self-argument
      """Update firmware / config PMC on switch"""

      # pylint: disable-msg=import-outside-toplevel
      from MicrosemiLib import MicrosemiConsts

      if config.firmware is True:
         if config.supervisor:
            config.firmware = MicrosemiConsts.firmwareImage
         else:
            config.firmware = None

      if config.firmwareConfig is True:
         if config.supervisor:
            config.firmwareConfig = MicrosemiConsts().configImage()
         else:
            config.firmwareConfig = None

      t0( f"firmware: {config.firmware}" )
      t0( f"firmwareConfig: {config.firmwareConfig}" )

      if not config.firmware and not config.firmwareConfig:
         config.subcmd.sp.error(
            "Need --firmware and/or --config "
            f"to {config.subcmd.name}" )

      with cls.recoverySession(
            config,
            openTty=True,
            recoveryMode=True ) as session:
         cc = ConsoleClient.fromTty( session.tty )

         if config.firmware:
            cc.updateImage( config.firmware )

         if config.firmwareConfig:
            cc.updateImage( config.firmwareConfig )

   @subcommand(
      option( "--image",
              metavar="IMAGE", nargs="+",
              choices=ConsoleClient.ImageOption.names(),
              help="Image(s) to dump ({})".format(
                 ", ".join( ConsoleClient.ImageOption.names() ) ) ),
      option( "--archive",
              action="store_true",
              help="Try dumping all images, skipping images reported invalid." ),
      option( "--output", '-o',
              metavar="PATH", required=True,
              help="File or directory" ) )
   def dumpImage( cls, config ): # pylint: disable-msg=no-self-argument
      """Dump individual image item(s) from switch"""

      if not config.image and config.archive:
         config.image = ConsoleClient.ImageOption.names()

      if not config.image:
         config.subcmd.sp.error( "Need at least one --image to dump." )

      i_multi = len( config.image ) > 1
      o_isdir = False
      try:
         st = os.stat( config.output )
         o_isdir = stat.S_ISDIR( st.st_mode )
      except FileNotFoundError as e:
         if i_multi:
            config.subcmd.sp.error( str( e ) )

      if i_multi and not o_isdir:
         config.subcmd.sp.error(
            "When dumping multiple files,"
            f" output {config.output!r} must be a directory" )

      for key in config.image:
         opt = ConsoleClient.ImageOption[ key ]
         if o_isdir:
            path = os.path.join( config.output,
                                 f"{key}.pmc" )
         else:
            path = config.output

         t0( f"{config.subcmd.name} {key} to {path!r}" )
         try:
            with cls.recoverySession(
                  config,
                  openTty=True,
                  recoveryMode=True ) as session:
               cc = ConsoleClient.fromTty( session.tty )
               cc.dumpImage( opt, path )

         except ConsoleClient.InvalidImage as e:
            t0( f"{config.subcmd.name} {key}: {e!r}" )
            if not config.archive:
               raise

   @subcommand(
      option( "--image",
              dest="flashImage", metavar="FILE", required=True,
              help="Input flash image" ) )
   def updateFlash( cls, config ): # pylint: disable-msg=no-self-argument
      """Update raw flash image on switch"""

      with cls.recoverySession(
            config,
            openTty=True,
            recoveryMode=True ) as session:
         cc = ConsoleClient.fromTty( session.tty )
         cc.updateFlash( config.flashImage )

   @subcommand(
      option( "--image",
              dest="flashImage", metavar="FILE", required=True,
              help="Output flash image" ) )
   def dumpFlash( cls, config ): # pylint: disable-msg=no-self-argument
      """Dump raw flash image from switch"""

      with cls.recoverySession(
            config,
            openTty=True,
            recoveryMode=True ) as session:
         cc = ConsoleClient.fromTty( session.tty )
         cc.dumpFlash( config.flashImage )

   @subcommand(
      option( "--recovery",
              dest="recoveryMode", action="store_true", default=False,
              help="Start session in recovery mode" ),
      option( "--opentty",
              dest="openTty", action="store_true", default=False,
              help="Start session with an open TTY" ) )
   def debugSession( cls, config ): # pylint: disable-msg=no-self-argument
      """Launch Pdb in RecoverySession"""

      with cls.recoverySession(
            config,
            openTty=config.openTty,
            recoveryMode=config.recoveryMode ) as session:

         tty = session.tty
         t0( f"session.tty: {tty!r}" )
         resetGpo = session.gpos.resetGpo if session.gpos else None
         t0( f"resetGpo: {resetGpo!r}" )
         recoveryGpo = session.gpos.recoveryGpo if session.gpos else None
         t0( f"recoveryGpo: {recoveryGpo!r}" )

         # pylint: disable-msg=import-outside-toplevel
         import pdb

         # pylint: disable-msg=forgotten-debug-statement
         pdb.set_trace() # A4NOCHECK

         # pylint: disable-msg=unnecessary-pass
         pass # Use 'c(ont(inue))' to leave session gracefully, or q(uit) to abort.

   @classmethod
   def gpoPair( cls, config ):

      if config.supervisor and \
         ( config.resetGpo is None or config.recoveryGpo is None ):
         supervisor = Supervisor.lookup()

         if config.resetGpo is None:
            config.resetGpo = supervisor.resetGpo

         if config.recoveryGpo is None:
            config.recoveryGpo = supervisor.recoveryGpo

      if config.resetGpo is None or config.recoveryGpo is None:
         config.subcmd.sp.error(
            "Missing GpoPair arguments, "
            f"resetGpo={config.resetGpo}, "
            f"recoveryGpo={config.recoveryGpo}" )

      return GpoPair.fromDescStrings( config.resetGpo,
                                      config.recoveryGpo )

   @classmethod
   def openTty( cls, config ):

      if config.supervisor and config.ttyDevPath is None:
         supervisor = Supervisor.lookup()

         if supervisor.ttySysDev:
            config.ttyDevPath = Tty.lookup( supervisor.ttySysDev )
            assert config.ttyDevPath, \
               f"Failed to resolve ttySysDev={supervisor.ttySysDev!r} " \
               f"for SID {supervisor.sid!r}"

      if config.ttyDevPath is None:
         config.subcmd.sp.error(
            "Missing --tty argument for terminal connection" )

      return Tty.open( config.ttyDevPath )

   @classmethod
   def recoverySession( cls, config, openTty=True, recoveryMode=True ):
      tty = None
      if openTty:
         tty = cls.openTty( config )

      gpos = None
      if recoveryMode:
         gpos = cls.gpoPair( config )

      return RecoverySession( gpos, tty )

class Supervisor:

   def __init__( self,
                 sid,
                 resetGpo=None,
                 recoveryGpo=None,
                 ttySysDev=None ):
      self.sid = sid
      self.resetGpo = resetGpo
      self.recoveryGpo = recoveryGpo
      self.ttySysDev = ttySysDev

   MAP = {
      'OtterLake.*' : dict(
         resetGpo=str( ScdResetGpo.gpoDesc( "9f:00.0", 0x4000, pinId=8 ) ),
         recoveryGpo=str( ScdResetGpo.gpoDesc( "9f:00.0", 0x4000, pinId=7 ) ),
         ttySysDev="usb:3-4:1.0" )
   }

   class SidNotFound ( Exception ):
      pass

   @classmethod
   def lookup( cls, sid=MicrosemiLib.sid() ):
      sup = None
      if not hasattr( cls, '_lookup' ):
         for exp, kwargs in cls.MAP.items():
            if re.match( exp, sid ):
               sup = cls( sid, **kwargs )
               break
         if sup is None:
            raise cls.SidNotFound(
               "Failed to resolve {!r} for SID {!r}".format(
                  cls.__name__, sid ) )
         t0( "{}.lookup(sid={}): {!r}".format(
            cls.__name__, sid, sup ) )
         cls._lookup = sup
      else:
         sup = cls._lookup # pylint: disable-msg=no-member

      return sup

   def __str__( self ):
      return f"{self.__class__.__name__}({self.sid})"

   def __repr__( self ):
      return \
         "{}({})".format(
            self.__class__.__name__,
            ", ".join( [ f"{k}={v!r}"
                         for k, v in self.__dict__.items() ] ) )

if __name__ == "__main__":

   def main():
      try:
         RecoveryUtil.main()
      except RecoveryUtil.ArgParser.Exit as e:
         if e.status:
            print( e, file=sys.stderr )
         sys.exit( 1 if e.status else 0 )

   main()
