#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This MlagPlugin handles creating and destroying the reactors that
# observe vxlan configuration changes on the mlag peers.

from __future__ import absolute_import, division, print_function
import Tracing
import MlagMountHelper
from MlagShared import configCheckDir as CCD
t2 = Tracing.trace2

# subdirDeprecated collection is being deprecated from EOS.
# To be compatible with past/future releases, follow these
# recommendations:
# Do not introduce any new calls to newSubdirDeprecated
# i.e call newEntity to create an entity
# rather than a subdir.
# If looking up a subdir, first look in entityPtr collection
# if the key is not found, look in subdirDeprecated collection
# See AID/7616 for more details

def Plugin( ctx ):
   # Create local directories to be shared with peer.
   # subdirDeprecated collection is being deprecated from EOS.
   # do not introduce new calls to newSubdirDeprecated,
   # use newEntity to create an entity instead.
   # see message on top of this file regarding subdirDeprecated
   # lookup/creation.
   vxlanDir = ctx.configCheckP2pDir.newSubdirDeprecated( 'vxlan' )
   p2pVxlanConfig = vxlanDir.newEntity( "MlagP2p::ConfigCheck::GenericConfig",
                                        "vxlanConfig" )
   p2pVtiConfig = vxlanDir.newEntity( "MlagP2p::ConfigCheck::GenericConfig",
                                      "vtiConfig" )

   vtiSmContainer = ctx.configCheckDir.newEntity( 'Mlag::ConfigCheck::SmContainer',
                                                  'VtiConfigCheckSm' )
   vtiConfigCheckDesc = \
       ctx.configCheckDir.newEntity( 'Mlag::ConfigCheck::ConfigCheckDesc',
                                     'Vti' )
   intfConfigEntT = vtiConfigCheckDesc.newNestedConfigCheckDesc( 'vtiConfig' )
   intfConfigEntT.attrName[ 'udpPort' ] = 'udp-port'
   intfConfigEntT.attrName[ 'srcIpIntf' ] = 'source-interface'
   intfConfigEntT.attrName[ 'arpLocalAddress' ] = 'arp-ip-address-local'
   intfConfigEntT.collectionName[ 'vlanToVniMap' ] = 'vlan-to-vni'
   intfConfigEntT.collectionName[ 'vrfToVniMap' ] = 'vrf-to-vni'
   intfConfigEntT.attrName[ 'mlagSharedRouterMacConfig' ] = \
      'mlag-shared-router-mac-config'
   intfConfigEntT.attrName[ 'mlagSharedRouterMacAddr' ] = \
      'mlag-shared-router-mac-addr'
   intfConfigEntT.attrName[ 'vtepToVtepBridgingConfigured' ] = \
      'bridging vtep-to-vtep'
   intfConfigEntT.attrName[ 'vtepSourcePruningAll' ] = \
      'source-vtep tx disabled all vteps'
   intfConfigEntT.collectionName[ 'vtepSetForSourcePruning' ] = \
      'source-vtep tx disabled'
   intfConfigEntT.attrName[ 'floodLearnedAll' ] = 'flood vtep learned data-plane'

   vxlanSmContainer = ctx.configCheckDir.newEntity( 'Mlag::ConfigCheck::SmContainer',
                                                    'VxlanConfigCheckSm' )
   vxlanConfigCheckDesc = \
       ctx.configCheckDir.newEntity( 'Mlag::ConfigCheck::ConfigCheckDesc',
                                     'Vxlan' )
   vxlanIntfConfigEntT = \
       vxlanConfigCheckDesc.newNestedConfigCheckDesc( 'vxlanConfig' )
   vxlanIntfConfigEntT.collectionName[ 'floodVtepList' ] = 'flood vtep list'
   vxlanIntfConfigEntT.collectionName[ 'vniToIpAclMap' ] = 'vni to ip access list'

   # Mount from sysdb.
   mg = ctx.entityManager.mountGroup()
   intfConfig = mg.mount( 'interface/config/eth/vxlan', 'Vxlan::VtiConfigDir',
                          'r' )
   vxlanIntfConfig = mg.mount( 'vxlan/config', 'Vxlan::VxlanConfigDir', 'r' )
   # Mount mlag/config, Mlag::Config and its dependent paths
   mlagConfig = MlagMountHelper.mountMlagConfig( mg )
   mlagStatus = MlagMountHelper.mountMlagStatus( mg )

   def peerRoot():
      return ctx.mountStatus.peerRoot

   def activeSupervisor():
      return ctx.agent.runMode == 'mlagActive'

   def handleActive( mlagState, failover ):
      t2( "handle mlag state Active in VxlanConfigCheck" )
      if activeSupervisor() and peerRoot():
         try:
            peerVtiConfig = None
            peerVxlanConfig = None
            configCheck = CCD( peerRoot() )
            # subdirDeprecated collection is being deprecated from EOS.
            # to be compatible with past/future releases
            # first look up in default collection ( entityPtr )
            # see message on top of this file regarding subdirDeprecated
            # lookup/creation.
            if configCheck.get( 'vxlan' ):
               peerVxlanConfig = configCheck[ 'vxlan' ].entity[ 'vxlanConfig' ]
               peerVtiConfig = configCheck[ 'vxlan' ].entity[ 'vtiConfig' ]
            else:
               # lookup in subdirDeprecated collection
               peerVxlanConfig = configCheck.subdirDeprecated[ 'vxlan' ] \
                  .entity[ 'vxlanConfig' ]
               peerVtiConfig = configCheck.subdirDeprecated[ 'vxlan' ] \
                  .entity[ 'vtiConfig' ]
         except KeyError:
            # This is an ISSU case. We will not have a configChecker for Vxlan.
            return
         if vtiSmContainer.configCheckSm or vxlanSmContainer.configCheckSm:
            if mlagState == 'primary':
               if failover:
                  vxlanConfigCheckCleanup()
         else:
            vxlanConfigCheck = ctx.configCheckPlugin( "vxlan" )
            vtiConfigCheck = ctx.configCheckPlugin( "vti" )
            # pluginName is different from feature name for vti
            vtiConfigCheck.pluginName = "vxlan"
            vtiSmContainer.configCheckSm = ( vtiConfigCheck, 'Vti',
                                             intfConfig, p2pVtiConfig,
                                             peerVtiConfig,
                                             vtiConfigCheckDesc, mlagConfig,
                                             mlagStatus )
            vxlanSmContainer.configCheckSm = ( vxlanConfigCheck, 'Vxlan',
                                               vxlanIntfConfig, p2pVxlanConfig,
                                               peerVxlanConfig,
                                               vxlanConfigCheckDesc, mlagConfig,
                                               mlagStatus )
      elif not peerRoot():
         t2( "Deleting configcheckSm because peerRoot is none" )
         vxlanConfigCheckCleanup()

   def vxlanConfigCheckCleanup():
      vtiSmContainer.configCheckSm = None
      vxlanSmContainer.configCheckSm = None
      if activeSupervisor():
         ctx.configCheckPluginDel( "vxlan" )
         ctx.configCheckPluginDel( "vti" )

   def handleInactive():
      t2( "handle mlag state Inactive in VxlanConfigCheck" )
      vxlanConfigCheckCleanup()

   def finishMounts():
      t2( "Mounts complete" )

      def handleMlagStateForVxlanConfigCheck( mlagState, failover ):
         t2( "mlagState:", mlagState )
         if mlagState in ( "primary", "secondary" ):
            handleActive( mlagState, failover )
         else:
            handleInactive()
      ctx.callbackIs( handleMlagStateForVxlanConfigCheck )

   mg.close( finishMounts )
