# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function
import os
import re
import types
from six.moves import zip
import EosVersion

# This matches Release Number, Major Number and Minor number from the
# version or release name eg 'EOS-4.14.6FX.-INT.12345.SOMETHING' will 
# have release number 4, major number as 14 and minor number as 6.
# We ignore everything else as suffix.
#
versionMatch = re.compile( r'(\d+\.\d+\.\d+).*' )
verToVerFieldMatch = re.compile( r'(\d+)\.(\d+)\.(\d+).*' )
releaseToVerMatch = re.compile( r'EOS-(\d+\.\d+\.\d+).*' ) 
releaseToVerFieldMatch = re.compile( r'EOS-(\d+)\.(\d+)\.(\d+).*' ) 

etcDir = os.environ.get( "VERSION_ETC_DIR", "/etc" )

# ----------------------------------------------------------------------
# Misc Utility functions 
# ----------------------------------------------------------------------

# Convert a ver to release. Works on lists too.
# For example returns 'EOS-4.3.14' for v='4.3.14' and 
# [ 'EOS-4.12.4', 'EOS4.15.6' ] for v=[ '4.12.4', '4.15.6' ] 
#
def verToRelease( v ):
   if isinstance( v, list ):
      return [ 'EOS-' + i for i in v ] 
   else:
      return 'EOS-' + v

# Convert a given release list/item to corresponding verlist/ver.
# For example 'EOS-4.15.6' => '4.15.6'
def releaseToVer( rel ):
   if isinstance( rel, list ):
      return [ re.match( releaseToVerMatch, i ).group( 1 ) for i in rel ]
   else:
      return re.match( releaseToVerMatch, rel ).group( 1 )

# Matches just the release, major numbers of two version strings. This ignores the
# minor version. eg '4.13.45.345' is same as '4.13.67.89'
def releaseMajVersionEqual( v1, v2 ):
   m1 = re.match( verToVerFieldMatch, v1 )
   m2 = re.match( verToVerFieldMatch, v2 )
   assert m1 and m2
   releaseMaj1 = m1.group( 1, 2 )
   releaseMaj2 = m2.group( 1, 2 )
   return releaseMaj1 == releaseMaj2

def cmpVersion ( ver1, ver2 ):
   return cmpRelease( verToRelease( ver1 ), verToRelease( ver2 ) )

# Matches just the release, major, minor numbers of two release strings.
# For example EOS-4.13.45.234.SOMETHING is same EOS-4.13.45.897.FX-BLAH and
# returns as 0 equality. For other cases returns the 'int' value. This  
# can be used as sorting function helper.
def cmpRelease( ver1 , ver2 ):
   m1 = re.match( releaseToVerFieldMatch, ver1 ) 
   m2 = re.match( releaseToVerFieldMatch, ver2 )
   assert m1 and m2 
   for v1, v2 in zip( m1.groups(), m2.groups() ):
      if v1 != v2:
         return int( v1 ) - int( v2 )
   return 0

# Returns a list of all matching Release and  Maj versions in the 
# release list. This ignores minor version.
def getAllMatchingVersions( ver, releaseList, generateRelease=True ):
   matchedVersions = []
   for i in releaseList:
      v = releaseToVer( i )
      if releaseMajVersionEqual( ver, v ):
         if generateRelease:
            matchedVersions.append( i )
         else:
            matchedVersions.append( v )
   return matchedVersions

def getCompatReleasesFromMlagData( mlagCompatData ):
   imageList = []
   if mlagCompatData:
      try:
         newMlagCompMod = types.ModuleType( 'dummyMlagCompatibilityMod' )
         exec( mlagCompatData, newMlagCompMod.__dict__ )
         # pylint: disable-next=consider-using-dict-items
         imageList = [ newMlagCompMod.__dict__[ s ] for s in 
                         newMlagCompMod.__dict__ 
                         if re.match( r'v\d+Images', s ) ]
      except Exception: # pylint: disable=broad-except
         # ignore all exceptions.
         pass
   return imageList

# Return version and compatible images as reported by the swi.
def getSwiVersionAndCompatibleImages( swiFile ):
   import zipfile # pylint: disable=import-outside-toplevel
   newVersion = None
   imageList = []
   try:
      with zipfile.ZipFile( swiFile ) as zp:
         vers = zp.read( 'version' ) if zp.getinfo( 'version' ) else None
         if vers:
            newVersion = EosVersion.VersionInfo( None, versionFile=vers ).\
                           version()
         mlagCompatData = zp.read( 'MlagCompatibility.data' ) if zp.getinfo( 
                           'MlagCompatibility.data' ) else None
         imageList = getCompatReleasesFromMlagData( mlagCompatData )
   except(  zipfile.BadZipfile , KeyError ):
      pass
   # flatten it
   imageList = [ i for y in imageList for i in y ] 
   return ( newVersion, imageList )

# Returns Mlag compatible images from our MlagCompatibility data
def getCompatibleImages():
   imageList = None
   mlagData = os.path.join( etcDir, 'MlagCompatibility.data' )
   with open( mlagData ) as f:
      mlagCompDataContent = f.read()
      imageList = getCompatReleasesFromMlagData( mlagCompDataContent )
      imageList = [ i for y in imageList for i in y ] 
   return imageList

#
# Suggest compatible image(s) if possible:
# These are possible permutation/combinations:
# case -1: No compatibility data in current image. Impossible case.
# case 0: No Comp data in new image. possible only in Downgrade case. 
#         Return matching release/major from compat set to newimage. 
#         There should be just one in normal case, but we don't care.
# case 1: if compatibility set overlaps then return all compatible images + other
# case 2: 
#    2.1: For Upgrade case, return matching release/major version from cur
#          comp set to newImage and compatible version from newImage compat data
#          to match cur image. Either case will be 2 hops away.
#    2.2: For downgrade case, do the same as above and we don't order in any 
#         particular way.
#
def suggestCompatibleVersions( curVer, curCompReleases, swiVer, swiCompReleases ):
   sugImages = set()
   myImages = set( curCompReleases )
   otherImages = set( swiCompReleases )
   compSet = myImages & otherImages
   if compSet:
      # These are all common images in compatibility data.
      sugImages |=  compSet
   # return all possible matches based on compatibility data analysis too. 
   sugImages |= set ( getAllMatchingVersions( swiVer, myImages ) + \
                      getAllMatchingVersions( curVer, otherImages ) ) 
   sugImages = sorted( sugImages, cmpRelease )
   return sugImages

