#!/usr/bin/env python3
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Plugins
import SharedMem
import Smash
import Tac
import Tracing
import MlagMountHelper

t2 = Tracing.trace2

# subdirDeprecated collection is being deprecated from EOS.
# To be compatible with past/future releases, follow these
# recommendations:
# Do not introduce any new calls to newSubdirDeprecated
# i.e call newEntity to create an entity
# rather than a subdir.
# If looking up a subdir, first look in entityPtr collection
# if the key is not found, look in subdirDeprecated collection
# See AID/7616 for more details

MLAG_ASU_BIT = Tac.Type( "Mlag::FailoverCauseBitMask" ).mlagAsu
MAINT_MODE_BIT = Tac.Type( "Mlag::FailoverCauseBitMask" ).maintMode

@Plugins.plugin( provides=( 'local/hostTable', ) )
def Plugin( ctx ):
   agent = ctx.agent

   # subdirDeprecated collection is being deprecated from EOS.
   # do not introduce new calls to newSubdirDeprecated,
   # use newEntity to create an entity instead.
   # see message on top of this file regarding subdirDeprecated
   # lookup/creation.

   #
   # Create the Entities in /ar/mlag/local that will be sent to the peer
   #
   # interface/status/eth
   intfDir = ctx.localDir.newSubdirDeprecated( 'interface' ) \
      .newSubdirDeprecated( 'status' )
   p2pLocalIntfDir = intfDir.newEntity( "MlagP2p::EthIntfStatusDir", "eth" )

   # interface/config/eth
   intfCfgDir = ctx.localDir.newSubdirDeprecated( 'interface' ) \
      .newSubdirDeprecated( 'config' )
   intfCfgDir.newEntity( "MlagP2p::EthIntfConfigDir", "eth" )

   # bridging/config
   bridgingDir = ctx.localDir.newSubdirDeprecated( 'bridging' )
   lbc = bridgingDir.newEntity( "MlagP2p::Bridging::Config", "config" )
   p2pVlanStatus = bridgingDir.newEntity( "MlagP2p::Bridging::VlanStatus",
                                          "vlanStatus" )
   p2pHostTable = ctx.localDir.newEntity( "MlagP2p::HostTable", "hostTable" )

   mg = ctx.entityManager.mountGroup()

   # bridging/status
   shmemEm = SharedMem.entityManager( sysdbEm=ctx.entityManager )
   smi = Smash.mountInfo( 'shadow' )
   bs = shmemEm.doMount( 'bridging/status', 'Smash::Bridging::Status', smi )

   # bridging
   #
   # Now mount Entities from Sysdb
   # 
   # Mount mlag/config, Mlag::Config and its dependent paths
   mlagCfg = MlagMountHelper.mountMlagConfig( mg )
   bc = mg.mount( 'bridging/config', 'Bridging::Config', 'r' )
   bim = mg.mount( 'bridging/input/config/mlag',
                   'Bridging::Input::Config', 'w' )
   isep = mg.mount( 'interface/status/eth/peer',
                    'Interface::PeerIntfStatusDir', 'w' )
   isei = mg.mount( 'interface/status/eth/intf',
                    'Interface::EthIntfStatusDir', 'r' )
   sysdbHostTable = mg.mount( 'mlag/hostTable', 'Mlag::HostTable', 'w' )
   asic = mg.mount( 'bridging/switchIntfConfig', 
                    'Bridging::SwitchIntfConfigDir', 'r' )
   sysdbVlanStatus = mg.mount( 'bridging/vlan/status',
                               'Bridging::VlanStatusDir', 'r' )
   lagConfig = mg.mount( 'lag/config', 'Lag::Config', 'r' )
   mg.mount( 'lag/input/config/cli', 'Lag::Input::Config', 'r' )
   mg.mount( 'interface/config/eth/lag', 'Interface::EthLagIntfConfigDir', 'r' )
   vMacConfigDir = mg.mount( 'mlag/virtualMac/config', 
                             'Mlag::VirtualMacConfigDir', 'r' )
   l2RibSource = mg.mount( 'bridging/l2Rib/inputDir/mlag',
                           'L2Rib::SourceDirEntry', 'w' )
   portIdDir = mg.mount( 'interface/eth/portid', 'Interface::EthIntfPortIdDir', 'r' )
   mlagFmInputDir = mg.mount( 'interface/input/forwardingmodel/mlag',
                              'Interface::IntfForwardingModelInputDir', 'w' )
   managedIntfList = mg.mount( 'interface/status/managedIntfList',
                               'Interface::IntfStatusPtrDir', 'w' )
   mlagStatus = ctx.status

   def peerRoot():
      return ctx.mountStatus.peerRoot

   def activeSupervisor():
      return ctx.agent.runMode == 'mlagActive'

   def handleActive( mlagState, failover ):
      peerLinkIntf = ctx.status.peerLinkIntf
      t2( 'handleActive called with state', mlagState,
            'failover', failover, 'mlagStatus::failoverCause',
            mlagStatus.failoverCause )

      if not agent.attrLogUtil:
         t2( "Creating AttrLogUtil" )
         agent.attrLogUtil = ( ctx.entityManager.cEm_, )

      if agent.bridgingSysdbToP2pSm:
         # Already primary or secondary

         if mlagState == 'primary':
            if agent.bridgingP2pToSysdbSm:
               t2( "Failover changed. Update peer state." )
               agent.bridgingP2pToSysdbSm.handleFailover( failover )
               agent.bridgingConfigP2pToSysdbSm.handleFailover( failover )
               if ( failover or ( mlagStatus.failoverCause & MLAG_ASU_BIT ) or
                    ( ( mlagStatus.failoverCause & MAINT_MODE_BIT ) and
                      not mlagStatus.fastMacRedirection ) ):
                  agent.peerHostTableSm.handleFailover()
                  # Do nothing if we're coming out of failover.
                  # peerHostTableSm will get a peerOffsetValid
                  # notification.
                  if not agent.peerMacPromotionSm:
                     t2( "Creating PeerMacPromotionSm" )
                     agent.peerMacPromotionSm = ( Tac.activityManager.clock,
                                                  bs, ctx.status, sysdbHostTable )
               else:
                  # Reset peerMacPromotionSm now.
                  t2( "Deleting PeerMacPromotionSm" )
                  agent.peerMacPromotionSm = None
      else:
         t2( "Creating VlanStatusSm" )
         agent.vlanStatusSm = ( Tac.activityManager.clock, sysdbVlanStatus,
                                peerLinkIntf )

         t2( "Creating BridgingSysdbToP2pSm" )
         agent.bridgingSysdbToP2pSm = (
            isei, peerLinkIntf, lagConfig, p2pLocalIntfDir,
            agent.vlanStatusSm.vlanStatus,
            p2pVlanStatus, ctx.protoStatus )

         t2( "Syncing Bridging::Config to p2p" )
         agent.bridgingConfigSysdbToP2pSm = (
            bc, asic, bim, agent.intfFilter.filteredIntf, ctx.protoStatus, lbc )

         t2( "Creating VirtualMacTable and VirtualMacConfigDirSm" )
         agent.virtualMacTable = Tac.newInstance( 'Mlag::Agent::VirtualMacTable',
                                                  'virtualMacTable' )
         agent.virtualMacConfigDirSm = ( vMacConfigDir, agent.virtualMacTable )

         t2( "Creating HostTableSm" )
         agent.hostTableSm = ( Tac.activityManager.clock,
                               bc, bs, p2pHostTable, mlagCfg,
                               ctx.status, ctx.protoStatus, agent.agentImpl,
                               peerLinkIntf, agent.vlanStatusSm.vlanStatus,
                               agent.virtualMacTable, agent.attrLogUtil )

      if not activeSupervisor() or agent.bridgingConfigP2pToSysdbSm \
             or not peerRoot():
         return

      # Set switchIntfConfigPriority to same value that Ebra uses
      # (there should be no conflicts)
      bim.switchIntfConfigPriority = 100

      t2( "Syncing Bridging::Config to Sysdb from p2p" )
      peerRootBridging = None
      # subdirDeprecated collection is being deprecated from EOS.
      # to be compatible with past/future releases 
      # first look up in default collection ( entityPtr )
      # see message on top of this file regarding subdirDeprecated
      # lookup/creation.
      if peerRoot().get( 'bridging' ):
         peerRootBridging = peerRoot()['bridging']
      else:
         # lookup in subdirDeprecated collection
         peerRootBridging = peerRoot().subdirDeprecated['bridging']

      pbc = peerRootBridging.entity['config']
      agent.bridgingConfigP2pToSysdbSm = ( pbc, bim, ctx.protoStatus )

      t2( "Creating BridgingP2pToSysdbSm" )
      peerRootInterface = None
      # subdirDeprecated collection is being deprecated from EOS.
      # to be compatible with past/future releases 
      # first look up in default collection ( entityPtr )
      # see message on top of this file regarding subdirDeprecated
      # lookup/creation.
      if peerRoot().get( 'interface' ):
         peerRootInterface = peerRoot()['interface']
      else:
         # lookup in subdirDeprecated collection
         peerRootInterface = peerRoot().subdirDeprecated['interface']

      peerRootInterfaceStatus = None
      # subdirDeprecated collection is being deprecated from EOS.
      # to be compatible with past/future releases 
      # first look up in default collection ( entityPtr )
      # see message on top of this file regarding subdirDeprecated
      # lookup/creation.
      if peerRootInterface.get( 'status' ):
         peerRootInterfaceStatus = peerRootInterface['status']
      else:
         # lookup in subdirDeprecated collection
         peerRootInterfaceStatus = peerRootInterface.subdirDeprecated['status']

      p2pPeerIntfDir = peerRootInterfaceStatus.entity['eth']
      agent.bridgingP2pToSysdbSm = ( p2pPeerIntfDir, isep, ctx.protoStatus,
                                     agent.kniRoot, portIdDir, mlagFmInputDir,
                                     managedIntfList )

      t2( "Creating PeerHostTableSm" )
      peerHostTable = peerRoot()[ 'hostTable' ]
      peerVlanStatus = peerRootBridging.entity[ 'vlanStatus' ]
      collectionInfo = []
      collectionInfo.append( ( 'host', agent.l2RibHostTableLen ) )
      l2RibSmi = Smash.mountInfo( 'writer', collectionInfo=collectionInfo )
      l2RibInput = shmemEm.doMount( agent.l2RibHostTableSmashPath,
                                    'L2Rib::HostInput', l2RibSmi )
      l2RibSource.hostTableSmashPath = agent.l2RibHostTableSmashPath
      l2RibSource.hostTableLen = agent.l2RibHostTableLen
      l2RibSource.source = 'sourceMlag'
      l2RibSource.initialized = True
      agent.peerHostTableSm = ( Tac.activityManager.clock, 
                                peerHostTable, p2pHostTable,
                                peerVlanStatus,  ctx.status,
                                ctx.protoStatus, sysdbHostTable,
                                mlagCfg, bs, agent.agentImpl, peerLinkIntf,
                                agent.vlanStatusSm.vlanStatus,
                                agent.virtualMacTable,
                                agent.attrLogUtil )
      agent.hostTableSm.peerHostTableSm = agent.peerHostTableSm
      agent.l2RibInput = l2RibInput

      l2RibOutSmi = Smash.mountInfo( 'reader', collectionInfo=collectionInfo )
      l2RibHostOutput = shmemEm.doMount( 'bridging/l2Rib/hostOutput',
                                         'L2Rib::HostOutput', l2RibOutSmi )
      agent.l2RibHostOutput = l2RibHostOutput

      t2( "Creating L2RibHostTableSm" )
      agent.l2RibHostTableSm = ( Tac.activityManager.clock,
                                 sysdbHostTable, l2RibInput, l2RibHostOutput )

   def handleFailoverCause( state, failoverCause ):
      t2( 'handleFailoverCause called with state', state,
          'mlagStatus::failoverCause', mlagStatus.failoverCause,
          'mlagStatus::failoverInitiated', mlagStatus.failoverInitiated )

      # Only create PeerMacPromotionSm on active supervisor and only
      # if this peer didn't initiate the failover
      if activeSupervisor() and not mlagStatus.failoverInitiated:
         if ( state == 'primary' and
              ( failoverCause & MLAG_ASU_BIT or
                ( ( failoverCause & MAINT_MODE_BIT ) and
                  not mlagStatus.fastMacRedirection ) ) ):
            if not agent.peerMacPromotionSm:
               t2( "Creating PeerMacPromotionSm for mlagAsu failoverCause" )
               agent.peerMacPromotionSm = ( Tac.activityManager.clock,
                                            bs, ctx.status, sysdbHostTable ) 
         elif state == 'secondary' or failoverCause == 0:
            t2( "Deleting PeerMacPromotionSm for unknown failoverCause" )
            agent.peerMacPromotionSm = None

   def handleInactive():
      t2( "Clean up bridging-related state" )
      agent.bridgingConfigSysdbToP2pSm = None
      agent.bridgingConfigP2pToSysdbSm = None
      agent.doP2pBridgingConfigCleanup( lbc )
      agent.bridgingSysdbToP2pSm = None
      agent.bridgingP2pToSysdbSm = None
      agent.vlanStatusSm = None
      agent.doP2pIntfCleanup( p2pLocalIntfDir )
      agent.hostTableSm = None
      agent.peerHostTableSm = None
      agent.virtualMacConfigDirSm = None
      agent.virtualMacTable = None
      agent.peerMacPromotionSm = None
      agent.l2RibHostTableSm = None
      if activeSupervisor():
         t2( "Redundancy mode is active. Clean up L2Rib smash and Sysdb." )
         agent.doL2RibInputCleanup()
         l2RibSource.initialized = False
         shmemEm.doUnmount( agent.l2RibHostTableSmashPath )
         bim.switchIntfConfigPriority = 0
         agent.doPeerIntfCleanup( isep )
         agent.doP2pHostTableCleanup( p2pHostTable )
         agent.doPeerBridgingConfigCleanup( bim )
         agent.doPeerHostTableCleanup( sysdbHostTable )

   def finishMounts():
      def handleMlagStateForBridging( mlagState, failover ):
         if mlagState in ( 'primary', 'secondary' ):
            handleActive( mlagState, failover )
         else:
            handleInactive()
      ctx.callbackIs( handleMlagStateForBridging )
      ctx.failoverCauseCallbackIs( handleFailoverCause )

   mg.close( finishMounts )
