# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from Toggles import VxlanToggleLib

import os
import fcntl
import Plugins
import SharedMem
import Smash
import Tac
import Tracing
import QuickTrace
import MlagMountHelper

qv = QuickTrace.Var
qt2 = QuickTrace.trace2
t2 = Tracing.trace2

# subdirDeprecated collection is being deprecated from EOS.
# To be compatible with past/future releases, follow these
# recommendations:
# Do not introduce any new calls to newSubdirDeprecated
# i.e call newEntity to create an entity
# rather than a subdir.
# If looking up a subdir, first look in entityPtr collection
# if the key is not found, look in subdirDeprecated collection
# See AID/7616 for more details

@Plugins.plugin( requires=( 'local/hostTable', ) )
def Plugin( ctx ):
   # Create /ar/mlag/mlagVxlan
   # subdirDeprecated collection is being deprecated from EOS.
   # do not introduce new calls to newSubdirDeprecated,
   # use newEntity to create an entity instead.
   # see message on top of this file regarding subdirDeprecated
   # lookup/creation.
   mlagVxlanDir = ctx.agent.parent.newSubdirDeprecated( 'mlagVxlan' )
   # Instantiate smContainer at /ar/mlag/mlagVxlan/stateMachine
   smC = mlagVxlanDir.newEntity( 'MlagVxlan::SmContainer', 'stateMachine' )
   p2pVxlanDir = ctx.localDir.newSubdirDeprecated( 'vxlan' )
   p2pVxlanStatus = p2pVxlanDir.newEntity( 'MlagP2p::Vxlan::Status', 'status' )
   p2pHostTable = ctx.localDir[ 'hostTable' ]

   vlanType = Tac.Type( 'Bridging::VlanId' )
   for vlan in range( vlanType.min, vlanType.max + 1 ):
      p2pVxlanStatus.sharedDynVlan[ vlan ] = True
   p2pVxlanStatus.arpByVlanIpCapable = True
   p2pVxlanStatus.scheduledMlagPeerArpSync = \
         VxlanToggleLib.toggleVxlanScheduledMlagPeerArpSyncEnabled()

   # Mount stuff from Sysdb
   mg = ctx.entityManager.mountGroup()
   sysdbHostTable = mg.mount( 'mlag/hostTable', 'Mlag::HostTable', 'r' )
   vxlanStatus = mg.mount( 'mlag/vxlan/status', 'Mlag::VxlanStatus', 'w' )
   controllerStatusBaseDir = mg.mount( 'vxlancontroller/version2/vni',
                                       'Tac::Dir', 'ri' )
   vtiStatusDir = mg.mount( 'interface/status/eth/vxlan',
                            'Vxlan::VtiStatusDir', 'r' )
   vtepStatusDir = mg.mount( 'vxlan/version2/vtepStatusDir',
                             'VxlanController::VtepStatusDirV2', 'r' )
   arpInputConfig = mg.mount( 'arp/input/config/vxlan',
                              'Arp::InputConfig', 'r' )
   # VxlanSwFwd agent is conditional so make sure we pass the "O" flag
   # to avoid Mlag ArcherEm observing mount timeouts.
   activeArpIpTable = mg.mount( 'vxlan/activeArpIpTable', 'Vxlan::ActiveArpIpTable',
                                'rO' )

   # bfd/status/peer
   bfdStatusPeer = mg.mount( 'bfd/status/peer', 'Bfd::StatusPeer', 'r' )
   bfdLauncherControl = mg.mount( 'bfd/launcherControl', 'Tac::Dir', 'wi' )

   # bridging/status
   shmemEm = SharedMem.entityManager( sysdbEm=ctx.entityManager )
   smi = Smash.mountInfo( 'shadow' )
   bridgingStatus = shmemEm.doMount( 'bridging/status', 'Smash::Bridging::Status',
                                     smi )
   smi = Smash.mountInfo( 'shadow' )
   vxlanFdbStatus = shmemEm.doMount( 'vxlan/hardware/fdbStatus',
                                     'Smash::Vxlan::FdbStatus',
                                     smi )

   # bridging/config
   bridgingConfig = mg.mount( 'bridging/config', 'Bridging::Config', 'r' )

   # routing l2-vpn
   l2VpnConfig = mg.mount( 'routing/l2vpn/config', "Routing::L2Vpn::Config", "r" )

   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagMountHelper.mountMlagStatus( mg )

   # Flood-list synchronization
   vcsSde = mg.mount( 'bridging/l2Rib/inputDir/vcs', 'L2Rib::SourceDirEntry', 'r' )

   def peerRoot():
      return ctx.mountStatus.peerRoot

   def activeSupervisor():
      return ctx.agent.runMode == 'mlagActive'

   def setVxPeerMacAddr( macAddr ):
      t2( 'Setting vxpeermac : %s' % macAddr )
      vxPeerMacStr = "vxpeermac %s" % macAddr
      # Pad it with "\0" until the total length is 128 which is required for
      # the ioctl call
      cmd = vxPeerMacStr + '\0' * ( 128 - len( vxPeerMacStr ) )
      fd = None
      try:
         fd = os.open( '/dev/fabric', os.O_RDWR )
         fcntl.ioctl( fd, 128, cmd )
      except OSError as e:
         qt2( "Setting vxpeermac failed with exception : %s" % str( e ) )
      finally:
         if fd is not None:
            try:
               os.close( fd )
            except OSError as e:
               qt2( "Closing fd : %d failed with exception : %s" % ( fd, str( e ) ) )

   def handleActive( mlagState, failover ):
      # tell dma-driver about the mlag peer to enable peer DAD frame suppression
      # for ipv6 address virtual configs
      setVxPeerMacAddr( mlagStatus.peerMacAddr )
      if not smC.remoteHostTableSm:
         t2( "Creating RemoteHostTableSm" )
         smC.remoteHostTableSm = ( p2pHostTable, sysdbHostTable, bridgingConfig,
                                   bridgingStatus, vxlanFdbStatus, ctx.protoStatus,
                                   ctx.status, controllerStatusBaseDir,
                                   vtiStatusDir )

      if not activeSupervisor() or not peerRoot():
         t2( activeSupervisor(), " ", peerRoot() )
         qt2( "active supervisor:", qv( activeSupervisor() ),
              "peerRoot:", qv( peerRoot() ) )
         if not peerRoot():
            if smC.activeArpP2pToSysdbSm:
               t2( "delete activeArpP2pToSysdbSm" )
               qt2( "delete activeArpP2pToSysdbSm" )
               smC.activeArpP2pToSysdbSm = None
            if smC.scheduledMlagPeerArpSyncSm:
               t2( "delete scheduledMlagPeerArpSyncSm" )
               qt2( "delete scheduledMlagPeerArpSyncSm" )
               smC.scheduledMlagPeerArpSyncSm = None
         return

      smC.remoteHostTableSm.recvP2pHostTable = peerRoot()[ 'hostTable' ]

      peerRootVxlan = None
      peerVxlanStatus = None
      # subdirDeprecated collection is being deprecated from EOS.
      # to be compatible with past/future releases
      # first look up in default collection ( entityPtr )
      # see message on top of this file regarding subdirDeprecated
      # lookup/creation.
      if peerRoot().get( 'vxlan' ):
         peerRootVxlan = peerRoot()[ 'vxlan' ]
      else:
         # lookup in subdirDeprecated collection
         peerRootVxlan = peerRoot().subdirDeprecated.get( 'vxlan' )

      if peerRootVxlan is not None:
         peerVxlanStatus = peerRootVxlan.entity[ 'status' ]

      if not smC.activeArpP2pToSysdbSm:
         if hasattr( peerVxlanStatus, 'activeArpIpTable' ):
            smC.activeArpP2pToSysdbSm = ( vtiStatusDir, l2VpnConfig,
                                          activeArpIpTable,
                                          peerVxlanStatus.activeArpIpTable, # in
                                          vxlanStatus.activeArpIpTable, # out
                                          activeArpIpTable, #dest state
                                          None, #rx dest state
                                          'activeArpP2pToSysdb' )
      if not smC.activeArpSysdbToP2pSm:
         smC.activeArpSysdbToP2pSm = ( vtiStatusDir, l2VpnConfig,
                                       activeArpIpTable,
                                       activeArpIpTable, # in
                                       p2pVxlanStatus.activeArpIpTable, # out
                                       peerVxlanStatus.activeArpIpTable, #dest state
                                       vxlanStatus.activeArpIpTable, #rx dest state
                                       'activeArpSysdbToP2p' )

      if not smC.scheduledMlagPeerArpSyncSm:
         smC.scheduledMlagPeerArpSyncSm = ( p2pVxlanStatus, peerVxlanStatus,
                                            vxlanStatus )

      if mlagState == 'primary':
         t2( "Mlag state is 'primary'" )
         qt2( "Mlag state is 'primary'" )
         # We do not care if it is in failover window i.e., the peer is down.
         # We just keep pushing received macs to p2p mount

         t2( "Creating vcsSdeSysdbToBumStatusP2pSm" )
         if not smC.vcsSdeSysdbToBumStatusP2pSm:
            smC.vcsSdeSysdbToBumStatusP2pSm = ( vcsSde,
                                                p2pVxlanStatus.bumStatus )

         t2( "Creating vcsSdeSysdbToP2pSm" )
         if not smC.vcsSdeSysdbToP2pSm:
            p2pVxlanStatus.vcsInputSupported = True
            smC.vcsSdeSysdbToP2pSm = ( vcsSde,
                                       p2pVxlanStatus.vcsSde, 'sysdbToP2p' )
         if smC.floodListInputSelectorSm:
            t2( "Destroying floodListInputSelectorSm" )
            smC.floodListInputSelectorSm.doCleanup()
            smC.floodListInputSelectorSm = None

         if smC.secondarySm:
            t2( "Destroying smCSecondarySm" )
            smC.secondarySm.doCleanup()
            smC.secondarySm = None

         if not smC.primarySm:
            t2( "Mlag state is 'primary'. Creating primarySm." )
            if peerRootVxlan is None or peerVxlanStatus is None:
               t2( "'vxlan/status' path not present on peer." )
               vxlanStatus.bfdSupported = False
               return
            if not hasattr( peerVxlanStatus, 'vniToDynVlanMap' ):
               t2( "'vxlan/status' has no vniToDynVlanMap support on peer." )
               vxlanStatus.bfdSupported = False
               return
            if not hasattr( peerVxlanStatus, 'arpStatus' ):
               t2( "'vxlan/status' has no arpStatus support on peer." )
               return
            vxlanStatus.bfdSupported = True
            smC.primarySm = ( p2pVxlanStatus, peerVxlanStatus, vxlanStatus,
                              vtiStatusDir, vtepStatusDir,
                              arpInputConfig, bfdStatusPeer )
      else:
         assert mlagState == 'secondary'
         t2( "Mlag state is 'secondary'" )
         qt2( "Mlag state is 'secondary'" )
         if peerRootVxlan is None or peerVxlanStatus is None:
            t2( "'vxlan/status' path not present on peer." )
            vxlanStatus.bfdSupported = False
            return

         # Destroy SMs that only run on primary
         smC.vcsSdeSysdbToBumStatusP2pSm = None
         smC.vcsSdeSysdbToP2pSm = None

         # Create SM to sync from primary to local VCS SDE.
         if not smC.floodListInputSelectorSm:
            smC.floodListInputSelectorSm = ( peerVxlanStatus,
                                             peerVxlanStatus.bumStatus,
                                             peerVxlanStatus.vcsSde,
                                             vxlanStatus.vcsSde, 'p2pToSysdb' )

         if smC.primarySm:
            t2( 'Destroying smcPrimarySm' )
            smC.primarySm.doCleanup()
            smC.primarySm = None

         if not smC.secondarySm:
            t2( "Mlag state is 'secondary'. Creating secondarySm." )
            if not hasattr( peerVxlanStatus, 'vniToDynVlanMap' ):
               t2( "'vxlan/status' has no vniToDynVlanMap support on peer." )
               vxlanStatus.bfdSupported = False
               return
            if not hasattr( peerVxlanStatus, 'arpStatus' ):
               t2( "'vxlan/status' has no arpStatus support on peer." )
               return
            vxlanStatus.bfdSupported = True
            smC.secondarySm = ( p2pVxlanStatus, peerVxlanStatus, vxlanStatus,
                                bridgingConfig, vtiStatusDir, vtepStatusDir,
                                mlagStatus )

   def handleInactive():
      t2( "Clean up state" )
      setVxPeerMacAddr( "00:00:00:00:00:00" )
      if smC.remoteHostTableSm:
         smC.remoteHostTableSm.doCleanup()
         smC.remoteHostTableSm = None
      if smC.vcsSdeSysdbToBumStatusP2pSm:
         smC.vcsSdeSysdbToBumStatusP2pSm.doCleanup()
         smC.vcsSdeSysdbToBumStatusP2pSm = None
      if smC.vcsSdeSysdbToP2pSm:
         smC.vcsSdeSysdbToP2pSm.doCleanup()
         smC.vcsSdeSysdbToP2pSm = None
      if smC.floodListInputSelectorSm:
         smC.floodListInputSelectorSm.doCleanup()
         smC.floodListInputSelectorSm = None
      if smC.secondarySm:
         smC.secondarySm.doCleanup()
         smC.secondarySm = None
      if smC.primarySm:
         smC.primarySm.doCleanup()
         smC.primarySm = None
      if smC.activeArpP2pToSysdbSm:
         smC.activeArpP2pToSysdbSm = None
      if smC.activeArpSysdbToP2pSm:
         smC.activeArpSysdbToP2pSm = None
      if smC.scheduledMlagPeerArpSyncSm:
         smC.scheduledMlagPeerArpSyncSm = None

   def finishMounts():
      t2( "Mounts complete" )
      if activeSupervisor():
         t2( "re-populating sharedDynVlan" )
         vlanType = Tac.Type( 'Bridging::VlanId' )
         for vlan in range( vlanType.min, vlanType.max + 1 ):
            vxlanStatus.sharedDynVlan[ vlan ] = True

      def handleMlagStateForVxlan( mlagState, failover ):
         if mlagState in ( 'primary', 'secondary' ):
            handleActive( mlagState, failover )
         else:
            handleInactive()
      ctx.callbackIs( handleMlagStateForVxlan )

   mg.close( finishMounts )
