#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from __future__ import absolute_import, division, print_function
import six
import struct
import Tracing

t0 = Tracing.trace0

class TlvError( Exception ):
   pass

# Known TLV types (16-bit, all starting with 'M')
TLV_TYPE_MLAG_STATE = 0x4D01 # mlag state in strings
TLV_TYPE_LOCAL_PORT_COUNT = 0x4D02 # number of linkUp/linkDown local ports

# Basic TLV support
# TLVs are aligned to word boundaries (4-byte)
def align( length ):
   return ( length + 3 ) // 4 * 4

def tlvs( data ):
   # assuming data is already aligned
   offset = 0
   while offset < len( data ):
      try:
         tlvType, length = struct.unpack( '!HH', data[ offset : offset + 4 ] )
         # pylint: disable-next=consider-using-f-string
         value = struct.unpack( '!%is' % length, 
                                data[ offset + 4 : offset + 4 + length ] )[0]
         t0( "TLV type", tlvType, "length", length )
      except struct.error:
         t0( "Unproper TLV structure found:", data[ offset: ],
             "size", len( data ) - offset )
         break
      yield tlvType, value
      # align to word boundary
      offset += 4 + align( length )

def encodeTlv( tlvType, data ):
   # word aligned
   # pylint: disable-next=consider-using-f-string
   padding = struct.pack( '!%dx' % ( align( len( data ) ) - len( data ), ) )
   return struct.pack( '!HH', tlvType, len( data ) ) + \
      six.ensure_binary( data ) + padding

def mlagStateEncode( status ):
   return str( status.mlagState )

def localPortCountEncode( status ):
   upCount = 0
   downCount = 0
   for intfStatus in six.itervalues( status.intfStatus ):
      if intfStatus.localLinkStatus == 'linkUp':
         upCount += 1
      else:
         downCount += 1
   return struct.pack( '!ii', upCount, downCount )
