#!/usr/bin/env python3
# Copyright (c) 2009-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=attribute-defined-outside-init
# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
import Agent, Tracing
from GenericReactor import GenericReactor
import Tac, os, socket, errno, Cell, SharedMem, Smash
from MlagShared import MLAG_PORT, MLAG_TTL, MLAG_TOS, IP_RECVTTL
from if_ether_arista import TC_PRIO_CONTROL
from Arnet.NsLib import DEFAULT_NS
import MlagMountHelper
import six
import QuickTrace

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

Tac.activityManager.useEpoll = True
qv = QuickTrace.Var
qt0 = QuickTrace.trace0

VrfName = Tac.Type( "L3::VrfName" )
IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )
Ipv6Address = Tac.Type( "Arnet::AddressFamily" ).ipv6

class VrfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatusLocal, master ):
      self.master_ = master
      self.vrfName = vrfStatusLocal.vrfName
      self.vrfStatusLocal = vrfStatusLocal
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      # we need to know about create as well
      self.master_.handleVrfState( self.vrfName )

   @Tac.handler( 'state' )
   def handleState( self ):
      self.master_.handleVrfState( self.vrfName )

class MlagTunnel( Agent.Agent ):
   """ The MlagTunnel agent allows an agent on one MLAG peer to send
   and/or receive protocol packets on the other peer's interfaces. """

   def __init__( self, entityManager ):
      self.mlagTunnelSm_ = None
      self.tapPamManager_ = None
      self.mlagConstants_ = Tac.Value( "Mlag::Constants" )
      self.udpSocketBoundTo_ = ""
      self.mlagConfigReactor_ = None
      self.mlagStatusReactor_ = None
      self.mlagTunnelStatusUpdater_ = None
      self.redundancyModeReactor_ = None
      self.localInterfaceStatusReactor_ = None
      self.kernelIntfStatusReactor_ = None
      self.localIntfStatusManager = None
      self.mlagStatus_ = None
      self.mlagConfig_ = None
      self.mlagTunnelStatus_ = None
      self.mlagTunnelCellStatus_ = None
      self.mlagTunnelConfig_ = None
      self.peerIntfs_ = None
      self.intfStatus_ = None
      self.intfStatusLocal_ = None
      self.ethIntfs_ = None
      self.cleanupActivity_ = None
      self.kniStatus_ = None
      self.netNs_ = None
      self.shmemEm_ = None

      Agent.Agent.__init__( self, entityManager )

      self.tunnelAgentRoot_ = self.agentRoot_.newEntity(
            "Mlag::MlagTunnelAgentRoot", "root" )

      if "QUICKTRACEDIR" in os.environ:
         qtfile = "%s%s.qt" % (self.agentName, "-%d" if "QUICKTRACEDIR"
                               not in os.environ else "" )
         QuickTrace.initialize( qtfile, "10,10,10,10,10,10,10,10,10,10" )

   def doInit( self, entityManager ):
      mg = entityManager.mountGroup()

      self.createLocalEntity( "EthIntfStatusDir",
                              "Interface::EthIntfStatusDir",
                              "interface/status/eth/intf" )
      self.createLocalEntity( "AllIntfStatusDir",
                              "Interface::AllIntfStatusDir",
                              "interface/status/all" )
      self.createLocalEntity( "AllIntfStatusLocalDir",
                              "Interface::AllIntfStatusLocalDir",
                              Cell.path( "interface/status/local" ) )
      self.localEntitiesCreatedIs( True )
      # Mount mlag/status, Mlag::Status and its dependent paths
      self.mlagStatus_ = MlagMountHelper.mountMlagStatus( mg )
      # Mount mlag/config, Mlag::Config and its dependent paths
      self.mlagConfig_ = MlagMountHelper.mountMlagConfig( mg )
      self.protoStatus_ = mg.mount( 'mlag/proto', 'Mlag::ProtoStatus', 'rO' )
      self.mlagTunnelStatus_ = mg.mount( 'mlag/tunnel/status',
                                         'Mlag::TunnelStatus', 'w' )
      self.mlagTunnelConfig_ = mg.mount( 'mlag/tunnel/config',
                                         'Mlag::TunnelConfig' )
      self.mlagTunnelCellStatus_ = mg.mount( Cell.path( 'mlag/tunnel/status' ),
                                         'Mlag::TunnelCellStatus', 'fw' )
      self.peerIntfs_ = mg.mount( 'interface/status/eth/peer',
                                  'Interface::PeerIntfStatusDir', 'r' )
      self.intfStatus_ = mg.mount( 'interface/status/all',
                                   'Interface::AllIntfStatusDir' )
      self.intfStatusLocal_ = mg.mount( Cell.path( 'interface/status/local' ),
                                        'Interface::AllIntfStatusLocalDir', 'r' )
      self.ethIntfs_ = mg.mount( 'interface/status/eth/intf',
                                 'Interface::EthIntfStatusDir' )
      self.vrfStatusLocal_ = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                       'Ip::AllVrfStatusLocal', 'r' )

      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      self.shmemEm_ = shmemEm

      self.kniStatus_ = shmemEm.doMount( "kni/ns/%s/status" % DEFAULT_NS,
                                         "KernelNetInfo::Status",
                                         Smash.mountInfo( 'keyshadow' ) )
      def _finishMounts():
         t2( 'finishMounts' )
         self.netnsIntfStatusManager_ = \
            Tac.newInstance( 'Interface::NetNsIntfStatusManagerRoot',
                             self.intfStatus_, self.intfStatusLocal_, True )

         self.tunnelAgentRoot_.intfStatusManager = (
            self.intfStatus_, self.intfStatusLocal_,
            self.tunnelAgentRoot_.deviceIntfStatusDir, self.kniStatus_ )

         self.tunnelAgentRoot_.intfStatusManager \
                              .intfStatusManager.omitNonRunningKernelDevices = True
         self.namespaceIs( self.mlagStatus_.localIntfNamespace )
         self.localInterfaceStatusReactor_ = GenericReactor(
            self.mlagStatus_, [ 'localInterface', 'localIntfNamespace' ],
            self.handleIntfStatusAndDeviceName )
         self.mlagConfigReactor_ = GenericReactor(
            self.mlagConfig_, [ 'domainId', 'peerAddress' ],
            self.handleMlagConfig )
         self.mlagStatusReactor_ = GenericReactor(
            self.mlagStatus_, [ 'mlagState' ],
            self.handleState, callBackNow=True )
         self.redundancyModeReactor_ = GenericReactor(
            self.redundancyStatus(), [ 'mode' ],
            self.handleRedundancyMode, callBackNow=True )
         self.vrfStatusLocalReactor_ = Tac.collectionChangeReactor(
            self.vrfStatusLocal_.vrf, VrfStatusReactor, reactorArgs=( self, ) )
      mg.close( _finishMounts )

   def namespaceIs( self, netNs ):
      t0( 'Changing namespace to %s from %s' % ( netNs, self.netNs_ ) )
      qt0( 'Changing namespace to %s from %s' % ( netNs, self.netNs_ ) )
      if netNs == self.netNs_:
         return
      del self.kernelIntfStatusReactor_
      if self.netNs_ and self.netNs_ != DEFAULT_NS:
         del self.netnsIntfStatusManager_.intfStatusManager[ self.netNs_ ]
         del self.vrfKernelIntfStatusDir_
         self.shmemEm_.doUnmount( "kni/ns/%s/status" % self.netNs_ )
      self.netNs_ = netNs
      if self.netNs_ != DEFAULT_NS:
         self.kniLocalStatus_ = self.shmemEm_.doMount(
            "kni/ns/%s/status" % self.netNs_, "KernelNetInfo::Status",
            Smash.mountInfo( 'keyshadow' ) )
         self.vrfKernelIntfStatusDir_ = Tac.newInstance(
            'Interface::DeviceIntfStatusDir', 'vrfdisd' )
         self.netnsIntfStatusManager_.intfStatusManager.newMember(
            self.vrfKernelIntfStatusDir_, self.kniLocalStatus_, self.netNs_ )

         self.localIntfStatusManager = \
            self.netnsIntfStatusManager_.intfStatusManager[ self.netNs_ ]
         self.localIntfStatusManager.omitNonRunningKernelDevices = True
      else:
         self.localIntfStatusManager = self.tunnelAgentRoot_.intfStatusManager
      self.kernelIntfStatusReactor_ = GenericReactor(
         self.localIntfStatusManager.kernelIntfStatusDir,
         [ 'intfStatusAndDeviceName' ], self.handleIntfStatusAndDeviceName )

   def primary( self ):
      return self.mlagStatus_.mlagState == 'primary'

   def secondary( self ):
      return self.mlagStatus_.mlagState == 'secondary'

   def mlagEnabled( self ):
      return self.mlagStatus_.mlagState in ( 'primary', 'secondary', 'inactive' )

   def handleMlagConfig( self, notifiee ):
      t2( "handleMlagConfig" )
      if not self.mlagConfig_.domainId or \
         self.mlagConfig_.peerAddress.isAddrZero:
         t2( "Mlag is not configured. Let mlag state transition handle cleanup" )
         return
      if not self.mlagTunnelSm_:
         return
      if ( self.mlagTunnelSm_.domainId != self.mlagConfig_.domainId or
           self.mlagTunnelSm_.onePam.txDstIpAddr != self.mlagConfig_.peerAddress or
           self.mlagTunnelSm_.onePam.netNs != self.netNs_ ):
         t2( "domainId or peerAddress or netns changed! Deleting mlagTunnelSm." )
         qt0( "domainId or peerAddress or netns changed! Deleting mlagTunnelSm." )
         self.cleanUp( complete=True )
         # Recreate necessary devices
         self.handleState( None )

   def localVrfReady( self ):
      netNs = self.mlagStatus_.localIntfNamespace
      if netNs == DEFAULT_NS:
         t2( "vrfName is default, always ready" )
         return True
      vrfName = VrfName.netNsVrf( netNs )
      t2( "vrfName is ", vrfName )
      if vrfName in self.vrfStatusLocal_.vrf:
         t2( "vrfState is ", self.vrfStatusLocal_.vrf[ vrfName ].state )
         qt0( "vrfState is ", qv( self.vrfStatusLocal_.vrf[ vrfName ].state ) )
         return self.vrfStatusLocal_.vrf[ vrfName ].state == "active"
      return False

   def handleState( self, notifiee ):
      t2( "MLAG state is " + self.mlagStatus_.mlagState )
      qt0( "MLAG state is ", qv( self.mlagStatus_.mlagState ) )
      if self.mlagEnabled():
         self.namespaceIs( self.mlagStatus_.localIntfNamespace )
         self.initNonIntfTaps()
      if self.primary() or self.secondary():
         self.mlagTunnelCellStatus_.running = True
         self.initTapPamManager()
         if self.mlagTunnelSm_:
            self.initIntfStateMachines()
      elif self.mlagEnabled():
         self.cleanUp()
      else:
         self.cleanUp( complete=True )

   def handleRedundancyMode( self, notifiee=None ):
      redMode = self.redundancyStatus().mode
      t0( "Redundancy mode is " + redMode )
      if redMode == 'active' and self.mlagTunnelSm_ \
             and not self.mlagTunnelStatusUpdater_:
         t0( "Creating MlagTunnelStatusUpdater" )
         self.mlagTunnelStatusUpdater_ = MlagTunnelStatusUpdater(
               self.mlagTunnelConfig_, self.mlagTunnelStatus_,
               self.mlagTunnelSm_ )
      elif redMode != 'active' or not self.mlagTunnelSm_:
         t0( "Deleting (or not creating) MlagTunnelStatusUpdater" )
         self.mlagTunnelStatusUpdater_ = None

   def handleVrfState( self, vrfName=None ):
      if not self.mlagStatus_.localInterface:
         qt0( "no localInterface..returning" )
         return
      nsName = self.mlagStatus_.localIntfNamespace
      # this function is never called for vrfName 'default'
      if VrfName( vrfName ).nsName() == nsName:
         self.maybeUpdateMlagTunnelSm( nsName=nsName )

   def initTapPamManager( self ):
      if not self.tapPamManager_:
         self.tapPamManager_ = Tac.newInstance(
            "Mlag::Agent::TapPamManager", "" )

   def initNonIntfTaps( self ):
      ''' Do all the initialization that doesn't require access to the
      interface directories in Sysdb. '''
      if self.mlagTunnelSm_:
         return
      if not self.localVrfReady():
         return
      if self.netNs_ is None:
         return
      if self.netNs_ != DEFAULT_NS:
         intfStatusAndDeviceName = self.localIntfStatusManager.\
            kernelIntfStatusDir.intfStatusAndDeviceName.get( \
            self.mlagStatus_.localInterface.intfId )
         t2( 'initNonIntfTaps int=%s, intfname=%s, nsName=%s'
             % ( intfStatusAndDeviceName, self.mlagStatus_.localInterface.intfId,
                 self.netNs_ ) )
         if not intfStatusAndDeviceName or not intfStatusAndDeviceName.deviceName:
            qt0( "localIntf not there... returning" )
            # just return if the namespace doesn't exist,
            # as the netns should change and re-execute this code.
            return

      # Create the MlagTunnelSm.
      self.mlagTunnelSm_ = self.newMlagTunnelSm()
      self.mlagTunnelSm_.mode = 'both'

      # Setup TapPam for UDP heartbeat
      pam = Tac.newInstance( "Arnet::TapPam", "MlagHeartbeat",
                             self.mlagConstants_.udpHeartbeatDevName )
      pam.mode = 'enabled'
      udpHeartbeatPortId = self.mlagConstants_.udpHeartbeatPortId
      self.mlagTunnelSm_.manyPam[ udpHeartbeatPortId ] = pam
      self.mlagTunnelSm_.localToPeerPortId[ udpHeartbeatPortId ] = \
          udpHeartbeatPortId

      # Setup TapPam for Stp stable device
      self.initStpStableDevice()
      # Bind to the local-interface if available
      self.maybeBindToLocalDevice()
      # Create status updater if redundancy mode is active.
      self.handleRedundancyMode()

   def initStpStableDevice( self ):
      if not self.mlagTunnelSm_:
         # Wait for mlag state to become inactive
         return
      udpStpStablePortId = self.mlagConstants_.udpStpStablePortId
      if udpStpStablePortId in self.mlagTunnelSm_.manyPam:
         # Already set up, nothing to do
         return
      # Setup TapPam for Stp stability communication
      stpStablePam = Tac.newInstance( "Arnet::TapPam", "MlagStpStable",
                                      self.mlagConstants_.udpStpStableDevName )
      stpStablePam.mode = 'enabled'
      self.mlagTunnelSm_.manyPam[ udpStpStablePortId ] = stpStablePam
      self.mlagTunnelSm_.localToPeerPortId[ udpStpStablePortId ] = \
          udpStpStablePortId

   def initIntfStateMachines( self ):
      ''' Do all the initialization that depends on the interfaces
      directories in Sysdb. '''
      self.tunnelAgentRoot_.fromPeerIntfSm = (
         self.peerIntfs_, self.mlagTunnelSm_,
         self.tapPamManager_.tapPams )
      self.tunnelAgentRoot_.fromPeerIntfSm.mode = 'enabled'

      if self.primary() and not self.tunnelAgentRoot_.toPeerIntfSmPrimary:
         if self.tunnelAgentRoot_.toPeerIntfSmSecondary:
            self.tunnelAgentRoot_.toPeerIntfSmSecondary.mode = 'disabled'
            self.tunnelAgentRoot_.toPeerIntfSmSecondary = None
         self.tunnelAgentRoot_.toPeerIntfSmPrimary = (
            self.mlagTunnelSm_, self.ethIntfs_,
            self.tunnelAgentRoot_.intfStatusManager, self.mlagStatus_ )
         self.tunnelAgentRoot_.toPeerIntfSmPrimary.mode = 'enabled'
      elif self.secondary() and not self.tunnelAgentRoot_.toPeerIntfSmSecondary:
         if self.tunnelAgentRoot_.toPeerIntfSmPrimary:
            self.tunnelAgentRoot_.toPeerIntfSmPrimary.mode = 'disabled'
            self.tunnelAgentRoot_.toPeerIntfSmPrimary = None
         assert self.mlagStatus_.peerLinkIntf
         self.tunnelAgentRoot_.toPeerIntfSmSecondary = (
            self.mlagTunnelSm_, self.ethIntfs_,
            self.tunnelAgentRoot_.intfStatusManager,
            self.mlagStatus_.peerLinkIntf, self.peerIntfs_, self.mlagStatus_ )
         self.tunnelAgentRoot_.toPeerIntfSmSecondary.mode = 'enabled'

   def deferredCleanUp( self ):
      # Kick the TapPamManager once to start flushing its queue of devices.
      if not self.cleanupActivity_ and self.tapPamManager_:
         self.tapPamManager_.processTapPamEventQueue()
         t0( 'deferred cleanup: %d devices remain queued' %
               len( self.tapPamManager_.tapPamEvent ) )

      # once no devices remain, it's safe to stop running.
      if not self.tapPamManager_ or len( self.tapPamManager_.tapPamEvent ) == 0:
         t0( 'deferred cleanup: complete' )
         self.mlagTunnelCellStatus_.running = False
         self.cleanupActivity_ = None
         return

      t0( 'waiting for cleanup: %d devices queued' %
               len( self.tapPamManager_.tapPamEvent ) )

      # poll once a second for the TapPamEvent queue to empty
      self.cleanupActivity_ = Tac.ClockNotifiee( self.deferredCleanUp )
      self.cleanupActivity_.timeMin = Tac.now() + 1.0

   def cleanUp( self, complete=False ):
      # If doing a complete cleanup, delete mlagTunnelSm
      # for initialization later.
      t0( "In cleanup, complete=", complete, " mlagTunnelSm=", self.mlagTunnelSm_ )
      if complete and self.mlagTunnelSm_:
         # Completely remove mlagTunnelSm_
         self.mlagTunnelSm_.mode = 'disabled'
         self.mlagTunnelSm_.onePam.mode = 'disabled'
         self.udpSocketBoundTo_ = ""
         t0( "Disabled pam" )
         # Remove TapPam for UDP heartbeat
         udpHeartbeatPortId = self.mlagConstants_.udpHeartbeatPortId
         pam = self.mlagTunnelSm_.manyPam.get( udpHeartbeatPortId )
         if pam:
            pam.mode = 'disabled'
            t0( "Disabled mlag_hb pam" )
            del self.mlagTunnelSm_.manyPam[ udpHeartbeatPortId ]
            del self.mlagTunnelSm_.localToPeerPortId[ udpHeartbeatPortId ]
         # Remove TapPam for Stp stability
         udpStpStablePortId = self.mlagConstants_.udpStpStablePortId
         stpStablePam = self.mlagTunnelSm_.manyPam.get( udpStpStablePortId )
         if stpStablePam:
            stpStablePam.mode = 'disabled'
            del self.mlagTunnelSm_.manyPam[ udpStpStablePortId ]
            del self.mlagTunnelSm_.localToPeerPortId[ udpStpStablePortId ]
         self.mlagTunnelSm_ = None
         # Delete MlagTunnelStatusUpdater
         self.handleRedundancyMode()

      if self.tunnelAgentRoot_.fromPeerIntfSm:
         self.tunnelAgentRoot_.fromPeerIntfSm.mode = 'disabled'
      if self.tunnelAgentRoot_.toPeerIntfSmPrimary:
         self.tunnelAgentRoot_.toPeerIntfSmPrimary.mode = 'disabled'
      if self.tunnelAgentRoot_.toPeerIntfSmSecondary:
         self.tunnelAgentRoot_.toPeerIntfSmSecondary.mode = 'disabled'
      self.tunnelAgentRoot_.toPeerIntfSmPrimary = None
      self.tunnelAgentRoot_.toPeerIntfSmSecondary = None
      self.tunnelAgentRoot_.fromPeerIntfSm = None

      if complete:
         self.deferredCleanUp()

   def maybeBindToLocalDevice( self, intfName=None ):
      nsName = self.mlagStatus_.localIntfNamespace
      if not self.mlagStatus_.localInterface:
         self.udpSocketBoundTo_ = ""
         qt0( "Cleared udpSocketBoundTo_" )
         return
      intfStatusAndDeviceName = self.localIntfStatusManager.\
                                kernelIntfStatusDir.intfStatusAndDeviceName.get( \
         self.mlagStatus_.localInterface.intfId )
      qt0( 'maybeBindToLocalDevice int=%s, intfname=%s, nsName=%s'
          % ( intfStatusAndDeviceName, intfName, nsName ) )
      if not intfStatusAndDeviceName or not intfStatusAndDeviceName.deviceName:
         self.udpSocketBoundTo_ = ""
         qt0( "Cleared udpSocketBoundTo_" )
         return
      if self.udpSocketBoundTo_ != intfStatusAndDeviceName.deviceName:
         self.setSockOptsOnLocalInterface()
         qt0( "Updated sockOpts" )

   def maybeUpdateMlagTunnelSm( self, intfName=None, nsName=None ):
      if not self.mlagTunnelSm_ or \
         ( self.mlagTunnelSm_ and self.mlagTunnelSm_.onePam.netNs != nsName ):
         # Gotta delete/recreate the pam
         qt0( "Updating tunnelsm" )
         self.handleState( None )
         self.handleMlagConfig( None )
         # return here since handleState will have done the maybeBindToLocalDevice
         return
      self.maybeBindToLocalDevice( intfName )

   def handleIntfStatusAndDeviceName( self, notif=None, intfName=None ):
      qt0( "handleIntfStatusAndDeviceName for ", qv( intfName ) )
      if ( not self.mlagStatus_.localInterface or
           ( intfName and intfName != self.mlagStatus_.localInterface.intfId ) ):
         t2( "no localInterface or interface name mismatch...returning" )
         qt0( "no localInterface or interface name mismatch...returning" )
         return

      nsName = self.mlagStatus_.localIntfNamespace
      self.maybeUpdateMlagTunnelSm( intfName, nsName )
  
   def setSockOptsOnLocalInterface( self ):
      if not self.mlagTunnelSm_:
         return
      pam = self.mlagTunnelSm_.onePam
      ipv6Enabled = self.mlagConfig_.peerAddress.af == Ipv6Address
      if ipv6Enabled:
         sock = socket.fromfd( pam.fileDesc.descriptor,
                              socket.AF_INET6, socket.SOCK_DGRAM )
      else:
         sock = socket.fromfd( pam.fileDesc.descriptor,
                              socket.AF_INET, socket.SOCK_DGRAM )
      devName = self.mlagStatus_.localInterface.deviceName
      t2( "setSockOptsOnLocalInterface", devName )
      try:
         sock.setsockopt( socket.SOL_SOCKET, socket.SO_BINDTODEVICE,
                          six.ensure_binary( devName ) )
         self.udpSocketBoundTo_ = devName
      except socket.error as e:
         if e.errno == errno.ENODEV:
            print( "Device %s went away" % devName )
            # Wait until it comes back.
         else:
            raise

   def newMlagTunnelSm( self ):
      ipv6Enabled = self.mlagConfig_.peerAddress.af == Ipv6Address
      if ipv6Enabled:
         socketOpts = [ ( socket.IPPROTO_IPV6, socket.IPV6_UNICAST_HOPS, MLAG_TTL ),
                        ( socket.IPPROTO_IPV6, socket.IPV6_TCLASS, MLAG_TOS ),
                        ( socket.IPPROTO_IPV6, socket.IPV6_RECVHOPLIMIT, 1 ) ]
      else:
         socketOpts = [ ( socket.IPPROTO_IP, socket.IP_TTL, MLAG_TTL ),
                        ( socket.IPPROTO_IP, socket.IP_TOS, MLAG_TOS ),
                        ( socket.IPPROTO_IP, IP_RECVTTL, 1 ) ]
      socketOpts.append( ( socket.SOL_SOCKET, socket.SO_PRIORITY, TC_PRIO_CONTROL ) )
      sm = Tac.newInstance( "Mlag::MlagTunnelSm",
                            self.newUdpPam( socketOptions=socketOpts ),
                            self.mlagConfig_.domainId,
                            self.protoStatus_ )
      return sm
   
   def newUdpPam( self, socketOptions=None ):
      ipv6Enabled = self.mlagConfig_.peerAddress.af == Ipv6Address
      pam = Tac.newInstance( "Arnet::UdpPam", "udpPam" )
      pam.maxRxPktData = 9900
      pam.rxPort = MLAG_PORT
      t0( "peerAddress = " + str( self.mlagConfig_.peerAddress ) + "; v4 = " +
          str( self.mlagConfig_.peerAddress.v4Addr ) )
      if ipv6Enabled:
         pam.ipv6En = True
         pam.rxIp6Addr = Tac.newInstance( "Arnet::IpGenAddr", "::" ).v6Addr
         pam.txDstIp6Addr = self.mlagConfig_.peerAddress.v6Addr
      else:
         pam.rxIpAddr = "0.0.0.0" # INADDR_ANY
         pam.txDstIpAddr = self.mlagConfig_.peerAddress.v4Addr
      # This environment variable can be set for testing purposes:
      pam.txPort = int( os.environ.get( 'MLAG_PEER_UDP_PORT',
                                        str( MLAG_PORT ) ) )
      pam.mode = 'server'
      qt0( "newUdpPam in namespace", qv( self.netNs_ ) )
      if self.netNs_ != DEFAULT_NS:
         t2( 'newUdpPam in nsName=%s' % self.netNs_ )
         pam.netNs = self.netNs_
      if socketOptions:
         # Here we assume that pam socket is AF_INET/SOCK_DGRAM (i.e. IP/UDP) 
         # but it does not really matter, because setsockopt() ignores these
         # parameters anyway.
         sockProto = socket.AF_INET6 if ipv6Enabled else socket.AF_INET
         sock = socket.fromfd( pam.fileDesc.descriptor,
                               sockProto, socket.SOCK_DGRAM )
         for (level, opt, val) in socketOptions:
            sock.setsockopt( level, opt, val )
      return pam

class MlagTunnelStatusUpdater( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, cfg, status, sm ):
      self.cfg_ = cfg
      self.status_ = status
      self.sm_ = sm
      self.statusUpdateActivity_ = Tac.ClockNotifiee( self.statusUpdate )
      self.clearCountersReactor_ = GenericReactor(
         self.cfg_, [ 'clearCounters' ], self.clearCounters )

   def __del__( self ):
      self.clearCountersReactor_ = None
      self.statusUpdateActivity_ = None
      self.sm_ = None

   def statusUpdate( self ):
      status = self.status_
      # for now we only have packet counters to update
      # Setting Pam counters to zero only changes the counters, 
      # but updating them in Sysdb may trigger more actions,  
      # so, just to be safe, we first reset Pam counters, and only 
      # then update the ones in Sysdb, keeping the counters consistent.
      rx = self.sm_.onePam.rxPkts
      self.sm_.onePam.rxPkts = 0
      tx = self.sm_.onePam.txPkts
      self.sm_.onePam.txPkts = 0
      enc = self.sm_.encPkts
      self.sm_.encPkts = 0
      dec = self.sm_.decPkts
      self.sm_.decPkts = 0
      ePkts = { }
      dPkts = { }
      allCounters = list( self.sm_.encDecByMacPkts.values() ) + \
          list( self.sm_.encDecByProtoPkts.values() ) + \
          list( self.sm_.encDecByStrPkts.values() )
        
      for counter in allCounters:
         name = counter.counterName
         ePkts[ name ] = counter.encPkts 
         counter.encPkts = 0
         dPkts[ name ] = counter.decPkts
         counter.decPkts = 0
      status.rxPkts += rx
      status.txPkts += tx
      status.encPkts += enc
      status.decPkts += dec
      for name in [ c.counterName for c in allCounters ]:
         if not name in status.encDecPkts:
            status.encDecPkts.newMember( name )
         counter = status.encDecPkts[ name ]
         counter.encPkts += ePkts[ name ]
         counter.decPkts += dPkts[ name ]

      # Reschedule one second in the future.
      self.statusUpdateActivity_.timeMin = Tac.now() + 1

   def clearCounters( self, notifiee ):
      status = self.status_
      status.rxPkts = 0
      status.txPkts = 0
      status.encPkts = 0 
      status.decPkts = 0 
      for key in status.encDecPkts:
         status.encDecPkts[ key ].encPkts = 0
         status.encDecPkts[ key ].decPkts = 0
