#!/usr/bin/env python3
# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arastra, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

'''
Generic version of the molex test, one version for sfp and one for qsfp.

The "port" object is required to have the following methods.

i2cWrite( deviceId, address, value ) -- do an i2c write.
i2cRead( deviceId, address ) -- do an i2c read, will return the value read as an int.
name() -- the name of the port, i.e. Ethernet1.

The "port" object is required to have the following addition methods for sfp tests.

rxLos() -- returns the state of the rxLos pin on the xcvr.
txFault() -- returns the state of the txFault pin on the xcvr.
modAbs() -- returns the state of the modAbs pin on the xcvr.

as0Is( bool ) -- Set the state of the as0 pin on the xcvr.
as1Is( bool ) -- returns the state of the as1 pin on the xcvr.
txDisableIs( bool ) -- returns the state of the txDisable pin on the xcvr.

The "port" object is required to have the following methods for qsfp tests.
resetIs( bool )
lpModeIs( bool )
intL()
modPresent()

TBD: Other functions.

In all functions TestFailure is a function pointer to be called when the test fails.
It should have the form TestFailure( string errorMessage ). It should not return.

'''

import time
import Tracing

myTraceFacility = "MolexDiagLib"

t0 = Tracing.Handle( myTraceFacility ).trace0

molexQsfpCmdDelay = 0.05   # 50 ms delay for uC to execute i2c cmd to wiggle GPIO

def eepromRead( _type, port, addr ):
   if _type == 'sfp':
      # the eeprom is at address 0xao
      return port.i2cRead( 0xa0, addr )
   elif _type == 'qsfp':
      return port.i2cRead( 0xa0, addr )
   else:
      assert 0

def sfpMolex( port, args, silent, init, TestFailure ):

   pinMap = { "TxFault": 0x80,
              "TxDisable": 0x40,
              "ModABS": 0x20,
              "RateSelect1": 0x10,
              "RateSelect0": 0x8,
              "RXLOS": 0x4,
              "VccT": 0x2,
              "VccR": 0x1 }
   configurationRegister = 0x5b
   polarityRegister = 0

   def initMolex( port ):
      # set polarity on all pins to normal
      port.i2cWrite( 0x3a, 0x02, polarityRegister )
      # set default configuration
      #   NOTE: we are now making I/O 4 an input, unlike the
      #         preliminary Molex data sheet, we believe this pin
      #         to be connected to RateSelect1 instead of WP EE.
      port.i2cWrite( 0x3a, 0x03, configurationRegister )

   def printReg( inputRegister, names ):
      rc = []
      for name in names:
         if configurationRegister & pinMap[ name ]:
            # if polarity is true then input pin is active low
            value = ( inputRegister ^ polarityRegister ) & pinMap[ name ]
            if not silent:
               print( value, name, value and "(true)\n" or "(false)\n" )
            rc.append( value )
         elif not silent:
            print( name, " is an output, skipping\n" )
            rc.append( -1 )
      return rc

   def molexUsage():
      TestFailure( "Molex configuration args are: <operation> [<signal>]\n"
                   "  <operation> -> read, set, clear\n"
                   "  [<signal>[<signal>[<signal>]...]] -> %s "
                   " (required for read and set only) " %
                   ", ".join( pinMap ) )

   if init:
      initMolex( port )

   if args[ 0 ] == "read":
      if len( args ) < 2:
         molexUsage()
      # fetch the current values from the Molex port expander
      inputRegister = port.i2cRead( 0x3b, 0x00 )
      if args[ 1 ] == 'all':
         regs = []
         for label in pinMap:
            regs.append( label )

         return printReg( inputRegister, regs )
      else:
         try:
            return printReg( inputRegister, args[ 1 : ] )
         except KeyError:
            TestFailure( "Invalid signal. Valid values are:",
                         ", ".join( pinMap ) )
   elif args[ 0 ] == "set":
      if len( args ) < 2:
         molexUsage()
      try:
         value = 0
         for reg in args[ 1 : ]:
            value |= pinMap[ reg ]
      except KeyError:
         TestFailure( "Invalid signal. Valid values are:",
                      ", ".join( pinMap ) )
      port.i2cWrite( 0x3a, 0x01, value )
      return 0
   elif args[ 0 ] == "clear":
      if len( args ) != 1:
         molexUsage()
      port.i2cWrite( 0x3a, 0x01, 0 )
   else:
      TestFailure( "Invalid operation [%s].\n"
                   "Valid ops are 'read', 'set', and 'clear'" )

def qsfpMolex( port, args, silent, init, TestFailure ):

   pinMap = { "LpMode": 0x80,
              "Vcc1": 0x40,
              "Reset": 0x20,
              "IntL": 0x10,
              "ModPresent": 0x8,
              "WP_EE_IO": 0x4,
              "VccRx": 0x2,
              "VccTx": 0x1 }

   registers = { "input": 0x0,
                 "output": 0x1,
                 "polarity": 0x2,
                 "config": 0x3 }

   # Reset and LPMode have reversed polarity.
   polarityRegister = 0xa0
   # Set ports LpMode and Reset as outputs.
   configurationRegister = 0xe7
   # Clearing register- needed because the logic is inverted for some pins.
   # So far, only intL and modPresent are used and both have inverted logic.
   clearingRegister = 0x18

   def initMolex( port ):
      # set polarity on all pins to normal
      port.i2cWrite( 0x3a, registers[ "polarity" ], 0 )
      # set default configuration
      port.i2cWrite( 0x3a, registers[ "config" ], configurationRegister )

   def printReg( inputRegister, names ):
      rc = []
      for name in names:
         if configurationRegister & pinMap[ name ]:
            # if polarity is true then input pin is active low
            value = ( inputRegister ^ polarityRegister ) & pinMap[ name ]
            if not silent:
               print( value, name, value and "(true)\n" or "(false)\n" )
            rc.append( value )
         elif not silent:
            print( name, " is an output, skipping\n" )
            rc.append( -1 )
      return rc

   def molexUsage():
      TestFailure( "Molex configuration args are: <operation> [<signal>]\n"
                   "  <operation> -> read, set, clear\n"
                   "  [<signal>[<signal>[<signal>]...]] -> %s "
                   " (required for read and set only) " %
                   ", ".join( pinMap ) )

   if init:
      initMolex( port )

   if not args:
      molexUsage()

   if args[ 0 ] == "read":
      if len( args ) < 2:
         molexUsage()
      # fetch the current values from the Molex port expander
      inputRegister = port.i2cRead( 0x3b, registers[ "input" ] )
      if args[ 1 ] == 'all':
         regs = []
         for label in pinMap:
            regs.append( label )

         return printReg( inputRegister, regs )
      else:
         try:
            return printReg( inputRegister, args[ 1 : ] )
         except KeyError:
            errString = "Invalid signal. Valid values are:"
            valString = ", ".join( pinMap )
            TestFailure( errString + valString )
   elif args[ 0 ] == "set":
      if len( args ) < 2:
         molexUsage()
      try:
         value = clearingRegister
         # Since the only two outputs we have are intL and modPresent
         # and since the logic is inverted for both of these, we start
         # with a value of 0x18 and CLEAR the bits we want to turn on.
         for reg in args[ 1 : ]:
            value ^= pinMap[ reg ]
      except KeyError:
         # pylint: disable-next=undefined-variable
         errString = "Invalid signal '%s'. Valid values are:" % e
         valString = ", ".join( pinMap )
         TestFailure( errString + valString )
      port.i2cWrite( 0x3a, registers[ "output" ], value )
      time.sleep( molexQsfpCmdDelay )
      return 0
   elif args[ 0 ] == "clear":
      if len( args ) != 1:
         molexUsage()
      port.i2cWrite( 0x3a, registers[ "output" ], clearingRegister )
   else:
      TestFailure( "Invalid operation [%s].\n"
                   "Valid ops are 'read', 'set', and 'clear'" )

# ----------------------------------------------------------------------------
#
# "molexTest"
#
# ----------------------------------------------------------------------------

def sfpMolexTest( port, TestFailure ):
   """Test the input and output signals on the PHY using a
      Molex Universal SPF+ Loopback Adapter (a fancy SFP with a built-in
      I/O port expander at an alternate I2C address, which allows us to
      set and check each low-speed I/O pin.
   """

   validSerialList = {
      # standard molex parts (green label)
      "Molex74765_0901": [ 0x00, 0x09, 0x3a,
                             0x37, 0x34, 0x37, 0x36, 0x35, 0x2d, 0x30, 0x39,
                             0x30, 0x31, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20 ],
      # arastra version (tan label)
      "Molex74765_0902": [ 0x00, 0x09, 0x3a,
                             0x37, 0x34, 0x37, 0x36, 0x35, 0x2d, 0x30, 0x39,
                             0x30, 0x32, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20 ],
      # arastra version (tan label) with no write protect
      "Molex74765_0912": [ 0x00, 0x09, 0x3a,
                             0x37, 0x34, 0x37, 0x36, 0x35, 0x2d, 0x30, 0x39,
                             0x31, 0x32, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20 ],
      # Leoni SFP single loopback
      "LeoniL45593-C915-A115": [ 0xa8, 0xb0, 0xae,
                             0x4c, 0x34, 0x35, 0x35, 0x39, 0x33, 0x2d, 0x43,
                             0x39, 0x31, 0x35, 0x2d, 0x41, 0x31, 0x31, 0x35 ],
      # Leoni SFP gang loopback untabbed
      "LeoniL45593-C915-A105": [ 0xa8, 0xb0, 0xae,
                             0x4c, 0x34, 0x35, 0x35, 0x39, 0x33, 0x2d, 0x43,
                             0x39, 0x31, 0x35, 0x2d, 0x41, 0x31, 0x30, 0x35 ],
      # Leoni SFP gang loopback tabbed
      "LeoniL45593-C915-B105": [ 0xa8, 0xb0, 0xae,
                             0x4c, 0x34, 0x35, 0x35, 0x39, 0x33, 0x2d, 0x43,
                             0x39, 0x31, 0x35, 0x2d, 0x42, 0x31, 0x30, 0x35 ],
    }

   def getSerialId( port ):
      seeprom = list() # pylint: disable=use-list-literal
      # read 3 byte OUI and 16 byte Part Number
      for offset in range( 19 ):
         seeprom.append( eepromRead( 'sfp', port, offset + 37 ) )
      sfp = "unknown"
      # pylint: disable-next=consider-using-dict-items,consider-iterating-dictionary
      for sfpType in validSerialList.keys():
         if validSerialList[ sfpType ] == seeprom:
            sfp = sfpType
            break
      if sfp == "unknown":
         TestFailure( "%s - no valid SFP found.\n"
                      "Do you have the correct Molex SFP inserted?" % port.name() )
      t0( "SFP %s" % sfp )
      return sfp

   def txFaultLow( port ):
      # initial test conditions, all molex outputs are low

      # Read tx fault to make sure it is clear
      if port.txFault() == 1:
         return "%s [7.1] TX Fault not detected Low\n" % port.name()
      return ""

   def modAbsLow( port ):
      # mod_abs should be false because the molex output is low
      if port.modAbs() == 1:
         return "%s [5.1] ModAbs not deteted High\n" % port.name()
      return ""

   def rxLosLow( port ):
      # rxLos should be false because the molex output is low
      if port.rxLos():
         return "%s [2.1] rxLos signal not detected Low \n" % port.name()
      return ""

   def txFaultHigh( port ):
      # Read tx fault, it should be set
      if not port.txFault():
         return "%s [7.2] TX Fault not detected High\n" % port.name()
      return ""

   def modAbsHigh( port ):
      if not port.modAbs():
         return "%s [5.2] ModAbs not detected High\n" % port.name()
      return ""

   def rxLosHigh( port ):
      # check that the rxLos signal detect is true
      if not port.rxLos():
         return "%s [2.2] rxLos not detected High\n" % port.name()
      return ""

   def testPhyInputs( port ):
      ''' Test port inputs:
      TX_FAULT (molex pin 7)
      MOD_ABS (molex pin 5)
      RX_LOS (molex pin 2)
      '''
      # initialize the molex by clearing all outputs
      sfpMolex( port, [ "clear" ], True, True, TestFailure )

      # check all signals are detected as low
      error7 = txFaultLow( port )
      error5 = modAbsLow( port )
      error2 = rxLosLow( port )

      # change the value of the molex outputs and make sure the port detects them
      # Enable TX Fault in SFP and make sure the port detects it
      sfpMolex( port, [ "set", "TxFault", "ModABS", "RXLOS" ], True, False,
                TestFailure )

      # check all signals are detected as high
      error7 += txFaultHigh( port )
      error5 += modAbsHigh( port )
      error2 += rxLosHigh( port )

      if error7 == "":
         t0( "%s [7] TX Fault passed" % port.name() )
      if error5 == "":
         t0( "%s [5] Mod ABS passed" % port.name() )
      if error2 == "":
         t0( "%s [2] Rx LOS passed" % port.name() )

      return error7 + error5 + error2

   def testPhyOutputs( port, sfpType ):
      ''' tests: rate select 0 & 1, tx disable, VccT and VccR signals'''
      error0 = ""
      error1 = ""
      error3 = ""
      error4 = ""
      error6 = ""

      # set port outputs to low (VccT and VccR cannot be set low)
      if sfpType != "Molex74765_0901":
         # the green molexs don't do rate select
         port.as1Is( False )
         port.as0Is( False )
      port.txDisableIs( False )

      # Signals should be low coming from the PHY, read and check them
      res = sfpMolex( port, [ "read", "RateSelect1", "RateSelect0",
                          "TxDisable" ], True, False, TestFailure )
      if sfpType != "Molex74765_0901":
         if res[ 0 ] != 0:
            error4 += "%s [4.1] Unable to clear RateSelect1\n" % port.name()
         if res[ 1 ] != 0:
            error3 += "%s [3.1] Unable to clear RateSelect0\n" % port.name()
      if res[ 2 ] != 0:
         error6 += "%s [6.1] Unable to clear TX Disable\n" % port.name()

      # set port outputs to high (VccT and VccR are already high)
      if sfpType != "Molex74765_0901":
         port.as1Is( True )
         port.as0Is( True )
      port.txDisableIs( True )

      # check signals are high
      res = sfpMolex( port, [ "read", "RateSelect1", "RateSelect0",
                          "TxDisable", "VccT", "VccR" ], True, False, TestFailure )
      if sfpType != "Molex74765_0901":
         if res[ 0 ] == 0:
            error4 += "%s [4.2] Failed to read RateSelect1 correctly\n" % port.name()
         if res[ 1 ] == 0:
            error3 += "%s [3.2] Failed to read RateSelect0 correctly\n" % port.name()
      if res[ 2 ] == 0:
         error6 += "%s [6.2] Failed to read TX Disable correctly\n" % port.name()
      if res[ 3 ] == 0:
         error1 += "%s [1.1] VccT not set\n" % port.name()
      if res[ 4 ] == 0:
         error0 += "%s [0.1] VccR not set\n" % port.name()

      if sfpType != "Molex74765_0901":
         if error4 == "":
            t0( "%s [4] RateSelect1 passed" % port.name() )
         if error3 == "":
            t0( "%s [3] RateSelect0 passed" % port.name() )
      else:
         t0( "%s [4] RateSelect1 skipped" % port.name() )
         t0( "%s [3] RateSelect0 skipped" % port.name() )
      if error6 == "":
         t0( "%s [6] TX Disable passed" % port.name() )
      if error1 == "":
         t0( "%s [1] VccT passed" % port.name() )
      if error0 == "":
         t0( "%s [0] VccR passed" % port.name() )

      return error6 + error4 + error3

   error = ""
   # first get the molex type (some tests change behaviour depending on which
   # molex is inserted)
   sfpType = getSerialId( port )

   try:
      #### Test Phy Inputs ####
      # Test TX_FAULT (Molex pin 7)
      # Test MOD_DET (Molex pin 5)
      # Test RX_LOS (Molex pin 2)
      error += testPhyInputs( port )

      #### Test Phy Outputs ####
      # Test Rate Select 1 and 0 (Molex pins 4 and 3)
      # Test TX_DISABLE (Molex pin 6)
      # Test VccT and VccR (Molex pins 1 and 0):
      error += testPhyOutputs( port, sfpType )
   finally:
      # really important if you want to use this molex again !!!
      # restore molex to a working state, see BUG6225
      sfpMolex( port, [ "clear" ], True, True, TestFailure )
   if error != "":
      for ln in error.splitlines():
         print( "%s ERROR (%s)" % ( port.name(), ln ) )
      return port
   else:
      print( "%s OK" % port.name() )
      return None

# Start here
def qsfpMolexTest( port, TestFailure ):
   """Test the input and output signals on the PHY using a
      Molex Universal SPF+ Loopback Adapter (a fancy SFP with a built-in
      I/O port expander at an alternate I2C address, which allows us to
      set and check each low-speed I/O pin.
   """

   validSerialList = {
      # arastra version (tan label)
      "Molex74763_0025": [ 0x00, 0x09, 0x3a,
                             0x37, 0x34, 0x37, 0x36, 0x33, 0x2d, 0x30, 0x30,
                             0x32, 0x35, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20 ],
      "LeoniL45593-D915-A115": [ 0xa8, 0xb0, 0xae,
                             0x4c, 0x34, 0x35, 0x35, 0x39, 0x33, 0x2d, 0x44,
                             0x39, 0x31, 0x35, 0x2d, 0x41, 0x31, 0x31, 0x35 ],
    }

   def getSerialId( port ):
      seeprom = list() # pylint: disable=use-list-literal
      # read 3 byte OUI and 16 byte Part Number
      for offset in range( 19 ):
         seeprom.append( eepromRead( 'qsfp', port, offset + 165 ) )
      qsfp = "unknown"
      # pylint: disable-next=consider-using-dict-items,consider-iterating-dictionary
      for qsfpType in validSerialList.keys():
         if validSerialList[ qsfpType ] == seeprom:
            qsfp = qsfpType
            break
      if qsfp == "unknown":
         TestFailure( "%s - no valid QSFP found.\n"
                      "Is the correct Molex QSFP inserted?" % port.name() )
      t0( "QSFP %s" % qsfp )
      return qsfp

   def intLLow( port ):
      # Read intL to make sure it is clear
      if port.intL() == 1:
         return "%s [7.1] IntL not detected Low\n" % port.name()
      return ""

   def modPresentLow( port ):
      # mod_present should be false because the molex output is low
      if port.modPresent() == 1:
         return "%s [5.1] ModPresent not deteted Low\n" % port.name()
      return ""

   def intLHigh( port ):
      # Read intL, it should be set
      if not port.intL():
         return "%s [7.2] intL not detected High\n" % port.name()
      return ""

   def modPresentHigh( port ):
      if not port.modPresent():
         return "%s [5.2] ModPresent not detected High\n" % port.name()
      return ""

   def testPhyInputs( port ):
      ''' Test port inputs:
      IntL (molex pin 4)
      MOD_Present (molex pin 3)
      '''
      # initialize the molex by clearing all outputs
      qsfpMolex( port, [ "clear" ], True, True, TestFailure )

      # check all signals are detected as low
      error4 = intLLow( port )
      error3 = modPresentLow( port )

      # Change the value of the molex outputs and make sure the port detects them.
      # ModPresent must come first since changes to IntL aren't reported when no
      # module is present.
      qsfpMolex( port, [ "set", "ModPresent", "IntL" ], True, False,
                TestFailure )

      # check all signals are detected as high
      error3 += modPresentHigh( port )
      error4 += intLHigh( port )

      if error3 == "":
         t0( "%s [5] Mod Present passed" % port.name() )
      if error4 == "":
         t0( "%s [7] IntL passed" % port.name() )

      return error4 + error3

   def testPhyOutputs( port, qsfpType ):
      ''' tests: LpMode, Reset, ... '''
      error5 = ""
      error7 = ""

      # We will test two transitions. Make the initial condition
      # both lines asserted.
      port.lpModeIs( True )
      port.resetIs( True )

      # Signals should be high coming from the PHY, read and check them
      res = qsfpMolex( port, [ "read", "LpMode", "Reset" ], True, False,
                       TestFailure )

      # For now, we won't test setting LPMODE.
      # if res[ 0 ] == 0:
      #   error7 += "%s [7.2] Failed to set LpMode\n" % port.name()
      if res[ 1 ] == 0:
         error5 += "%s [5.2] Failed to set Reset\n" % port.name()

      # Deassert the lines
      port.lpModeIs( False )
      port.resetIs( False )

      # Signals should be low coming from the PHY, read and check them
      res = qsfpMolex( port, [ "read", "LpMode", "Reset" ], True, False,
                       TestFailure )
      # For now, we won't test setting LPMODE.
      # if res[ 0 ] != 0:
      #   error7 += "%s [7.1] Unable to clear LpMode\n" % port.name()
      if res[ 1 ] != 0:
         error5 += "%s [5.1] Unable to clear Reset\n" % port.name()

      port.lpModeIs( True )
      port.resetIs( True )

      # Signals should be high coming from the PHY, read and check them
      res = qsfpMolex( port, [ "read", "LpMode", "Reset" ], True, False,
                       TestFailure )

      # For now, we won't test setting LPMODE.
      # if res[ 0 ] == 0:
      #   error7 += "%s [7.2] Failed to set LpMode\n" % port.name()
      if res[ 1 ] == 0:
         error5 += "%s [5.2] Failed to set Reset\n" % port.name()

      if error7 == "":
         t0( "%s [7] LpMode passed" % port.name() )
      if error5 == "":
         t0( "%s [5] Reset passed" % port.name() )

      return error7 + error5

   # Function body starts here.
   error = ""
   # first get the molex type (some tests change behaviour depending on which
   # molex is inserted)
   qsfpType = getSerialId( port )

   try:
      #### Test Phy Inputs ####
      # Test IntL (Molex pin 4)
      # Test MOD_Present (Molex pin 3)
      error += testPhyInputs( port )

      #### Test Phy Outputs ####
      # Test LpMode (Molex pin 7)
      # Test Reset (Molex pin 5)
      error += testPhyOutputs( port, qsfpType )
   finally:
      # really important if you want to use this molex again !!!
      # restore molex to a working state, see BUG6225
      qsfpMolex( port, [ "clear" ], True, True, TestFailure )

      # The clear command above would have put the molex in a clean 'absent' state.
      # Restore module presence.
      qsfpMolex( port, [ "set", "ModPresent" ], True, False, TestFailure )
   if error != "":
      for ln in error.splitlines():
         print( "%s ERROR (%s)" % ( port.name(), ln ) )
      return port
   print( "%s OK" % port.name() )
   return None
