# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ClientCommonLib import (
   getRsvpFec,
   getRsvpTunnelInfo,
   resolveNexthop,
)

# Currently viaInfo is calculated here only for RSVP ping
# If there is a need to get curent nexthop info for each ping iteration
# for other protocols, add the via info calculation here
# rather than in MplsPingHandlers.
# And then a call can be made from ClientCore to the calculateViaInfo function.
def calculateViaInfo( mount, rsvpTunnel, rsvpSubTunnel, session, lsp, state ):
   lspIds = []
   subTunnels = []
   if rsvpTunnel:
      nextHopsAndLabels, _, rsvpSpIds, rsvpSenderAddr, subTunnels = \
         getRsvpTunnelInfo( mount, rsvpTunnel, rsvpSubTunnel )
   else:
      nextHopsAndLabels, _, rsvpSpIds, rsvpSenderAddr, lspIds = \
         getRsvpFec( mount, session, lsp )
   if nextHopsAndLabels is None or rsvpSpIds is None:
      return None, None, None, None

   # All SPs have the same destination, so just take the first one.
   prefix = rsvpSpIds[ 0 ].sessionId.dstIp

   viaInfo, clientIdToVias, clientIdToLsp, clientIdToSubTunnels = [], {}, {}, {}
   for idx in range( 0, len( rsvpSpIds ) ):
      # Map L3 nexthop to L2 nexthop
      nexthop, label, _ = nextHopsAndLabels[ idx ]
      # label can be a list or single label. Converting it to a list here makes the
      # rest of the logic simpler
      labelStack = [ label ] if not isinstance( label, list ) else label
      nexthopIntf, nexthopEthAddr = resolveNexthop( mount, state, nexthop )
      if nexthopIntf and nexthopEthAddr:
         clientIdToVias[ idx ] = [ ( nexthop, labelStack, nexthopIntf ) ]
         # The label list could potentially contain the implicit null label. We leave
         # it in the list above when we pass it to the clientIdToVias because it is
         # nice and consistent to see it in the ping output, however we remove the
         # label before passing it to viaInfo so that the rest of the processing does
         # not have to special case it (the label isn't actually in the stack).
         # However, if the list ONLY contains the implicit null, we leave it in,
         # since the client config needs to have at least one label.
         if any( label != 3 for label in labelStack ):
            labelStack = [ l for l in labelStack if l != 3 ]

         viaInfo.append( ( nexthopIntf, labelStack, nexthopEthAddr ) )
         # No need to save the lspIds if the lsp was specified in the ping command.
         if not lsp and lspIds:
            clientIdToLsp[ idx ] = lspIds[ idx ]
         if subTunnels:
            clientIdToSubTunnels[ idx ] = subTunnels[ idx ]
   renderArg = ( clientIdToVias, prefix, 'RSVP',
                 clientIdToLsp, clientIdToSubTunnels )

   if not viaInfo:      # No resolved interface
      print( "Failed to find a valid output interface" )
      return None, None, None, None
   return viaInfo, renderArg, rsvpSpIds, rsvpSenderAddr
