#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Agent
from Arnet.NsLib import DEFAULT_NS
from Arnet.MplsLib import labelStackToString
import Cell
from ForwardingHelper import forwardingHelperKwFactory
from IpLibConsts import DEFAULT_VRF
import Tac
from TypeFuture import TacLazyType
import SharedMem
import Shark
import Smash
import SmashLazyMount

import Toggles.RsvpToggleLib

AgentName = 'MplsUtilLsp'
fwdingHelper = None

TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )

def frwdHelper( allowEncap=True, l3IntfId="", fecId=None, selectOneVia=False ):
   if fecId:
      nexthopInfos = fwdingHelper.resolveHierarchical( allowEncap=allowEncap,
                                                       fecId=fecId,
                                                       selectOneVia=selectOneVia )
   else:
      nexthopInfos = fwdingHelper.resolveHierarchical( allowEncap=allowEncap,
                                                       l3IntfId=l3IntfId,
                                                       selectOneVia=selectOneVia )
   nextHops = []
   for nexthopInfo in nexthopInfos:
      if nexthopInfo.resolved:
         nextHops.append( ( str( nexthopInfo.resolved ), nexthopInfo.dstMac,
                            nexthopInfo.nexthopIp, nexthopInfo.intf,
                            labelStackToString( nexthopInfo.labelStack ),
                            ','.join( nexthopInfo.srcIntfIdStack ) ) )
   return nextHops

class MplsUtilLsp( Agent.Agent ):
   def __init__( self, entityManager ):
      Agent.Agent.__init__( self, entityManager, agentName=AgentName )
      self.root_ = None
      self.vrfNameStatus_ = None
      self.arpSmash_ = None
      self.allIntfStatusLocalDir_ = None
      self.timerWheel_ = None
      self.ipStatus_ = None
      self.allIntfStatusDir_ = None
      self.ip6Status_ = None
      self.kniStatus_ = None
      self.bridgingConfig_ = None
      self.mplsHwStatus_ = None
      self.routingHwRouteStatus_ = None
      self.mplsStatus_ = None
      self.isisV4SmashColl_ = None
      self.isisV6SmashColl_ = None
      self.ospfSmashColl_ = None
      self.ldpCommonLibSmashColl_ = None
      self.bgpLuCommonLibSmashColl_ = None
      self.p2mpCommonLibSmashColl_ = None
      self.rsvpStatus_ = None
      self.rsvpSharkStatus_ = None
      self.mldpOpqTable_ = None
      self.ldpStatusColl_ = None
      self.decapLfib_ = None
      self.transitLfib_ = None
      self.mldpTransitLfib_ = None
      self.rsvpLfib_ = None
      self.forwardingStatus_ = None
      self.forwarding6Status_ = None
      self.forwardingGenStatus_ = None
      self.intfConfigDir_ = None
      self.routingStatus_ = None
      self.routing6Status_ = None
      self.trie_ = None
      self.v6trie_ = None
      self.bgpLuTunnelTable_ = None
      self.ldpTunnelTable_ = None
      self.srTeSegmentListTunnelTable_ = None
      self.srTunnelTable_ = None
      self.ospfSrTunnelTable_ = None
      self.isisFlexAlgoTunnelTable_ = None
      self.staticTunnelTable_ = None
      self.tiLfaTunnelTable_ = None
      self.l3Config_ = None
      self.packetTracerConfig_ = None
      self.packetTracerStatus_ = None
      self.pwLfib_ = None
      self.pwLabelBindingColl_ = None
      self.pwAttrColl_ = None
      self.bgpLuLfib_ = None
      self.bgpLuRib_ = None
      self.bgpLuLocalRouteTable_ = None
      self.rdConfig_ = None
      self.rdAuto_ = None
      self.rdVrf_ = None
      self.subIntfConfigDir_ = None
      self.tunnelFib_ = None
      self.srTeForwardingStatus_ = None
      self.srTeSegmentListTunnelTable_ = None

   def doInit( self, entityManager ):
      mg = entityManager.mountGroup()
      self.l3Config_ = mg.mount( "l3/config", "L3::Config", "r" )
      # pylint: disable-msg=W0201

      def _onMountsComplete():
         self.fecModeStatus_ = Tac.newInstance( "Smash::Fib::FecModeStatus",
                                                "fecModeStatus" )
         self.fecModeSm_ = Tac.newInstance( "Ira::FecModeSm", self.l3Config_,
                                            self.fecModeStatus_ )
         self.fecModeRestartSm_ = Tac.newInstance( "Ira::FecModeRestartSm",
                                                   self.fecModeStatus_ )
         self.doInitStage2( entityManager )

      mg.close( _onMountsComplete )

   def doInitStage2( self, entityManager ):
      # pylint: disable=attribute-defined-outside-init, W0201
      mg = entityManager.mountGroup()
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
      self.ipStatus_ = mg.mount( "ip/status", "Ip::Status", "r" )
      self.ip6Status_ = mg.mount( "ip6/status", "Ip6::Status", "r" )
      self.mplsHwStatus_ = mg.mount( "routing/hardware/mpls/status",
                                     "Mpls::Hardware::Status", "rS" )
      self.routingHwRouteStatus_ = mg.mount( "routing/hardware/route/status",
                                             "Routing::Hardware::RouteStatus", "r" )
      self.bridgingConfig_ = mg.mount( "bridging/config", "Bridging::Config", "r" )
      RsvpSysdbStatus = Tac.Type( "Rsvp::RsvpSysdbStatus" )
      self.rsvpStatus_ = mg.mount( RsvpSysdbStatus.mountPath,
                                   "Rsvp::RsvpSysdbStatus", "rS" )
      RsvpLerSysdbStatus = Tac.Type( "Rsvp::RsvpLerSysdbStatus" )
      self.rsvpLerStatus_ = mg.mount( RsvpLerSysdbStatus.mountPath,
                                      "Rsvp::RsvpLerSysdbStatus", "rS" )
      self.mplsStatus_ = mg.mount( "mpls/status", "Mpls::Api::Status", "rS" )
      self.mplsUtilsConfig_ = mg.mount( 'mplsutils/config',
                                        'MplsUtils::Config', 'r' )
      self.mldpOpqTable_ = mg.mount( "mpls/ldp/mldpOpaqueValueTable",
                                     "Mpls::MldpOpaqueValueTable", "r" )
      self.ldpStatusColl_ = mg.mount( "mpls/ldp/ldpStatusColl",
                                      "Ldp::LdpStatusColl", "r" )
      self.packetTracerActiveClient_ = mg.mount( 'packettracer/activeClient',
                                                 'Tac::Dir', 'w' )
      # pylint: disable-next=consider-using-f-string
      self.packetTracerConfig_ = mg.mount( 'packettracer/config/%s' % AgentName,
                                           'PacketTracer::ClientConfig',
                                           'w' )
      # pylint: disable-next=consider-using-f-string
      self.packetTracerStatus_ = mg.mount( 'packettracer/status/%s' % AgentName,
                                           'PacketTracer::ClientStatus',
                                           'r' )
      self.packetTracerHwStatus_ = mg.mount( 'packettracer/hwstatus',
                                             'PacketTracer::HwStatus',
                                             'r' )
      self.packetTracerSwStatus_ = mg.mount( 'packettracer/swstatus',
                                             'PacketTracer::SwStatus',
                                             'r' )
      self.subIntfConfigDir_ = mg.mount( 'interface/config/subintf',
                                         'Interface::SubIntfConfigDir',
                                         'r' )
      self.timerWheel_ = Tac.newInstance( "Ark::TimerWheel",
                                          Tac.activityManager.clock, 100,
                                          10 * 60 * 5, True, 1024 )
      self.schedulerRoot_ = Tac.Type( "Ark::TaskSchedulerRoot" ).\
                            findOrCreateScheduler()

      if self.fecModeStatus_.fecMode == 'fecModeUnified':
         forwardingStatusPath = "forwarding/unifiedStatus"
         forwarding6StatusPath = "forwarding6/unifiedStatus"
         forwardingGenStatusPath = "forwardingGen/unifiedStatus"
      else:
         forwardingStatusPath = "forwarding/status"
         forwarding6StatusPath = "forwarding6/status"
         forwardingGenStatusPath = "forwardingGen/status"

      self.allIntfStatusLocalDir_ = self.createLocalEntity(
         "AllIntfStatusLocalDir",
         "Interface::AllIntfStatusLocalDir",
         Cell.path( "interface/status/local" ) )

      self.allIntfStatusDir_ = self.createLocalEntity(
         "AllIntfStatusDir",
         "Interface::AllIntfStatusDir",
         "interface/status/all" )

      self.vrfNameStatus_ = mg.mount( Cell.path( 'vrf/vrfNameStatus' ),
                                      'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

      self.localEntitiesCreatedIs( True )

      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

      # Shark mounts
      shmemMg = shmemEm.getMountGroup()
      RsvpSharkStatus = Tac.Type( "Rsvp::RsvpSharkStatus" )
      self.rsvpSharkStatus_ = shmemMg.doMount( RsvpSharkStatus.mountPath,
                                               "Rsvp::RsvpSharkStatus",
                                               Shark.mountInfo( 'shadow' ) )
      shmemMg.doClose()

      # Smash mounts
      fibInfo = Tac.Value( 'Smash::Fib::TableInfo' )
      forwardingMountInfo = fibInfo.forwardingInfo( 'reader' )
      readerInfo = Smash.mountInfo( 'reader' )
      keyShadowInfo = Smash.mountInfo( 'keyshadow' )
      # Only default VRF is supported for now. We will need to mount kniStatus
      # per VRF when multi-VRF support exists. See BUG124566.
      # kniStatus is a keyshadow instead of reader because MplsUtils passes it to
      # Interface::IntfStatusManager which reacts to it.
      # pylint: disable-next=consider-using-f-string
      self.kniStatus_ = shmemEm.doMount( "kni/ns/%s/status" % DEFAULT_NS,
                                         "KernelNetInfo::Status", keyShadowInfo )
      self.forwardingStatus_ = shmemEm.doMount( forwardingStatusPath,
                                                "Smash::Fib::ForwardingStatus",
                                                readerInfo )
      self.forwarding6Status_ = shmemEm.doMount( forwarding6StatusPath,
                                                 "Smash::Fib6::ForwardingStatus",
                                                 readerInfo )
      self.forwardingGenStatus_ = shmemEm.doMount( forwardingGenStatusPath,
                                                 "Smash::FibGen::ForwardingStatus",
                                                 readerInfo )

      self.routingStatus_ = shmemEm.doMount( "routing/status",
                                             "Smash::Fib::RouteStatus",
                                             readerInfo )
      self.routing6Status_ = shmemEm.doMount( "routing6/status",
                                              "Smash::Fib6::RouteStatus",
                                              readerInfo )
      self.mplsUtilsConfig_ = mg.mount( 'mplsutils/config',
                                       'MplsUtils::Config', 'r' )

      self.isisV4SmashColl_ = shmemEm.doMount( "mpls/labelBindingTables/isisV4",
                                            "CommonLibSmash::LabelBindingTable",
                                            keyShadowInfo )
      self.isisV6SmashColl_ = shmemEm.doMount( "mpls/labelBindingTables/isisV6",
                                            "CommonLibSmash::LabelBindingTable",
                                            keyShadowInfo )
      self.ospfSmashColl_ = shmemEm.doMount( "mpls/labelBindingTables/ospf",
                                            "CommonLibSmash::LabelBindingTable",
                                            keyShadowInfo )
      self.ldpCommonLibSmashColl_ = shmemEm.doMount(
         "mpls/labelBindingTables/ldp", "CommonLibSmash::LabelBindingTable",
         keyShadowInfo )
      self.bgpLuCommonLibSmashColl_ = shmemEm.doMount(
         "mpls/labelBindingTables/bgpLu", "CommonLibSmash::LabelBindingTable",
         keyShadowInfo )
      self.p2mpCommonLibSmashColl_ = shmemEm.doMount(
         "mpls/labelBindingTables/mldp", "CommonLibSmash::LabelBindingTable",
         keyShadowInfo )

      self.transitLfib_ = shmemEm.doMount( 'mpls/transitLfib', 'Mpls::LfibStatus',
                                           readerInfo )
      self.bgpLuTunnelTable_ = shmemEm.doMount(
         'tunnel/table/bgpLu', 'Tunnel::TunnelTable::BgpLuTunnelTable', readerInfo )
      self.ldpTunnelTable_ = shmemEm.doMount(
         'tunnel/table/ldp', 'Tunnel::TunnelTable::LdpTunnelTable', readerInfo )
      self.srTeSegmentListTunnelTable_ = shmemEm.doMount(
         'tunnel/table/srTeSegmentList',
         'Tunnel::TunnelTable::SrTeSegmentListTunnelTable',
         readerInfo )
      self.srTunnelTable_ = shmemEm.doMount(
         'tunnel/table/sr', 'Tunnel::TunnelTable::SrTunnelTableBase', readerInfo )
      self.ospfSrTunnelTable_ = shmemEm.doMount(
         'tunnel/table/ospfSr', 'Tunnel::TunnelTable::SrTunnelTableBase',
         readerInfo )
      self.isisFlexAlgoTunnelTable_ = shmemEm.doMount(
         'tunnel/table/isisFlexAlgoTunnel',
         'Tunnel::TunnelTable::IsisFlexAlgoTunnelTable', readerInfo )
      self.staticTunnelTable_ = shmemEm.doMount(
         'tunnel/table/static',
         'Tunnel::TunnelTable::StaticTunnelTable',
         readerInfo )
      self.decapLfib_ = shmemEm.doMount( 'mpls/decapLfib', 'Mpls::LfibStatus',
                                         readerInfo )
      self.tiLfaTunnelTable_ = shmemEm.doMount(
         'tunnel/table/tiLfa', 'Tunnel::TunnelTable::TiLfaTunnelTable', readerInfo )
      self.mldpTransitLfib_ = shmemEm.doMount( 'mpls/protoLfibInputDir/mldp',
                                               'Mpls::LfibStatus', readerInfo )
      if Toggles.RsvpToggleLib.toggleRsvpP2mpUnifiedForP2pEnabled():
         self.rsvpLfib_ = shmemEm.doMount( 'mpls/protoLfibInputDir/rsvp',
                                           'Mpls::LfibStatus', readerInfo )
      else:
         self.rsvpLfib_ = shmemEm.doMount( 'mpls/protoLfibInputDir/rsvpAlt',
                                           'Mpls::LfibStatus', readerInfo )
      self.pwLfib_ = shmemEm.doMount( 'mpls/protoLfibInputDir/pseudowire',
                                      'Mpls::LfibStatus', readerInfo )
      self.pwLabelBindingColl_ = mg.mount( 'pseudowire/agent/labelBinding',
                                'Pseudowire::LabelBindingColl', 'r' )
      self.pwAttrColl_ = mg.mount( 'pseudowire/agent/pseudowireAttributeStatusColl',
                                  'Pseudowire::PseudowireAttributeStatusColl', 'r' )
      self.bgpLuLfib_ = shmemEm.doMount( 'mpls/protoLfibInputDir/bgpLu',
                                         'Mpls::LfibStatus', readerInfo )
      self.bgpLuRib_ = shmemEm.doMount( 'tunnel/protoTunnelRib/bgpLu',
                                        'Tunnel::TunnelTable::ColoredTunnelRib',
                                        readerInfo )
      self.bgpLuLocalRouteTable_ = shmemEm.doMount(
         'routing/bgp/export/luLocalRouteTable',
         'Routing::Bgp::RouteExportTable',
         readerInfo )
      self.srTeForwardingStatus_ = shmemEm.doMount(
         'forwarding/srte/status',
         'Smash::Fib::ForwardingStatus',
         forwardingMountInfo )

      tableInfo = TunnelTableMounter.getMountInfo(
         TunnelTableIdentifier.srTeSegmentListTunnelTable ).tableInfo
      self.srTeSegmentListTunnelTable_ = SmashLazyMount.mount(
         entityManager, tableInfo.mountPath, tableInfo.tableType,
         Smash.mountInfo( 'reader' ) )

      self.trie_ = Tac.newInstance( "Routing::Trie", "trie" )
      self.v6trie_ = Tac.newInstance( "Routing6::Trie", "v6trie" )
      self.arpSmash_ = shmemEm.doMount( 'arp/status', 'Arp::Table::Status',
                                        Smash.mountInfo( 'reader' ) )
      self.intfConfigDir_ = mg.mount( 'l3/intf/config', 'L3::Intf::ConfigDir', 'r' )

      self.tunnelFib_ = shmemEm.doMount( 'tunnel/tunnelFib',
                                         'Tunnel::TunnelFib::TunnelFib',
                                         Smash.mountInfo( 'reader' ) )

      # RD paths in precedence order to get the vrf associated
      self.rdConfig_ = mg.mount( 'ip/vrf/routeDistinguisherInputDir/config/bgp',
                                 'Ip::BgpRouteDistinguisherInput', 'r' )
      self.rdAuto_ = mg.mount( 'ip/vrf/routeDistinguisherInputDir/auto/bgpAuto',
                               'Ip::BgpAutoRouteDistinguisherInput', 'r' )
      self.rdVrf_ = mg.mount( 'ip/vrf/status/global',
                              'Ip::AllVrfStatusGlobal', 'r' )
      vrf = DEFAULT_VRF
      vrfRoutingStatus = { vrf : self.routingStatus_ }
      vrfRouting6Status = { vrf : self.routing6Status_ }
      vrfTrie = { vrf : self.trie_ }
      vrfTrie6 = { vrf : self.v6trie_ }
      resolveHelperKwargs = {
         'vrfRoutingStatus' : vrfRoutingStatus,
         'vrfRouting6Status' : vrfRouting6Status,
         'forwardingStatus' : self.forwardingStatus_,
         'forwarding6Status' : self.forwarding6Status_,
         'forwardingGenStatus' : self.forwardingGenStatus_,
         'arpSmash' : self.arpSmash_,
         'trie4' : vrfTrie,
         'trie6' : vrfTrie6,
         'vrfNameStatus' : self.vrfNameStatus_,
         'intfConfigDir' : self.intfConfigDir_,
         'tunnelFib' : self.tunnelFib_,
         'srTeForwardingStatus' : self.srTeForwardingStatus_,
         'srTeSegmentListTunnelTable' : self.srTeSegmentListTunnelTable_,
      }
      global fwdingHelper
      fwdingHelper = forwardingHelperKwFactory( **resolveHelperKwargs )

      local = entityManager.root().parent
      self.root_ = None


      # pylint: disable-msg=W0201
      def _onMountsComplete():
         print ( "Mounts complete. Initializing" )
         self.root_ = local[ AgentName ].newEntity(
                                                   'MplsUtils::MplsUtilsAgentRoot',
                                                   'mplsUtilsAgentRoot' )
         self.root_.mplsUtilsRootSm = ( self.ipStatus_, self.ip6Status_,
                                        self.mplsHwStatus_, 
                                        self.mplsStatus_,
                                        self.isisV4SmashColl_,
                                        self.isisV6SmashColl_,
                                        self.ospfSmashColl_,
                                        self.ldpCommonLibSmashColl_,
                                        self.bgpLuCommonLibSmashColl_,
                                        self.p2mpCommonLibSmashColl_,
                                        self.mldpOpqTable_,
                                        self.ldpStatusColl_,
                                        self.rdConfig_,
                                        self.rdAuto_,
                                        self.rdVrf_,
                                        self.decapLfib_,
                                        self.transitLfib_,
                                        self.mldpTransitLfib_,
                                        self.rsvpLfib_,
                                        self.routingHwRouteStatus_,
                                        self.forwardingStatus_,
                                        self.forwarding6Status_,
                                        self.forwardingGenStatus_,
                                        self.routingStatus_,
                                        self.routing6Status_,
                                        entityManager.cEntityManager(),
                                        self.bgpLuTunnelTable_,
                                        self.ldpTunnelTable_,
                                        self.srTeSegmentListTunnelTable_,
                                        self.srTunnelTable_,
                                        self.ospfSrTunnelTable_,
                                        self.isisFlexAlgoTunnelTable_,
                                        self.staticTunnelTable_,
                                        self.tiLfaTunnelTable_,
                                        self.allIntfStatusDir_,
                                        self.allIntfStatusLocalDir_,
                                        self.kniStatus_,
                                        self.mplsUtilsConfig_,
                                        self.bridgingConfig_,
                                        self.rsvpStatus_,
                                        self.rsvpSharkStatus_,
                                        self.rsvpLerStatus_,
                                        self.packetTracerActiveClient_,
                                        self.packetTracerConfig_,
                                        self.packetTracerStatus_,
                                        self.packetTracerHwStatus_,
                                        self.packetTracerSwStatus_,
                                        self.subIntfConfigDir_,
                                        self.timerWheel_,
                                        self.pwLfib_,
                                        self.pwLabelBindingColl_,
                                        self.pwAttrColl_,
                                        self.bgpLuLfib_,
                                        self.bgpLuRib_,
                                        self.bgpLuLocalRouteTable_,
                                        self.schedulerRoot_ )
         print( "%s started" % AgentName ) # pylint: disable=consider-using-f-string

      mg.close( _onMountsComplete )

def main():
   container = Agent.AgentContainer( [ MplsUtilLsp ] )
   container.runAgents()

if __name__ == "__main__":
   main()
