# Copyright (c) 2016-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable-msg=W0611, W0212

# pylint: disable=consider-using-f-string

import re
import os
import sys
import pkgutil
import time
import datetime
from ipaddress import IPv4Network, IPv4Address, NetmaskValueError
from ipaddress import summarize_address_range
import json
import Tac
import PyClient
import Tracing, BothTrace
import Logging
from IpUtils import Mask
import MssPolicyMonitor.Plugin
from Arnet import IpGenAddrWithMask


AGENT_NAME = 'MssPolicyMonitor'       # seen by users from cli 'show agent'
MPM_TRACE_HANDLE =  AGENT_NAME
PLUGIN_PACKAGE = 'MssPolicyMonitor.Plugin'
PATH_FOR_API = 'MssPolicyMonitor.CliApiAccess'
DEFAULT_POLICY_TAG = 'Arista_MSS'
BTEST_WAITFOR_TIMEOUT = int( os.environ.get( 'MSSPM_TEST_TIMEOUT', 600 ) )

# Import into modules that support BothTrace to ensure proper trace facility name
__defaultTraceHandle__ = Tracing.Handle( MPM_TRACE_HANDLE )

t0 = __defaultTraceHandle__.trace0
t1 = __defaultTraceHandle__.trace1
t2 = __defaultTraceHandle__.trace2
t3 = __defaultTraceHandle__.trace3
t4 = __defaultTraceHandle__.trace4
t5 = __defaultTraceHandle__.trace5
t6 = __defaultTraceHandle__.trace6
t7 = __defaultTraceHandle__.trace7  # high-level timing info
t8 = __defaultTraceHandle__.trace8  # low-level timing info
t9 = __defaultTraceHandle__.trace9

v = BothTrace.Var
b0 = BothTrace.tracef0
b1 = BothTrace.tracef1
b2 = BothTrace.tracef2
b3 = BothTrace.tracef3
b4 = BothTrace.tracef4
b5 = BothTrace.tracef5
b6 = BothTrace.tracef6
b7 = BothTrace.tracef7
b8 = BothTrace.tracef8
b9 = BothTrace.tracef9

ENCRYPTION_KEY = '@$3m0*49ZK5ja^td4utw}8>#2!I?hS5ZlL(01O];' # ToDo: use EOS key mgmt
ZONE_A = 'zoneA'
ZONE_B = 'zoneB'
PORT_CHANNEL = 'Port-Channel'
MLAG = 'Mlag'
VIRTUAL_WIRE = 'vwire'  # interface or zone type
LAYER2 = 'layer2'  # interface or zone type
LAYER3 = 'layer3'  # interface or zone type
LINK_STATE_UP = 'Up'
LINK_STATE_DOWN = 'Down'
LINK_STATE_UNKNOWN = 'Unknown'
HA_ACTIVE_PASSIVE = 'Active-passive'
HA_ACTIVE_ACTIVE = 'Active-active'
HA_ACTIVE = 'Active'
HA_PASSIVE = 'Passive'
HA_ACTIVE_PRIMARY = 'Active-primary'
HA_ACTIVE_SECONDARY = 'Active-secondary'
HA_DEVICE_SUSPENDED = 'HA-device-suspended'
TIMESTAMP_FORMAT = '%Y %b %d, %H:%M:%S'
ARISTA_OUI = [ '001c73', '444ca8', '28993a' ]
SKIP_ATTRIBS = [ 'tacHasAttrLogInterface', 'isNondestructing', 'entity', 'name',
                 'fullName', 'parent', 'parentAttr' ]
HTTP_STATUS_CODES = {
   200: 'OK', 201: 'Created', 202: 'Accepted', 203: 'Non-Authoritative Information',
   204: 'No Content', 205: 'Reset Content', 206: 'Partial Content',
   400: 'Bad Request', 401: 'Unauthorized', 402: 'Payment Required',
   403: 'Forbidden', 404: 'Not Found', 405: 'Method Not Allowed',
   406: 'Not Acceptable', 407: 'Proxy Authentication Required',
   408: 'Request Timeout', 409: 'Conflict', 410: 'Gone', 411: 'Length Required',
   412: 'Precondition Failed', 413: 'Request Entity Too Large',
   414: 'Request-URI Too Long', 415: 'Unsupported Media Type',
   416: 'Requested Range Not Satisfiable', 417: 'Expectation Failed',
   100: 'Continue', 101: 'Switching Protocols', 300: 'Multiple Choices',
   301: 'Moved Permanently', 302: 'Found', 303: 'See Other', 304: 'Not Modified',
   305: 'Use Proxy', 306: '(Unused)', 307: 'Temporary Redirect',
   500: 'Internal Server Error', 501: 'Not Implemented', 502: 'Bad Gateway',
   503: 'Service Unavailable', 504: 'Gateway Timeout',
   505: 'HTTP Version Not Supported' }
IP_PROTOCOL = {
   1: 'ICMP', 2: 'IGMP', 4: 'IPv4', 6: 'TCP', 8: 'EGP', 17: 'UDP', 41: 'IPv6',
   46: 'RSVP', 47: 'GRE', 89: 'OSPFIGP', 132: 'SCTP' }
IP_PROTOCOL_TAC_VALUE = { proto: Tac.Value( 'Arnet::IpProto', num )
                          for num, proto in IP_PROTOCOL.items() }

# enum labels below must be in sync with tac models
HA_DISABLED_STATE = Tac.Type( 'MssPolicyMonitor::HaState' ).Disabled
HA_ACTIVE_STATE = Tac.Type( 'MssPolicyMonitor::HaState' ).Active
HA_STANDBY_STATE = Tac.Type( 'MssPolicyMonitor::HaState' ).Standby

MSSL3_ACTION_REDIRECT = Tac.Type( 'MssL3::MssAction' ).redirect
MSSL3_ACTION_BYPASS = Tac.Type( 'MssL3::MssAction' ).bypass
MSSL3_ACTION_DROP = Tac.Type( 'MssL3::MssAction' ).drop

VIRTUAL_WIRE_ZONE = Tac.Type( 'MssPolicyMonitor::ZoneType' ).vwire
LAYER3_ZONE = Tac.Type( 'MssPolicyMonitor::ZoneType' ).layer3
UNKNOWN_ZONE_TYPE = Tac.Type( 'MssPolicyMonitor::ZoneType' ).unknown

INTERCEPT_ZONE = Tac.Type( 'MssPolicyMonitor::ZoneInterceptClass' ).intercept
NON_INTERCEPT_ZONE = Tac.Type( 'MssPolicyMonitor::ZoneInterceptClass' ).nonIntercept
UNCLASSIFIED_ZONE = Tac.Type( 'MssPolicyMonitor::ZoneInterceptClass' ).unclassified

BYPASS_MODE = Tac.Type( 'MssPolicyMonitor::ExceptionHandlingMode' ).bypass
REDIRECT_MODE = Tac.Type( 'MssPolicyMonitor::ExceptionHandlingMode' ).redirect

VERBATIM = Tac.Type( 'MssPolicyMonitor::ModifierTagType' ).verbatim
FORWARD_ONLY = Tac.Type( 'MssPolicyMonitor::ModifierTagType' ).forwardOnly

INITIALIZING = Tac.Type( 'MssPolicyMonitor::MonitoringState' ).initializing
SHUTDOWN = Tac.Type( 'MssPolicyMonitor::MonitoringState' ).shutdown
ACTIVE = Tac.Type( 'MssPolicyMonitor::MonitoringState' ).active
DRY_RUN = Tac.Type( 'MssPolicyMonitor::MonitoringState' ).dryRun
SUSPEND = Tac.Type( 'MssPolicyMonitor::MonitoringState' ).suspend

MSSPM_STATE_TO_MSS_STATE = {
   SHUTDOWN: Tac.Type( 'Mss::PolicyConfigState' ).disabled,
   ACTIVE:   Tac.Type( 'Mss::PolicyConfigState' ).enabled,
   DRY_RUN:  Tac.Type( 'Mss::PolicyConfigState' ).dryrun,
}

VALID = Tac.Type( "Mgmt::Security::Ssl::ProfileState" ).valid
INVALID = Tac.Type( "Mgmt::Security::Ssl::ProfileState" ).invalid
SslFeature = Tac.Type( "Mgmt::Security::Ssl::SslFeature" )
SSL_ERROR_MSG = 'SSL Error'

UNASSIGNED = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).unassigned
TEST_PLUGIN = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).testMssPM
TEST_AGG_MGR_PLUGIN = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).testMssPMAggMgr
PAN_FW_PLUGIN = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).PaloAltoFirewall
PANORAMA_PLUGIN = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).PaloAltoPanorama
FORTIMGR_PLUGIN = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).FortinetFortiManager
CHKP_MS_PLUGIN = Tac.Type(
   'MssPolicyMonitor::PolicySourceType' ).CheckPointMgmtServer

OFFLOAD = "offload"
REDIRECT = "redirect"

CLI_TIMEOUT = 4
API_ACCESS_MSG = 'Accessing external device(s), this may take a few seconds...\n'
TIMEOUT_MSG = 'No reply from external device. Waited for {0} seconds. ' \
              'Use cli-timeout option to wait longer.'
GROUP_MEM_MSG = \
   '% This is an Aggregation Manager, network/neighbor/resources/high-availability' \
   ' information valid only for group-members of this aggregation device'

ALLOW_ACTION_SYNONYMS = [ 'ALLOW', 'ACCEPT', 'PERMIT' ]
DENY_ACTION_SYNONYMS = [ 'DENY', 'DROP', 'BLOCK' ]

# feature toggle flag for running MssPM in MultiVwire Mode
# which supporst multiple vwire without any Aggregated Ethernet Interface
# in the firewall
mssMultiVwireEnabled = False
MssL3MssPriority = Tac.Value( 'MssL3::MssPriority' )
TEST_DATA_ENV_VAR = 'MSSPM_TEST_DATA'

IP_REGEX = re.compile( r'(\d{1,3}).(\d{1,3}).(\d{1,3}).(\d{1,3})$' )
# Match <a.b.c.d>/<a.b.c.d> or <a.b.c.d>|<a.b.c.d>
IP_SUBNET_REGEX = re.compile( r'(/|\|)(\d{1,3}).(\d{1,3}).(\d{1,3}).(\d{1,3})$' )
# Match a | or /, used for splitting ip and subnet
IP_SPLIT_REGEX = re.compile( r'/|\|' )
HOST_NAME_REGEX = re.compile( r'([a-zA-Z].*?)\..*' )
SUBNET_SPLIT_REGEX = re.compile( r'\||/' )

DeviceName = Tac.Type( 'Mss::DeviceName' )

class MssDeviceName:
   def __init__( self, devName, vinstName, fwType, v2 ):
      self.devName = devName
      self.vinstName = vinstName
      self.fwType = fwType
      self.v2 = v2

   @classmethod
   def fromString( cls, strName, v2 ):
      if v2:
         deviceName = DeviceName( strName )
         devName = deviceName.getPhyInstanceName()
         vinstName = deviceName.getVirtInstanceName()
         fwType = deviceName.getFwType()
      else:
         devName = strName
         vinstName = ''
         fwType = ''
      return MssDeviceName( devName, vinstName, fwType, v2 )

   def name( self ):
      if self.v2:
         return DeviceName.encode( self.devName, self.vinstName, self.fwType )
      return self.devName

   def __str__( self ):
      return self.name()

def loadTestData( testDataEnvVarName ):
   if testDataEnvVarName in os.environ:
      with open( os.environ[ testDataEnvVarName ] ) as f:
         t2( '$$ loading test data from temp file:', f.name )
         return json.load( f )
   return {}


def shortenIntfName( aStr ):
   aStr = aStr.replace( 'Ethernet', 'Et' )
   aStr = aStr.replace( PORT_CHANNEL, 'Po' )
   return aStr


def shortenPluginName( aStr ):
   aStr = aStr.replace( 'PaloAltoFirewall', 'panFW' )
   aStr = aStr.replace( 'PaloAltoPanorama', 'panorama' )
   aStr = aStr.replace( 'FortinetFortiManager', 'fortiMgr' )
   aStr = aStr.replace( 'CheckPointMgmtServer', 'chkptMS' )
   return aStr


def bothTraceInit( name ):
   BothTrace.initialize( name + '-%d.qt' )

#-----------------------------------------------------------------------------------
def isMssMultiVwireEnabled():
   return mssMultiVwireEnabled

def isMssL3EnabledAndL3Policy( devicePolicy ):
   return bool( not devicePolicy.isL2Policy )

def isMssL3EnabledAndL3Zone( serviceDeviceZone ):
   # pylint: disable-next=consider-using-in
   return bool ( serviceDeviceZone.zoneType == LAYER3_ZONE or
                 serviceDeviceZone.zoneType == UNKNOWN_ZONE_TYPE )

def isL2RawPolicy( policy ):
   return bool(
      policy.srcZoneType == VIRTUAL_WIRE and policy.dstZoneType == VIRTUAL_WIRE or
      policy.srcZoneType == LAYER2 and policy.dstZoneType == LAYER2 )


def checkSslProfileStatus( deviceConfig, sslStatus ):
   sslProfileName = deviceConfig[ 'sslProfileName' ]
   if sslProfileName:
      profileStatus = sslStatus.profileStatus.get( sslProfileName )
      if profileStatus and profileStatus.state == VALID:
         deviceConfig[ 'trustedCertsPath' ] = profileStatus.trustedCertsPath
      elif not profileStatus or profileStatus.state == INVALID:
         deviceConfig[ 'trustedCertsPath' ] = ''


def httpStatus( code ):
   ''' Return http response status code and text description if available
   '''
   if code in HTTP_STATUS_CODES:
      return f'{code} {HTTP_STATUS_CODES[ code ]}'
   else:
      return code


def getTimestamp():
   return Tac.utcNow()


def timestampToStr( ts ):
   return datetime.datetime.fromtimestamp( ts ).strftime( TIMESTAMP_FORMAT )


def isManagementIntfLinkUp( intfStatusDir ):
   ''' Check /ar/Sysdb/interface/status/eth/intf for a Management interface
       with attribute linkStatus="linkUp".
   '''
   for intfName, intfStatus in intfStatusDir.items():
      if intfName.startswith( 'Management' ) and intfStatus.linkStatus == 'linkUp':
         return True
   return False


def waitForManagementIntfReady( intfStatusDir, fibReadyDir, vrfReadyStatusDir,
                                sleep=5 ):
   while not isManagementIntfLinkUp( intfStatusDir ):
      t2( 'Management interface is not up, checking again in', sleep, 'seconds' )
      time.sleep( sleep )


def hidePassword( deviceConfig ):
   return { k: v for k, v in deviceConfig.items()
            if k not in [ 'username', 'password' ] }


def createNewDeviceSet( cfg, deviceSetName ):
   return cfg.newDeviceSet( deviceSetName )


def setPolicySourceType( config, psType ):
   config.policySourceType = psType


def createNewServiceDevice( cfg, ipdns ):
   return cfg.newServiceDevice( ipdns )


def listAllDeviceSets( cfg, name ):
   return [ k for k in cfg.deviceSet
            if cfg.deviceSet[ k ].policySourceType == name ]


def setAggrMgr( cfg, val ):
   cfg.isAggregationMgr = val


def getInterfaceMap( serviceDevice ):
   return { intf: dictFromTacNominal( imap )
            for intf, imap in serviceDevice.intfMap.items() }


def getAggMgrIntfMap( deviceSet ):
   aggMgrIntfMap = {}
   for deviceId, serviceDevice in deviceSet.serviceDevice.items():
      if serviceDevice.isAccessedViaAggrMgr:  # find any map-only device entries
         intfMap = getInterfaceMap( serviceDevice )
         if intfMap:
            aggMgrIntfMap[ deviceId ] = intfMap
   return aggMgrIntfMap


def mergeNeighborsWithIntfMap( lldpNeighbors, deviceId, deviceSet ):
   ''' Merges the passed lldpNeighbor dict with any overrides configured
       via CLI in the service device submode via 'map device-interface'
   '''
   t3( 'mergeNeighborsWithIntfMap deviceId:', deviceId, 'deviceSet:', deviceSet )
   cliDeviceIntfMap = {}
   if deviceSet.isAggregationMgr:
      aggMgrIntfMap = getAggMgrIntfMap( deviceSet )
      if deviceId in aggMgrIntfMap: # pylint: disable=consider-using-get
         cliDeviceIntfMap = aggMgrIntfMap[ deviceId ]
   else:
      serviceDevice = deviceSet.serviceDevice[ deviceId ]
      cliDeviceIntfMap = getInterfaceMap( serviceDevice )

   if cliDeviceIntfMap:
      t3( 'merging device intfMap', cliDeviceIntfMap, 'w/ LLDP nbors',
          lldpNeighbors )
      lldpNeighbors.update( cliDeviceIntfMap )
   return lldpNeighbors

def getMssAction( devPolicy ):
   '''
   Get MssL3::Action value from MssPM status device policy
   '''
   if devPolicy.isOffloadPolicy:
      # offload or offload verbatim
      if devPolicy.action.upper() in ALLOW_ACTION_SYNONYMS:
         return MSSL3_ACTION_BYPASS
      if devPolicy.action.upper() in DENY_ACTION_SYNONYMS:
         return MSSL3_ACTION_DROP
      b0( 'unknown offload policy action:', v( devPolicy.action ) )
      return ''
   # redirect verbatim
   return MSSL3_ACTION_REDIRECT

def isRawPolicyVrfValid( rawPolicy, policyList ):
   if ( len( policyList.firewallVrf ) > 1 and
      not rawPolicy.srcZoneInterfaces and
      not rawPolicy.dstZoneInterfaces ):
      b1( 'skip policy ', v( rawPolicy.name ),
          ' zones not defined while device manages multiple vrfs' )
      return False
   return True

def getDefaultVrfName( firewallType ):
   if firewallType == FORTIMGR_PLUGIN:
      return '0'
   return 'default'

def setDefaultVrf( routingTable, firewallType ):
   '''
   Set default VRF name as "default".
   This is to be used in best-effort mode since VRF/NetVRF mapping is not handled.
   '''
   defaultVrfName = getDefaultVrfName( firewallType )
   if defaultVrfName != "default":
      routingTable[ "default" ] = routingTable[ defaultVrfName ]
      del routingTable[ defaultVrfName ]

def getDevicePolicyVrf( policyList, ceEnabled ):
   '''
   Get device policy VRF. It is assumed that both source and destination zones are
   on the same VRF.
   '''
   policies = list( policyList.devicePolicy.values() )
   if not ceEnabled:
      # Only 'default' VRF is supported
      policies.sort( key=lambda pol: pol.number )
      return { "default" : policies }

   if len( policyList.firewallVrf ) <= 1:
      # Only one VRF in the virtual instance
      policies.sort( key=lambda pol: pol.number )
      return { next( iter( policyList.firewallVrf ) ) : policies }

   perVrfPolicies = {}
   for pol in policies:
      if pol.zoneA.link:
         policyVrf = list( pol.zoneA.link.values() )[ 0 ].vrf
      elif pol.zoneB.link:
         policyVrf = list( pol.zoneB.link.values() )[ 0 ].vrf
      else:
         assert False, "Either source or destination zone must be defined"
      perVrfPolicies.setdefault( policyVrf, [] ).append( pol )

   for polList in perVrfPolicies.values():
      polList.sort( key=lambda pol: pol.number )

   return perVrfPolicies

def getDeviceVrf( serviceDevice, deviceSet ):
   '''
   Collect all the service device vrf (i.e. virtual routers, vrf id) that must be
   handled by the policy monitor plugin.
   '''
   vrfs = set()
   for vInstConfig in serviceDevice.virtualInstance.values():
      if vInstConfig.firewallVrf:
         vrfs |= set( vInstConfig.firewallVrf.keys() )
      else:
         vrfs.add( getDefaultVrfName( deviceSet.policySourceType )  )

   return vrfs

def getDeviceConfig( deviceSet, serviceDevice ):
   ''' Generate service device config dict for passing to policy monitor
       plugin constructor.
   '''
   tags = list( deviceSet.policyTag ) + list( deviceSet.offloadTag )
   return {
      # device set attribs
      'deviceSet': deviceSet.name,
      'serviceDeviceType': deviceSet.policySourceType,
      'queryInterval': deviceSet.queryInterval,
      'timeout': deviceSet.timeout,
      'retries': deviceSet.retries,
      'policyTags': tags,
      'isAggregationMgr': deviceSet.isAggregationMgr,
      'verifyCertificate': deviceSet.verifyCertificate,
      'exceptionMode': deviceSet.exceptionHandling,
      'virtualDomain': deviceSet.virtualDomain, # Fortinet VDOM
      'adminDomain': deviceSet.adminDomain, # Fortinet ADOM
      'extAttr': deviceSet.extAttr, # extra attributes used only for btests

      # service device attribs
      'vrf': serviceDevice.vrf,
      'virtualInstance' : list( serviceDevice.virtualInstance ),
      'vrouters' : getDeviceVrf( serviceDevice, deviceSet ),
      'ipAddress': serviceDevice.ipAddrOrDnsName,
      'username': serviceDevice.username,
      'password': serviceDevice.password,
      'protocol': serviceDevice.protocol,
      'protocolPortNum': serviceDevice.protocolPortNum,
      'sslProfileName': serviceDevice.sslProfileName,
      'group': serviceDevice.group, # for aggregationManagers
      'mgmtIntfVdom': serviceDevice.mgmtIntfVirtualDomain  # Fortinet VDOM
      }

def checkAggrMgrPair( deviceSet ):
   ''' Check if there are two aggregation managers in config
       The caller must grab the ActivityLock
   '''
   return deviceSet.isAggregationMgr and sum( not servDev.isAccessedViaAggrMgr
                                              for servDev in
                                              deviceSet.serviceDevice.values() ) == 2

def printObjectType( obj ):
   print( 'OBJ= %s\nTYPE= %s\nDir=%s  Entity=%s  Value=%s  Coll=%s  '
         'EntityProxy=%s  isColl=%s  ' % (
      obj, type( obj ),
      isinstance( obj, Tac.Type( "Tac::Dir" ) ),
      isinstance( obj, Tac.Type( "Tac::Entity" ) ),
      Tac.isValue( obj ),
      Tac.isCollection( obj ),
      isinstance( obj, PyClient.EntityProxy ),
      obj.isCollection() if isinstance( obj, PyClient.EntityProxy ) else ''
       ) )

def dumpSysdbObj( obj, indent=0, attrTypes=False ):
   ''' basic type is one of: Entity, Value, Collection, other (e.g. bool)
       Entities and values have attributes, Entity also has a name
       Collections have keys and values (no attributes or name)
       printObjectType( obj )
   '''
   spc = ' ' * indent
   if isinstance( obj, Tac.Type( "Tac::Dir" ) ):
      print( 'object is a Tac::Dir, subdirs: %s' % list( obj.subdir ) )
      for subdir in obj.subdir:
         dumpSysdbObj( subdir, indent + 3 )
   elif ( isinstance( obj, Tac.Type( "Tac::Entity" ) ) or
          Tac.isValue( obj ) or
          ( isinstance( obj, PyClient.EntityProxy ) and not obj.isCollection() ) ):
      for attrName in obj.attributes:
         if attrName in SKIP_ATTRIBS:
            continue
         try:
            if isinstance( obj, PyClient.EntityProxy ):
               attr = obj.__getattr__( attrName )
            else:
               attr = obj.__getattribute__( attrName )
         except Exception as ex:  # pylint: disable-msg=W0703
            print( 'Error: Unable to get attribute:', attrName, ex )
            continue
         atype = type( attr ) if attrTypes else ''
         if ( Tac.isCollection( attr ) or
              ( isinstance( attr, PyClient.EntityProxy ) and attr.isCollection ) ):
            if obj.tacType.attr( attrName ).instantiating:
               print( 'IC %s%-25s  %-40s %s' % ( spc, attrName, '', atype ) )
            else:
               print( 'KC %s%-25s  %-40s %s' % ( spc, attrName, '', atype ) )
               #print 'KC %s%-25s  %-40s %s' % ( spc, attrName, attr.keys(), atype)
            dumpSysdbObj( attr, indent + 3 )
         elif isinstance( attr, Tac.Type( "Tac::Entity" ) ):
            print( 'EN %s%-25s  %-40s %s' % ( spc, attrName, attr, atype ) )
         elif isinstance( attr, PyClient.EntityProxy ):
            print( 'EP %s%-25s  %-40s %s' % ( spc, attrName, attr, atype ) )
         elif Tac.isValue( attr ):
            print( 'VL %s%-25s  %-40s %s' % ( spc, attrName, attr, atype ) )
         else:
            print( '.  %s%-25s  %-40s %s' % ( spc, attrName, attr, atype ) )
   elif ( Tac.isCollection( obj ) or
          ( isinstance( obj, PyClient.EntityProxy ) and obj.isCollection() ) ):
      for key, valObj in obj.items():
         print( 'C  %s%-25s  %-40s' % ( spc, key,
                                       type( valObj ) if attrTypes else '' ) )
         dumpSysdbObj( valObj, indent )
         print()
   elif isinstance( obj, bool ):
      pass  # skip bools
   else:
      print( f'Error: {obj} has unknown type: {type( obj )}' )
      printObjectType( obj )


def dumpTacEntity( obj, indent=0, attrTypes=False ):
   spaces = ' ' * indent
   print( f'{spaces}{obj.name}  {type( obj )}' )
   for attrName in obj.attributes:
      if attrName in SKIP_ATTRIBS:
         continue
      attr = obj.__getattribute__( attrName )
      print( '%s%-25s  %-40s %s' % ( spaces, attrName, attr,
                                    type( attr ) if attrTypes else '' ) )


def dumpDevicePolicy( dp ):
   out = '{}  Dev: {}  {}  Action: {}  Pol#: {}  Tags: {}  L4: {}'.format(
      dp.policyName, dp.serviceDeviceId, dp.serviceDeviceType, dp.action,
      dp.number, list( dp.tag ),
      [ ( l4.protocol, list( l4.port ) ) for l4 in dp.dstL4App.values() ] )
   out += '\n ZoneA: %s' % dumpServiceDeviceZone( dp.zoneA )
   out += '\n ZoneB: %s' % dumpServiceDeviceZone( dp.zoneB )
   if dp.childServicePolicy:
      out += '\n ChildServicePolicy: %s' % list( dp.childServicePolicy )
   return out

def dumpIpAndZone( srcIps, srcIpRanges, dstIps, dstIpRanges, dp ):
   out = ''
   if srcIps:
      out += 'srcIps: %s ' % [ ip.stringValue for ip in srcIps ]
   if srcIpRanges:
      out += 'srcIpRange: %s ' % (
         [ f'{ipRange.startIp.stringValue}-{ipRange.endIp.stringValue}'
           for ipRange in srcIpRanges ] )
   if dstIps:
      out += 'dstIps: %s ' % [ ip.stringValue for ip in dstIps ]
   if dstIpRanges:
      out += 'dstIpRange: %s ' % (
         [ f'{ipRange.startIp.stringValue}-{ipRange.endIp.stringValue}'
           for ipRange in dstIpRanges ] )
   if dp.zoneA.zoneName:
      out += 'zoneA: %s ' % dp.zoneA.zoneName
   if dp.zoneB.zoneName:
      out += 'zoneB: %s ' % dp.zoneB.zoneName
   return out

def dumpServiceDeviceZone( zone ):
   return '{}  Type: {}  Link: {}  Icept: {}'.format(
      zone.zoneName, zone.zoneType, dumpServiceDeviceLink( zone.link ),
      # pylint: disable-next=unnecessary-comprehension
      [ icpt for icpt in zone.rawIntercept ] )
      # [ icpt.ipAddr.stringValue for icpt in zone.intercept.values() ] )


def dumpServiceDeviceLink( links, msg='' ):
   out = 'Dumping: %s \n' % msg if msg else ''
   for link in links.values():
      ip = link.ipAddr
      out += '{}  IP: {}  SW: {}  {}  PO: {}  MLAG: {}  MLAGSID: {}  VL={}\n'.format(
         link.serviceDeviceIntf,
         ip if ip and not ( ip.isAddrZero and not ip.isUnspecified ) else '',
         link.switchId, link.switchIntf, link.portChannel, link.mlag,
         link.mlagSystemId, ','.join( link.allowedVlan.keys() ) )
   return out


def dumpMssL3ServiceDevice( svcDevice ):
   out = 'Dumping MssL3.ServiceDevice: %s\n' % svcDevice.name
   for vrf in svcDevice.vrf.values():
      out += ' vrf: %s\n' % vrf.vrfName
      for intf in vrf.l3Intf.values():
         out += '  intf: {}  subnets: {}\n'.format(
            intf.ip, [ i.stringValue for i in intf.reachableSubnet ] )
   return out


def dumpMssL3PolicySet( policySet ):
   out = 'Dumping MssL3.PolicySet: %s\n' % policySet.name
   for policyCfg in policySet.policy.values():
      out += ' policyConfig for VRF: %s\n' % policyCfg.vrfName
      for rule in policyCfg.rule.values():
         out += '  rule priority: {}  action: {}  seq {} \n'.format(
            rule.priority, rule.action, rule.seqNo )
         out += '   match: src{}{} L4:{}  dst{}{} L4:{}{} L3:{}\n'.format(
            [ ip.stringValue for ip in rule.match.srcIp ],
            [ f'{r.startIp.stringValue}-{r.endIp.stringValue}'
              for r in rule.match.srcIpRange ],
            [ ( IP_PROTOCOL.get( p.proto, p.proto ), p.port )
              for p in rule.match.srcL4App ],
            [ ip.stringValue for ip in rule.match.dstIp ],
            [ f'{r.startIp.stringValue}-{r.endIp.stringValue}'
              for r in rule.match.dstIpRange ],
            [ ( IP_PROTOCOL.get( p.proto, p.proto ), p.port )
              for p in rule.match.dstL4App ],
            [ ( IP_PROTOCOL.get( p.proto, p.proto ),
                f'{p.startPort}-{p.endPort}' )
              for p in rule.match.dstL4AppRange ],
            [ IP_PROTOCOL.get( p, '' ) for p in rule.match.l3App ]
         )
   return out


def dumpTopology( topologyStatus, msg='' ):
   # caller should have Tac.ActivityLock, if other threads may access topologyStatus
   out = 'Dumping: %s \n' % msg if msg else ''
   for mlag in topologyStatus.mlagHost.values():
      out += f'\n\nMlag: {mlag.systemId}  {mlag.domainId}'
      for mlagPort in mlag.mlagPort:
         out += '\nmlagPort : %s' % mlagPort.name
         for peer, po in mlagPort.portGroup.items():
            out += f'\npeer: {peer} {po.name}'

   for host in topologyStatus.host.values():
      out += f'\n\nHost: {host.name}  {host.hostname}'
      out += '\nmlagSystemId: %s' % host.mlagSystemId
      out += '\nmlagDomainId: %s' % host.mlagDomainId
      out += '\nmlagState: %s' % host.mlagState
      out += '\nmlagPeerLink: %s' % host.mlagPeerLink

      for logicalPort in host.logicalPort:
         out += '\nlogicalPort: %s' % logicalPort
      for name, port in host.port.items():
         if port.portGroup and port.portGroup.mlag:
            out += '\nport: {} {} {}'.format(
               name, port.portGroup.name, port.portGroup.mlag.name )
         elif port.portGroup:
            out += f'\nport: {name} {port.portGroup.name}'
      for po in host.portGroup:
         out += '\nportGroup: %s' % po
   return out

def dictFromTacNominal( nom ):
   independentAttributes = [
      attr.name for attr in nom.tacType.attributeQ if attr.isIndependentDomainAttr ]
   return { k: getattr( nom, k ) for k in independentAttributes }


def hostnameOnly( fqdn ):
   mo = HOST_NAME_REGEX.match( fqdn )
   return mo.group( 1 ) if mo else fqdn


def isValidIpAddr( ip ):
   ''' check if valid IPv4 address
   '''
   if ip and ip != '0.0.0.0':
      mo = IP_REGEX.match( ip )
      if mo:
         return all( int( mo.group( j ) ) <= 255 for j in range( 1, 5 ) )
   return False


def isIpRange( addrField ):
   if '-' in addrField:
      startIp = addrField.split( '-' )[ 0 ].strip()
      endIp = addrField.split( '-' )[ 1 ].strip()
      if isValidIpAddr( startIp ) and isValidIpAddr( endIp ):
         return True
   return False


def isIpSubnet( addrField ):
   def maskToPrefixLen( mask ):
      if mask.isdigit():
         return int( mask )
      if isValidIpAddr( mask ):
         try:
            return IPv4Network( str( '255.255.255.255/%s' % mask ),
                                strict=False ).prefixlen
         except NetmaskValueError:
            return -1
      return -1

   parts = SUBNET_SPLIT_REGEX.split( addrField )
   if len( parts ) == 2:
      ip = parts[ 0 ].strip()
      mask = maskToPrefixLen( parts[ 1 ].strip() )

      return isValidIpAddr( ip ) and isValidArnetSubnet( f'{ip}/{mask}' )
   return False

def convertIpMaskSyntax( addrField ):
   addrField = addrField.replace( " ", "" )
   # Convert <ip>|<subnet> and <ip>/<subnet> to <ip>/<mask> notation
   if IP_SUBNET_REGEX.search( addrField ):
      ip, mask = IP_SPLIT_REGEX.split( addrField )
      try:
         mask = Mask( mask )
      except ValueError as ex:
         t0( 'Invalid Mask:', addrField, ex )
         return addrField
      return ip + '/' + str( mask.maskLen )
   return addrField

def isIpV4AddrOrRangeOrSubnet( ipStr ):
   try:
      return isValidIpAddr( ipStr ) or isIpRange( ipStr ) or isIpSubnet( ipStr )
   except Exception as ex:  # pylint: disable-msg=W0703
      t2( 'isIpV4AddrOrRangeOrSubnet arg:', ipStr, 'exc:', ex )
      return False

def isValidArnetSubnet( subnet, strictCheck=False ):
   try:
      if not strictCheck:
         # If strictCheck is disabled allow auto-correcting of the IPv4 prefix
         Tac.Value( 'Arnet::IpGenPrefix',
                     IpGenAddrWithMask( subnet ).subnet.v4Prefix.stringValue )
      else:
         # strictCheck enabled, use the IPv4 subnet as provided
         Tac.Value( 'Arnet::IpGenPrefix', subnet )
   except ( IndexError, ValueError ):  # pylint: disable-msg=W0703
      t2( 'invalid Arnet subnet:', subnet )
      return False
   return True

def expandIpRange( addrField ):
   ''' Convert address ranges and subnets to a list of individual IP addresses '''
   addrs = []
   if isValidIpAddr( addrField ):  # /32 host
      return [ addrField ]
   elif isIpRange( addrField ):
      try:
         startIp = IPv4Address( str(
      addrField.split( '-' )[ 0 ].strip() ) )
         endIp = IPv4Address( str( addrField.split( '-' )[ 1 ].strip() ) )
         return [ str( ip ) for subnet in summarize_address_range( startIp, endIp )
                  for ip in subnet ]
      except Exception as ex:  # pylint: disable-msg=W0703
         t0( 'unable to expand IP range:', addrField, ex )
   elif isIpSubnet( addrField ):
      try:
         if '|' in addrField:  # internal format
            addrField = addrField.replace( '|', '/' )

         try: # Try to parse the subnet
            ips = IPv4Network( str( addrField ) )
         except ValueError: # Subnet has host bits set, ie: 1.0.0.10/30
            # If IPv4Network cannot parse the IP, then it should be able to parse
            # the IP after being corrected 1.0.0.8/30 to align with the subnet
            # boundary
            ips = IPv4Network( str(
                  IpGenAddrWithMask( addrField ).subnet.v4Prefix.stringValue ) )
         finally:
            addrs = [ str( ip ) for ip in ips ]

      except Exception as ex:  # pylint: disable-msg=W0703
         t0( 'unable to expand IP subnet:', addrField, ex )
   else:
      t0( 'Invalid IP address field:', addrField )
   return addrs


def filterOnStartsWith( aString, prefixList ):
   return any( aString.startswith( prefix ) for prefix in prefixList )


def appendOrExtend( aList, item ):
   if isinstance( item, list ):
      aList.extend( item )
   else:
      aList.append( item )


def purgeNonCurrentKeys( aDict, currentKeys ):
   for key in aDict:
      if key not in currentKeys:
         del aDict[ key ]


def dumpDict( aDict, msg='' ):
   out = 'Dumping: %s \n' % msg if msg else ''
   for k, val in aDict.items():
      out += f'{k}   {val}\n'
   return out


def dumpList( aList, msg='' ):
   out = 'Dumping: %s \n' % msg if msg else ''
   for item in aList:
      out += '%s\n' % item
   return out


def agentStats( mpmStatus ):
   out = 'MssPM Agent Stats: startTime: {}  numPoliciesProcessed: {}'.format(
      mpmStatus.startTime, mpmStatus.numPoliciesProcessed )
   return out


def getPlugin( policySourceType ):
   ''' Returns the requested PolicyMonitor plugin if it exists.
   '''
   if policySourceType in _registeredPlugins:
      return _registeredPlugins[ policySourceType ]
   else:
      _loadPolicymonPlugins()  # load/reload plugins
      if policySourceType in _registeredPlugins:
         return _registeredPlugins[ policySourceType ]
      else:
         return None


def registerPlugin( pluginName, pluginModule ):
   ''' called by each plugin to register itself
   '''
   t4( 'register MPM plugin for:', pluginName, '=', pluginModule )
   _registeredPlugins[ pluginName ] = pluginModule

#-----------------------------------------------------------------------------------
# private functions
_registeredPlugins = {}

def _loadPolicymonPlugins():
   ''' force loading/reloading of all modules in plugin directory
   '''
   pluginPath = sys.modules[ PLUGIN_PACKAGE ].__path__[ 0 ]
   t1( 'loading policy monitor plugins from path:', pluginPath )
   for _, moduleName, _ in pkgutil.iter_modules( path=[ pluginPath ] ):
      __import__( PLUGIN_PACKAGE + '.' + moduleName )
   t2( 'registered msspm plugins:\n', dumpDict( _registeredPlugins ) )


#-----------------------------------------------------------------------------------
# unit tests
def test():
   print( 'Running unit tests' )
   # print expandIpRange( 'ip-addr' )
   # print expandIpRange( '5.4.3.2.1-1.1' )
   # print expandIpRange( '4.3.2.1-4.3.2.0' )
   # print expandIpRange( '5.4.3.2.1/30' )
   # print expandIpRange( '10.0.0.0|0.0.0.0' )
   # print expandIpRange( '10.0.0.0|abc' )
   # print expandIpRange( '10.0.0.0|255.255.248.248' )
   # print expandIpRange( '10.0.0.1/32' )
   # print expandIpRange( '10.0.0.1|255.255.255.255' )
   # print expandIpRange( '10.0.0.1-10.0.0.1' )
   # print expandIpRange( '10.0.0.1-10.0.0.10' )
   # print expandIpRange( '10.0.0.0/30' )
   # print expandIpRange( '10.0.0.4/30' )
   # print expandIpRange( '10.0.0.0/24' )
   # print expandIpRange( '10.0.0.0/22' )

   print( expandIpRange( '10.0.0.1/32' ) )
   print( expandIpRange( '10.0.0.1-10.0.0.6' ) )
   print( expandIpRange( '10.0.0.0/30' ) )
   print( expandIpRange( '10.0.0.0/255.255.255.252' ) )
   print( expandIpRange( '10.0.0.0|255.255.255.252' ) )

   assert expandIpRange( '10.0.0.1/32' ) == ['10.0.0.1']
   assert expandIpRange( '10.0.0.1-10.0.0.6' ) == \
      ['10.0.0.1', '10.0.0.2', '10.0.0.3', '10.0.0.4', '10.0.0.5', '10.0.0.6']
   assert expandIpRange( '10.0.0.0/30' ) == \
          expandIpRange( '10.0.0.0|255.255.255.252' )
   assert expandIpRange( '10.0.0.0/30' ) == ['10.0.0.0', '10.0.0.1', '10.0.0.2',
                                             '10.0.0.3']
   assert expandIpRange( '10.0.0.0/29' ) == \
          expandIpRange( '10.0.0.0|255.255.255.248' )
   assert expandIpRange( '10.0.0.0/29' ) == \
          ['10.0.0.0', '10.0.0.1', '10.0.0.2', '10.0.0.3', '10.0.0.4', '10.0.0.5',
           '10.0.0.6', '10.0.0.7']
   assert expandIpRange( '10.0.0.0/24' ) == expandIpRange( '10.0.0.0|255.255.255.0' )
   assert expandIpRange( '10.0.0.10/30' ) == expandIpRange( '10.0.0.8/30' )
   assert len( expandIpRange( '10.0.0.0/24' ) ) == 256
   assert len( expandIpRange( '10.0.0.0/20' ) ) == 4096
   assert len( expandIpRange( '10.0.0.0/16' ) ) == 65536
   print( 'ALL tests PASS' )

if __name__ == '__main__':
   test()

