#!/usr/bin/env python3
#
# Copyright (c) 2017-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable=W0231

import sys
from MssPolicyMonitor.Plugin import CheckPointApi
from MssPolicyMonitor.PluginLib import ( IServiceDevicePlugin, IAggregationMgrPlugin,
                                         IHAStatePlugin, IPolicyPlugin )
from MssPolicyMonitor.Lib import registerPlugin, CHKP_MS_PLUGIN

registerPlugin( CHKP_MS_PLUGIN, sys.modules[ __name__ ] )

managementServers = {}

def getAggMgrPluginObj( deviceConfig ):
   aggMgrPluginObj = CheckPointMgmtServerPlugin( deviceConfig )
   managementServers[ deviceConfig[ 'ipAddress' ] ] = aggMgrPluginObj.deviceApi
   return aggMgrPluginObj


def getPluginObj( deviceConfig, aggMgrMemberId=None ):
   return CheckPointGatewayPlugin( deviceConfig, aggMgrMemberId )


####################################################################################
class CheckPointMgmtServerPlugin( IAggregationMgrPlugin, IPolicyPlugin ):
   ''' MSS Service Policy Monitor Plugin for Check Point Software
       Security Management Server.
   '''
   def __init__( self, deviceConfig ):
      self.deviceApi = CheckPointApi.CheckPointMgmtServer( deviceConfig )
      self.deviceConfig = deviceConfig

   def getDeviceInfo( self ):
      ''' Returns selected device info
      '''
      return self.deviceApi.getDeviceInfo()

   def getAggMgrGroupMembers( self, groupName=None ):
      ''' Return a list of firewall/gateways accessible from this
          Security Management Server
      '''
      # getAllGatewayIPAddrs() returns a dict. The list of members is the keys
      return list( self.deviceApi.getAllGatewayIPAddrs() )

   def getPolicies( self, mssTags=None ):
      ''' Returns a list of ServiceDevicePolicy objects
      '''
      return self.deviceApi.getPolicies( mssTags=mssTags )

   def closeApiConnection( self ):
      ''' Close any open connections to the service device
      '''
      return self.deviceApi.closeApiConnection()

####################################################################################
class CheckPointGatewayPlugin( IHAStatePlugin, IServiceDevicePlugin ):
   ''' MSS Service Policy Monitor Plugin for Check Point Firewalls
   '''
   def __init__( self, deviceConfig, mgmtIp ):

      if deviceConfig[ 'ipAddress' ] in managementServers:
         mgmtServer = managementServers[ deviceConfig[ 'ipAddress' ] ]
      else:
         mgmtServer = None
      self.deviceApi = CheckPointApi.CheckPointGateway(
         deviceConfig, mgmtIp, mgmtServer=mgmtServer )

   def getDeviceInfo( self ):
      ''' Return dict with at least these keys: 'ipAddr', 'name', 'model'
      '''
      return self.deviceApi.getDeviceInfo()

   def getHighAvailabilityState( self ):
      ''' Returns a ServiceDeviceHAState object with current
          High Availability State for the service device.
      '''
      return self.deviceApi.getHighAvailabilityState()

   # pylint: disable-next=invalid-overridden-method
   def isSingleLogicalDeviceHaModel( self ):
      return False

   def getInterfacesInfo( self, resolveZoneNames=True ):
      ''' Returns a list of NetworkInterface objects
      '''
      return self.deviceApi.getInterfacesInfo( resolveZoneNames=resolveZoneNames )

   def getInterfaceNeighbors( self ):
      ''' Returns a dict of service device neighbor links
      '''
      return self.deviceApi.getInterfaceNeighbors()

   def getDeviceResources( self ):
      ''' Returns a dict with device resource info
      '''
      return self.deviceApi.getDeviceResources()

   def getDeviceRoutingTables( self ):
      ''' Returns a ServiceDeviceRoutingTables object
      '''
      return self.deviceApi.getDeviceRoutingTables()

   def closeApiConnection( self ):
      ''' Close any open connections to the service device
      '''
      return self.deviceApi.closeApiConnection()


####################################################################################
# tests
def testAll():
   devDict = {
      'ipAddress': 'bizdev-chkp-ms', 'username': 'admin', 'password': 'arista123',
      'protocol': 'https', 'protocolPortNum': 443, 'method': 'tls',
      'verifyCertificate': False, 'timeout': 5, 'retries': 2,
      'exceptionMode': 'bypass', 'group': 'MSS_Layer',
      'interfaceMap': {
         'eth1-01': {
            'switchIntf': 'Ethernet11', 'switchChassisId': '001c.7374.819e' },
         'eth1-02': {
            'switchIntf': 'Ethernet12', 'switchChassisId': '001c.7374.819e' }, } }

   ms = getAggMgrPluginObj( devDict )
   info = ms.getDeviceInfo()
   print( 'MS DeviceInfo:', info, '\n' )

   policies = ms.getPolicies()
   print( 'Policies from Management Server:' )
   for device, pols in policies.items():
      print( 'Firewall:', device )
      for pol in pols:
         print( ' ', pol, '\n' )

   aggMembers = ms.getAggMgrGroupMembers()
   print( 'AggMgrGroupMembers:', aggMembers )
   ms.closeApiConnection()

   gateways = list( policies )
   for gwIP in gateways:
      print( '\n' +'Gateway API:', gwIP, '\n', '-' * 60, '\n' )
      gw = getPluginObj( devDict, gwIP )

      info = gw.getDeviceInfo()
      print( 'GW DeviceInfo:', info, '\n' )

      ha = gw.getHighAvailabilityState()
      print( '\n' +'HAState:', ha, 'isHaPassiveOrSecondary:',
             ha.isHaPassiveOrSecondary(), '\n' )

      r = gw.getDeviceResources()
      print( 'Resources:\n', r[ 'resourceInfo' ] )

      intfs = gw.getInterfacesInfo()
      print( 'Interfaces:' )
      for intf in intfs:
         print( intf )

      nbors = gw.getInterfaceNeighbors()
      print( '\nLLDP_Neighbors:' )
      for intf, nbor in nbors.items():
         print( intf, 'neighbor:', nbor )

      gw.closeApiConnection()

if __name__ == "__main__":
   testAll()
