# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import abc

from collections import namedtuple
from enum import Enum

from MssCliLib import MssL3PolicyAction, MssL3V2PolicyAction
from MssCliLib import mssL3ModifierVerbatim, mssL3V2ModifierVerbatim
from MssCliLib import mssL3ModifierForwardOnly, mssL3V2ModifierForwardOnly
from MssCliLib import mssL3ModifierReverseOnly, mssL3V2ModifierReverseOnly
from MssCliLib import l3Protocol, l4Protocol
import Tac

defaults = Tac.Value( "Mss::CliDefaults" )

Direction = namedtuple( 'Direction', 'forward reverse' )
ERROR_MSG_DEFAULT_VRF = ( 'Non-defaut VRF are not supported in ' +
                          'best-effort policy consistency' )
ERROR_MSG_MULTI_IPREDIRECT = 'Only one IP redirect rule is supported.'
ERROR_MSG_PRIORITY = 'An IP redirect rule should have the lowest priority.'
ERROR_MSG_REDUNDANT_ATTRIBUTE = 'An IP redirect rule requires only IP addresses.'
ERROR_MSG_IP_REDIRECT_DIRECTION = 'Direction cannot be set for an IP redirect rule'
ERROR_MSG_EXCEED_VERBATIM_LIMIT = 'The number of verbatim rules exceeds the limit.'

DeviceName = Tac.Type( 'Mss::DeviceName' )
mssL3DeviceCliVInst = 'root'

mssL3PolicyCliSource = 'Static'
mssL3PolicyCliTag = 'CLI'

MssL3PolicyPriority = Tac.Type( 'MssL3::MssPriority' )
mssL3MaxVerbatimRuleNum = MssL3PolicyPriority.max - MssL3PolicyPriority.min

MssL3V2PolicyPriority = Tac.Type( 'MssL3V2::MssPriority' )
mssL3V2MaxVerbatimRuleNum = MssL3V2PolicyPriority.max - MssL3V2PolicyPriority.min

defaultMssL3PolicyAction = MssL3PolicyAction.drop

MssL3PolicyActionCli = Enum( 'MssL3PolicyActionCli',
                             'drop, forward, redirect, ipRedirect' )
MssL3PolicyActionCliToTac = {
   MssL3PolicyActionCli.drop: MssL3PolicyAction.drop,
   MssL3PolicyActionCli.forward: MssL3PolicyAction.bypass,
   MssL3PolicyActionCli.redirect: MssL3PolicyAction.redirect,
   MssL3PolicyActionCli.ipRedirect : MssL3PolicyAction.redirect
}
MssL3V2PolicyActionCliToTac = {
   MssL3PolicyActionCli.drop: MssL3V2PolicyAction.drop,
   MssL3PolicyActionCli.forward: MssL3V2PolicyAction.bypass,
   MssL3PolicyActionCli.redirect: MssL3V2PolicyAction.redirect,
   MssL3PolicyActionCli.ipRedirect : MssL3V2PolicyAction.redirect
}
MssL3PolicyActionTacToCli = {
   MssL3PolicyAction.drop: MssL3PolicyActionCli.drop,
   MssL3PolicyAction.bypass: MssL3PolicyActionCli.forward,
   MssL3PolicyAction.redirect: MssL3PolicyActionCli.redirect,
}
MssL3V2PolicyActionTacToCli = {
   MssL3V2PolicyAction.drop: MssL3PolicyActionCli.drop,
   MssL3V2PolicyAction.bypass: MssL3PolicyActionCli.forward,
   MssL3V2PolicyAction.redirect: MssL3PolicyActionCli.redirect,
}
MssL3V2PolicyActionTacToV1 = {
   MssL3V2PolicyAction.drop: MssL3PolicyAction.drop,
   MssL3V2PolicyAction.bypass: MssL3PolicyAction.bypass,
   MssL3V2PolicyAction.redirect: MssL3PolicyAction.redirect,
}
MssL3PolicyActionTacToV2 = {
   MssL3PolicyAction.drop: MssL3V2PolicyAction.drop,
   MssL3PolicyAction.bypass: MssL3V2PolicyAction.bypass,
   MssL3PolicyAction.redirect: MssL3V2PolicyAction.redirect,
}
MssL3PolicyActionStrToCli = {
   'drop': MssL3PolicyActionCli.drop,
   'forward': MssL3PolicyActionCli.forward,
   'redirect': MssL3PolicyActionCli.redirect,
   'ip-redirect': MssL3PolicyActionCli.ipRedirect
}

l3ProtoStrToTac = { proto: Tac.Value( 'Arnet::IpProto', num )
                          for num, proto in l3Protocol.items() }
l4ProtoStrToTac = { proto: Tac.Value( 'Arnet::IpProto', num )
                          for num, proto in l4Protocol.items() }

def modifierSetToDirection( modifierSet ):
   forward = 'forwardOnly' in modifierSet
   reverse = 'reverseOnly' in modifierSet
   # As we don't support reverseOnly scheme, forward must be present.
   # Moreover, return value None means no config for direction is present
   return Direction( forward=forward, reverse=reverse ) if forward else None

class InvalidIpAddress( Exception ):
   def __init__( self, ipAddr ):
      self.ipAddr = ipAddr

class DowngradeException( Exception ):
   def __init__( self, msg ):
      self.msg = msg

def tacSetCopy( dstSet, srcSet ):
   change = False

   for element in dstSet:
      if element not in srcSet:
         dstSet.remove( element )
         change = True

   for element in srcSet:
      if element not in dstSet:
         dstSet.add( element )
         change = True

   return change

#----------------------------------------------------------------------------
# Class wrapping L3 next hop and routing table
#----------------------------------------------------------------------------
class ServiceDeviceL3Intf:
   def __init__( self, l3Intf ):
      self.l3Intf = l3Intf
      self.dirty = False

   @property
   def ip( self ):
      return str( self.l3Intf.ip )

   @staticmethod
   def copyL3Intf( l3IntfDst, l3IntfSrc ):
      for subnet in l3IntfSrc.reachableSubnet:
         l3IntfDst.reachableSubnet.add( subnet )

   #----------------------------------------------------------------------------
   # Add subnet route if not present.
   # This function is called by "route <a.b.c.d/e>" command
   #----------------------------------------------------------------------------
   def addReachableSubnet( self, subnet ):
      if subnet not in self.l3Intf.reachableSubnet:
         self.dirty = True
         self.l3Intf.reachableSubnet.add( subnet )

   #----------------------------------------------------------------------------
   # Remove subnet route if not present.
   # This function is called by "no route <a.b.c.d/e>" command
   #----------------------------------------------------------------------------
   def removeReachableSubnet( self, subnet ):
      self.dirty = True
      self.l3Intf.reachableSubnet.remove( subnet )

#----------------------------------------------------------------------------
# Class wrapping rule match/action
#----------------------------------------------------------------------------
class ServiceDeviceMatch( metaclass=abc.ABCMeta ):
   def __init__( self, action=None, match=None ):
      self.action = action
      self.match = self._newMatchConfig()
      if match:
         self.copyMatch( self.match, match )

   @abc.abstractmethod
   def _newMatchConfig( self ):
      pass

   #----------------------------------------------------------------------------
   # Copy srcMatch into dstMatch
   #----------------------------------------------------------------------------
   @staticmethod
   def copyMatch( dstMatch, srcMatch ):
      change = tacSetCopy( dstMatch.srcIp, srcMatch.srcIp )
      change = tacSetCopy( dstMatch.srcIpRange, srcMatch.srcIpRange ) or change
      change = tacSetCopy( dstMatch.dstIp, srcMatch.dstIp ) or change
      change = tacSetCopy( dstMatch.dstIpRange, srcMatch.dstIpRange ) or change
      change = tacSetCopy( dstMatch.l3App, srcMatch.l3App ) or change
      change = tacSetCopy( dstMatch.srcL4App, srcMatch.srcL4App ) or change
      change = tacSetCopy( dstMatch.srcL4AppRange, srcMatch.srcL4AppRange ) or change
      change = tacSetCopy( dstMatch.dstL4App, srcMatch.dstL4App ) or change
      change = tacSetCopy( dstMatch.dstL4AppRange, srcMatch.dstL4AppRange ) or change
      return change

   def _addAddresses( self, addrSet, addrs ):
      for address in addrs:
         try:
            ipPrefix = Tac.Value( 'Arnet::IpGenPrefix', address )
            addrSet.add( ipPrefix )
         except IndexError as e:
            raise InvalidIpAddress( address ) from e

   def _removeAddresses( self, addrSet, addrs ):
      for address in addrs:
         try:
            ipPrefix = Tac.Value( 'Arnet::IpGenPrefix', address )
            self.match.srcIp.remove( ipPrefix )
         except IndexError as e:
            raise InvalidIpAddress( address ) from e

   #----------------------------------------------------------------------------
   # Add source IP address match
   #----------------------------------------------------------------------------
   def addSrcAddress( self, addresses ):
      self._addAddresses( self.match.srcIp, addresses )

   #----------------------------------------------------------------------------
   # Remove source IP address match
   #----------------------------------------------------------------------------
   def removeSrcAddress( self, addresses ):
      self._removeAddresses( self.match.srcIp, addresses )

   #----------------------------------------------------------------------------
   # Remove all source IP address match
   #----------------------------------------------------------------------------
   def clearSrcAddress( self ):
      self.match.srcIp.clear()

   #----------------------------------------------------------------------------
   # Add destination IP address match
   #----------------------------------------------------------------------------
   def addDstAddress( self, addresses ):
      self._addAddresses( self.match.dstIp, addresses )

   #----------------------------------------------------------------------------
   # Remove destination IP address match
   #----------------------------------------------------------------------------
   def removeDstAddress( self, addresses ):
      self._removeAddresses( self.match.dstIp, addresses )

   #----------------------------------------------------------------------------
   # Remove all destination IP address match
   #----------------------------------------------------------------------------
   def clearDstAddress( self ):
      self.match.dstIp.clear()

   #----------------------------------------------------------------------------
   # Add protocol match
   #----------------------------------------------------------------------------
   def addProtocol( self, proto ):
      self.match.l3App.add( l3ProtoStrToTac[ proto ] )

   #----------------------------------------------------------------------------
   # Remove protocol match
   #----------------------------------------------------------------------------
   def removeProtocol( self, proto ):
      self.match.l3App.remove( l3ProtoStrToTac[ proto ] )

   @abc.abstractmethod
   def _buildL4App( self, l4Proto, l4Port ):
      pass

   def _addL4App( self, l4AppSet, protocol, ports ):
      l4Proto = l4ProtoStrToTac[ protocol ]
      for port in ports:
         l4Port = Tac.Value( 'Arnet::Port', port )
         l4App = self._buildL4App( l4Proto, l4Port )
         l4AppSet.add( l4App )

   def _removeL4App( self, l4AppSet, protocol, ports ):
      l4Proto = l4ProtoStrToTac[ protocol ]
      if ports:
         for port in ports:
            l4Port = Tac.Value( 'Arnet::Port', port )
            l4App = self._buildL4App( l4Proto, l4Port )
            l4AppSet.remove( l4App )
      else:
         for l4App in l4AppSet:
            if l4App.proto == l4Proto.value:
               l4AppSet.remove( l4App )

   #----------------------------------------------------------------------------
   # Add source L4 port match
   #----------------------------------------------------------------------------
   def addSrcL4App( self, protocol, ports ):
      self._addL4App( self.match.srcL4App, protocol, ports )

   #----------------------------------------------------------------------------
   # Remove source L4 port match
   #----------------------------------------------------------------------------
   def removeSrcL4App( self, protocol, ports=None ):
      self._removeL4App( self.match.srcL4App, protocol, ports )

   #----------------------------------------------------------------------------
   # Add destination L4 port match
   #----------------------------------------------------------------------------
   def addDstL4App( self, protocol, ports ):
      self._addL4App( self.match.dstL4App, protocol, ports )

   #----------------------------------------------------------------------------
   # Remove source L4 port match
   #----------------------------------------------------------------------------
   def removeDstL4App( self, protocol, ports=None ):
      self._removeL4App( self.match.dstL4App, protocol, ports )

   #---------------------------------------------------------------------------------
   # Check if the rule config is valid:
   #    1. For IP redirect rule, if there is only source IP address config
   #---------------------------------------------------------------------------------
   def validate( self ):
      if ( self.isIpRedirect() and
           ( self.match.dstIp or self.match.dstIpRange or
             self.match.l3App or self.match.srcL4App or
             self.match.srcL4AppRange or self.match.dstL4App or
             self.match.dstL4AppRange ) ):
         return ERROR_MSG_REDUNDANT_ATTRIBUTE
      else:
         return None

   def isIpRedirect( self ):
      return self.action == MssL3PolicyActionCli.ipRedirect

class ServiceDeviceMatchV1( ServiceDeviceMatch ):
   def _newMatchConfig( self ):
      return Tac.newInstance( "MssL3::MssMatchConfig" )

   def _buildL4App( self, l4Proto, l4Port ):
      return Tac.Value( "MssL3::MssL4App", l4Proto, l4Port )

class ServiceDeviceMatchV2( ServiceDeviceMatch ):
   def _newMatchConfig( self ):
      return Tac.newInstance( "MssL3V2::MssMatchConfig" )

   def _buildL4App( self, l4Proto, l4Port ):
      return Tac.Value( "MssL3V2::MssL4App", l4Proto, l4Port )

#----------------------------------------------------------------------------
# Classes wrapping static rules
#----------------------------------------------------------------------------
class ServiceDeviceRule( metaclass=abc.ABCMeta ):
   def __init__( self, ruleName, isVerbatim, action, source=None, tags=None,
                 match=None, direction=None ):
      self.ruleName = ruleName
      self.source = source
      self.tags = tags
      self.backupRuleCfg = None
      self.currentRuleCfg = None
      self.direction = direction
      self.backupDirection = None

   #----------------------------------------------------------------------------
   # Backup commited rule internally.
   # This is used while a rule is being edited.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def backup( self ):
      pass

   #----------------------------------------------------------------------------
   # Restore backed up rule.
   # This is used after an "abort" command.
   #----------------------------------------------------------------------------
   def restore( self ):
      self.direction = self.backupDirection
      self.currentRuleCfg = self.backupRuleCfg

   def validate( self ):
      return self.currentRuleCfg.validate()

   #----------------------------------------------------------------------------
   # Commit current rule configuration in Sysdb
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def commit( self, rule, policyDirection ):
      pass

   def setDirection( self, forward, reverse ):
      self.direction = Direction( forward=forward, reverse=reverse )

   def noDirection( self ):
      self.direction = None

   def computeDirection( self, policyDirection ):
      return ( self.direction or policyDirection
               or Direction( forward=True, reverse=True ) )

class ServiceDeviceRuleV1( ServiceDeviceRule ):
   def __init__( self, ruleName, isVerbatim, action, source=None, tags=None,
                 match=None, direction=None ):
      super().__init__( ruleName, isVerbatim, action,
                        source=source, tags=tags,
                        match=match, direction=direction )
      if isVerbatim:
         action = MssL3PolicyActionTacToCli[ action ]
      else:
         action = MssL3PolicyActionCli.ipRedirect

      self.currentRuleCfg = ServiceDeviceMatchV1( action, match )

   def backup( self ):
      self.backupDirection = self.direction
      self.backupRuleCfg = ServiceDeviceMatchV1( self.currentRuleCfg.action,
                                                 self.currentRuleCfg.match )

   def getNewModifierSet( self, policyDirection ):
      assert not self.currentRuleCfg.isIpRedirect(), "Shouldn't be called for "\
            "ip-redirect"
      newModifierSet = { mssL3ModifierVerbatim }
      newDirection = self.computeDirection( policyDirection )
      assert newDirection.forward, "Forward direction must be set"
      if newDirection.forward and not newDirection.reverse:
         newModifierSet.add( mssL3ModifierForwardOnly )
      return newModifierSet

   @staticmethod
   def copyRuleFromV2( ruleV1, ruleV2 ):
      ruleOriginV2 = next( iter( ruleV2.origin.values() ) )
      ruleOriginV1 = ruleV1.newOrigin( ruleOriginV2.policyName )

      ruleV1.match = ()
      ServiceDeviceMatch.copyMatch( ruleV1.match, ruleOriginV2.match )
      ruleV1.action = MssL3V2PolicyActionTacToV1[ ruleOriginV2.action ]

      ruleOriginV1.match = ()
      if ruleV2.priority == MssL3PolicyPriority.min:
         ServiceDeviceMatch.copyMatch( ruleOriginV1.match, ruleOriginV2.match )
      ruleOriginV1.action = MssL3V2PolicyActionTacToV1[ ruleOriginV2.action ]
      ruleOriginV1.tags = ruleOriginV2.tags
      ruleOriginV1.source = mssL3PolicyCliSource
      if ruleV2.priority != MssL3V2PolicyPriority.min:
         # String value of the modifiers for both V1 and V2 are same. So, not adding
         # unnecessary conversion logic here.
         for i in ruleOriginV2.policyModifierSetCli:
            ruleOriginV1.policyModifierSetCli.add( i )
         for i in ruleOriginV2.policyModifierSet:
            ruleOriginV1.policyModifierSet.add( i )

      ruleV1.seqNo += 1

   def commit( self, rule, policyDirection ):
      def updatePolicyModifiers():
         if self.currentRuleCfg.isIpRedirect():
            return
         originRule.policyModifierSetCli.clear() # This clear is ok as nobody reacts
         if self.direction:
            if self.direction.forward:
               originRule.policyModifierSetCli.add( mssL3ModifierForwardOnly )
            if self.direction.reverse:
               originRule.policyModifierSetCli.add( mssL3ModifierReverseOnly )
         # The following is implicit. We add it in this collection for the sake of
         # consistency
         originRule.policyModifierSetCli.add( mssL3ModifierVerbatim )

         # Add derived modifiers. Entries in this set are used in Mss orchestration
         # logic. For example, reverseOnly attribute won't be added in this set
         originRule.policyModifierSet.clear()
         for i in self.getNewModifierSet( policyDirection ):
            originRule.policyModifierSet.add( i )

      originRule = rule.newOrigin( self.ruleName )
      originRule.match = ()
      originRule.source = self.source
      originRule.tags = self.tags
      updatePolicyModifiers()
      if self.currentRuleCfg.isIpRedirect():
         ServiceDeviceMatch.copyMatch( originRule.match,
                                       self.currentRuleCfg.match )
      originRule.action = MssL3PolicyActionCliToTac[ self.currentRuleCfg.action ]
      rule.action = MssL3PolicyActionCliToTac[ self.currentRuleCfg.action ]
      ServiceDeviceMatch.copyMatch( rule.match, self.currentRuleCfg.match )
      rule.seqNo += 1

      # V1 doesn't to report any change because commit is triggered by seqNo
      return False

class ServiceDeviceRuleV2( ServiceDeviceRule ):
   def __init__( self, ruleName, isVerbatim, action, source=None, tags=None,
                 match=None, direction=None ):
      super().__init__( ruleName, isVerbatim, action,
                        source=source, tags=tags,
                        match=match, direction=direction )
      if isVerbatim:
         action = MssL3V2PolicyActionTacToCli[ action ]
      else:
         action = MssL3PolicyActionCli.ipRedirect

      self.currentRuleCfg = ServiceDeviceMatchV2( action, match )

   def backup( self ):
      self.backupDirection = self.direction
      self.backupRuleCfg = ServiceDeviceMatchV2( self.currentRuleCfg.action,
                                                 self.currentRuleCfg.match )

   @staticmethod
   def copyRuleFromV1( ruleV2, ruleV1 ):
      ruleOriginV1 = next( iter( ruleV1.origin.values() ) )
      ruleOriginV2 = ruleV2.newOrigin( ruleOriginV1.policyName )

      ruleOriginV2.match = ()
      ServiceDeviceMatch.copyMatch( ruleOriginV2.match, ruleV1.match )
      ruleOriginV2.tags = ruleOriginV1.tags
      ruleOriginV2.action = MssL3PolicyActionTacToV2[ ruleV1.action ]
      if ruleV1.priority != MssL3PolicyPriority.min:
         # Modifiers string value are same. So, not adding unnecessary conversion
         # logic
         for i in ruleOriginV1.policyModifierSetCli:
            ruleOriginV2.policyModifierSetCli.add( i )
         for i in ruleOriginV1.policyModifierSet:
            ruleOriginV2.policyModifierSet.add( i )

   def getNewModifierSet( self, originRule, policyDirection ):
      assert not self.currentRuleCfg.isIpRedirect(), "Shouldn't be called for "\
            "ip-redirect"
      currentModifierSet = set( originRule.policyModifierSet )
      newModifierSet = { mssL3V2ModifierVerbatim }
      newDirection = self.computeDirection( policyDirection )
      assert newDirection.forward, "Forward direction must be set"
      if newDirection.forward and not newDirection.reverse:
         newModifierSet.add( mssL3V2ModifierForwardOnly )

      return ( currentModifierSet != newModifierSet, newModifierSet )

   def commit( self, rule, policyDirection ):
      def updatePolicyModifiers():
         if self.currentRuleCfg.isIpRedirect():
            return False
         # Updating CLI's modifiers
         originRule.policyModifierSetCli.clear() # This clear is ok as nobody reacts
         if self.direction:
            if self.direction.forward:
               originRule.policyModifierSetCli.add( mssL3V2ModifierForwardOnly )
            if self.direction.reverse:
               originRule.policyModifierSetCli.add( mssL3V2ModifierReverseOnly )
         # The following is implicit. We add it in this collection for the sake of
         # consistency
         originRule.policyModifierSetCli.add( mssL3V2ModifierVerbatim )

         # Add derived modifiers.Entries in this set are used in Mss orchestration
         # logic. For example, reverseOnly attribute won't be added in this set
         changed, newModifiers = self.getNewModifierSet( originRule,
                                                         policyDirection )
         if changed:
            originRule.policyModifierSet.clear()
            for i in newModifiers:
               originRule.policyModifierSet.add( i )
         return changed

      change = False
      originRule = rule.origin.get( self.ruleName )
      if not originRule:
         if not self.currentRuleCfg.isIpRedirect():
            # New exact match rule replaces an old one.
            rule.origin.clear()
            change = True
         originRule = rule.newOrigin( self.ruleName )
         originRule.match = ()

      originRule.tags = self.tags # tags are always the same
      change = updatePolicyModifiers() or change

      if ( originRule.action !=
           MssL3V2PolicyActionCliToTac[ self.currentRuleCfg.action ] ):
         change = True
         originRule.action = \
               MssL3V2PolicyActionCliToTac[ self.currentRuleCfg.action ]

      change = ServiceDeviceMatch.copyMatch( originRule.match,
                                             self.currentRuleCfg.match ) or change
      return change

#----------------------------------------------------------------------------
# Class wrapping static device VRF configuration
#----------------------------------------------------------------------------
class ServiceDeviceVrf( metaclass=abc.ABCMeta ):
   def __init__( self, vrfName, serviceDeviceVrf, policySetVrf, direction=None,
                 dirty=False ):
      self.vrfName = vrfName
      self.serviceDeviceVrf = serviceDeviceVrf
      self.policySetVrf = policySetVrf
      self.direction = direction

      # map interface name to ServiceDeviceL3Intf object
      self.intfs = {}
      self._initRoutingTable()

      # list of rule name in priority order
      self.ruleNameList = []
      # map rule name to ServiceDeviceRule object
      self.candidateRules = {}
      self._initCandidateRules()

      # Dirty flag to help commit changes in routing table
      self.serviceDeviceDirty = dirty

      # Backup data
      self.backupDirection = None

   @staticmethod
   def copyRoutingTable( dstRoutingTable, srcRoutingTable ):
      for ip, l3Config in srcRoutingTable.l3Intf.items():
         l3Intf = dstRoutingTable.newL3Intf( ip )
         ServiceDeviceL3Intf.copyL3Intf( l3Intf, l3Config )

   #----------------------------------------------------------------------------
   # Initializes existing routing table
   #----------------------------------------------------------------------------
   def _initRoutingTable( self ):
      for ip, l3Intf in self.serviceDeviceVrf.l3Intf.items():
         self.intfs[ str( ip ) ] = ServiceDeviceL3Intf( l3Intf )

   #----------------------------------------------------------------------------
   # Initializes existing rules
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def _initCandidateRules( self ):
      pass

   #----------------------------------------------------------------------------
   # Creates new L3 interface if not exists in the current VRF.
   # Returns corresponding ServiceDeviceL3Intf object
   # This function is called by "next-hop address <a.b.c.d>" command
   #----------------------------------------------------------------------------
   def getOrCreateL3Intf( self, ip ):
      l3Intf = self.serviceDeviceVrf.l3Intf.get( ip )
      if l3Intf:
         # interface already exists
         return self.intfs[ str( ip ) ]

      self.serviceDeviceDirty = True
      l3Intf = self.serviceDeviceVrf.newL3Intf( ip )
      intfObj = ServiceDeviceL3Intf( l3Intf )
      self.intfs[ str( ip ) ] =  intfObj

      return intfObj

   #----------------------------------------------------------------------------
   # Removes L3 interface from the current VRF.
   # This function is called by "no next-hop address <a.b.c.d>" command
   #----------------------------------------------------------------------------
   def removeL3Intf( self, ip ):
      if str( ip ) in self.intfs:
         self.serviceDeviceDirty = True
         del self.intfs[ str( ip ) ]
         del self.serviceDeviceVrf.l3Intf[ ip ]

   def hasRule( self, ruleName ):
      return ruleName in self.candidateRules

   @abc.abstractmethod
   def _createRule( self, ruleName, isVerbatim, action, source=None, tags=None,
                    match=None, direction=None ):
      pass

   #----------------------------------------------------------------------------
   # Add new rule in the current VRF.
   # This function is called by "rule <ruleName> [(after|before) <refName>]"
   #----------------------------------------------------------------------------
   def addRule( self, ruleName, order=None, refRuleName=None ):
      if ruleName in self.candidateRules:
         rule = self.candidateRules[ ruleName ]
      else:
         rule = self._createRule( ruleName, True, defaultMssL3PolicyAction,
                                  source=mssL3PolicyCliSource,
                                  tags=mssL3PolicyCliTag )
         self.candidateRules[ ruleName ] = rule

      self.setRulePriority( ruleName, order, refRuleName )
      return rule

   #----------------------------------------------------------------------------
   # Remove rule from the current VRF.
   # This function is called by "no rule <ruleName>"
   #----------------------------------------------------------------------------
   def removeRule( self, ruleName ):
      if ruleName in self.candidateRules:
         self.ruleNameList.remove( ruleName )
         del self.candidateRules[ ruleName ]

   #----------------------------------------------------------------------------
   # Clear all rules from current VRF.
   # This function is called by "no traffic-policy" command.
   #----------------------------------------------------------------------------
   def clearAllRules( self ):
      self.ruleNameList = []
      self.candidateRules.clear()
      self.commitRules()

   #----------------------------------------------------------------------------
   # Set rule priority. This function may be called at rule creation or by the
   # "move <ruleName> (after|before) <refName>" command.
   #----------------------------------------------------------------------------
   def setRulePriority( self, ruleName, order=None, refRuleName=None ):
      if order and refRuleName:
         if ruleName in self.ruleNameList:
            self.ruleNameList.remove( ruleName )

         insertingPoint = self.ruleNameList.index( refRuleName )
         if order == 'after':
            insertingPoint += 1
         self.ruleNameList.insert( insertingPoint, ruleName )
      else:
         if ruleName not in self.ruleNameList:
            self.ruleNameList.append( ruleName )

   def setDirection( self, forward=False, reverse=False ):
      self.direction = Direction( forward=forward, reverse=reverse )

   def backup( self ):
      self.backupDirection = self.direction

   def restore( self ):
      self.direction = self.backupDirection

   #---------------------------------------------------------------------------------
   # Check if the policy config valid
   #    1. if there is only one ip-redirect rule
   #    2. if the ip-redirect rule has the lowest priority
   #    3. direction is not set for the ip-redirect rule
   #    4. if the number of verbatim rules is within limit
   #---------------------------------------------------------------------------------
   def validate( self ):
      redirectNum = 0
      verbatimNum = 0

      for rule in self.candidateRules.values():
         if rule.currentRuleCfg.isIpRedirect() :
            # IP redirect rule should have the lowest priority
            # i.e. its ruleName is at the end of ruleNameList
            if ( self.ruleNameList.index( rule.ruleName ) !=
                 len( self.ruleNameList ) - 1 ):
               return ERROR_MSG_PRIORITY
            if rule.direction:
               return ERROR_MSG_IP_REDIRECT_DIRECTION
            redirectNum += 1
         else:
            verbatimNum += 1

      if redirectNum > 1:
         return ERROR_MSG_MULTI_IPREDIRECT

      if verbatimNum > mssL3MaxVerbatimRuleNum:
         return ERROR_MSG_EXCEED_VERBATIM_LIMIT

      return None

   #----------------------------------------------------------------------------
   # Commit routes into Sysdb
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def commitRoutes( self ):
      pass

   #----------------------------------------------------------------------------
   # Commit policies into Sysdb
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def commitRules( self ):
      pass

class ServiceDeviceVrfV1( ServiceDeviceVrf ):
   def _createRule( self, ruleName, isVerbatim, action, source=None, tags=None,
                    match=None, direction=None ):
      return ServiceDeviceRuleV1( ruleName, isVerbatim, action, source=source,
                    tags=tags, match=match, direction=direction )

   def _initCandidateRules( self ):
      for rule in self.policySetVrf.rule.values():
         for ruleName, ruleOrigin in rule.origin.items():
            self.ruleNameList.append( ruleName )
            self.candidateRules[ ruleName ] = self._createRule(
                  ruleName, mssL3ModifierVerbatim in ruleOrigin.policyModifierSet,
                  rule.action, source=ruleOrigin.source, tags=ruleOrigin.tags,
                  match=rule.match,
                  direction=modifierSetToDirection( ruleOrigin.policyModifierSetCli )
                  )

   @staticmethod
   def copyVrfFromV2( vrfV1, vrfV2 ):
      ServiceDeviceVrf.copyRoutingTable( vrfV1, vrfV2 )

   @staticmethod
   def copyPolicyFromV2( polV1, polV2 ):
      for i in polV2.policyModifierSet:
         polV1.policyModifierSet.add( i )
      for priority, ruleConfig in polV2.rule.items():
         ruleV1 = polV1.newRule( priority )
         ServiceDeviceRuleV1.copyRuleFromV2( ruleV1, ruleConfig )

   def commitRoutes( self ):
      # V1 doesn't use any commit semantic for L3 interfaces
      pass

   def commitRules( self ):
      self.policySetVrf.policyModifierSet.clear()
      if self.direction:
         if self.direction.forward:
            self.policySetVrf.policyModifierSet.add( mssL3ModifierForwardOnly )
         if self.direction.reverse:
            self.policySetVrf.policyModifierSet.add( mssL3ModifierReverseOnly )

      candidateRulePriorities = set()
      candidatePriority = MssL3PolicyPriority.max
      # Commit all rules
      for ruleName in self.ruleNameList:
         candidateRuleConfig = self.candidateRules[ ruleName ]

         if candidateRuleConfig.currentRuleCfg.isIpRedirect():
            candidatePriority = MssL3PolicyPriority.ipRedirect
         candidateRulePriorities.add( candidatePriority )

         if candidatePriority not in self.policySetVrf.rule:
            rule = self.policySetVrf.newRule( candidatePriority )
            rule.match = ()
         else:
            rule = self.policySetVrf.rule[ candidatePriority ]
            rule.origin.clear()

         candidateRuleConfig.commit( rule, self.direction )
         candidatePriority -= 1

      # Delete unused rules
      for priority in self.policySetVrf.rule:
         if priority not in candidateRulePriorities:
            del self.policySetVrf.rule[ priority ]

class ServiceDeviceVrfV2( ServiceDeviceVrf ):
   def _createRule( self, ruleName, isVerbatim, action, source=None, tags=None,
                    match=None, direction=None ):
      return ServiceDeviceRuleV2( ruleName, isVerbatim, action, source=source,
                                  tags=tags, match=match, direction=direction )

   def _initCandidateRules( self ):
      for rule in self.policySetVrf.rule.values():
         for ruleName, ruleOrigin in rule.origin.items():
            self.ruleNameList.append( ruleName )
            self.candidateRules[ ruleName ] = self._createRule(
                  ruleName, mssL3ModifierVerbatim in ruleOrigin.policyModifierSet,
                  ruleOrigin.action, source=None, tags=ruleOrigin.tags,
                  match=ruleOrigin.match,
                  direction=modifierSetToDirection( ruleOrigin.policyModifierSetCli )
                  )

   @staticmethod
   def copyVrfFromV1( vrfV2, vrfV1 ):
      ServiceDeviceVrf.copyRoutingTable( vrfV2, vrfV1 )
      vrfV2.timestamp = Tac.now()

   @staticmethod
   def copyPolicyFromV1( polV2, polV1 ):
      for i in polV1.policyModifierSet:
         polV2.policyModifierSet.add( i )
      for priority, ruleConfig in polV1.rule.items():
         ruleV2 = polV2.newRule( priority )
         ServiceDeviceRuleV2.copyRuleFromV1( ruleV2, ruleConfig )

      if MssL3PolicyPriority.max in polV1.rule:
         # set timestamp only if non-ip-redirect rules are present
         # when it's the case, max priority should be in the rule collection
         polV2.timestamp = Tac.now()

   def commitRoutes( self ):
      dirty = False
      for l3IntfObj in self.intfs.values():
         if l3IntfObj.dirty:
            l3IntfObj.dirty = False
            dirty = True

      if self.serviceDeviceDirty or dirty:
         self.serviceDeviceVrf.timestamp = Tac.now()
         self.serviceDeviceDirty = False

   def commitRules( self ):
      self.policySetVrf.policyModifierSet.clear()
      if self.direction:
         if self.direction.forward:
            self.policySetVrf.policyModifierSet.add( mssL3V2ModifierForwardOnly )
         if self.direction.reverse:
            self.policySetVrf.policyModifierSet.add( mssL3V2ModifierReverseOnly )

      policySetDirty = False
      candidateRulePriorities = set()
      candidatePriority = MssL3V2PolicyPriority.max

      # Commit all rules
      for ruleName in self.ruleNameList:
         candidateRuleConfig = self.candidateRules[ ruleName ]

         if candidateRuleConfig.currentRuleCfg.isIpRedirect():
            candidatePriority = MssL3V2PolicyPriority.ipRedirect
         candidateRulePriorities.add( candidatePriority )

         rule = ( self.policySetVrf.rule.get( candidatePriority ) or
                  self.policySetVrf.newRule( candidatePriority ) )
         if candidateRuleConfig.commit( rule, self.direction ):
            # Update timestamp only if current rule isn't ip-redirect
            policySetDirty |= not candidateRuleConfig.currentRuleCfg.isIpRedirect()
         candidatePriority -= 1

      # Delete unused rules
      for priority in self.policySetVrf.rule:
         if priority not in candidateRulePriorities:
            del self.policySetVrf.rule[ priority ]
            if priority != MssL3V2PolicyPriority.ipRedirect:
               policySetDirty = True

      # Set timestamp
      if policySetDirty:
         self.policySetVrf.timestamp = Tac.now()

#-------------------------------------------------------------------------------
# Class wrapping a service device within the CLI.
#-------------------------------------------------------------------------------
class ServiceDevice( metaclass=abc.ABCMeta ):
   def __init__( self, deviceName, deviceConfig, policySetConfig ):
      self.deviceName = deviceName
      self.deviceConfig = deviceConfig
      self.policySetConfig = policySetConfig

   #----------------------------------------------------------------------------
   # Creates the vrf named "vrfName". This function is called whenever
   # "vrf <vrfName>" mode is entered.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def addVrf( self, vrfName ):
      pass

   #----------------------------------------------------------------------------
   # Removes the vrf named "vrfName". This function is called by
   # "no vrf <vrfName>" command.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def removeVrf( self, vrfName ):
      pass

   #----------------------------------------------------------------------------
   # Returns list of existing vrf names. This function is used for auto-complete.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def getVrfNames( self ):
      pass

   #----------------------------------------------------------------------------
   # Set traffic inspection (local and/or outbound) of static device.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def setTrafficInspection( self, trafficInspection ):
      pass

class ServiceDeviceV1( ServiceDevice ):
   def addVrf( self, vrfName ):
      policyConfig = self.policySetConfig.newPolicy( vrfName )
      return ServiceDeviceVrfV1(
            vrfName, self.deviceConfig.newVrf( vrfName ), policyConfig,
            direction=modifierSetToDirection( policyConfig.policyModifierSet ) )

   def removeVrf( self, vrfName ):
      del self.deviceConfig.vrf[ vrfName ]
      del self.policySetConfig.policy[ vrfName ]

   @staticmethod
   def copyServiceDeviceFromV2( deviceV1, deviceV2 ):
      if len( deviceV2.netVrf ) > 1 or "default" not in deviceV2.netVrf:
         # Multi-VRF is not supported in V1
         raise DowngradeException( ERROR_MSG_DEFAULT_VRF )
      for vrfName, vrfConfig in deviceV2.netVrf.items():
         vrfV1 = deviceV1.newVrf( vrfName )
         ServiceDeviceVrfV1.copyVrfFromV2( vrfV1, vrfConfig )

   @staticmethod
   def copyPolicySetFromV2( polSetV1, polSetV2 ):
      for vrfName, polConfig in polSetV2.policy.items():
         vrfV1 = polSetV1.newPolicy( vrfName )
         ServiceDeviceVrfV1.copyPolicyFromV2( vrfV1, polConfig )

   def getVrfNames( self ):
      return list( self.deviceConfig.vrf )

   def setTrafficInspection( self, trafficInspection ):
      # Traffic inspection is not supported in V1
      pass

class ServiceDeviceV2( ServiceDevice ):
   def addVrf( self, vrfName ):
      policyConfig = self.policySetConfig.newPolicy( vrfName )
      direction = modifierSetToDirection( policyConfig.policyModifierSet )
      if vrfName in self.deviceConfig.netVrf:
         netVrfConfig = self.deviceConfig.netVrf[ vrfName ]
         sdVrf = ServiceDeviceVrfV2( vrfName, netVrfConfig, policyConfig,
                                     direction=direction, dirty=False )
      else:
         netVrfConfig = self.deviceConfig.newNetVrf( vrfName )
         sdVrf = ServiceDeviceVrfV2( vrfName, netVrfConfig, policyConfig,
                                     direction=direction, dirty=True )
      return sdVrf

   def removeVrf( self, vrfName ):
      del self.deviceConfig.netVrf[ vrfName ]
      del self.policySetConfig.policy[ vrfName ]

   def getVrfNames( self ):
      return list( self.deviceConfig.netVrf )

   def setTrafficInspection( self, trafficInspection ):
      self.deviceConfig.trafficInspection = trafficInspection

   @staticmethod
   def copyServiceDeviceFromV1( deviceV2, deviceV1 ):
      for vrfName, vrfConfig in deviceV1.vrf.items():
         vrfV2 = deviceV2.newNetVrf( vrfName )
         ServiceDeviceVrfV2.copyVrfFromV1( vrfV2, vrfConfig )

   @staticmethod
   def copyPolicySetFromV1( polSetV2, polSetV1 ):
      for vrfName, polConfig in polSetV1.policy.items():
         vrfV2 = polSetV2.newPolicy( vrfName )
         ServiceDeviceVrfV2.copyPolicyFromV1( vrfV2, polConfig )

#------------------------------------------------------------------------------------
# Class for operation on MssL3(V2)::MssPolicySourceConfig / ServiceDeviceSourceConfig
#------------------------------------------------------------------------------------
class ServiceDeviceDir( metaclass=abc.ABCMeta ):
   def __init__( self, deviceConfigDir, policySetConfigDir ):
      self.deviceConfigDir = deviceConfigDir
      self.policySetConfigDir = policySetConfigDir

   #----------------------------------------------------------------------------
   # Creates the service device corresponding to this name. This function is
   # called whenever "static device device-name" mode is entered
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def createDevice( self, deviceName ):
      pass

   #----------------------------------------------------------------------------
   # Destroys the service device corresponding to this name. This function is
   # called by "no static service device device-name" command.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def destroyDevice( self, deviceName ):
      pass

   #----------------------------------------------------------------------------
   # Returns current service device present in configuration.
   # This is use for auto-completion.
   #----------------------------------------------------------------------------
   @abc.abstractmethod
   def serviceDevice( self ):
      pass

   #----------------------------------------------------------------------------
   # Destroys all service devices configured via CLI. This function is called by
   # "no service mss" command.
   #----------------------------------------------------------------------------
   def clearAllDevices( self ):
      self.deviceConfigDir.serviceDevice.clear()
      self.policySetConfigDir.policySet.clear()

class ServiceDeviceDirV1( ServiceDeviceDir ):
   def createDevice( self, deviceName ):
      return ServiceDeviceV1(
         deviceName,
         self.deviceConfigDir.newServiceDevice( deviceName ),
         self.policySetConfigDir.newPolicySet( deviceName ) )

   def destroyDevice( self, deviceName ):
      del self.deviceConfigDir.serviceDevice[ deviceName ]
      del self.policySetConfigDir.policySet[ deviceName ]

   def serviceDevice( self ):
      return self.deviceConfigDir.serviceDevice

   @staticmethod
   def copyServiceDeviceDirFromV2( deviceDirV1, deviceDirV2 ):
      for devName, devConfig in deviceDirV2.serviceDevice.items():
         devV1 = deviceDirV1.newServiceDevice( devName.getPhyInstanceName() )
         ServiceDeviceV1.copyServiceDeviceFromV2( devV1, devConfig )

   @staticmethod
   def copyPolicySetDirFromV2( polSetDirV1, polSetDirV2 ):
      for devName, polSetConfig in polSetDirV2.policySet.items():
         devV1 = polSetDirV1.newPolicySet( devName.getPhyInstanceName() )
         ServiceDeviceV1.copyPolicySetFromV2( devV1, polSetConfig )

class ServiceDeviceDirV2( ServiceDeviceDir ):
   def createDevice( self, deviceName ):
      # static cli device name format: dev1:root:Static
      mssDeviceName = DeviceName( DeviceName.encode( deviceName,
         mssL3DeviceCliVInst, mssL3PolicyCliSource ) )

      serviceDevice = self.deviceConfigDir.serviceDevice.get( mssDeviceName )
      if not serviceDevice:
         serviceDevice = self.deviceConfigDir.newServiceDevice( mssDeviceName )

      return ServiceDeviceV2( deviceName, serviceDevice,
                              self.policySetConfigDir.newPolicySet( mssDeviceName ) )

   def destroyDevice( self, deviceName ):
      # static cli device name format: dev1:root:Static
      mssDeviceName = DeviceName( DeviceName.encode( deviceName,
         mssL3DeviceCliVInst, mssL3PolicyCliSource ) )

      del self.deviceConfigDir.serviceDevice[ mssDeviceName ]
      del self.policySetConfigDir.policySet[ mssDeviceName ]

   def serviceDevice( self ):
      return [ d.getPhyInstanceName()
               for d in self.deviceConfigDir.serviceDevice ]

   @staticmethod
   def copyServiceDeviceDirFromV1( deviceDirV2, deviceDirV1 ):
      for devName, devConfig in deviceDirV1.serviceDevice.items():
         devV2 = deviceDirV2.newServiceDevice( DeviceName(
            DeviceName.encode( devName, mssL3DeviceCliVInst,
                               mssL3PolicyCliSource ) ) )
         ServiceDeviceV2.copyServiceDeviceFromV1( devV2, devConfig )

   @staticmethod
   def copyPolicySetDirFromV1( polSetDirV2, polSetDirV1 ):
      for devName, polSetConfig in polSetDirV1.policySet.items():
         devV2 = polSetDirV2.newPolicySet( DeviceName(
            DeviceName.encode( devName, mssL3DeviceCliVInst,
                               mssL3PolicyCliSource ) ) )
         ServiceDeviceV2.copyPolicySetFromV1( devV2, polSetConfig )

def copyStaticV1ToV2( staticServiceDeviceV2Dir, staticPolicySetV2Dir,
                      staticServiceDeviceDir, staticPolicySetDir ):
   # copy
   ServiceDeviceDirV2.copyServiceDeviceDirFromV1(
         staticServiceDeviceV2Dir, staticServiceDeviceDir )
   ServiceDeviceDirV2.copyPolicySetDirFromV1(
         staticPolicySetV2Dir, staticPolicySetDir )

   # cleanup
   staticServiceDeviceDir.serviceDevice.clear()
   staticPolicySetDir.policySet.clear()

def copyStaticV2ToV1( staticServiceDeviceDir, staticPolicySetDir,
                      staticServiceDeviceV2Dir, staticPolicySetV2Dir ):
   try:
      # copy
      ServiceDeviceDirV1.copyServiceDeviceDirFromV2(
            staticServiceDeviceDir, staticServiceDeviceV2Dir )
      ServiceDeviceDirV1.copyPolicySetDirFromV2(
            staticPolicySetDir, staticPolicySetV2Dir )

      # cleanup
      staticServiceDeviceV2Dir.serviceDevice.clear()
      staticPolicySetV2Dir.policySet.clear()
   except DowngradeException:
      # if we cannot process with downgrade, cleanup V1 entries
      staticServiceDeviceDir.serviceDevice.clear()
      staticPolicySetDir.policySet.clear()
      raise
