# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import email

import Ark
import FpgaUtil
from BmcUtil import BmcType

class Context:
   def __init__( self, prefdl ):
      self.prefdl = prefdl
      self.prefdlParams = email.message_from_bytes( self.prefdl )
      self.sid = self.prefdlParams.get( 'SID', '' )
      self.hwApi = self.prefdlParams.get( 'HwApi', '' )
      self.cpuSid = self.prefdlParams.get( 'CpuSID', '' )
      self.cpuHwApi = self.prefdlParams.get( 'CpuHwApi', '' )
      self.cpuPca = self.prefdlParams.get( 'CpuPCA', '' )
      self.hwEpoch = self.prefdlParams.get( 'HwEpoch', '00.00' )
      self.macAddrBase = self.prefdlParams.get( 'MacAddrBase', '' )
      self.flavor = FpgaUtil.translateSwiFlavor()
      self.platform = Ark.getPlatform()
      self.bmcType = BmcType.NONE
      self.preFpgaUpgradeFns = set()
      self.postFpgaUpgradeFns = set()

   def registerPreFpgaUpgradeFn( self, fn, platformGuardFn ):
      """ Register 'fn' to be called before upgrading the FPGAs iff 'platformGuardFn'
         returns True.

         platformGuardFn (function): Takes SID as input and returns bool
      """

      if platformGuardFn( self.sid ):
         self.preFpgaUpgradeFns.add( fn )

   def updatePreFpgaUpgrade( self ):
      for fn in self.preFpgaUpgradeFns:
         fn( self )

   def registerPostFpgaUpgradeFn( self, fn, platformGuardFn ):
      """ Register 'fn' to be called after upgrading the FPGAs iff 'platformGuardFn'
          returns True.

          platformGuardFn (function): Takes SID as input and returns bool
      """

      if platformGuardFn( self.sid ):
         self.postFpgaUpgradeFns.add( fn )

   def updatePostFpgaUpgrade( self ):
      for fn in self.postFpgaUpgradeFns:
         fn( self )
