# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from types import MethodType
import binascii
import struct
import sys
from EepromDecoder import EepromDecoder, DecoderException

FALLBACK_EEPROM_DUMP_FILE = '/mnt/flash/eeprom.dump'
MAX_ONIE_EEPROM_LENGTH = 2048
ONIE_HEADER_LENGTH = 11

class OnieEepromDecoder( EepromDecoder ):
   def __init__( self, bus=None, address=None, filename=None, sequentialRead=None ):
      ''' This constructor is not meant to be called directly.  The class methods
          below are used to create instances of this decoder class with different
          functionalities.

          For example:
          decoder = OnieEepromDecoder.fromI2c( bus, ... )
          decoder.read()
      '''
      super().__init__()
      self.bus = bus
      self.address = address
      self.filename = filename
      self.sequentialRead = sequentialRead
      self.raw_ = None

   @classmethod
   def fromI2cDumpFile( cls, filename ):
      # pylint: disable-msg=W0212, protected-access
      self = cls( filename=filename )
      self._read = MethodType( cls._readFromFile, self )
      return self

   @classmethod
   def fromI2c( cls, bus, address, sequentialRead=False ):
      # pylint: disable-msg=W0212, protected-access
      self = cls( bus=bus, address=address, sequentialRead=sequentialRead )
      self._read = MethodType( cls._readFromI2cWithFallback, self )
      return self

   def _read( self ): # pylint: disable-msg=E0202, method-hidden
      ''' The implementation of this function will be set by the "constructor"
          class methods above
      '''
      raise NotImplementedError

   def _readFromFile( self ):
      dump = open( self.filename ).read() # pylint: disable=consider-using-with
      self.raw_ = bytesFromI2cDump( dump )
      self._parseOnieEepromContent()

   def _readFromI2cWithFallback( self ):
      self._rawFromI2c()
      try:
         self._parseOnieEepromContent()
      except DecoderException as e:
         # pylint: disable-next=consider-using-f-string
         print( "%s" % str( e ), file=sys.stderr )
         print( "NOTE: attempting to read from fallback file", file=sys.stderr )
         self.filename = FALLBACK_EEPROM_DUMP_FILE
         self._readFromFile()

   def _rawFromI2c( self ):
      if self.sequentialRead:
         # Set internal data word address
         self.bus.write8( self.address, 0x00, 0x00 )
         nextByte = lambda _: self.bus.recvByte( self.address )
      else:
         nextByte = lambda i: self.bus.read8( self.address, i )

      # Read Tlv header that contains length of the data
      header = [ nextByte( i ) for i in range( ONIE_HEADER_LENGTH ) ]

      # Extract length
      dataLen = header[ 9 ] << 8
      dataLen += header[ 10 ]
      dataLen = min( dataLen, MAX_ONIE_EEPROM_LENGTH )

      # Read remaining data
      TlvData = [ nextByte( i + ONIE_HEADER_LENGTH ) for i in range( dataLen ) ]
      self.raw_ = header + TlvData

   def _parseOnieEepromContent( self ):
      buf = self.raw_
      # Confirm we are looking at an ONIE eeprom TLV we understand
      signature = "TlvInfo\0" #pylint: disable-msg=W1401
      if len( buf ) < len( signature ):
         raise DecoderException( "Invalid EEPROM: Does not contain a valid header" )

      for i in range( len( signature ) ): # pylint: disable=consider-using-enumerate
         if buf[ i ] != ord( signature[ i ] ):
            raise DecoderException( "Invalid EEPROM: Signature does not match" )

      version = buf[ len( signature ) ]
      if version != 1:
         # pylint: disable-next=consider-using-f-string
         raise DecoderException( "Unable to parse EEPROM:"
                                 " version %d not supported" % version )

      datalen = buf[ 9 ] << 8
      datalen += buf[ 10 ]
      buf = buf[ 0: ONIE_HEADER_LENGTH + datalen ] # Truncate out of range data

      # Check crc over all the contents except the value of the crc itself
      # pylint: disable-next=consider-using-f-string
      hex_data = "".join( [ "%02x" % n for n in buf[ :-4 ] ] )
      bin_data = binascii.a2b_hex( hex_data )
      computedCrc = binascii.crc32( bin_data )

      crc = buf[ -1 ]
      crc += buf[ -2 ] << 8
      crc += buf[ -3 ] << 16
      crc += buf[ -4 ] << 24

      if (crc & 0xffffffff) != (computedCrc & 0xffffffff):
         raise DecoderException( "Invalid EEPROM: CRC does not match" )

      # We have a supported format and the data isn't corrupt, delete the header and
      # start decoding
      #                       version datalen
      buf = buf[ len( signature ) + 1 + 2: ]

      # Mapping of type codes to a (name, encoding) tuple. Encodings are functions
      # which convert from the list of integers to whatever string format is
      # appropriate.
      def ascii( l ): # pylint: disable=redefined-builtin
         return "".join( [ chr( n ) for n in l ] )
      def macAddress( l ):
         # pylint: disable-next=consider-using-f-string
         return ":".join( [ "%02x" % i for i in l ] )
      def date( l ):
         # In format is "MM/DD/YYYY HH:NN:SS", out format is "YYYYMMDDHHNNSS"
         d = [ chr( i ) for i in l ]
         out = [ d[6:10], d[0:2], d[3:5], d[11:13], d[14:16], d[17:19] ]
         return "".join( [ "".join( i ) for i in out ] )
      def uint16( l ):
         # u16 big endian format
         s = b"".join( struct.pack( '>B', x ) for x in l[ : 2 ] )
         return str( struct.unpack( ">H", s )[ 0 ] )
      def hwRev( l ):
         return '%02d.00' % l[ 0 ] # pylint: disable=consider-using-f-string

      # Some type field labels have been changed to stay consistent with Arista's
      # format. The original labels are provided in the comments
      types = {
            # ProductName
            0x21: ( "SKU", ascii ),
            0x22: ( "PartNumber", ascii ),
            0x23: ( "SerialNumber", ascii ),
            # MacBase
            0x24: ( "MacAddrBase", macAddress ),
            # ManufactureDate
            0x25: ( "MfgTime", date ),
            # DeviceVersion
            0x26: ( "HwRev", hwRev ),
            0x27: ( "LabelRevision", ascii ),
            0x28: ( "PlatformName", ascii ),
            0x29: ( "OnieVersion", ascii ),
            0x2a: ( "NumMacs", uint16 ),
            0x2b: ( "Manufacturer", ascii ),
            0x2c: ( "CountryCode", ascii ),
            0x2d: ( "Vendor", ascii ),
            0x2e: ( "DiagVersion", ascii ),
            0x2f: ( "ServiceTag", ascii ),
            }

      while len( buf ) > 0:
         t = buf[ 0 ] # Type
         l = buf[ 1 ] # Length

         if len( buf ) >= 2 + l:
            v = buf[ 2: 2 + l ] # Value
         else:
            v = buf[ 2: ]

         if t in types:
            field, encoding = types[ t ]
            self.preFdl[ field ] = encoding( v )

         buf = buf[ 2 + l: ]

def bytesFromI2cDump( dump ):
   # Converts the output of running i2cdump on the eeprom device into a byte
   # array used for parsing
   lines = dump.split( "\n" )
   del lines[0]
   del lines[-1]
   buf = []
   for line in lines:
      data, _ = line.split( "    " )
      data = data.split( " " )
      del data[0]
      buf.extend( [ int( d, base=16 ) for d in data ] )
   return buf
