#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import requests
import collections
import re

import Tac
import Tracing

t0 = Tracing.trace0
t9 = Tracing.trace9

authUrlRegEx = re.compile(
   r"(?P<protocol>http[s]?)://(?P<address>[a-zA-Z0-9\-\.]+):"
   "(?P<port>[0-9]+).*" )

VALID_STATUS_CODES = frozenset( ( 200, 201, 202, 204 ) )

# Timeout the request if no response was received for 30 seconds
REQUEST_TIMEOUT = 30

class OpenStackException( Exception ):
   ''' Base class for OpenStack exceptions '''

class OpenStackApiException( OpenStackException ):
   ''' Exception raised when the OpenStack Api fails '''

class OpenStackConnectException( OpenStackException ):
   ''' Exception raised if OpenStack servers are not reachable '''

def execWithRetry( method, exception, timeout=60, kwargs=None ):
   ''' The given method is executed and if the execution results in the expected
       exception, then the method is retried until the timeout period is reached '''

   timeInterval = 5
   seenException = None
   timeoutAt = Tac.now() + timeout
   while Tac.now() < timeoutAt:
      try:
         if kwargs:
            return method( **kwargs )
         else:
            return method()
      except exception as e:
         Tac.runActivities( timeInterval )
         t0( "Retrying... ", method )
         t0( e )
         seenException = e
   if seenException:
      # pylint: disable-msg=E0702
      raise seenException

class HttpClient:
   ''' HttpClient provides a GET and POST method for making HTTP requests. '''

   def __init__( self, protocol, serverAddress, port ):
      # The HttpClient is associated with a server when it is created.
      self.protocol = protocol
      self.serverAddress = serverAddress
      self.port = port

   def __makeRequest( self, path, headers=None, params=None, data=None,
                      method=None ):
      '''
      path: The request path
      headers (dict): Additional headers that needs to be passed to OpenStack
      params (dict): The GET parameters that should be appended to the request
      data (dict): The POST data that is sent as part of the request.
      method: Whether the request is a GET request or a POST request.
      '''

      requestHeaders = {}
      requestHeaders[ 'Content-Type' ] = 'application/json'
      requestHeaders[ 'Accept' ] = 'application/json'
      if headers:
         requestHeaders.update( headers )

      # pylint: disable-next=consider-using-f-string
      url = "{}://{}:{}/{}".format( self.protocol, self.serverAddress, self.port,
                                    path )
      response = None
      kwargs = {}

      if params:
         kwargs[ 'params' ] = params
      if data:
         kwargs[ 'data' ] = json.dumps( data )

      kwargs[ 'headers' ] = requestHeaders

      try:
         if method == 'GET':
            response = requests.get( url, verify=False, timeout=REQUEST_TIMEOUT,
                                     **kwargs )
         elif method == 'POST':
            response = requests.post( url, verify=False, timeout=REQUEST_TIMEOUT,
                                      **kwargs )
         else:
            raise OpenStackApiException( "Unsupported method: ", method )
      except requests.exceptions.ConnectionError as e:
         t9( "Socket timed out" )
         t9( e )
         raise e
      except requests.exceptions.Timeout as e:
         t9( "Request timed out" )
         t9( e )
         raise e
      # pylint: disable-msg=w0703
      except Exception as e:
         t9( "Exception!" )
         t9( e )
         raise e

      if response.status_code not in VALID_STATUS_CODES:
         raise OpenStackApiException( response.reason, url, requestHeaders, params,
                                      data )

      return response.json()

   def get( self, url, headers=None, params=None ):
      return self.__makeRequest( url, headers=headers, params=params, method='GET' )

   def post( self, url, headers=None, params=None, data=None ):
      return self.__makeRequest( url, headers=headers, params=params, data=data,
                                 method='POST' )

class OpenStackHttpClient:
   ''' This class uses an HttpClient to send the requests to OpenStack cluster '''
   def __init__( self, protocol, ip, port ):
      self.client = HttpClient( protocol, ip, port )

   def get( self, url, headers=None, params=None ):
      return self.client.get( url, headers=headers, params=params )

   def post( self, url, headers=None, params=None, data=None ):
      return self.client.post( url, headers=headers, params=params, data=data )

   def pagedGet( self, url, headers=None, dataToken="", params=None,
                 pageLimit=10 ):
      '''
      dataToken: Token used to identify the data section
                 If the response does not contain the dataToken,
                 then the KeyError is raised
      pageItems: Number of items retrived in each request
      '''

      reqParams = params or collections.OrderedDict()
      reqParams.update( { 'limit' : pageLimit } )

      data = []
      moreData = True
      while moreData:
         response = self.get( url, headers=headers, params=reqParams )
         # pylint: disable-next=use-implicit-booleaness-not-len
         if not len( response[ dataToken ] ):
            break
         marker = None
         for item in response[ dataToken ]:
            marker = item[ 'id' ]
            data.append( item )
         reqParams.update( { 'marker' : marker } )

      return data

class KeystoneClient( OpenStackHttpClient ):
   '''
   KeyStoneClient provides an interface for OpenStack identity requests
   '''

   def __init__( self, keystoneProtocol='http', keystoneIp="", keystonePort=None,
                 userName=None, password=None, tenantName=None ):
      super().__init__( keystoneProtocol,
                        keystoneIp,
                        keystonePort )

      self.keystoneIp = keystoneIp
      self.keystonePort = keystonePort
      self.userName = userName
      self.password = password
      self.tenantName = tenantName
      self.accessInfo = None

   def __getAccessInfo( self ):
      if self.accessInfo:
         return self.accessInfo

      postData = {}
      auth = {}
      passwordCredentials = {}
      passwordCredentials[ 'username' ] = self.userName
      passwordCredentials[ 'password' ] = self.password
      auth[ 'passwordCredentials' ] = passwordCredentials
      auth[ 'tenantName' ] = self.tenantName
      postData[ 'auth' ] = auth
      url = "/v2.0/tokens"
      self.accessInfo = self.post( url, data=postData )
      return self.accessInfo

   def getAuthToken( self ):
      accessInfo = self.__getAccessInfo()
      authToken = accessInfo[ 'access' ][ 'token' ][ 'id' ]
      return authToken

   def getEndpoint( self, service ):
      accessInfo = self.__getAccessInfo()
      for s in accessInfo[ 'access' ][ 'serviceCatalog' ]:
         if s[ 'name' ] == service:
            # ToDo: Can there be multiple endpoints?
            ep = s[ 'endpoints' ][ 0 ][ 'publicURL' ]
            matched = authUrlRegEx.match( ep )
            if matched:
               # Return the 'ip':'port' tuple
               return ( matched.group( 'protocol' ),
                        matched.group( 'address' ),
                        matched.group( 'port' ) )

      return ( None, None, None )

   # This returns the id of the tenant that was used to LOG INTO keystone.
   # (Be careful about the difference between this and say, a tenant that we want to
   # query information about! They are not necessarily the same.)
   def getLoginTenantId( self ):
      accessInfo = self.__getAccessInfo()
      return accessInfo[ 'access' ][ 'token' ][ 'tenant' ][ 'id' ]

class KeystoneAdminClient( KeystoneClient ):
   '''
   Admin client uses a token from a user+tenant which has an admin role to connect to
   the keystone service.
   '''

   def __init__( self, keystoneProtocol='http', keystoneIp="", keystonePort=None,
                 userName=None, password=None, tenantName=None ):
      super().__init__( keystoneProtocol, keystoneIp,
                        keystonePort, userName, password,
                        tenantName )
      self.authToken = self.getAuthToken()

   def __doListTenants( self ):
      url = "/v2.0/tenants"
      headers = { 'X-Auth-Token' : self.authToken }
      return self.pagedGet( url, headers=headers, dataToken='tenants' )

   def listTenants( self ):
      return execWithRetry( self.__doListTenants,
                            requests.exceptions.ConnectionError )

   # pylint: disable-next=unused-private-member
   def __doGetTenantByName( self, tenantName ):
      url = "/v2.0/tenants"
      headers = { 'X-Auth-Token' : self.authToken }
      params = { 'name' : tenantName }
      data = self.get( url, headers=headers, params=params )
      # pylint: disable-next=unidiomatic-typecheck
      if type( data ) == dict and 'tenant' in data:
         return data[ 'tenant' ]
      else:
         t9( "Data: ", data )
         raise OpenStackApiException( 'KeyError: tenant', url, data )

   def __doGetTenantById( self, tenantId ):
      url = "/v2.0/tenants/%s" % tenantId # pylint: disable=consider-using-f-string
      headers = { 'X-Auth-Token' : self.authToken }
      data = self.get( url, headers=headers )
      # pylint: disable-next=unidiomatic-typecheck
      if type( data ) == dict and 'tenant' in data:
         return data[ 'tenant' ]
      else:
         t9( "Data: ", data )
         raise OpenStackApiException( 'KeyError: tenant', url, data )

   def getTenantById( self, tenantId ):
      kwargs = { 'tenantId' : tenantId }
      return execWithRetry( self.__doGetTenantById,
                            requests.exceptions.ConnectionError,
                            kwargs=kwargs )

class NovaClient( OpenStackHttpClient ):
   '''
   NovaClient is used provides an interface for the VM related OpenStack services
   '''

   def __init__( self, novaProtocol, novaIp, novaPort, authToken, ksTenantId ):
      super().__init__( novaProtocol, novaIp, novaPort )
      self.authToken = authToken

      # Stores the ID of the tenant that was used to authenticate to keystone before
      # acquiring the nova client.
      self.ksTenantId = ksTenantId

   def listTenantServers( self, tenantId ):
      # pylint: disable-next=consider-using-f-string
      url = "/v2/%s/servers/detail" % self.ksTenantId
      # Get 10 items per page.
      params = collections.OrderedDict( [ ( 'all_tenants', 1 ), ( 'limit', 10 ),
                                          ( 'project_id', tenantId ) ] )
      headers = { 'X-Auth-Token' : self.authToken }
      data = self.client.get( url, headers=headers, params=params )
      # pylint: disable-next=unidiomatic-typecheck
      if type( data ) == dict and 'servers' in data:
         return data[ 'servers' ]
      else:
         t9( "Server data: ", data )
         raise OpenStackApiException( 'KeyError: servers', url, data )

   def listServers( self, marker="" ):
      # pylint: disable-next=consider-using-f-string
      url = "/v2/%s/servers/detail" % self.ksTenantId
      # Get 10 items per page.
      params = collections.OrderedDict( [ ( 'all_tenants', 1 ), ( 'limit', 10 ) ] )
      if marker != "":
         params.update( { 'marker' : marker } )

      headers = { 'X-Auth-Token' : self.authToken }
      data = self.client.get( url, headers=headers, params=params )
      # pylint: disable-next=unidiomatic-typecheck
      if type( data ) == dict and 'servers' in data:
         return data[ 'servers' ]
      else:
         t9( "Server data: ", data )
         raise OpenStackApiException( 'KeyError: servers', url, data )

   def serverDetails( self, serverId ):
      # pylint: disable-next=consider-using-f-string
      url = "/v2/%s/servers/detail" % self.ksTenantId
      params = collections.OrderedDict(
         [ ( 'all_tenants', 1 ), ( 'uuid', serverId ) ]
      )

      headers = { 'X-Auth-Token' : self.authToken }
      data = self.client.get( url, headers=headers, params=params )
      # pylint: disable-next=unidiomatic-typecheck
      if type( data ) == dict and 'servers' in data:
         if data[ 'servers' ]:
            return data[ 'servers' ][ 0 ]
         else:
            return None
      else:
         t9( "Server data: ", data )
         raise OpenStackApiException( 'KeyError: servers', url, data )

class OpenStackClientLite:
   '''
   OpenStackClientLite provides a minimal set of OpenStack apis. This class relies
   only on the python-requests library to talk to OpenStack services as opposed to
   the OpenStackClient class which uses python-openstackclients. This is a light
   weight class used by the PollingAgent.
   '''

   def __init__( self, keystoneProtocol, keystoneAddress, keystonePort,
                 osUser, osPass, tenantName ):
      self.keystoneProtocol = keystoneProtocol
      self.keystoneAddress = keystoneAddress
      self.keystonePort = keystonePort
      self.userName = osUser
      self.userPass = osPass
      self.tenantName = tenantName
      self.authToken = None
      self.ksTenantId = None

      # Clients
      self.keystoneClient_ = None
      self.novaClient_ = None

   def __keystoneClient( self ):
      if not self.keystoneClient_:
         self.keystoneClient_ = KeystoneClient(
               keystoneProtocol=self.keystoneProtocol,
               keystoneIp=self.keystoneAddress,
               keystonePort=self.keystonePort,
               userName=self.userName,
               password=self.userPass,
               tenantName=self.tenantName )
      return self.keystoneClient_

   def getEndpoint( self, service ):
      ksClient = self.__keystoneClient()
      return ksClient.getEndpoint( service )

   def getAuthInfo( self ):
      # Do lazy auth to speed things up.
      if not ( self.authToken or self.ksTenantId ):
         ksClient = self.__keystoneClient()
         self.authToken = ksClient.getAuthToken()
         self.ksTenantId = ksClient.getLoginTenantId()

      return self.authToken, self.ksTenantId

   def __novaClient( self ):
      if not self.novaClient_:
         protocol, ip, port = self.getEndpoint( 'nova' )
         authToken, ksTenantId = self.getAuthInfo()
         self.novaClient_ = NovaClient( protocol, ip, port, authToken, ksTenantId )

      return self.novaClient_

   def __doListTenantServers( self, tenantId ):
      novaClient = self.__novaClient()
      servers = novaClient.listTenantServers( tenantId )
      return servers

   def listTenantServers( self, tenantId ):
      kwargs = { 'tenantId' : tenantId }
      return execWithRetry( self.__doListTenantServers,
                            requests.exceptions.ConnectionError,
                            kwargs=kwargs )

   def __doListServers( self, marker="" ):
      novaClient = self.__novaClient()
      servers = novaClient.listServers( marker=marker )
      return servers

   def listServers( self, marker="" ):
      kwargs = { 'marker' : marker }
      return execWithRetry( self.__doListServers,
                            requests.exceptions.ConnectionError,
                            kwargs=kwargs )

   def __doServerDetails( self, serverId=None ):
      novaClient = self.__novaClient()
      server = novaClient.serverDetails( serverId )
      return server

   def serverDetails( self, serverId="" ):
      kwargs = { 'serverId' : serverId }
      return execWithRetry( self.__doServerDetails,
                            requests.exceptions.ConnectionError,
                            kwargs=kwargs )
