#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SmbusUtil
import PLSmbusUtil
import Tac

def agentIsRunning( agentName="PlutoSmbus" ):
   try:
      Tac.run( [ "pgrep", agentName ], stdout=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return False
   return True

#
# Convenient wrapper class around PLSmbusUtil
#
# PLSmbusClient *indirectly* read/write from/to PCI by sending socket
# requests to PlutoSmbus daemon.
#
# In other words, it's a *client* of PlutoSmbus daemon.
#
# This class requires PlutoSmbus daemon running.
#
# If PlutoSmbus daemon is not running, PLSmbusClient's read/write requests will
# fail.
#

# pylint: disable-msg=W0223
class PLSmbusClient( SmbusUtil.BaseSmbusHelper ):
   def __init__( self, accelId, busId, deviceId, addrSize, pciAddress, backend,
                 readDelay=0, writeDelay=0, extraDelay=0 ):
      SmbusUtil.BaseSmbusHelper.__init__( self )
      assert accelId >= 0
      assert 0 <= busId <= 0xf
      self.accelId = accelId
      self.busId = busId
      self.deviceId = deviceId
      self.addrSize = addrSize
      self.pciAddress = pciAddress
      self.backend = backend
      self.readDelay = readDelay
      self.writeDelay = writeDelay
      self.extraDelay = extraDelay
      self.sock = None
      self.pci = None

      self.connect()

   def connect( self ):
      if self.pciAddress and agentIsRunning() and not self.sock:
         self.pci = PLSmbusUtil.encodePCIAddress( self.pciAddress )
         self.sock = PLSmbusUtil.connect()

   def read( self, address, count, accessSize, raw=False, pec=False,
             readCurrent=False ):
      self.connect()
      data = PLSmbusUtil.read( self.sock, self.pci, self.accelId, self.busId,
                                 self.deviceId, address, count=count,
                                 readCurrent=readCurrent,
                                 backend=self.backend,
                                 delay=self.readDelay,
                                 extraDelay=self.extraDelay )

      # PLSmbusUtil uses bytes but SmbusUtil requires a list of ints. This
      # conversion can be removed if SmbusUtil is updated to use bytes natively.
      # BUG756158
      if raw:
         return data
      return list( data )

   def write( self, address, data, accessSize=1, stride=0, pec=False ):
      self.connect()
      # PLSmbusUtil uses bytes but SmbusUtil requires a list of ints. This
      # conversion can be removed if SmbusUtil is updated to use bytes natively.
      # BUG756158
      dataStr = b''.join( data )
      PLSmbusUtil.write( self.sock, self.pci, self.accelId, self.busId,
                         self.deviceId, address, dataStr, backend=self.backend,
                         delay=self.writeDelay,
                         extraDelay=self.extraDelay )

#
# Convenient wrapper function to construct raw PCI helper using SmbusUtil,
# with reasonable default values for most Pluto systems.
#
# PLSmbusRawPciHelper *directly* reads/writes to PCI.
#
# Thus, this class requires PlutoSmbus NOT running.
#
# If PlutoSmbus daemon is running, PLSmbusRawPciHelper's read/write requests will
# fail.
#
def PLSmbusRawPciHelper( accelId, busId, deviceId, addrSize,
                         readDelayMs='delay50ms', writeDelayMs='delay50ms',
                         busTimeout='busTimeout1000ms',
                         writeNoStopReadCurrent=False, smbusAddrSpacing=0x80,
                         smbusAgentId=None, pciAddress=None,
                         extraDelay=0 ):
   bug30005ExtraDelays = extraDelay > 0
   factory = SmbusUtil.Factory()
   device = factory.device( accelId, busId, deviceId, addrSize,
                            readDelayMs=readDelayMs, writeDelayMs=writeDelayMs,
                            busTimeout=busTimeout,
                            writeNoStopReadCurrent=writeNoStopReadCurrent,
                            smbusAddrSpacing=smbusAddrSpacing,
                            smbusAgentId=smbusAgentId,
                            pciAddress=pciAddress,
                            bug30005ExtraDelays=bug30005ExtraDelays )
   return device

#
# Convenient wrapper class around both classes (PLSmbusClient & PLSmbusRawPciHelper)
# above.
#
# PLSmbusPciDevice can both *directly* or *indirectly* reads/writes to PCI:
#
# 1. If PlutoSmbus is running, PLSmbusPciDevice sends PCI read/write requests to
# PlutoSmbus daemon for the daemon to proceed.
#
# 2. If PlutoSmbus is NOT running, PLSmbusPciDevice directly reads/writes to PCI
# using raw PCI helper functions defined in SmbusUtil.
#
# 3. If PlutoSmbus daemon changes its state (from "running" to "NOT running" and
# vice versa), PLSmbusPciDevice automatically adjusts its actions between the 2
# ways listed above accordingly.
#
# This class can be used in scenarios where the code must be adaptive to
# PlutoSmbus's running or NOT running status. If PlutoSmbus's status is expected
# to persist, it's probably better to use PLSmbusClient or PLSmbusRawPciHelper
# classes above.
#
class PLSmbusPciDevice( PLSmbusClient ):
   def __init__( self, accelId, busId, deviceId, addrSize, pciAddress, backend ):
      super().__init__( accelId, busId, deviceId,
                        addrSize, pciAddress, backend )
      self.rawPciHelper = PLSmbusRawPciHelper( accelId, busId, deviceId,
                                               addrSize, pciAddress=pciAddress )

   def read( self, address, count, accessSize, raw=False, pec=False,
             readCurrent=False ):
      if agentIsRunning():
         try:
            return super().read( address, count, accessSize,
                                 raw, pec, readCurrent )
         except OSError:
            pass
      # writeNoStopReadCurrent is set in the constructor for the raw helper,
      # so passing it here is not supported. BUG972526 tracks adding support.
      assert not readCurrent, 'readCurrent not supported with raw helper read'
      return self.rawPciHelper.read( address, count, accessSize, raw, pec )

   def write( self, address, data, accessSize=1, stride=0, pec=False ):
      if agentIsRunning():
         try:
            super().write( address, data, accessSize,
                           stride, pec )
            return
         except OSError:
            pass
      self.rawPciHelper.write( address, data, accessSize, stride, pec )
