# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import os
import json
import shutil
import errno
import Tracing
import SecretCli
import Tac
from BothTrace import traceX as bt
from AaaPluginLib import TR_ERROR

traceHandle = Tracing.Handle( 'PasswordPolicyLib' )
error = traceHandle.trace0
debug = traceHandle.trace8

AaaDirectoryPath = "/persist/secure/aaa/"
PasswordHistoryFileName = "password_history"
EnablePasswordHistoryPath = "/persist/secure/aaa/enable_password_history"

def verifyPassword( encrypted, unencrypted ):
   return SecretCli.encrypt( unencrypted, encrypted ) == encrypted

def applyMinChangedCharactersPolicy( mode, username, policy, secretValue,
                                     currEncryptedPasswd ):
   # Check if minimum characters to change policy is applied
   if ( policy and policy.minChanged and secretValue and secretValue.newPass ):
      # Only apply policy if a pwd different from the existing one is entered
      if ( currEncryptedPasswd and not
           verifyPassword( currEncryptedPasswd, secretValue.newPass ) ):
         import getpass # pylint: disable=import-outside-toplevel
         currentPassword = getpass.getpass( f"Enter password for {username}:" )
         if not verifyPassword( currEncryptedPasswd, currentPassword ):
            mode.addError( "Entered password does not match existing password "
                           f"for {username}" )
            return False
         secretValue.validateMinChangedCharacters( currentPassword,
                                                   policy.minChanged )
         currentPassword = None
   return True

def applyDenyUsernamePolicy( username, policy, secretValue ):
   # Check if deny username policy is applied
   if ( policy and policy.denyUsername and secretValue and secretValue.secretHash ):
      secretValue.validateDenyUsername( username )

def getInitPasswordHistory():
   initPasswordHistory = {
      'version': '1.0.0',
      'username': '',
      'passwords': []
   }
   return initPasswordHistory

def writeJsonFile( path, jsonObj ):
   with open( path, 'w' ) as file:
      json.dump(jsonObj, file )

def loadJsonFile( path ):
   obj = {}
   if os.path.exists( path ):
      with open( path, 'r' ) as file:
         obj = json.load( file )
   return obj

def changePermissionEosadmin( path ):
   Tac.run( [ "chgrp", "eosadmin", path ], asRoot=True )
   Tac.run( [ "chmod", "775", path ], asRoot=True )

def createJsonFile( jsonObj, path ):
   Tac.run( [ "touch", path ], asRoot=True )
   changePermissionEosadmin( path )
   writeJsonFile( path, jsonObj )

def createDir( path ):
   # need to make sure the parent dir is exist
   try:
      Tac.run( [ "mkdir", path ], asRoot=True)
   except OSError as creatDirErr:
      if creatDirErr.errno == errno.EEXIST:
         debug( "Directory already exists: ", path )
      else:
         error( "Cannot create directory: ", path, "errno: ", creatDirErr.strerror )
         raise
   changePermissionEosadmin( path )

def doTrimPasswordHistoryFile( passwordHistoryPath, trimTarget ):
   try:
      passwordHistory = {}
      if os.path.exists( passwordHistoryPath ):      
         passwordHistory = loadJsonFile( passwordHistoryPath )
         passwordQueue = passwordHistory[ 'passwords' ]   
         if len( passwordQueue ) > trimTarget:
            debug( f"trim {passwordHistoryPath} history size to {trimTarget}")
            del passwordQueue[ :len( passwordQueue )-trimTarget ]
         writeJsonFile( passwordHistoryPath, passwordHistory )
      debug( f"Done trim {passwordHistoryPath}" )
   except OSError:
      bt( TR_ERROR, f"Failed to trim {passwordHistoryPath}" )

def trimPasswordHistory( trimTarget, accounts ):
   debug( f"Start trim with trimTarget:{trimTarget}, accounts:{accounts}" )
   # trim enable_password_history
   doTrimPasswordHistoryFile( EnablePasswordHistoryPath, trimTarget )
   for username in accounts:
      debug( f"Start trim username {username}" )
      passwordHistoryPath = os.path.join( AaaDirectoryPath, username, 
                                          PasswordHistoryFileName )
      doTrimPasswordHistoryFile( passwordHistoryPath, trimTarget )

def clearNonExistPasswordHistory( accounts ):
   # get all directories for password histories under aaa directory
   if os.path.exists( AaaDirectoryPath ):
      orphanDir = set(os.listdir(AaaDirectoryPath))-set(accounts)
      enablePasswordFile = "enable_password_history"
      if enablePasswordFile in orphanDir:
         debug( "Do not clear enable password history file")
         orphanDir.remove( enablePasswordFile )

      # remaining usernames in orphanDir is not in LocalUser.acct, remove the 
      # corresponding directory
      for name in orphanDir:
         debug( f"Removing password history for username:{name}")
         clearPasswordHistory( name )

def updatePasswordHistory( policy, passwordHistoryPath, secretValue ):
   # update password history with new password
   passwordHistory = loadJsonFile( passwordHistoryPath )

   clearTextPasswd = secretValue.newPass
   encryptedPasswd = SecretCli.encrypt( clearTextPasswd, hashAlgorithm="sha512" )

   passwordQueue = passwordHistory[ 'passwords' ]
   queueSize = policy.denyLastPasswd

   if not queueSize:
      return
   if len( passwordQueue ) == queueSize:
      # the current history queue is full
      passwordQueue.pop( 0 )
      passwordQueue.append( encryptedPasswd )
   else:
      passwordQueue.append( encryptedPasswd )

   writeJsonFile( passwordHistoryPath, passwordHistory )


def clearPasswordHistory( username ):
   passwordHistoryDirPath = os.path.join( AaaDirectoryPath, username )
   if os.path.exists( passwordHistoryDirPath ):
      shutil.rmtree( passwordHistoryDirPath )

def clearEnablePasswordHistory():
   debug( "Enter clearEnablePasswordHistory()")
   if os.path.exists( EnablePasswordHistoryPath ):
      try:
         debug( "Clearing enable_password_history" )
         os.remove( EnablePasswordHistoryPath )
      except OSError:
         error( "Failed to clear enable_password_history")
         

def applyDenyLastPolicy( mode, username, policy, secretValue, enable=False ):
   # check newPass as this policy only apply for passwords in clear-text
   # keep updating password history if previous_count > 0
   if secretValue and secretValue.newPass and policy and policy.denyLastPasswd:
      # create password history if there is not one
      if not enable:
         passwdHistoryDirPath = os.path.join( AaaDirectoryPath, username )
         passwdHistoryFilePath = os.path.join( passwdHistoryDirPath,
                                               PasswordHistoryFileName )
      else:
         passwdHistoryDirPath = AaaDirectoryPath
         passwdHistoryFilePath = EnablePasswordHistoryPath
         username = ""
      if not os.path.exists( passwdHistoryFilePath ):
         if not os.path.exists( AaaDirectoryPath ):
            try:
               createDir( AaaDirectoryPath )
               debug( f"Created dir {AaaDirectoryPath}" )
            except OSError:
               mode.addError( f"Failed to create directory: {AaaDirectoryPath}" )
               return
         if not os.path.exists( passwdHistoryDirPath ):
            try:
               createDir( passwdHistoryDirPath )
               debug( f"Created dir {passwdHistoryDirPath}" )
            except OSError:
               mode.addError( f"Failed to create directory: {passwdHistoryDirPath}" )
               return
         passwordHistory = getInitPasswordHistory()
         passwordHistory[ 'username' ] = username
         createJsonFile( passwordHistory, passwdHistoryFilePath )
         debug( f"Created JSON file {passwdHistoryFilePath}")
      else:
         passwordHistory = loadJsonFile( passwdHistoryFilePath )

      # validate password history only when deny last applied
      if not secretValue.validateDenyLastPasswd( passwordHistory[ 'passwords' ] ):
         return
      
      # Update password history with the new password
      updatePasswordHistory( policy, passwdHistoryFilePath, secretValue )
      
class DenyLastReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::PasswordPolicy"

   def __init__( self, policy, agent, name ):
      self.agent_ = agent
      Tac.Notifiee.__init__( self, policy )
      self.policy_ = policy
      self.name_ = name
      self.handleDenyLast()
   
   @Tac.handler( 'denyLastPasswd' )
   def handleDenyLast( self ):
      debug( f"DenyLastReactor triggered for {self.name_}" )
      # the policy the user modifying is the currently applied policy
      trimPasswordHistory( self.policy_.denyLastPasswd,
                           self.agent_.localUserConfig.acct )
      

class CurrentPolicyReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::Config"

   def __init__( self, entity, agent, policyName ):
      self.agent_ = agent
      Tac.Notifiee.__init__( self, entity )
      self.policyName_ = policyName
      self.denyLastReactor_ = None

      self.handlePasswordPolicies( self.policyName_ )

   @Tac.handler( "passwordPolicies" )
   def handlePasswordPolicies( self, name ):
      if name != self.policyName_:
         debug( f"Configuring policy f{name}, not currently applied policy, return")
         return
      passwordPolicies = self.agent_.mgmtSecConfig.passwordPolicies      
      policy = passwordPolicies.get( self.policyName_ )
      if policy:
         self.denyLastReactor_ = DenyLastReactor( policy, self.agent_,
                                                     self.policyName_ )
      else:
         if self.denyLastReactor_:
            debug( "Current policy not exists, close reactor" )
            self.denyLastReactor_.close()
            self.denyLastReactor_ = None
         debug( "Current policy not exists, trim password to 0" )
         trimPasswordHistory( 0, self.agent_.localUserConfig.acct )
   
   def close( self ):
      if self.denyLastReactor_:
         self.denyLastReactor_.close()
      Tac.Notifiee.close( self )
   
class ApplyPolicyReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Account::Config"

   def __init__( self, entity, agent ):
      self.agent_ = agent
      Tac.Notifiee.__init__( self, entity )
      self.currentPolicyReactor_ = None
      # check and update trim status file after reload agent
      self.handleApplyPolicy()
      clearNonExistPasswordHistory( self.agent_.localUserConfig.acct )
      
   @Tac.handler( 'passwordPolicy' )
   def handleApplyPolicy( self ):  
      currentPolicyName = self.agent_.mgmtAcctConfig.passwordPolicy
      if self.currentPolicyReactor_:
         debug( "Delete reactor for previous policy" )
         self.currentPolicyReactor_.close()
         self.currentPolicyReactor_ = None

      if currentPolicyName:  
         debug( f"Create reactor for {currentPolicyName}" )
         self.currentPolicyReactor_ = \
                                    CurrentPolicyReactor( self.agent_.mgmtSecConfig,
                                                          self.agent_, 
                                                          currentPolicyName )
      else:
         debug( "unapply policy, trim password history to 0" )
         trimPasswordHistory( 0, self.agent_.localUserConfig.acct )

