# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Tac

picassoConfigSysdbPath = "picasso/config"

tacPicassoMode = Tac.Type( "Picasso::PicassoMode" )
tacChunkMode = Tac.Type( "Picasso::ChunkMode" )
tacChunkType = Tac.Type( "Picasso::ChunkType" )
tacChunkRegionAccess = Tac.Type( "Picasso::ChunkRegionAccess" )
tacDmamemMode = Tac.Type( "Picasso::DmamemMode" )
tacReallocMode = Tac.Type( "Picasso::ReallocDataMode" )

"""
Predefined PicassoMode values.
"""
PICASSO_MODE_READONLY = tacPicassoMode.picassoModeReadOnly
PICASSO_MODE_READWRITE = tacPicassoMode.picassoModeReadWrite

"""
Predefined chunkMode values - for now, they're just aliases for the TAC values.
It might, however, change in the future
"""
CHUNK_MODE_READONLY = tacChunkMode.chunkModeReadOnly
CHUNK_MODE_READWRITE = tacChunkMode.chunkModeReadWrite

"""
Predefined ChunkRegionAccess values
"""
CHUNK_REGION_ACCESS_READONLY = tacChunkRegionAccess.chunkRegionAccessReadOnly
CHUNK_REGION_ACCESS_READWRITE = tacChunkRegionAccess.chunkRegionAccessReadWrite

"""
Predefined chunkType values - for now, they're just aliases for the TAC values.
It might, however, change in the future
"""
CHUNK_TYPE_NORMAL = tacChunkType.chunkTypeNormal
CHUNK_TYPE_SHARED = tacChunkType.chunkTypeShared

"""
Predefined DmamemMode values
"""
DMAMEM_MODE_32BIT = tacDmamemMode.dmamemMode32Bit
DMAMEM_MODE_64BIT = tacDmamemMode.dmamemMode64Bit

"""
Predefined ReallocMode values.
"""
REALLOC_DATA_MODE_PRESERVE = tacReallocMode.reallocDataModePreserve
REALLOC_DATA_MODE_NO_PRESERVE = tacReallocMode.reallocDataModeNoPreserve
REALLOC_DATA_MODE_PRESERVE_SAME_SIZE = \
      tacReallocMode.reallocDataModePreserveSameSize

class PicassoManagerBase:
   """ Base class for Python PicassoManager objects - not to be used directly

   Attributes
   ----------
   manager : object
      TAC Picasso::PicassoManagerBase object instance
   name : str
      Name of the Picasso instance
   mode : PicassoMode
      Determines the operating mode of Picasso:
      - PICASSO_MODE_READONLY - Only pre-existing chunks can be accessed,
        audits and memory replication are not in use.
        Minimal amount of read-only Sysdb mounts.
      - PICASSO_MODE_READWRITE - full functionality
   """

   def __init__( self, name, mode = PICASSO_MODE_READWRITE ):
      """Object constructor

      Parameters
      ----------
      name : str
         PicassoManager name
      mode : PicassoMode
         Determines the operating mode of Picasso:
         - PICASSO_MODE_READONLY - Only pre-existing chunks can be accessed,
           audits and memory replication are not in use.
           Minimal amount of read-only Sysdb mounts.
         - PICASSO_MODE_READWRITE - full functionality
      """
      self.manager = None
      self.name = name
      self.mode = mode

   def _initManager( self, audit, auditRequestDir, activeStatus,
                     standbyStatus, activeChunkDir, standbyChunkDir, transport ):
      """Creates and initializes the internal TAC PicassoManager object"""
      manager = Tac.newInstance( "Picasso::PicassoManagerBase", self.name )
      manager.mode = self.mode
      if self.mode == PICASSO_MODE_READWRITE:
         manager.audit = audit
         manager.auditRequestDir = auditRequestDir
         manager.activeChunkDir = activeChunkDir
         manager.standbyChunkDirView = standbyChunkDir
         manager.transport = transport
      manager.activeChunkDirView = activeChunkDir
      manager.activeStatus = activeStatus
      manager.standbyStatus = standbyStatus
      manager.initialized = True
      self.manager = manager

   def createChunk( self, chunkName, size, chunkMode, dmamemMode ):
      """Creates a Picasso memory chunk

      Wrapper function for TAC Picasso::PicassoManagerBase::createChunk()

      Parameters
      ----------
      chunkName : str
      size : int
         Size of the chunk in bytes
      chunkMode : Tac.Type( "Picasso::ChunkMode" )
         Can be set to one of the two predefined values:
         CHUNK_MODE_READONLY
         CHUNK_MODE_READWRITE
         Note: if Picasso is in the read-only mode, only read-only chunks can be
         requested!
      dmamemMode : Tac.Type( "Picasso:DmamemMode" )
         Dmamem mode to use for the chunk allocation:
         DMAMEM_MODE_32BIT
         DMAMEM_MODE_64BIT
      """
      return self.manager.createChunk( chunkName, size, chunkMode, dmamemMode )

   def createSharedChunk( self, chunkName, size, sharedAuditRequest,
                          chunkMode, dmamemMode ):
      """Creates a shared Picasso chunk

      Wrapper function for TAC Picasso::PicassoManagerBase::createSharedChunk()

      Parameters
      ----------
      chunkName : str
      size : int
         Size of the chunk in bytes
      sharedAuditRequest : Tac.Type( "Picasso::AuditRequest" )
      chunkMode : Tac.Type( "Picasso::ChunkMode" )
         Can be set to one of the two predefined values:
         CHUNK_MODE_READONLY
         CHUNK_MODE_READWRITE
         Note: if Picasso is in the read-only mode, only read-only chunks can be
         requested!
      dmamemMode : Tac.Type( "Picasso:DmamemMode" )
         Dmamem mode to use for the chunk allocation:
         DMAMEM_MODE_32BIT
         DMAMEM_MODE_64BIT
      """
      return self.manager.createSharedChunk( chunkName, size, sharedAuditRequest,
            chunkMode, dmamemMode )

   def reallocChunk( self, chunkName, size, reallocDataMode, dmamemMode ):
      """Reallocate a Picasso memory chunk

      Wrapper function for TAC Picasso::PicassoManagerBase::reallocChunk()

      Parameters
      ----------
      chunkName : str
      size : int
         Size of the chunk in bytes
      reallocDataMode : Tac.Type( "Picasso::ReallocDataMode" )
         Can be set to one of the three predefined values:
         REALLOC_DATA_MODE_PRESERVE - data from the existing chunk will be copied to
                                      the new one
         REALLOC_DATA_MODE_NO_PRESERVE - new zeroed memory chunk will me created
         REALLOC_DATA_MODE_PRESERVE_SAME_SIZE - data will be copied only from the
                                                existing chunk of the same size
      dmamemMode : Tac.Type( "Picasso:DmamemMode" )
         Dmamem mode to use for the chunk allocation:
         DMAMEM_MODE_32BIT
         DMAMEM_MODE_64BIT
      """
      return self.manager.reallocChunk( chunkName, size, reallocDataMode,
                                        dmamemMode )

   def freeChunk( self, chunkName ):
      """Frees the specified chunk

      Releases the chunk (if it's in use) and frees the underlying dmamem memory
      Any PicassoChunk object that are still in use are deactivated and will
      no longer be in "ready" state.
      Care must be taken not to use addresses that were previously obtained from
      PicassoChunk::virtAddr() and PicassoChunk::dmaAddr() as they will be
      no longer valid.

      This function may be used for reallocating chunks with a different size.
      If the chunk does not exist this function does nothing.

      NOTE: This function works only when the dynamic allocation is enabled.

      Parameters
      ----------
      chunkName : str
      """
      self.manager.freeChunk( chunkName )

   def getChunkSize( self, chunkName ):
      """Returns size of the specified chunk
      Returns the chunk's size or zero if it doesn't exists.
      The chunk doesn't have to created/opened first - one can use this method
      to check for chunk's existence and its size ie. to see if a reallocation
      is required prior to use.

      Parameters
      ----------
      chunkName : str
      """
      return self.manager.getChunkSize( chunkName )

   def releaseChunk( self, chunkName ):
      """Releases a specified chunk

      This function releases the chunk by deactivating and removing its state
      machines and unmapping  the underlying dmamem chunk, leaving any PicassoChunk
      objects that are still in use inactive.
      Note that the dmamem chunk is not freed - memory contents are preserved
      and the chunk may be re-created (re-opened) later.

      Parameters
      ----------
      chunkName : str
      """
      self.manager.releaseChunk( chunkName )

   def releaseAllChunks( self ):
      """Releases all allocated chunks

      Releases all chunks managed by this instance of PicassoManager.
      See releaseChunk() for more details.
      """
      self.manager.releaseAllChunks()

   @property
   def initialized( self ):
      """Returns true if the underlying TAC PicassoManager object is initialized"""
      return ( self.manager is not None ) and self.manager.initialized

class PicassoManager( PicassoManagerBase ):
   """PicassoManager Python wrapper

   This object provides a way to instantiate a Picasso::PicassoManagerBase
   object using Python Sysdb APIs.

   Attributes
   ----------
   auditsEnabled : bool
      Flag indicating if memory auditing is enabled and thus if Picasso managed
      memory is writeable
   manager : object
      TAC Picasso::PicassoManagerBase object instance
   name : str
      Name of the Picasso instance

   """

   def __init__( self, name, mode = PICASSO_MODE_READWRITE ):
      """PicassoManager constructor.

      Parameters
      ----------
      name : str
         Name of the Picasso instance.
      mode : PicassoMode
         Determines the operating mode of Picasso:
         - PICASSO_MODE_READONLY - Only pre-existing chunks can be accessed,
           audits and memory replication are not in use.
           Minimal amount of read-only Sysdb mounts.
         - PICASSO_MODE_READWRITE - full functionality
      """
      super().__init__( name, mode )
      self.transport_ = None
      # Picasso Sysdb objects
      self.audit_ = None
      self.auditRequestDir_ = None
      self.activeStatus_ = None
      self.standbyStatus_ = None
      self.activeChunkDir_ = None
      self.standbyChunkDir_ = None

   def mountObjects( self, mg ):
      """Mounts Sysdb objects used by Picasso.

      Parameters
      ----------
      mg : MountGroup
         MountGroup object used to mount the Sysdb entities.
      """
      def getLocalPath( name ):
         return f"picasso/{name}/cell/{Cell.cellId()}"

      def getPeerPath( name ):
         return f"picasso/{name}/cell/{Cell.peerCellId()}"

      # Mount audit and chunk entities depending on the mode
      if self.mode == PICASSO_MODE_READWRITE:
         self.audit_ = mg.mount( getPeerPath( "audit" ), "Picasso::Audit", "r" )
         self.auditRequestDir_ = mg.mount( getLocalPath( "auditrequest" ),
                                           "Tac::Dir", "wi" )
         self.activeChunkDir_ = mg.mount( getLocalPath( "chunk" ),
                                          "Tac::Dir", "wi" )
         self.standbyChunkDir_ = mg.mount( getPeerPath( "chunk" ),
                                           "Tac::Dir", "ri" )
      else:
         self.activeChunkDir_ = mg.mount( getLocalPath( "chunk" ), "Tac::Dir", "ri" )

      self.activeStatus_ = mg.mount( getLocalPath( "status" ),
                                     "Picasso::Status", "r" )
      self.standbyStatus_ = mg.mount( getPeerPath( "status" ),
                                      "Picasso::Status", "r" )
      transportProvider = Tac.newInstance( "Picasso::MemoryTransportProvider" )
      self.transport_ = transportProvider.createTransport(
            "supToSupTransportTypeNone" )

   def initialize( self ):
      """Finishes the Picasso initialization.

      Must be called after the mountObjects() and after all of the Picasso mounts
      are completed (when their EnitityFuture objects morph to Entity)
      - usually after a call to MountGroup.close()
      After the initalization is done, PicassoManager TAC object may be accessed
      via the "manager" field.
      """
      self._initManager( self.audit_, self.auditRequestDir_,
                         self.activeStatus_, self.standbyStatus_,
                         self.activeChunkDir_, self.standbyChunkDir_,
                         self.transport_ )

class PicassoManagerMock( PicassoManagerBase ):
   """ Mock PicassoManager object which allows tests to provide their own
   Sysdb entities

   Attributes
   ----------
   manager : Picasso::PicassoManagerBase
      PicassoManagerBase TAC object
   name : str
      Name of the Picasso instance

   """

   def __init__( self, name, audit, auditRequestDir, activeStatus,
                 standbyStatus, activeChunkDir, standbyChunkDir, transport,
                 mode=PICASSO_MODE_READWRITE ):
      """ PicassoManagerMock constructor

      Parameters
      ----------
      name : str
         Name of the Picasso instance
      audit : Entity
         Picasso::Audit instance. May be None if the audits are disabled and Picasso
         operates in read-only mode.
      auditRequestDir : Tac::Dir
         Tac::Dir used for storing audit requests
      activeStatus : Entity
         Active SUP Picasso::ActiveStatus instance
      standbyStatus : Entity
         Standby SUP Picasso::ActiveStatus instance
      activeChunkDir : Tac::Dir
         Active SUP chunk directory (must be writeable in r/w mode)
      standbyChunkDir : Tac::Dir
         Read-only standby SUP chunk directory
      transport : Entity
         Picasso::MemoryTransport instance
      mode : PicassoMode
         Determines the operating mode of Picasso:
         - PICASSO_MODE_READONLY - Only pre-existing chunks can be accessed,
           audits and memory replication are not in use.
           Minimal amount of read-only Sysdb mounts.
         - PICASSO_MODE_READWRITE - full functionality
      """
      super().__init__( name, mode )
      self._initManager( audit, auditRequestDir, activeStatus,
                         standbyStatus, activeChunkDir, standbyChunkDir, transport )
